/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/walla.js — Rol: Captura de credenciales Wallapop + export/import de anuncios.
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

import { stGet } from './storage.js';
import { swRetryWithBackoff } from './utils.js';
import { apiFetch, withDeviceHeaders } from './api_client.js';

// ✅ v10.5.115: Flag DEBUG controlado por debugMode en storage
let DEBUG_MODE = false;
chrome.storage.local.get(['debugMode'], (r) => { DEBUG_MODE = !!r.debugMode; });
chrome.storage.onChanged.addListener((changes) => {
  if (changes.debugMode) DEBUG_MODE = !!changes.debugMode.newValue;
});
const dlog = (...args) => DEBUG_MODE && console.log(...args);
const dwarn = (...args) => DEBUG_MODE && console.warn(...args);

export const UPLOAD_URL = 'https://es.wallapop.com/app/catalog/upload';

const WCREDS_KEY = 'walla.creds';
let wallaCreds = { token: null, deviceId: null, appVersion: null, deviceOs: null, ts: 0 };

// ============================================================================
// ✅ v10.5.60: Decodificar JWT para obtener edad REAL del token
// ============================================================================

/**
 * Decodifica un JWT y extrae el payload.
 * @param {string} token - Token JWT
 * @returns {object|null} - Payload decodificado o null si falla
 */
function decodeJWT(token) {
  if (!token) return null;
  try {
    const parts = token.split('.');
    if (parts.length !== 3) return null;
    // El payload es la segunda parte, codificada en base64url
    const payload = parts[1];
    // Convertir base64url a base64 estándar
    const base64 = payload.replace(/-/g, '+').replace(/_/g, '/');
    // Decodificar
    const decoded = atob(base64);
    return JSON.parse(decoded);
  } catch (e) {
    console.error('[TOKEN] Error decodificando JWT:', e);
    return null;
  }
}

/**
 * Obtiene la edad REAL del token desde el campo 'iat' del JWT.
 * Esto es diferente de wallaCreds.ts que indica cuándo lo CAPTURAMOS.
 */
export function getTokenRealAge() {
  if (!wallaCreds.token) return { ms: null, formatted: 'sin token', isExpired: true };
  
  const payload = decodeJWT(wallaCreds.token);
  if (!payload) return { ms: null, formatted: 'no decodificable', isExpired: true };
  
  const now = Math.floor(Date.now() / 1000); // Tiempo actual en segundos
  
  // iat = issued at (cuándo se emitió)
  // exp = expiration (cuándo expira)
  const iat = payload.iat || null;
  const exp = payload.exp || null;
  
  if (!iat) return { ms: null, formatted: 'sin iat', isExpired: exp ? now > exp : true };
  
  const ageSeconds = now - iat;
  const ageMs = ageSeconds * 1000;
  const minutes = Math.floor(ageSeconds / 60);
  const formatted = minutes > 0 ? `${minutes}m ${ageSeconds % 60}s` : `${ageSeconds}s`;
  
  // Calcular tiempo restante si hay exp
  let remainingSeconds = null;
  let remainingFormatted = 'sin exp';
  let isExpired = false;
  
  if (exp) {
    remainingSeconds = exp - now;
    isExpired = remainingSeconds <= 0;
    if (remainingSeconds > 0) {
      const remMin = Math.floor(remainingSeconds / 60);
      remainingFormatted = remMin > 0 ? `${remMin}m ${remainingSeconds % 60}s` : `${remainingSeconds}s`;
    } else {
      remainingFormatted = 'EXPIRADO';
    }
  }
  
  return {
    ms: ageMs,
    seconds: ageSeconds,
    minutes,
    formatted,
    iat: iat ? new Date(iat * 1000).toLocaleTimeString() : null,
    exp: exp ? new Date(exp * 1000).toLocaleTimeString() : null,
    remainingSeconds,
    remainingFormatted,
    isExpired
  };
}

// ✅ v10.5.46: Función para obtener edad del token en formato legible (desde CAPTURA)
export function getTokenAge() {
  if (!wallaCreds.ts) return { ms: null, formatted: 'sin timestamp' };
  const ms = Date.now() - wallaCreds.ts;
  const seconds = Math.floor(ms / 1000);
  const minutes = Math.floor(seconds / 60);
  const formatted = minutes > 0 ? `${minutes}m ${seconds % 60}s` : `${seconds}s`;
  return { ms, seconds, minutes, formatted };
}

// ✅ v10.5.60: Obtener info de diagnóstico del token (incluye edad REAL)
export function getTokenDiagnostics() {
  const captureAge = getTokenAge();
  const realAge = getTokenRealAge();
  return {
    hasToken: !!wallaCreds.token,
    hasDeviceId: !!wallaCreds.deviceId,
    // Edad desde captura (lo que teníamos antes)
    tokenAge: captureAge,
    // ✅ v10.5.60: Edad REAL desde JWT
    realAge: realAge,
    tokenFirst6: wallaCreds.token ? wallaCreds.token.substring(0, 6) + '...' : null,
    capturedAt: wallaCreds.ts ? new Date(wallaCreds.ts).toLocaleTimeString() : null
  };
}

export async function initWallaCredsFromStorage() {
  try { 
    const s = await stGet([WCREDS_KEY]); 
    if (s[WCREDS_KEY]) {
      wallaCreds = s[WCREDS_KEY];
      const age = getTokenAge();
      dlog(`[TOKEN] 📂 Cargado de storage. Edad: ${age.formatted}`);
    }
  } catch {}
}

export function setupWallaInterceptors() {
  // ✅ v10.5.106: Interceptor restaurado EXACTAMENTE como v10.5.76 (que funcionaba)
  if (chrome.webRequest?.onBeforeSendHeaders) {
    chrome.webRequest.onBeforeSendHeaders.addListener(
      (details) => {
        let tok = null, dev = null, appV = null, devOs = null;
        for (const h of (details.requestHeaders || [])) {
          const n = (h.name || '').toLowerCase();
          const v = (h.value || '').trim();
          if (n === 'authorization' && /^bearer\s+/i.test(v)) tok = v.replace(/^bearer\s+/i, '');
          if (n === 'x-deviceid' || n === 'x-device-id') dev = v;
          if (n === 'x-appversion') appV = v;
          if (n === 'x-deviceos') devOs = v;
        }
        if (tok || dev || appV || devOs) {
          const oldToken = wallaCreds.token;
          const fresh = Date.now() - (wallaCreds.ts || 0) > 5_000;
          if (tok)   wallaCreds.token     = tok;
          if (dev)   wallaCreds.deviceId  = dev;
          if (appV)  wallaCreds.appVersion= appV;
          if (devOs) wallaCreds.deviceOs  = devOs;
          if (tok || dev || fresh) wallaCreds.ts = Date.now();
          chrome.storage.local.set({ [WCREDS_KEY]: wallaCreds });
          
          if (tok && tok !== oldToken) {
            const realAge = getTokenRealAge();
            dlog(`[TOKEN] 🔄 NUEVO token! (${tok.substring(0, 6)}...) | Expira: ${realAge.exp} | Restante: ${realAge.remainingFormatted}`);
          }
        }
      },
      { urls: ['https://api.wallapop.com/*'], types: ['xmlhttprequest', 'other'] },
      ['requestHeaders', 'extraHeaders']  // ✅ v10.5.106: Igual que v10.5.76 (funcionaba)
    );
    dlog('[TOKEN] 👂 Interceptor de Wallapop activo');
  } else {
    dwarn('[TOKEN] ⚠️ webRequest no disponible');
  }
}

export async function kickOnceOnActiveTab() {
  try {
    // Intentar primero con tab activo
    let [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    // Si el activo no es Wallapop, buscar cualquier tab de Wallapop abierto
    if (!tab?.url || !/wallapop\.(com|es)$/i.test(new URL(tab.url).hostname)) {
      const allTabs = await chrome.tabs.query({});
      const wallaTabs = allTabs.filter(t => 
        t?.url && /wallapop\.(com|es)$/i.test(new URL(t.url).hostname)
      );
      
      if (wallaTabs.length > 0) {
        tab = wallaTabs[0]; // Usar primer tab de Wallapop encontrado
      }
    }
    
    if (tab?.id && tab?.url && /wallapop\.(com|es)$/i.test(new URL(tab.url).hostname)) {
      await chrome.tabs.sendMessage(tab.id, { type: 'CAPTURE.KICK_ONCE' }).catch?.(() => {});
    } else {
    }
  } catch (e) {
    console.error('[WALLA] ❌ Error en kickOnceOnActiveTab:', e);
  }
}

export async function ensureWallaCreds({ timeoutMs=15000, kick=true } = {}) {
  if (wallaCreds.token && wallaCreds.deviceId) return wallaCreds;
  const st = await stGet([WCREDS_KEY]);
  if (st[WCREDS_KEY]?.token && st[WCREDS_KEY]?.deviceId) { wallaCreds = st[WCREDS_KEY]; return wallaCreds; }
  const t0 = Date.now();
  if (kick) kickOnceOnActiveTab();
  return await new Promise((resolve) => {
    const iv = setInterval(() => {
      if (wallaCreds.token && wallaCreds.deviceId) { clearInterval(iv); resolve(wallaCreds); }
      else if (Date.now() - t0 > timeoutMs) { clearInterval(iv); resolve(null); }
    }, 200);
  });
}

/**
 * ✅ v10.5.62: Refresca el token de Wallapop SOLO si está por expirar
 * 
 * CAMBIOS v10.5.62:
 * - NO borramos el token actual hasta tener uno nuevo
 * - Si el refresh falla, usamos el token actual (puede que aún funcione)
 * - Buffer aumentado a 45s para dar más margen
 * 
 * @param {number} tabId - ID del tab de Wallapop
 * @param {number} _maxAgeMs - IGNORADO (mantenido por compatibilidad)
 * @param {boolean} forceRefresh - Si true, ignora todo y siempre refresca
 * @returns {Promise<{ok: boolean, refreshed: boolean, creds: Object|null}>}
 */
export async function refreshWallaCredsIfOld(tabId, _maxAgeMs = 90000, forceRefresh = false) {
  // ✅ v10.5.78: SIEMPRE refrescar via iframe (sin comprobar tiempos)
  // El usuario puede poner delays de horas entre anuncios
  const realAge = getTokenRealAge();
  dlog(`[TOKEN] 🔄 Refrescando via iframe... (actual: ${realAge.remainingFormatted})`);
  
  // Guardamos el token actual para comparar
  const oldToken = wallaCreds.token;
  const oldTs = wallaCreds.ts || 0;
  
  // ✅ v2.2.1: Reintentos del iframe (máx 3 intentos)
  const MAX_RETRIES = 3;
  
  for (let attempt = 1; attempt <= MAX_RETRIES; attempt++) {
    // Usar iframe para refrescar token
    try {
      dlog(`[TOKEN] 📦 Intento ${attempt}/${MAX_RETRIES} - Solicitando refresh via iframe...`);
      const refreshResult = await chrome.tabs.sendMessage(tabId, { type: 'TOKEN.REFRESH_VIA_IFRAME' });
      dlog(`[TOKEN] 📦 Resultado iframe intento ${attempt}:`, refreshResult?.ok ? '✅' : '❌');
    } catch (e) {
      dwarn(`[TOKEN] ⚠️ Intento ${attempt} - No se pudo ejecutar iframe refresh:`, e.message);
    }
    
    // Esperar a que el interceptor capture el token nuevo (máx 5s por intento)
    const t0 = Date.now();
    while (Date.now() - t0 < 5000) {
      // Verificar si llegó un token diferente
      if (wallaCreds.token && wallaCreds.token !== oldToken) {
        const newAge = getTokenRealAge();
        dlog(`[TOKEN] ✅ Token NUEVO en intento ${attempt}! Restante: ${newAge.remainingFormatted}`);
        return { ok: true, refreshed: true, creds: wallaCreds };
      }
      
      // Verificar en memoria por timestamp
      if (wallaCreds.token && wallaCreds.ts > oldTs) {
        const currentAge = getTokenRealAge();
        if (currentAge.remainingSeconds !== null && currentAge.remainingSeconds > 60) {
          dlog(`[TOKEN] ✅ Token refrescado en intento ${attempt}! Restante: ${currentAge.remainingFormatted}`);
          return { ok: true, refreshed: true, creds: wallaCreds };
        }
      }
      
      // Verificar en storage
      const st = await stGet([WCREDS_KEY]);
      if (st[WCREDS_KEY]?.token && st[WCREDS_KEY]?.token !== oldToken) {
        wallaCreds = st[WCREDS_KEY];
        const stAge = getTokenRealAge();
        dlog(`[TOKEN] ✅ Token desde storage en intento ${attempt}! Restante: ${stAge.remainingFormatted}`);
        return { ok: true, refreshed: true, creds: wallaCreds };
      }
      
      await new Promise(r => setTimeout(r, 300));
    }
    
    // Si no es el último intento, esperar antes de reintentar
    if (attempt < MAX_RETRIES) {
      dlog(`[TOKEN] ⏳ Intento ${attempt} sin éxito, reintentando en 1s...`);
      await new Promise(r => setTimeout(r, 1000));
    }
  }
  
  // Si no llegó token nuevo después de todos los intentos, usar el actual si aún es válido
  const finalAge = getTokenRealAge();
  if (wallaCreds.token && !finalAge.isExpired && finalAge.remainingSeconds > 10) {
    dwarn(`[TOKEN] ⚠️ No llegó token nuevo tras ${MAX_RETRIES} intentos, usando actual (restante: ${finalAge.remainingFormatted})`);
    return { ok: true, refreshed: false, creds: wallaCreds };
  }
  
  // Token expirado
  console.error(`[TOKEN] ❌ Sesión de Wallapop expirada tras ${MAX_RETRIES} intentos - Refresca la página de Wallapop`);
  return { ok: false, refreshed: false, creds: null, reason: 'session_expired' };
}

export async function invalidateWallaCredsInSW() {
  
  // Limpiar en memoria
  wallaCreds = { token: null, deviceId: null, appVersion: null, deviceOs: null, ts: 0 };
  
  // Limpiar en storage
  try {
    await chrome.storage.local.remove([WCREDS_KEY]);
  } catch (e) {
    console.error('[WALLA] ❌ Error invalidando storage:', e);
  }
}

export function wallaHeaders() {
  return {
    'Accept':'application/json',
    'Authorization':`Bearer ${wallaCreds.token}`,
    'X-Device-Id': wallaCreds.deviceId
  };
}

export function aliasFromProfileUrl(u){
  try {
    const last = new URL(u).pathname.split('/').filter(Boolean).pop() || '';
    return last.replace(/-\d+$/, '');
  } catch { return null; }
}

export async function importIntoAPI({ wallaUserId, profileUrl, items }) {
  const payload = {
    walla_user_id: wallaUserId,
    profile_url: profileUrl,
    account_alias: aliasFromProfileUrl(profileUrl),
    items
  };
  const headers = withDeviceHeaders({ 'Content-Type': 'application/json' });
  const r = await apiFetch('/api/walla/listings', { method:'POST', headers, body: JSON.stringify(payload) });
  const j = await r.json().catch(()=> ({}));
  return { ok: r.ok, data: j };
}

export async function getUserIdFromProfileUrl(profileUrl) {
  const res = await fetch(profileUrl, { method:'GET' });
  if (!res.ok) throw new Error('profile_http_'+res.status);
  const html = await res.text();
  const m = html.match(/<script id="__NEXT_DATA__"[^>]*>([\s\S]*?)<\/script>/);
  if (!m) throw new Error('nextdata_not_found');
  let next = null;
  try { next = JSON.parse(m[1]); } catch { throw new Error('nextdata_parse'); }
  const root = next?.props?.pageProps || next?.pageProps || {};
  const guesses = [
    root?.user?.id,
    root?.profile?.user?.id,
    root?.userProfile?.id,
    root?.itemSeller?.id,
  ].filter(Boolean);
  if (guesses.length) return guesses[0];
  let found = null;
  (function scan(o){
    if (!o || found) return;
    if (typeof o === 'object') {
      if (typeof o.id === 'string' && /^[-_a-z0-9]{8,32}$/i.test(o.id)) { found = o.id; return; }
      for (const v of Object.values(o)) scan(v);
    }
  })(root);
  if (!found) throw new Error('user_id_not_found');
  return found;
}

export async function listAllItemsAPI({ userId, limit=40, cursorParam='since' } = {}) {
  let cursor = null, all = [];
  const requestTimes = [];
  const MAX_REQUESTS_PER_SECOND = 10;
  const WINDOW_MS = 1000;
  for (let page=0; page<500; page++) {
    const now = Date.now();
    // limpiar tiempos antiguos
    while (requestTimes.length && (now - requestTimes[0] > WINDOW_MS)) requestTimes.shift();
    if (requestTimes.length >= MAX_REQUESTS_PER_SECOND) {
      const wait = WINDOW_MS - (now - requestTimes[0]);
      if (wait > 0) await new Promise(r => setTimeout(r, wait));
      requestTimes.shift();
    }
    requestTimes.push(Date.now());

    const u = new URL(`https://api.wallapop.com/api/v3/users/${userId}/items`);
    u.searchParams.set('limit', String(limit));
    if (cursor) u.searchParams.set(cursorParam, cursor);

    const r = await fetch(u.toString(), { headers: wallaHeaders() });
    if (r.status === 401) throw new Error('auth_401');
    if (!r.ok) throw new Error('http_'+r.status);

    const j = await r.json();
    const items = j.data || j.items || [];
    all.push(...items);
    cursor = j.next || j.since || j.meta?.next || null;
    if (!cursor || items.length === 0) break;
  }
  return all;
}

export async function getItemDetailAPI(id) {
  const r = await fetch(`https://api.wallapop.com/api/v3/items/${id}`, { headers: wallaHeaders() });
  if (r.status === 401) throw new Error('auth_401');
  if (!r.ok) throw new Error('http_'+r.status);
  return await r.json();
}

export async function getItemDetailFromPage(slug) {
  const r = await fetch(`https://es.wallapop.com/item/${slug}`);
  if (!r.ok) throw new Error('page_http_'+r.status);
  const html = await r.text();
  const m = html.match(/<script id="__NEXT_DATA__"[^>]*>([\s\S]*?)<\/script>/);
  if (!m) throw new Error('nextdata_not_found');
  const next = JSON.parse(m[1]);
  const d = next?.props?.pageProps?.item || next?.pageProps?.item || null;
  if (!d) throw new Error('item_not_found');
  return d;
}

export async function enrichAll(items, { concurrency=8 } = {}) {
  const out = new Array(items.length);
  let i = 0;
  async function worker(){
    while (true) {
      const idx = i++;
      if (idx >= items.length) break;
      const it = items[idx];
      try {
        let d;
        try { d = await getItemDetailAPI(it.id); }
        catch { d = await getItemDetailFromPage(it.slug); }
        out[idx] = {
          id: d.id,
          slug: d.slug,
          url: `https://es.wallapop.com/item/${d.slug}`,
          title: d.title,
          description: d.description,
          category_id: d.category_id || d.categoryId || d.taxonomy?.[d.taxonomy.length-1]?.id || null,
          price_amount: (d.price?.amount ?? d.price?.cash?.amount) ?? null,
          price_currency: (d.price?.currency ?? d.price?.cash?.currency) ?? null,
          images: (d.images || []).map(im => im.urls?.big || im.urls?.medium || im.urls?.small).filter(Boolean),
          location: d.location || null,
          attributes: d.type_attributes || d.attributes || d.goodsAndFashionInfo || null,
          status: d.status,
          flags: d.flags,
          seller: d.user || { id: d.userId },
          created_at: d.creation_date || d.creationDate || d.modified_date || d.modifiedDate || null,
          raw: d
        };
      } catch (e) {
        out[idx] = { id: it.id, slug: it.slug, error: String(e?.message || e) };
      }
    }
  }
  const workers = Math.max(1, Math.min(concurrency, items.length || 1));
  await Promise.all(Array.from({ length: workers }, () => worker()));
  return out;
}

export async function exportAll(profileUrl) {
  const creds = await ensureWallaCreds({ timeoutMs: 8000, kick: true });
  if (!creds) throw new Error('no_creds');
  const wallaUserId = await getUserIdFromProfileUrl(profileUrl);
  const base = await swRetryWithBackoff(
    async () => {
      try {
        return await listAllItemsAPI({ userId: wallaUserId, limit: 40, cursorParam: 'since' });
      } catch (e) {
        if (String(e.message) === 'auth_401') { await ensureWallaCreds({ timeoutMs: 8000, kick: true }); throw e; }
        throw e;
      }
    },
    { maxRetries: 2, baseDelay: 1000 }
  );
  const items = await enrichAll(base, { concurrency: 8 });
  const ok = items.filter(x => !x.error).length;
  return { wallaUserId, items, stats: { total: items.length, ok, errors: items.length - ok } };
}

export async function fullExportByProfileUrl(profileUrl) {
  const { items, stats } = await exportAll(profileUrl);
  const okItems = items.filter(x => !x.error);
  return { total: stats.total, sample: okItems[0] || items[0] || null };
}
