/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/utils.js — Rol: Utilidades: retry/backoff, fetch con timeout, withTimeout, uuidv4, singleFlight.
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

/* =========================================================
   Utils: Retry con backoff exponencial
   ========================================================= */
export async function swRetryWithBackoff(fn, opts = {}) {
  const { maxRetries = 3, baseDelay = 250, maxDelay = 5000, timeoutMs = null, onRetry = null } = opts;
  let lastError = null;
  for (let attempt = 0; attempt < maxRetries; attempt++) {
    try {
      if (timeoutMs) {
        return await Promise.race([
          fn(),
          new Promise((_, reject) => setTimeout(() => reject(new Error('timeout')), timeoutMs))
        ]);
      }
      return await fn();
    } catch (err) {
      lastError = err;
      if (attempt === maxRetries - 1) throw err;
      const delay = Math.min(baseDelay * Math.pow(2, attempt), maxDelay);
      if (onRetry) onRetry(attempt + 1, delay, err);
      await new Promise(r => setTimeout(r, delay));
    }
  }
  throw lastError;
}

/* =========================================================
   Utils: fetch con timeout + wrapper withTimeout
   ========================================================= */
export function fetchWithTimeout(url, opts = {}, ms = 12000) {
  const ctl = new AbortController();
  const t = setTimeout(() => ctl.abort(), ms);
  return fetch(url, { ...opts, signal: ctl.signal }).finally(() => clearTimeout(t));
}

export function withTimeout(promise, ms, fallback) {
  let timer;
  return Promise.race([
    promise,
    new Promise((resolve) => { timer = setTimeout(() => resolve(fallback), ms); })
  ]).finally(() => { if (timer) clearTimeout(timer); });
}

/* =========================================================
   Utils: uuid v4 + singleFlight
   ========================================================= */
export function uuidv4(){
  return ([1e7]+-1e3+-4e3+-8e3+-1e11)
    .replace(/[018]/g,c=>(c^crypto.getRandomValues(new Uint8Array(1))[0]&15>>c/4).toString(16));
}

const FLIGHTS = new Map();
export async function singleFlight(key, fn) {
  if (FLIGHTS.has(key)) return await FLIGHTS.get(key);
  const p = (async () => { try { return await fn(); } finally { FLIGHTS.delete(key); } })();
  FLIGHTS.set(key, p);
  return await p;
}

/* =========================================================
   Utils: Extracción segura de título desde snapshot
   Maneja estructura {original: "texto"} o string directo
   ========================================================= */
export function extractTitle(titleField, fallback = 'Sin título') {
  if (!titleField) return fallback;
  
  // Si es un objeto con propiedad original
  if (typeof titleField === 'object' && titleField !== null) {
    const original = titleField.original;
    // Verificar que original sea un string no vacío
    if (typeof original === 'string' && original.trim() !== '') {
      return original;
    }
    // Fallback si original está vacío o no existe
    return fallback;
  }
  
  // Si es un string directo
  if (typeof titleField === 'string' && titleField.trim() !== '') {
    return titleField;
  }
  
  return fallback;
}
