/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/token_refresher.js — Rol: Auto-renovación proactiva de tokens (cada 5 min)
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

import { isAccessNearExpiry, doRefresh, getAuthState, AUTH_STATES } from './api_client.js';

/* =========================================================
   AUTO-REFRESH PROACTIVO
   Renueva tokens automáticamente cada 5 minutos
   ANTES de que expiren, eliminando errores de expiración
   ========================================================= */

let refreshInterval = null;
let lastRefreshAttempt = 0;
const REFRESH_CHECK_INTERVAL = 5 * 60 * 1000;  // ✅ v1.8.0: 5 minutos
const REFRESH_BEFORE_EXPIRY = 3 * 60;  // ✅ v1.8.0: Renovar 3 min antes de expirar
const MIN_TIME_BETWEEN_REFRESHES = 60 * 1000;  // No más de 1 refresh por minuto

/**
 * Inicia el auto-refresh proactivo de tokens
 * Se ejecuta automáticamente al cargar la extensión si hay sesión activa
 */
export function startAutoRefresh() {
  if (refreshInterval) {
    return;
  }
  
  
  // Primera comprobación inmediata
  checkAndRefresh();
  
  // Luego cada 5 minutos
  refreshInterval = setInterval(() => {
    checkAndRefresh();
  }, REFRESH_CHECK_INTERVAL);
}

/**
 * Detiene el auto-refresh (al hacer logout)
 */
export function stopAutoRefresh() {
  if (refreshInterval) {
    clearInterval(refreshInterval);
    refreshInterval = null;
  }
}

/**
 * Verifica si el token necesita renovación y lo renueva proactivamente
 */
async function checkAndRefresh() {
  const now = Date.now();
  
  // Evitar refreshes muy seguidos
  if (now - lastRefreshAttempt < MIN_TIME_BETWEEN_REFRESHES) {
    return;
  }
  
  // Solo refrescar si hay sesión activa
  if (getAuthState() !== AUTH_STATES.AUTHENTICATED) {
    return;
  }
  
  // Verificar si el token expira pronto
  if (!isAccessNearExpiry(REFRESH_BEFORE_EXPIRY)) {
    return;
  }
  
  // Renovar token proactivamente
  try {
    lastRefreshAttempt = now;
    
    await doRefresh();
    
  } catch (error) {
    console.error('[AUTO-REFRESH] ❌ Error renovando token:', error.message || error);
    
    // Si falló porque no hay sesión, detener auto-refresh
    if (error.message === 'no_refresh_or_device' || error.message === 'max_refresh_fails_exceeded') {
      stopAutoRefresh();
    }
  }
}

/**
 * Resetea el cooldown (útil tras un login exitoso)
 */
export function resetRefreshCooldown() {
  lastRefreshAttempt = 0;
}

/**
 * Verifica estado del auto-refresh
 */
export function isAutoRefreshActive() {
  return refreshInterval !== null;
}

/* =========================================================
   EXPORTS
   ========================================================= */
export default {
  start: startAutoRefresh,
  stop: stopAutoRefresh,
  reset: resetRefreshCooldown,
  isActive: isAutoRefreshActive
};
