/**
 * Estado global del Service Worker
 * v6.7.0 - Tracking de modo resume
 * v10.5.35 - Estado persistente para delays largos
 */

// ✅ v6.7.0: Estado de publicación para evitar reload innecesario
export const publishState = {
  isResumeMode: false,   // true = reanudar, false = publicación nueva
  resetAt: 0,            // timestamp del último reset
  processedCount: 0,     // ✅ v7.0.7: Contador para pausas largas
  breakInterval: null    // ✅ v7.0.7: Intervalo de pausa (8-12)
};

// =============================================================================
// ✅ v10.5.35: Estado persistente para delays largos (usa chrome.storage)
// Cuando el SW se duerme durante un delay largo, este estado permite continuar
// =============================================================================

const PENDING_PUBLISH_KEY = 'publish.pending_item';

/**
 * Guarda el estado del item pendiente de publicar antes de un delay largo
 * @param {Object} data - Datos del item pendiente
 * @param {number} data.itemId - ID interno del item
 * @param {number} data.accountId - ID de la cuenta
 * @param {number} data.tabId - ID de la pestaña de Wallapop
 * @param {boolean} data.skipDelete - Si se saltó el delete (ya eliminado)
 * @param {Object} data.snapshot - Snapshot del item para logs
 */
export async function savePendingPublish(data) {
  try {
    const stateToSave = {
      ...data,
      savedAt: Date.now()
    };
    await chrome.storage.local.set({
      [PENDING_PUBLISH_KEY]: stateToSave
    });
    console.log(`[SW] 💾 Estado pendiente GUARDADO:`, { itemId: data.itemId, accountId: data.accountId, tabId: data.tabId });
  } catch (e) {
    console.error('[SW] ❌ Error guardando estado pendiente:', e);
  }
}

/**
 * Recupera el estado del item pendiente de publicar
 * @returns {Promise<Object|null>} Datos del item o null si no hay
 */
export async function getPendingPublish() {
  try {
    const result = await chrome.storage.local.get([PENDING_PUBLISH_KEY]);
    const state = result[PENDING_PUBLISH_KEY] || null;
    if (state) {
      console.log(`[SW] 📖 Estado pendiente LEÍDO:`, { itemId: state.itemId, accountId: state.accountId, savedAt: state.savedAt });
    } else {
      console.log(`[SW] 📖 No hay estado pendiente guardado`);
    }
    return state;
  } catch (e) {
    console.error('[SW] ❌ Error leyendo estado pendiente:', e);
    return null;
  }
}

/**
 * Limpia el estado del item pendiente (después de publicar o en caso de error)
 */
export async function clearPendingPublish() {
  try {
    await chrome.storage.local.remove([PENDING_PUBLISH_KEY]);
    console.log(`[SW] 🗑️ Estado pendiente LIMPIADO`);
  } catch (e) {
    console.error('[SW] ❌ Error limpiando estado pendiente:', e);
  }
}
