/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/realtime.js — Rol: Cliente WebSocket para actualizaciones en tiempo real
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  
  VERSIÓN: 3.2.0 - CONTRATO ÚNICO WebSocket
  
  ═══════════════════════════════════════════════════════════════════
  📡 CONTRATO ÚNICO DE MENSAJES WEBSOCKET
  ═══════════════════════════════════════════════════════════════════
  
  REGLA DE ORO: Los mensajes del backend llegan al panel SIN MODIFICACIONES
  
  Estructura única en toda la cadena:
  {
    type: 'backup_progress',      // Nombre del evento (sin traducir)
    account_id: 1116,             // ID de cuenta (nivel superior)
    data: {                       // Datos del evento
      items_processed: 50,
      items_total: 173
    },
    timestamp: '2025-11-03...'    // Marca temporal
  }
  
  ❌ NO HACER:
  - Traducir nombres de eventos (backup_progress → RUN_PROGRESS)
  - Añadir prefijos (WS., EVENT., etc)
  - Anidar estructuras ({payload: {data: {}}})
  
  ✅ SÍ HACER:
  - Reenviar TAL CUAL al panel
  - Mantener mismo 'type' en toda la cadena
  - Documentar nuevos eventos en CONTRATO_WEBSOCKET.md
  
  Ver: /outputs/CONTRATO_WEBSOCKET.md para detalles completos
  ═══════════════════════════════════════════════════════════════════
*/

import { CONFIG, doRefresh, session, isAccessNearExpiry, setAuthState, AUTH_STATES } from './api_client.js';
import { stopAutoRefresh } from './token_refresher.js';
import { stRemove } from './storage.js';

/**
 * Cliente WebSocket para comunicación en tiempo real con el backend
 * 
 * Maneja:
 * - Conexión/desconexión automática
 * - Reconexión con backoff exponencial
 * - Ping/pong para mantener conexión viva
 * - Emisión de eventos a paneles abiertos
 * - Gestión de múltiples conexiones por usuario
 */
class RealtimeClient {
  constructor() {
    this.ws = null;
    this.connectionId = null;
    this.reconnectTimer = null;
    this.reconnectAttempts = 0;
    this.maxReconnectAttempts = 10; // 10 reintentos
    this.reconnectDelay = 1000; // 1s inicial
    this.pingInterval = null;
    this.isConnecting = false;
    this.isConnected = false; // 🆕 Estado de conexión
    this.eventHandlers = new Map();
    this.accessToken = null;
    this.isIntentionallyClosed = false;
    
    // 🆕 HEARTBEAT WATCHDOG
    this.lastHeartbeatTime = null;
    this.heartbeatWatchdog = null;
    this.heartbeatTimeout = 60000; // 60s (2x intervalo del servidor = 30s)
    this.missedHeartbeats = 0;
    this.maxMissedHeartbeats = 2; // Desconectar tras 2 fallos
    
  }

  /**
   * Conectar WebSocket al backend
   * 
   * @param {string} accessToken - JWT access token para autenticación
   * @returns {Promise<boolean>} - True si conectó exitosamente
   */
  async connect(accessToken) {
    if (!accessToken) {
      console.error('[WS] ❌ No hay access token');
      return false;
    }

    if (this.isConnecting || (this.ws && this.ws.readyState === WebSocket.OPEN)) {
      return true;
    }

    this.isConnecting = true;
    this.isIntentionallyClosed = false;
    this.accessToken = accessToken;
    
    // Generar ID único de conexión
    this.connectionId = crypto.randomUUID();
    
    // Construir URL WebSocket
    const wsUrl = CONFIG.API_BASE
      .replace('https://', 'wss://')
      .replace('http://', 'ws://');
    
    const url = `${wsUrl}/ws?token=${accessToken}&conn_id=${this.connectionId}`;
    
    try {
      // 🔒 Token NO se loguea por seguridad
      
      this.ws = new WebSocket(url);
      
      // Evento: Conexión abierta
      this.ws.onopen = () => {
        this.isConnecting = false;
        this.isConnected = true;
        this.reconnectAttempts = 0;
        
        // Iniciar ping periódico
        this.startPingInterval();
        
        // 🆕 Iniciar heartbeat watchdog
        this.startHeartbeatWatchdog();
        
        // 🟢 Notificar panel que WebSocket está conectado
        this.notifyPanel('connected', { 
          data: { connectionId: this.connectionId },
          account_id: null 
        });
      };
      
      // Evento: Mensaje recibido
      this.ws.onmessage = (event) => {
        try {
          const message = JSON.parse(event.data);
          this.handleMessage(message);
        } catch (e) {
          console.error('[WS] ❌ Error parseando mensaje:', e);
        }
      };
      
      // Evento: Error
      this.ws.onerror = (error) => {
        console.error('[WS] ❌ Error en conexión:', error);
        console.error('[WS] ❌ ReadyState:', this.ws?.readyState);
        console.error('[WS] ❌ URL:', wsUrl + '/ws');
        this.isConnecting = false;
      };
      
      // Evento: Conexión cerrada
      this.ws.onclose = (event) => {
        this.isConnecting = false;
        this.isConnected = false;
        this.stopPingInterval();
        this.stopHeartbeatWatchdog(); // 🆕 Detener watchdog
        
        // 🔴 Notificar panel que WebSocket está desconectado
        this.notifyPanel('disconnected', { 
          data: { code: event.code, reason: event.reason },
          account_id: null 
        });
        
        // Reconectar automáticamente si no fue cierre intencional
        if (!this.isIntentionallyClosed && event.code !== 1000 && event.code !== 1001) {
          this.scheduleReconnect();
        }
      };
      
      return true;
      
    } catch (error) {
      console.error('[WS] ❌ Error creando WebSocket:', error);
      this.isConnecting = false;
      this.scheduleReconnect();
      return false;
    }
  }

  /**
   * Maneja mensajes recibidos del servidor
   * 
   * @param {Object} message - Mensaje del servidor con formato { type, data, account_id, timestamp }
   */
  handleMessage(message) {
    const { type, data, account_id, timestamp } = message;
    
    // 💓 HEARTBEAT: Responder con pong y resetear watchdog
    if (type === 'HEARTBEAT') {
      this.handleHeartbeat(message);
      return; // No propagar
    }
    
    // 🔴 CRITICAL: Manejar sesión revocada INMEDIATAMENTE
    if (type === 'SESSION_REVOKED') {
      console.error('[WS] 🔴 SESIÓN REVOCADA - Cerrando sesión...');
      this.handleSessionRevoked(message);
      return; // No propagar más
    }
    
    // 🔒 CRITICAL: Manejar modo mantenimiento INMEDIATAMENTE
    if (type === 'MAINTENANCE_MODE') {
      this.handleMaintenanceMode(message);
      return; // No propagar más
    }
    
    // Ejecutar handlers registrados
    const handlers = this.eventHandlers.get(type);
    if (handlers && handlers.size > 0) {
      handlers.forEach(handler => {
        try {
          handler(data, account_id);
        } catch (e) {
          console.error(`[WS] ❌ Error en handler ${type}:`, e);
        }
      });
    }
    
    // Notificar al panel (para actualizar UI)
    this.notifyPanel(type, { data, account_id, timestamp });
  }
  
  /**
   * Maneja HEARTBEAT del servidor
   * Responde con pong y resetea watchdog
   */
  handleHeartbeat(message) {
    // ✅ v4.84.8: Log eliminado (ruido innecesario)
    
    // Actualizar timestamp del último heartbeat
    this.lastHeartbeatTime = Date.now();
    this.missedHeartbeats = 0;
    
    // Responder con pong
    if (this.ws && this.ws.readyState === WebSocket.OPEN) {
      try {
        this.ws.send(JSON.stringify({
          type: 'pong',
          timestamp: new Date().toISOString()
        }));
        // ✅ v4.84.8: Log eliminado (ruido innecesario)
      } catch (e) {
        console.error('[WS] ❌ Error enviando pong:', e);
      }
    }
    
    // Notificar panel que servidor está vivo (DOT verde)
    this.notifyPanel('SERVER_ALIVE', { 
      data: { timestamp: message.timestamp },
      account_id: null 
    });
  }
  
  /**
   * Maneja revocación de sesión (login en otro dispositivo)
   * Limpia todo y redirige a login
   */
  async handleSessionRevoked(message) {
    const { reason, message: msg, force_logout } = message;
    
    console.error('[WS] 🔴 Sesión revocada:', { reason, message: msg });
    
    // Detener auto-refresh
    stopAutoRefresh();
    
    // Desconectar WebSocket
    this.disconnect();
    
    // Limpiar sesión del storage
    await stRemove(['session.access', 'session.refresh', 'session.exp', 'session.user']);
    
    // Actualizar estado global
    setAuthState(AUTH_STATES.UNAUTHENTICATED);
    
    // Notificar al panel para que muestre mensaje y redirige
    chrome.runtime.sendMessage({
      type: 'FORCE_LOGOUT',
      reason: reason,
      message: msg || 'Has iniciado sesión en otro dispositivo',
      timestamp: Date.now()
    }).catch(() => {
      // Panel no abierto
    });
    
  }
  
  /**
   * Maneja evento de modo mantenimiento
   * Notifica al panel para verificar si debe bloquearse
   */
  async handleMaintenanceMode(message) {
    const { enabled, message: msg } = message;
    
    
    // Notificar al panel para que maneje el modal/banner
    chrome.runtime.sendMessage({
      type: 'MAINTENANCE_MODE_CHANGED',
      enabled: enabled,
      message: msg || (enabled ? 'Sistema en mantenimiento' : 'Mantenimiento finalizado'),
      timestamp: Date.now()
    }).catch(() => {
      // Panel no abierto
    });
    
  }

  /**
   * Notifica al panel de eventos WebSocket
   * 
   * CONTRATO ÚNICO: Reenvía el mensaje TAL CUAL viene del backend
   * Sin traducciones, sin prefijos, sin modificaciones
   * 
   * @param {string} type - Tipo de evento del backend
   * @param {Object} payload - {data, account_id, timestamp}
   */
  notifyPanel(type, payload) {
    // Mensaje con CONTRATO ÚNICO - igual que backend
    chrome.runtime.sendMessage({
      type: type,                    // Tal cual: 'backup_progress', 'connected', etc
      account_id: payload.account_id,
      data: payload.data,
      timestamp: payload.timestamp
    }).catch(() => {
      // Panel no abierto, ignorar
    });
  }

  /**
   * Registra un handler para un tipo de evento
   * 
   * @param {string} eventType - Tipo de evento (backup_progress, publish_complete, etc)
   * @param {Function} handler - Función handler(data, account_id)
   */
  on(eventType, handler) {
    if (!this.eventHandlers.has(eventType)) {
      this.eventHandlers.set(eventType, new Set());
    }
    this.eventHandlers.get(eventType).add(handler);
  }

  /**
   * Elimina un handler
   * 
   * @param {string} eventType - Tipo de evento
   * @param {Function} handler - Función handler a eliminar
   */
  off(eventType, handler) {
    if (this.eventHandlers.has(eventType)) {
      this.eventHandlers.get(eventType).delete(handler);
    }
  }

  /**
   * Envía un mensaje al servidor
   * 
   * @param {Object} message - Mensaje a enviar
   */
  send(message) {
    if (this.ws && this.ws.readyState === WebSocket.OPEN) {
      this.ws.send(JSON.stringify(message));
    } else {
    }
  }

  /**
   * Ping periódico para mantener conexión viva
   */
  startPingInterval() {
    this.stopPingInterval();
    
    this.pingInterval = setInterval(() => {
      this.send({ type: 'ping' });
    }, 30000); // Cada 30s
  }

  /**
   * Detener ping
   */
  stopPingInterval() {
    if (this.pingInterval) {
      clearInterval(this.pingInterval);
      this.pingInterval = null;
    }
  }

  /**
   * Inicia watchdog que detecta falta de heartbeats
   * Si no llegan heartbeats → servidor caído → DOT rojo
   */
  startHeartbeatWatchdog() {
    this.stopHeartbeatWatchdog();
    
    this.lastHeartbeatTime = Date.now();
    this.missedHeartbeats = 0;
    
    this.heartbeatWatchdog = setInterval(() => {
      const now = Date.now();
      const timeSinceLastHeartbeat = now - this.lastHeartbeatTime;
      
      // Si pasaron > 60s sin heartbeat → servidor probablemente caído
      if (timeSinceLastHeartbeat > this.heartbeatTimeout) {
        this.missedHeartbeats++;
        
        
        // Notificar panel que servidor está muerto (DOT rojo)
        this.notifyPanel('SERVER_DEAD', { 
          data: { 
            missed: this.missedHeartbeats,
            last_heartbeat: timeSinceLastHeartbeat 
          },
          account_id: null 
        });
        
        // Si falló 2 veces → forzar reconexión
        if (this.missedHeartbeats >= this.maxMissedHeartbeats) {
          console.error('[WS] 💀 Servidor sin respuesta, forzando reconexión...');
          this.forceReconnect();
        }
      }
    }, 30000); // Verificar cada 30s
    
  }

  /**
   * Detiene watchdog
   */
  stopHeartbeatWatchdog() {
    if (this.heartbeatWatchdog) {
      clearInterval(this.heartbeatWatchdog);
      this.heartbeatWatchdog = null;
    }
  }

  /**
   * Fuerza reconexión cerrando conexión actual
   */
  forceReconnect() {
    
    if (this.ws) {
      try {
        this.ws.close();
      } catch (e) {
        console.error('[WS] Error cerrando WebSocket:', e);
      }
    }
    
    // Reconectar tras 2s
    setTimeout(() => {
      if (this.accessToken && !this.isIntentionallyClosed) {
        this.connect(this.accessToken);
      }
    }, 2000);
  }

  /**
   * Programa reconexión automática con backoff exponencial
   * ✅ OPTIMIZADO: Intenta refrescar token antes de reconectar
   */
  async scheduleReconnect() {
    if (this.isIntentionallyClosed) {
      return;
    }

    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      // Ya no detener, solo usar delay más largo
    }
    
    this.reconnectAttempts++;
    
    // Backoff exponencial con máximo de 30s
    const delay = Math.min(
      this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1),
      30000
    );
    
    
    this.reconnectTimer = setTimeout(async () => {
      // ✅ OPTIMIZADO: Intentar refrescar token ANTES de reconectar
      try {
        
        const refreshed = await doRefresh();
        
        if (refreshed) {
          // Obtener nuevo token
          const newToken = session.access;
          this.accessToken = newToken; // ✅ ACTUALIZAR TOKEN
          this.connect(newToken);
        } else {
          // Si no se pudo refrescar, intentar con token actual
          if (this.accessToken) {
            this.connect(this.accessToken);
          }
        }
      } catch (error) {
        console.error('[WS] ❌ Error refrescando token:', error);
        
        // ✅ v6.8.3: Verificar si token actual está expirado antes de usarlo
        try {
          // Si el token expira en menos de 2 minutos, no intentar reconectar
          if (isAccessNearExpiry(120)) {
            this.reconnectAttempts = this.maxReconnectAttempts; // Detener reintentos
            return;
          }
        } catch (checkErr) {
          console.error('[WS] Error verificando expiración:', checkErr);
        }
        
        // Fallback: intentar conectar con token actual solo si no está expirado
        if (this.accessToken) {
          this.connect(this.accessToken);
        }
      }
    }, delay);
  }

  /**
   * Desconectar manualmente
   */
  disconnect() {
    
    this.isIntentionallyClosed = true;
    
    // Cancelar reconexión
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
      this.reconnectTimer = null;
    }
    
    // Detener ping
    this.stopPingInterval();
    
    // Cerrar WebSocket
    if (this.ws) {
      this.ws.close(1000, 'Disconnect requested');
      this.ws = null;
    }
    
    this.reconnectAttempts = 0;
    this.isConnecting = false;
    this.accessToken = null;
  }

  /**
   * Verifica si está conectado
   * 
   * @returns {boolean}
   */
  isConnected() {
    return this.ws && this.ws.readyState === WebSocket.OPEN;
  }
}

// ============================================
// INSTANCIA GLOBAL
// ============================================

let realtimeClient = null;

/**
 * Inicializa el cliente WebSocket
 * 
 * @param {string} accessToken - JWT access token
 * @returns {RealtimeClient}
 */
export function initRealtimeClient(accessToken) {
  // Desconectar cliente anterior si existe
  if (realtimeClient) {
    realtimeClient.disconnect();
  }
  
  realtimeClient = new RealtimeClient();
  realtimeClient.connect(accessToken);
  return realtimeClient;
}

/**
 * Desconecta el cliente WebSocket
 */
export function disconnectRealtimeClient() {
  if (realtimeClient) {
    realtimeClient.disconnect();
  }
}

/**
 * Obtiene la instancia del cliente
 * 
 * @returns {RealtimeClient|null}
 */
export function getRealtimeClient() {
  return realtimeClient;
}
