/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/handlers/publish-process-next.js — Rol: Handler PUBLISH.PROCESS_NEXT (renovación de anuncios)
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

// ✅ v10.5.115: Flag DEBUG controlado por debugMode en storage
let DEBUG_MODE = false;
chrome.storage.local.get(['debugMode'], (r) => { DEBUG_MODE = !!r.debugMode; });
chrome.storage.onChanged.addListener((changes) => {
  if (changes.debugMode) DEBUG_MODE = !!changes.debugMode.newValue;
});
const dlog = (...args) => DEBUG_MODE && console.log(...args);
const dwarn = (...args) => DEBUG_MODE && console.warn(...args);

// ✅ v2.2.0: Verificar si está pausado y esperar hasta que se reanude
async function checkPauseAndWait() {
  let wasPaused = false;
  while (true) {
    const result = await chrome.storage.local.get(['publish.paused']);
    if (!result['publish.paused']) {
      if (wasPaused) {
        console.log('[PUBLISH] ▶️ Reanudado');
      }
      return; // No está pausado, continuar
    }
    
    if (!wasPaused) {
      console.log('[PUBLISH] ⏸️ Pausado - esperando...');
      wasPaused = true;
    }
    
    // Esperar 500ms y volver a verificar
    await new Promise(r => setTimeout(r, 500));
  }
}

// Imports necesarios desde otros módulos del SW
import { apiFetch, session, loadSession, CONFIG } from '../api_client.js';
import { ensureWallaCreds, getTokenDiagnostics, getTokenRealAge, refreshWallaCredsIfOld } from '../walla.js';
import { extractTitle } from '../utils.js';
import { 
  publishState,
  savePendingPublish,
  getPendingPublish,
  clearPendingPublish
} from '../state.js';  // ✅ v10.5.35: Estado persistente para delays largos
import { 
  SW_TIMEOUTS, 
  SW_DELAYS, 
  SW_RETRY, 
  CS_MESSAGES, 
  SW_ERROR_CODES,
  calculateBackoffDelay,
  humanDelay,
  getCSTimeout
} from '../constants.js';

/**
 * ✅ v10.5.53: Enviar mensaje de paso al panel
 * @param {string} message - Mensaje a mostrar (ej: "Seleccionando categoría...")
 * @param {number} step - Número de paso (1-7) para color progresivo
 */
function sendStepMessage(message, step = 1) {
  try {
    chrome.runtime.sendMessage({
      type: 'PUBLISH.STEP_MESSAGE',
      message,
      step
    }).catch(() => {}); // Ignorar error si panel cerrado
  } catch (e) {
    // Panel no disponible, ignorar
  }
}

/**
 * ✅ v10.5.130: Refresh inteligente de token con feedback visual
 * Solo refresca si el token tiene menos tiempo restante del necesario
 * 
 * @param {number} tabId - ID de la pestaña de Wallapop
 * @param {number} minRemainingSeconds - Segundos mínimos que debe tener el token
 * @param {string} feedbackMessage - Mensaje a mostrar si hay refresh (opcional)
 * @param {number} step - Paso actual para el color del mensaje
 * @returns {Promise<{ok: boolean, refreshed: boolean, creds: object}>}
 */
async function smartRefreshToken(tabId, minRemainingSeconds = 120, feedbackMessage = null, step = 1) {
  const diag = getTokenDiagnostics();
  const remaining = diag.realAge?.remainingSeconds;
  
  // Si el token tiene suficiente tiempo, no hacer nada
  if (remaining !== null && remaining > minRemainingSeconds) {
    dlog(`[TOKEN] ✅ Token OK (${diag.realAge.remainingFormatted} restante, necesita ${minRemainingSeconds}s)`);
    return { ok: true, refreshed: false, creds: { token: true } }; // creds simplificado, no se usa
  }
  
  // Token necesita refresh - mostrar feedback si se proporcionó mensaje
  if (feedbackMessage) {
    sendStepMessage(feedbackMessage, step);
  }
  dlog(`[TOKEN] 🔄 Refrescando... (${remaining !== null ? diag.realAge.remainingFormatted : 'sin token'} restante, necesita ${minRemainingSeconds}s)`);
  
  const result = await refreshWallaCredsIfOld(tabId, 90000, true); // forceRefresh = true
  
  if (result.refreshed) {
    const newDiag = getTokenDiagnostics();
    dlog(`[TOKEN] ✅ Refrescado! Nuevo restante: ${newDiag.realAge.remainingFormatted}`);
  } else if (!result.ok) {
    // ✅ v2.1.1: Mensaje claro de sesión expirada
    console.error('[TOKEN] ❌ Sesión de Wallapop expirada - Refresca la página de Wallapop');
  }
  
  return result;
}

/**
 * ✅ v10.5.130: Lanzar refresh en background durante countdown
 * Retorna una promesa que se puede esperar después del countdown
 * 
 * @param {number} tabId - ID de la pestaña
 * @param {number} delaySeconds - Duración del countdown
 * @returns {Promise} - Promesa del refresh (para await después)
 */
function startBackgroundRefresh(tabId, delaySeconds) {
  const diag = getTokenDiagnostics();
  const remaining = diag.realAge?.remainingSeconds || 0;
  
  // Solo lanzar refresh si el token expirará durante o justo después del delay
  // Margen de 60s para que el token esté fresco cuando termine el countdown
  if (remaining > delaySeconds + 60) {
    dlog(`[TOKEN] ⏳ Token durará todo el delay (${diag.realAge.remainingFormatted} > ${delaySeconds}s + 60s margen)`);
    return Promise.resolve({ ok: true, refreshed: false, background: true });
  }
  
  dlog(`[TOKEN] 🔄 Iniciando refresh en background (token: ${diag.realAge.remainingFormatted}, delay: ${delaySeconds}s)`);
  
  // Lanzar refresh sin await - se ejecuta en paralelo
  return refreshWallaCredsIfOld(tabId, 90000, true).then(result => {
    if (result.refreshed) {
      const newDiag = getTokenDiagnostics();
      dlog(`[TOKEN] ✅ Background refresh OK! Nuevo: ${newDiag.realAge.remainingFormatted}`);
    }
    return { ...result, background: true };
  }).catch(e => {
    dwarn('[TOKEN] ⚠️ Background refresh error:', e.message);
    return { ok: false, refreshed: false, background: true, error: e.message };
  });
}

/**
 * Crea un error estructurado para guardar en publish_error
 * @param {string} errorType - Tipo de error
 * @param {string} message - Mensaje descriptivo
 * @param {Object} details - Detalles del error
 * @param {Object} context - Contexto (item, account)
 * @param {string} previousError - Error anterior (JSON string) para preservar flags
 * @returns {string} JSON stringificado
 */
function createPublishError(errorType, message, details = {}, context = {}, previousError = null) {
  // ✅ v6.0.0: Preservar flags críticos del error anterior
  let preservedFlags = {};
  if (previousError) {
    try {
      const prevObj = typeof previousError === 'string' ? JSON.parse(previousError) : previousError;
      // Preservar deleted_pending_publish si existía (como campo o como error_type)
      if (prevObj.deleted_pending_publish === true || prevObj.error_type === 'deleted_pending_publish') {
        preservedFlags.deleted_pending_publish = true;
      }
    } catch (e) {
      // Si no es JSON válido, ignorar
    }
  }

  return JSON.stringify({
    timestamp: new Date().toISOString(),
    error_type: errorType,
    error_message: message,
    ...preservedFlags, // ← Incluir flags preservados al nivel raíz
    details: {
      ...details,
      step: details.step || 'process_next'
    },
    item: {
      internal_id: context.itemId || null,
      wallapop_id: context.wallapopId || null,
      title: context.title || null
    },
    account: {
      walla_account_id: context.accountId || null
    }
  });
}

/**
 * ✅ v6.5.4: Helper para actualizar estado de error de un item
 * Simplifica la actualización de errores con estructura consistente
 * 
 * @param {string} itemId - ID del item
 * @param {string} errorType - Tipo de error (ej: 'cs_not_responding')
 * @param {string} message - Mensaje descriptivo
 * @param {Object} details - Detalles adicionales
 * @param {Object} context - Contexto del item
 * @param {string} previousError - Error anterior para preservar flags
 * @returns {Promise<void>}
 */
async function updateItemError(itemId, errorType, message, details = {}, context = {}, previousError = null) {
  const structuredError = createPublishError(errorType, message, details, context, previousError);
  
  await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
    method: 'PATCH',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ 
      publish_status: 'error_retry', 
      publish_error: structuredError 
    })
  });
  
  
  // ✅ v6.6.8: Enviar status error_retry al panel para el monitor (sync con BD)
  try {
    chrome.runtime.sendMessage({
      type: 'PUBLISH_ITEM_UPDATE',
      data: {
        id: itemId,
        status: 'error_retry',
        error: message || errorType
      }
    }).catch(() => {});
  } catch (e) {
  }
}

/**
 * Handler principal: PUBLISH.PROCESS_NEXT
 * 
 * Este handler maneja el proceso completo de renovación/republicación de un anuncio:
 * 1. Obtiene el siguiente item pendiente de la cola
 * 2. Verifica que sea una categoría soportada (no Coches/Inmobiliaria)
 * 3. Marca el item como 'processing'
 * 4. Descuenta el crédito del usuario
 * 5. Elimina el anuncio viejo de Wallapop (si aplica)
 * 6. Espera delay configurable
 * 7. Publica el anuncio nuevo
 * 8. Captura y valida el nuevo ID
 * 9. Actualiza la BD con el estado final
 * 
 * @param {Object} msg - Mensaje recibido con { type, account_id }
 * @param {Object} sender - Información del remitente
 * @param {Function} sendResponse - Función para enviar respuesta
 */
export async function handleProcessNext(msg, sender, sendResponse) {
            let item = null; // ← Declarar aquí para que esté disponible en catch
            const accountId = msg.account_id; // ✅ v2.6.0: Mover fuera del try para catch
            
            try {
              if (!accountId) {
                sendResponse({ ok: false, error: 'account_id requerido' });
                return;
              }


              // ✅ v7.2.26: Obtener delay configurado AL INICIO para calcular timeout dinámico
              let userDelaySeconds = 30;
              try {
                const prefsRes = await apiFetch('/api/user/preferences', { method: 'GET' });
                if (prefsRes.ok) {
                  const prefsData = await prefsRes.json();
                  userDelaySeconds = prefsData?.ui_preferences?.publish_delay_seconds ?? 30;
                }
              } catch (e) {
              }
              
              // ✅ v7.2.26: Timeout dinámico = delay_usuario + 240s (4min margen)
              const dynamicTimeout = (userDelaySeconds + 240) * 1000; // Convertir a ms

              // 1. Obtener siguiente item pendiente
              const nextRes = await apiFetch(`/api/walla/publish2/next-item/${accountId}`, {
                method: 'GET',
                cache: 'no-store'
              });

              if (!nextRes.ok) {
                sendResponse({ ok: false, error: `HTTP ${nextRes.status}` });
                return;
              }

              const nextData = await nextRes.json();

              if (!nextData.has_next) {
                // No hay más items, proceso terminado
                
                // ✅ v2.6.0: Notificar fin de proceso (proceso completado)
                try {
                  await apiFetch('/api/walla/publish2/session/end-delay', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                      account_id: accountId,
                      reason: 'completed'
                    })
                  });
                } catch (endErr) {
                  console.error('[SW] ⚠️ Error notificando fin de proceso:', endErr);
                }
                
                // ✅ v4.84.0: Obtener stats SIN limpiar (backend ya no limpia automáticamente)
                let stats = { total: 0, published: 0, errors: 0 };
                try {
                  const statsRes = await apiFetch(`/api/walla/publish2/stats/${accountId}`);
                  
                  if (!statsRes.ok) {
                    console.error('[SW] ❌ Stats response no OK:', statsRes.status);
                  }
                  
                  stats = await statsRes.json();
                } catch (statsErr) {
                  console.error('[SW] ❌ Error obteniendo estadísticas:', statsErr);
                }
                
                // Enviar mensaje al panel para mostrar modal de éxito
                const statsToSend = {
                  total: stats.total || 0,
                  published: stats.published || 0,
                  errors: stats.errors || 0
                };
                
                
                try {
                  chrome.runtime.sendMessage({
                    type: 'PUBLISH.SHOW_SUCCESS_MODAL',
                    stats: statsToSend,
                    accountId: accountId  // ✅ v4.84.0: Enviar accountId para cleanup posterior
                  });
                  
                } catch (msgErr) {
                  console.error('[SW] ⚠️ Error enviando modal:', msgErr);
                }
                
                // ✅ v4.84.0: NO llamar a cleanup aquí - se hará después del modal
                
                sendResponse({ ok: true, done: true, has_next: false });
                return;
              }

              item = nextData.item; // ← Asignar (no declarar)
              const snapshot = item.snapshot || (item.json_data ? JSON.parse(item.json_data) : {});
              const oldId = item.id_wallapop || snapshot.id;
              const itemStatus = item.status || 'queued_delete';
              

              // ✅ v4.82.1: 2. Verificar que no sea Coches/Inmuebles (no soportados)
              const firstTaxonomyName = snapshot?.taxonomy?.[0]?.name || '';
              const taxonomyNorm = firstTaxonomyName.toLowerCase()
                .normalize('NFD').replace(/\p{Diacritic}+/gu,'').trim();
              
              if (taxonomyNorm === 'coches' || taxonomyNorm === 'inmobiliaria') {
                const verticalName = taxonomyNorm === 'coches' ? 'Coches' : 'Inmobiliaria';
                
                // Marcar como error y quitar de lote
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({
                    publish_status: null,  // Limpiar
                    publish_idx: null,     // Quitar de lote
                    publish_error: JSON.stringify({
                      type: 'unsupported_vertical',
                      vertical: taxonomyNorm,
                      message: `El tipo de anuncio "${firstTaxonomyName}" no está soportado. Este anuncio ha sido omitido.`
                    })
                  })
                });
                
                // Continuar con siguiente item
                sendResponse({ ok: true, skipped: true, reason: 'unsupported_vertical', item_id: item.id });
                return;
              }

              // Detectar si ya fue eliminado (basado en publish_error del item)
              const publishError = item.publish_error || '';
              const isDeletedPending = publishError.includes('deleted_pending_publish') || 
                                       publishError.includes('Anuncio eliminado') ||
                                       publishError.includes('pendiente republicar');
              const skipDelete = isDeletedPending;
              if (skipDelete) {
              }

              // 3. Marcar como 'processing'
              const processingPayload = { publish_status: 'processing' };
              
              const processingRes = await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(processingPayload)
              });

              if (!processingRes.ok) {
                const errorText = await processingRes.text().catch(() => 'No body');
                console.error('[SW] ❌ No se pudo marcar como processing:', {
                  status: processingRes.status,
                  statusText: processingRes.statusText,
                  body: errorText,
                  itemId: item.id
                });
                sendResponse({ ok: false, error: `No se pudo marcar como processing: ${errorText}` });
                return;
              }
              
              // ✅ v6.6.0: Enviar datos del item al panel para el monitor
              try {
                const itemTitle = extractTitle(snapshot?.title, item.title || 'Sin título');
                const itemPrice = snapshot?.price?.cash?.amount || snapshot?.price?.amount || item.price_amount || 0;
                const firstImage = snapshot?.images?.[0];
                let itemImageUrl = '';
                
                if (firstImage) {
                  const localUrl = firstImage.local_url || firstImage.file_url;
                  if (localUrl) {
                    // ✅ v10.5.33: Usar cfg.API_BASE (fuente única de verdad)
                    const apiBase = await chrome.storage.local.get('cfg.API_BASE').then(d => d['cfg.API_BASE'] || 'https://www.mitiklive.com/fa');
                    itemImageUrl = apiBase.replace(/\/$/, '') + (localUrl.startsWith('/') ? localUrl : '/' + localUrl);
                  } else {
                    itemImageUrl = firstImage.urls?.small || firstImage.url || '';
                  }
                }
                
                chrome.runtime.sendMessage({
                  type: 'PUBLISH_ITEM_UPDATE',
                  data: {
                    id: item.id,
                    title: itemTitle,
                    price: itemPrice,
                    image_url: itemImageUrl,
                    status: 'processing',
                    error: null
                  }
                }).catch(() => {}); // Ignorar error si panel no está escuchando
                
              } catch (e) {
              }

              // 🆕 DESCONTAR CRÉDITO AL INICIO (antes de hacer nada)
              // Si falla después, NO se devuelve → el usuario "consumió" el servicio
              try {
                
                let userId = session?.user?.id;
                
                if (!userId) {
                  console.error('[SW] ❌❌❌ NO SE PUDO OBTENER user_id - Session:', JSON.stringify(session, null, 2));
                  // Intentar cargar session de nuevo
                  await loadSession();
                  userId = session?.user?.id;  // ✅ Actualizar userId
                  
                  if (!userId) {
                    console.error('[SW] ❌ CRÍTICO: No hay user_id después de recargar session');
                    // Continuar SIN descontar (mejor que bloquear)
                  }
                }
                
                if (userId) {  // ✅ v4.82.5: Usar userId (puede venir de retry)
                  
                  const deductRes = await apiFetch('/api/billing/deduct-credit', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                      user_id: userId,
                      walla_account_id: accountId,
                      internal_listing_id: item.id  // Usar ID de BD para idempotencia
                    })
                  });
                  
                  
                  if (!deductRes.ok) {
                    const errorData = await deductRes.json().catch(() => ({}));
                    console.error('[SW] ❌ Error descontando crédito:', errorData);
                    
                    // Si no tiene créditos, detener el proceso
                    if (deductRes.status === 402) {
                      await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                        method: 'PATCH',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ 
                          publish_status: 'error_retry',  // ✅ v4.82.4: Corregido
                          publish_error: 'Sin créditos disponibles' 
                        })
                      });
                      sendResponse({ 
                        ok: false, 
                        error: 'Sin créditos disponibles. Recarga en mitiklive.com' 
                      });
                      return;
                    }
                    
                    // Para otros errores, continuar pero logear
                  } else {
                    const deductData = await deductRes.json();
                    
                    // Notificar al panel para actualizar contador
                    try {
                      chrome.runtime.sendMessage({
                        type: 'CREDIT_DEDUCTED',
                        payload: {
                          credits_remaining: deductData.credits_remaining,
                          publications_used: deductData.publications_used
                        }
                      }).catch(() => {
                      });
                    } catch (e) {
                    }
                  }
                }
              } catch (creditErr) {
                console.error('[SW] ❌ Error crítico descontando crédito:', creditErr);
                // Continuar con la publicación de todas formas
              }

              // 3. Buscar pestaña Wallapop
              const tabs = await chrome.tabs.query({ url: '*://*.wallapop.com/*' });
              
              if (!tabs.length) {
                console.error('[SW] ❌ No hay pestaña de Wallapop abierta');
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ 
                    publish_status: 'error_retry', 
                    publish_error: createPublishError(
                      'no_wallapop_tab',
                      'No hay pestaña de Wallapop abierta',
                      { step: 'find_tab' },
                      { 
                        itemId: item.id, 
                        wallapopId: oldId, 
                        title: extractTitle(snapshot?.title),
                        accountId 
                      },
                      item.publish_error // ← Preservar flags del error anterior
                    )
                  })
                });
                sendResponse({ ok: false, error: 'Abre una pestaña de Wallapop' });
                return;
              }

              const tab = tabs[0];

              // ✅ v4.82.3: Helper mejorado para detectar pestaña cerrada
              // ✅ v7.2.26: Usa timeout dinámico basado en delay del usuario
              const sendToTab = async (message, timeout = dynamicTimeout) => {
                // Verificar que la pestaña siga activa
                try {
                  await chrome.tabs.get(tab.id);
                } catch (e) {
                  throw new Error('La pestaña de Wallapop se cerró');
                }
                
                return new Promise((resolve, reject) => {
                  const timer = setTimeout(() => {
                    reject(new Error(`Timeout después de ${timeout}ms`));
                  }, timeout);
                  
                  chrome.tabs.sendMessage(tab.id, message, (response) => {
                    clearTimeout(timer);
                    
                    if (chrome.runtime.lastError) {
                      reject(new Error(chrome.runtime.lastError.message));
                    } else {
                      resolve(response);
                    }
                  });
                });
              };

              // Helper para verificar que CS está vivo
              const ensureCS = async () => {
                try {
                  const response = await sendToTab({ type: 'SYS.PING' }, 3000);
                  return response?.ok === true;
                } catch {
                  return false;
                }
              };

              // ✅ v2.2.0: Verificar pausa antes de navegar
              await checkPauseAndWait();

              // 4. Navegar a /upload y refrescar SOLO si hay datos previos
              // ✅ v6.6.7: NAVEGACIÓN ROBUSTA - Maneja todos los edge cases
              const currentUrl = tab.url || '';
              const isOnUpload = /\/app\/catalog\/upload(?:\/|$)/.test(currentUrl);
              
              
              // Helper: Verificar que tab sigue activo
              const isTabAlive = async () => {
                try {
                  await chrome.tabs.get(tab.id);
                  return true;
                } catch (e) {
                  console.error('[SW] ❌ Tab cerrado');
                  return false;
                }
              };
              
              // Verificar tab antes de cualquier operación
              if (!(await isTabAlive())) {
                await updateItemError(
                  item.id,
                  'tab_closed',
                  'La pestaña de Wallapop se cerró',
                  { step: 'check_tab' },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'La pestaña de Wallapop se cerró' });
                return;
              }
              
              // 5. Verificar CS
              let csReady = await ensureCS();
              if (!csReady) {
                try {
                  await chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    files: ['content_script.js']
                  });
                  await new Promise(r => setTimeout(r, 1000));
                  csReady = await ensureCS();
                } catch (e) {
                  console.error('[SW] ❌ Error reinyectando CS:', e);
                }
              }
              
              if (!csReady) {
                await updateItemError(
                  item.id,
                  'cs_not_responding',
                  'Content script no responde',
                  { step: 'verify_cs_initial', attempts: 2 },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'Content script no responde' });
                return;
              }
              
              // ✅ v7.2.24: Detectar si es reanudación (hay más items después de este)
              // Si es reanudación, limpiar formulario SIEMPRE
              let isResume = false;
              try {
                const statusRes = await apiFetch(`/api/walla/publish2/status?account_id=${accountId}`, {
                  method: 'GET',
                  cache: 'no-store'
                });
                if (statusRes.ok) {
                  const statusData = await statusRes.json();
                  // Es reanudación si hay más de 1 pendiente (este + otros)
                  isResume = (statusData.pending || 0) > 1 || (statusData.processing || 0) > 0 || (statusData.errors || 0) > 0;
                }
              } catch (e) {
              }
              
              if (isResume) {
                try {
                  // ✅ v10.5.44: Permitir navegación interna sin aviso
                  await chrome.tabs.sendMessage(tab.id, { type: 'PUBLISH.ALLOW_NAVIGATION' }).catch(() => {});
                  
                  await chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    func: () => {
                      location.href = 'https://es.wallapop.com/app/catalog/upload';
                    }
                  });
                  
                  // Esperar navegación
                  await new Promise(resolve => {
                    const listener = (tabId, info) => {
                      if (tabId === tab.id && info.status === 'complete') {
                        chrome.tabs.onUpdated.removeListener(listener);
                        resolve();
                      }
                    };
                    chrome.tabs.onUpdated.addListener(listener);
                    setTimeout(resolve, 10000);
                  });
                  
                  
                  // ✅ v7.2.25: Verificar CS de nuevo después de navegar
                  csReady = await ensureCS();
                  if (!csReady) {
                    try {
                      await chrome.scripting.executeScript({
                        target: { tabId: tab.id },
                        files: ['content_script.js']
                      });
                      await new Promise(r => setTimeout(r, 1000));
                      csReady = await ensureCS();
                    } catch (e) {
                      console.error('[SW] ❌ Error reinyectando CS:', e);
                    }
                  }
                  
                  if (!csReady) {
                    await updateItemError(
                      item.id,
                      'cs_not_responding_after_nav',
                      'Content script no responde después de navegar',
                      { step: 'verify_cs_after_navigation' },
                      { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                    );
                    sendResponse({ ok: false, error: 'CS no responde después de navegar' });
                    return;
                  }
                  
                  
                } catch (e) {
                }
              }

              // ✅ v6.6.4: VERIFICACIÓN INTELIGENTE - Esperar a que la página de upload esté realmente lista
              // Esto es crítico después de publicar cuando Wallapop muestra modal y navega automáticamente
              sendStepMessage('Esperando formulario...', 1);
              try {
                const uploadReadyRes = await sendToTab({
                  type: 'PUBLISH.WAIT_UPLOAD_READY'
                }, 30000); // ✅ v6.6.5: Aumentado a 30s (CS tiene 25s interno + margen)
                
                if (!uploadReadyRes?.ok) {
                  console.error('[SW] ❌ Página de upload no está lista:', uploadReadyRes);
                  await updateItemError(
                    item.id,
                    'upload_page_not_ready',
                    `Página de upload no se cargó: ${uploadReadyRes?.error || 'timeout'}`,
                    { 
                      step: 'wait_upload_ready', 
                      response: uploadReadyRes,
                      elapsed_ms: uploadReadyRes?.elapsed_ms 
                    },
                    { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                  );
                  sendResponse({ ok: false, error: `Página de upload no está lista: ${uploadReadyRes?.error || 'timeout'}` });
                  return;
                }
              } catch (waitErr) {
                console.error('[SW] ❌ Error esperando página de upload:', waitErr);
                await updateItemError(
                  item.id,
                  'upload_wait_timeout',
                  'Timeout esperando página de upload (30s)',
                  { step: 'wait_upload_ready', error: waitErr.message },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'Timeout esperando página de upload' });
                return;
              }

              // 6. Seleccionar vertical
              sendStepMessage('Seleccionando categoría...', 2);
              const verticalRes = await sendToTab({
                type: 'PUBLISH.SELECT_VERTICAL',
                snapshot
              }, 20000);
              
              
              if (!verticalRes?.ok) {
                console.error('[SW] ❌ Error en vertical:', verticalRes?.error || verticalRes?.code || 'Sin respuesta');
                await updateItemError(
                  item.id,
                  'vertical_selection_failed',
                  `Error seleccionando vertical: ${verticalRes?.error || verticalRes?.code || 'Sin respuesta'}`,
                  { 
                    step: 'select_vertical', 
                    code: verticalRes?.code,
                    response: verticalRes
                  },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: `Error seleccionando vertical: ${verticalRes?.error || verticalRes?.code || 'Sin respuesta'}` });
                return;
              }
              
              // ✅ v10.4.0: Delay después de seleccionar vertical para dar tiempo al DOM
              await new Promise(r => setTimeout(r, humanDelay(800, 0.3))); // 560-1040ms

              // 7. Verificar CS después de vertical
              if (!(await ensureCS())) {
                await updateItemError(
                  item.id,
                  'cs_not_responding',
                  'CS no responde después de vertical',
                  { step: 'verify_cs_after_vertical' },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'CS no responde después de vertical' });
                return;
              }

              // 8. Rellenar campos básicos (summary) - con retry si falla
              sendStepMessage('Rellenando título y descripción...', 3);
              let summaryRes = await sendToTab({
                type: 'PUBLISH.FILL_BASIC_FIELDS',
                snapshot
              }, 25000);  // ✅ v10.4.0: Aumentado a 25s (fillSummary espera 15s interno)
              
              // ✅ v10.4.0: Si falla, reintentar una vez con reload
              if (!summaryRes?.ok) {
                const errorCode = summaryRes?.code || summaryRes?.fields?.summary?.code || 'unknown';
                console.warn(`[SW] ⚠️ fillSummary falló (${errorCode}), reintentando con reload...`);
                
                // Reload y re-seleccionar vertical
                try {
                  // ✅ v10.5.44: Permitir navegación interna sin aviso
                  await chrome.tabs.sendMessage(tab.id, { type: 'PUBLISH.ALLOW_NAVIGATION' }).catch(() => {});
                  
                  await chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    func: () => { location.href = 'https://es.wallapop.com/app/catalog/upload'; }
                  });
                  
                  await new Promise(resolve => {
                    const listener = (tabId, info) => {
                      if (tabId === tab.id && info.status === 'complete') {
                        chrome.tabs.onUpdated.removeListener(listener);
                        resolve();
                      }
                    };
                    chrome.tabs.onUpdated.addListener(listener);
                    setTimeout(resolve, 12000);
                  });
                  
                  // Esperar página lista
                  await sendToTab({ type: 'PUBLISH.WAIT_UPLOAD_READY' }, 30000);
                  
                  // Re-seleccionar vertical
                  const retryVerticalRes = await sendToTab({
                    type: 'PUBLISH.SELECT_VERTICAL',
                    snapshot
                  }, 20000);
                  
                  if (retryVerticalRes?.ok) {
                    await new Promise(r => setTimeout(r, humanDelay(1000, 0.3)));
                    
                    // Reintentar fillSummary
                    summaryRes = await sendToTab({
                      type: 'PUBLISH.FILL_BASIC_FIELDS',
                      snapshot
                    }, 25000);
                  }
                } catch (retryErr) {
                  console.error('[SW] ❌ Error en retry de fillSummary:', retryErr);
                }
              }
              
              if (!summaryRes?.ok) {
                const errorCode = summaryRes?.code || summaryRes?.fields?.summary?.code || 'unknown';
                const errorDetail = summaryRes?.error || summaryRes?.fields?.summary?.code || 'Sin detalle';
                await updateItemError(
                  item.id,
                  'summary_fill_failed',
                  `Error rellenando summary: ${errorDetail}`,
                  { step: 'fill_summary', code: errorCode, response: summaryRes },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: `Error rellenando summary: ${errorDetail}` });
                return;
              }
              
              // ✅ v6.5.4: Delay humano con varianza
              await new Promise(r => setTimeout(r, humanDelay(SW_DELAYS.BETWEEN_ITEMS, 0.3)));

              // 9. Click continuar (summary → fotos)
              const continue1Res = await sendToTab({
                type: CS_MESSAGES.CLICK_CONTINUE,
                which: 'title',
                delay: humanDelay(SW_DELAYS.BETWEEN_STEPS, 0.3), // ✅ v6.5.4: Varianza ±30%
                timeout: SW_TIMEOUTS.CS_SELECT_VERTICAL
              }, SW_TIMEOUTS.CS_DEFAULT);
              
              if (!continue1Res?.ok) {
                await updateItemError(
                  item.id,
                  'click_continue_failed',
                  'Error en click continuar (título → fotos)',
                  { step: 'click_continue_title', response: continue1Res },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'Error en click continuar' });
                return;
              }

              // 11. Verificar CS antes de imágenes (sin delay fijo)
              // ✅ v6.5.4: CS esperará el DOM inteligentemente
              if (!(await ensureCS())) {
                await updateItemError(
                  item.id,
                  'cs_not_responding',
                  'CS no responde antes de imágenes',
                  { step: 'verify_cs_before_images' },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'CS no responde antes de imágenes' });
                return;
              }

              // ✅ v2.2.0: Verificar pausa antes de subir imágenes
              await checkPauseAndWait();

              // 12. Subir imágenes
              sendStepMessage('Subiendo imágenes...', 4);
              const imagesRes = await sendToTab({
                type: CS_MESSAGES.UPLOAD_IMAGES,
                snapshot,
                apiBase: CONFIG.API_BASE || ''
              }, SW_TIMEOUTS.CS_UPLOAD_IMAGES);
              
              if (!imagesRes?.ok) {
                await updateItemError(
                  item.id,
                  'image_upload_failed',
                  'Error subiendo imágenes',
                  { 
                    step: 'upload_images', 
                    response: imagesRes,
                    imageCount: snapshot?.images?.length || 0
                  },
                  { itemId: item.id, wallapopId: item.wallapop_id, title: extractTitle(snapshot?.title), accountId: accountId }
                );
                sendResponse({ ok: false, error: 'Error subiendo imágenes' });
                return;
              }

              // 13. Click continuar (fotos → next)
              let continue2Res = null;
              try {
                continue2Res = await sendToTab({
                  type: CS_MESSAGES.CLICK_CONTINUE,
                  which: 'photo',
                  delay: humanDelay(SW_DELAYS.BEFORE_FILL_FORM, 0.3), // ✅ v6.5.4: Varianza ±30%
                  timeout: SW_TIMEOUTS.CS_CLICK_CONTINUE
                }, SW_TIMEOUTS.CS_DEFAULT);
              } catch (e) {
                continue2Res = { ok: false, code: 'timeout' };
              }

              // 14. Seleccionar categoría (sin delay fijo)
              // ✅ v6.5.4: CS esperará el DOM inteligentemente
              if (snapshot.taxonomy?.length) {
                sendStepMessage('Seleccionando categoría principal...', 5);
                await sendToTab({
                  type: 'PUBLISH.SELECT_CATEGORY',
                  taxonomy: snapshot.taxonomy
                }, 20000);
              }

              // ✅ v2.5.2: Deshabilitar botón Wallapop AQUÍ (después de categorías, antes de rellenar)
              // En este momento el formulario completo ya está visible y el botón existe
              try {
                await chrome.tabs.sendMessage(tab.id, { 
                  type: 'WALLAPOP.DISABLE_BUTTON',
                  timeout: 8000  // Más tiempo porque el form tarda en renderizar
                });
                dlog('[SW] ✅ Botón Wallapop deshabilitado (después de categorías)');
              } catch (e) {
                // No es crítico, continuar
                dlog('[SW] ⚠️ No se pudo deshabilitar botón:', e.message);
              }

              // 15. Escanear formulario (sin delay fijo)
              // ✅ v6.5.4: CS esperará el DOM inteligentemente
              if (!(await ensureCS())) {
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ publish_status: 'error_retry', publish_error: 'CS no responde antes de escanear' })
                });
                sendResponse({ ok: false, error: 'CS no responde antes de escanear' });
                return;
              }
              
              await sendToTab({ type: 'FORM.SCAN_WIDGETS' }, 8000);

              // 16. Rellenar formulario completo (sin delay fijo)
              // ✅ v6.5.4: CS esperará el DOM inteligentemente
              if (!(await ensureCS())) {
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ publish_status: 'error_retry', publish_error: 'CS no responde antes de rellenar' })
                });
                sendResponse({ ok: false, error: 'CS no responde antes de rellenar' });
                return;
              }
              
              
              const fillStartTime = Date.now();
              sendStepMessage('Rellenando formulario...', 6);
              const fillResult = await sendToTab({
                type: 'FORM.FILL_WIDGETS',
                snapshot
              }, 45000); // ✅ v5.x: Aumentado a 45s para Brand/Model/Storage con esperas AJAX
              
              const fillElapsed = Date.now() - fillStartTime;
              

              // ✅ v4.81.0: MEJORADO - Procesar errores detallados
              if (!fillResult?.ok) {
                console.error('%c[SW] ❌ LLENADO DE FORMULARIO FALLIDO', 'color: white; background: #dc3545; font-weight: bold; font-size: 14px; padding: 6px;');
                console.error('[SW] ⏱️ Tiempo transcurrido:', fillElapsed, 'ms');
                console.error('[SW] 📊 fillResult:', fillResult);
                
                // 🔍 DIAGNÓSTICO: Detectar si fue timeout
                if (fillElapsed >= 44000) { // Más de 44 segundos = timeout de 45s
                  console.error('%c[SW] ⏰ TIMEOUT DETECTADO', 'color: white; background: #ff0000; font-weight: bold; font-size: 16px; padding: 8px;');
                  console.error('[SW] El content script no respondió en 45 segundos');
                  console.error('[SW] Posibles causas:');
                  console.error('  1. Se quedó atascado en un campo (probablemente Condición)');
                  console.error('  2. El selector cambió y no encuentra el elemento');
                  console.error('  3. El dropdown no abre correctamente');
                  console.error('[SW] 💡 Revisar logs del content script en la pestaña de Wallapop');
                }
                
                // ✅ v4.84.6: DEBUGGING - Mostrar campos problemáticos
                if (fillResult?.missingRequired && fillResult.missingRequired.length > 0) {
                  console.error('[SW] 📋 CAMPOS REQUERIDOS FALTANTES:', fillResult.missingRequired);
                  fillResult.missingRequired.forEach((field, idx) => {
                    console.error(`[SW] ❌ Campo ${idx + 1}:`, {
                      field: field.name || field,        // ← v4.85.1: Puede ser string o objeto
                      reason: field.reason || '(sin razón)',
                      selector: field.selector,
                      label: field.label,
                      jsonValue: field.jsonValue
                    });
                  });
                }
                if (fillResult?.skipped && fillResult.skipped.length > 0) {
                  fillResult.skipped.forEach((field, idx) => {
                  });
                }
                
                // Construir mensaje de error detallado
                let errorMessage = 'Formulario incompleto';
                let errorDetails = {};
                
                // ✅ NUEVO: Contexto de cuenta e item para diagnóstico
                // Extraer primera imagen - PRIORIDAD: local_url (servidor) > CDN Wallapop
                const apiBase = CONFIG.API_BASE || '';
                const buildImageUrl = (img) => {
                  if (!img) return null;
                  const localUrl = img.local_url || img.file_url;
                  if (localUrl) {
                    // Construir URL absoluta del servidor
                    const base = String(apiBase).replace(/\/$/, '');
                    return base + (localUrl.startsWith('/') ? localUrl : '/' + localUrl);
                  }
                  // Fallback a CDN de Wallapop
                  return img.urls?.small || img.urls?.medium || img.url || null;
                };
                
                const firstImage = buildImageUrl(snapshot?.images?.[0]);
                
                const errorContext = {
                  timestamp: new Date().toISOString(),
                  item: {
                    internal_id: item.id,
                    wallapop_id: oldId,
                    title: extractTitle(snapshot?.title, item.title || 'Sin título'),
                    image_url: firstImage // ✅ URL local del servidor
                  },
                  account: {
                    walla_account_id: accountId
                  }
                };
                
                if (fillResult?.criticalError) {
                  // Error crítico específico
                  const err = fillResult.criticalError;
                  errorMessage = `Campo crítico "${err.field}" falló: ${err.error}`;
                  errorDetails = {
                    ...errorContext,
                    error_type: 'critical_field_error',
                    error_message: errorMessage,
                    details: {
                      field: err.field,
                      error: err.error,
                      selector: err.selector,
                      jsonValue: err.jsonValue,
                      label: err.label || err.field,
                      step: 'fillForm'
                    }
                  };
                } else if (fillResult?.errors && fillResult.errors.length > 0) {
                  // Múltiples errores
                  const criticalErrors = fillResult.errors.filter(e => e.critical);
                  const nonCriticalErrors = fillResult.errors.filter(e => !e.critical);
                  
                  if (criticalErrors.length > 0) {
                    errorMessage = `${criticalErrors.length} campo(s) crítico(s) con errores`;
                    errorDetails = {
                      ...errorContext,
                      error_type: 'multiple_critical_errors',
                      error_message: errorMessage,
                      details: {
                        criticalCount: criticalErrors.length,
                        errors: criticalErrors.map(e => ({
                          field: e.field,
                          error: e.error,
                          selector: e.selector,
                          jsonValue: e.jsonValue,
                          available_options: e.available_options
                        })),
                        step: 'fillForm'
                      }
                    };
                  } else {
                    errorMessage = `${nonCriticalErrors.length} campo(s) con errores`;
                    errorDetails = {
                      ...errorContext,
                      error_type: 'non_critical_errors',
                      error_message: errorMessage,
                      details: {
                        errorCount: nonCriticalErrors.length,
                        errors: nonCriticalErrors.map(e => ({
                          field: e.field,
                          error: e.error,
                          selector: e.selector,
                          jsonValue: e.jsonValue
                        })),
                        step: 'fillForm'
                      }
                    };
                  }
                } else if (fillResult?.missingRequired && fillResult.missingRequired.length > 0) {
                  // Campos requeridos faltantes con detalles
                  const missingFields = fillResult.missingRequired;
                  errorMessage = missingFields.map(m => m.reason).join(', ');
                  errorDetails = {
                    ...errorContext,
                    error_type: 'missing_required_fields',
                    error_message: errorMessage,
                    details: {
                      fields: missingFields.map(m => ({
                        field_name: m.name,
                        reason: m.reason,
                        selector: m.selector || m.details?.selector,
                        json_value: m.jsonValue || m.details?.json_value,
                        available_options: m.details?.available_options,
                        error_type: m.error_type || m.details?.error_type
                      })),
                      step: 'fillForm'
                    }
                  };
                } else {
                  errorDetails = {
                    ...errorContext,
                    error_type: 'unknown_error',
                    error_message: errorMessage,
                    details: {
                      rawResult: fillResult,
                      step: 'fillForm'
                    }
                  };
                }
                
                console.error('[SW] 💾 Guardando error detallado en BD:', errorDetails);
                
                // Actualizar estado en BD con error detallado en JSON
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ 
                    publish_status: 'error_retry',
                    publish_error: JSON.stringify(errorDetails) // ← JSON detallado
                  })
                });
                
                // ✅ v6.6.8: Enviar status error_retry al panel para el monitor (sync con BD)
                try {
                  console.log('[SW] 📤 Enviando PUBLISH_ITEM_UPDATE al panel:', {
                    id: item.id,
                    status: 'error_retry',
                    error: errorMessage
                  });
                  
                  chrome.runtime.sendMessage({
                    type: 'PUBLISH_ITEM_UPDATE',
                    data: {
                      id: item.id,
                      status: 'error_retry',
                      error: errorMessage || 'Error en formulario',
                      missingFields: fillResult?.missingRequired?.map(m => m.name || m) || [] // ✅ v6.8.8: Agregar campos faltantes
                    }
                  }).catch((err) => {
                    console.warn('[SW] ⚠️ Error enviando PUBLISH_ITEM_UPDATE:', err.message);
                  });
                } catch (e) {
                  console.error('[SW] ❌ Excepción enviando PUBLISH_ITEM_UPDATE:', e);
                }
                
                // ❌ v6.6.0: ELIMINADO - Ya no mostramos la modal vieja de error
                // Ahora el monitor de publicación muestra los errores en la lista
                
                sendResponse({ 
                  ok: false, 
                  error: errorMessage,
                  errorDetails,
                  missingRequired: fillResult?.missingRequired 
                });
                return;
              }
              

              // 17. Re-escanear para verificar
              await sendToTab({ type: 'FORM.SCAN_WIDGETS' }, 8000);

              // 18. Verificar botón publicar
              sendStepMessage('Verificando...', 7);
              const btnCheckRes = await sendToTab({
                type: 'PUBLISH.CHECK_BUTTON'
              }, 5000);
              
              if (!btnCheckRes?.ok || !btnCheckRes?.visible) {
                console.error('[SW] ❌ Botón publicar no visible');
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ publish_status: 'error_retry', publish_error: 'Botón publicar no visible' })
                });
                sendResponse({ ok: false, error: 'Botón publicar no visible' });
                return;
              }
              

              // 18b. 🗑️ DELETE DEL ANUNCIO VIEJO (solo si no está en ready_publish)
              if (!skipDelete) {
                try {
                  // ✅ v10.5.130: Refresh inteligente con feedback visual
                  // Solo refresca si token tiene <2min restante, mostrando mensaje al usuario
                  const refreshResult = await smartRefreshToken(
                    tab.id, 
                    120, // 2 minutos mínimo
                    'Preparando sesión...', 
                    6
                  );
                  
                  if (!refreshResult.ok) {
                    // ✅ v2.1.1: Mensaje claro de sesión expirada
                    const errorMsg = 'Sesión de Wallapop expirada. Refresca la página de Wallapop y vuelve a intentar.';
                    console.error('[SW] ❌ ' + errorMsg);
                    await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                      method: 'PATCH',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify({ publish_status: 'error_retry', publish_error: errorMsg })
                    });
                    // ✅ v2.2.4: Continuar con el siguiente anuncio en vez de parar
                    // Al navegar a /upload para el siguiente, probablemente se capture token nuevo
                    console.log('[SW] ⏭️ Continuando con siguiente anuncio (token puede recuperarse al navegar)');
                    sendResponse({ ok: true, item_id: item.id, skipped: true, error: errorMsg });
                    return;
                  }
                  
                  // Mostrar mensaje de eliminación
                  sendStepMessage('Descartando anuncio...', 8);
                  
                  // Obtener credenciales actuales
                  const creds = await ensureWallaCreds({ timeoutMs: 5000, kick: false });
                  
                  if (!creds || !creds.token || !creds.deviceId) {
                    // ✅ v2.1.1: Mensaje claro de sesión expirada
                    const errorMsg = 'Sesión de Wallapop expirada. Refresca la página de Wallapop y vuelve a intentar.';
                    console.error('[SW] ❌ ' + errorMsg);
                    await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                      method: 'PATCH',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify({ publish_status: 'error_retry', publish_error: errorMsg })
                    });
                    // ✅ v2.2.4: Continuar con el siguiente anuncio en vez de parar
                    console.log('[SW] ⏭️ Continuando con siguiente anuncio (token puede recuperarse al navegar)');
                    sendResponse({ ok: true, item_id: item.id, skipped: true, error: errorMsg });
                    return;
                  }
                
                
                const deleteUrl = `https://api.wallapop.com/api/v3/items/${encodeURIComponent(oldId)}`;
                
                // ✅ v10.5.54: Función helper para DELETE con reintento en 401
                const doDelete = async (token, deviceId) => {
                  return await fetch(deleteUrl, {
                    method: 'DELETE',
                    headers: {
                      'Accept': 'application/json',
                      'Authorization': `Bearer ${token}`,
                      'X-DeviceId': deviceId
                    }
                  });
                };
                
                let deleteRes = await doDelete(creds.token, creds.deviceId);
                
                // ✅ v10.5.60: Log resultado del DELETE con edad REAL del token
                let diagDelete = getTokenDiagnostics();
                dlog(`[TOKEN] 📊 DELETE resultado: HTTP ${deleteRes.status} | Real: ${diagDelete.realAge.formatted} | Restante: ${diagDelete.realAge.remainingFormatted}`);
                
                // ✅ v10.5.54: Si 401, refrescar token y reintentar UNA vez
                if (deleteRes.status === 401) {
                  dwarn(`[TOKEN] ⚠️ DELETE 401 - Token expirado. Real: ${diagDelete.realAge.formatted} | Restante: ${diagDelete.realAge.remainingFormatted}`);
                  
                  // Forzar refresh (forceRefresh = true)
                  const refreshResult = await refreshWallaCredsIfOld(tab.id, 90000, true);
                  
                  if (refreshResult.ok && refreshResult.creds?.token) {
                    dlog(`[TOKEN] 🔄 Token refrescado, reintentando DELETE...`);
                    deleteRes = await doDelete(refreshResult.creds.token, refreshResult.creds.deviceId);
                    diagDelete = getTokenDiagnostics();
                    dlog(`[TOKEN] 📊 DELETE reintento: HTTP ${deleteRes.status} | Real: ${diagDelete.realAge.formatted} | Restante: ${diagDelete.realAge.remainingFormatted}`);
                  } else {
                    console.error(`[TOKEN] ❌ No se pudo refrescar token para reintento`);
                  }
                }
                
                if (deleteRes.status === 200 || deleteRes.status === 204) {
                  
                  // ✅ Marcar como error_retry por seguridad (si falla publicación, queda protegido)
                  const newError = createPublishError(
                    'deleted_pending_publish',
                    'Anuncio eliminado de Wallapop, pendiente republicar',
                    { step: 'delete_old', old_id: oldId },
                    { itemId: item.id, wallapopId: oldId, title: extractTitle(snapshot?.title), accountId }
                  );
                  
                  await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                    method: 'PATCH',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                      publish_status: 'error_retry',
                      publish_error: newError
                    })
                  });
                  
                  // ✅ v6.0.0: Actualizar item en memoria para preservar flag en errores posteriores
                  item.publish_error = newError;
                  
                } else if (deleteRes.status === 404 || deleteRes.status === 410) {
                  
                  // También marcar como error_retry (ya no existe = ok para publicar)
                  const newError = createPublishError(
                    'deleted_pending_publish',
                    'Anuncio ya no existe en Wallapop (404/410), pendiente republicar',
                    { step: 'delete_old', old_id: oldId, http_status: deleteRes.status },
                    { itemId: item.id, wallapopId: oldId, title: extractTitle(snapshot?.title), accountId }
                  );
                  
                  await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                    method: 'PATCH',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                      publish_status: 'error_retry',
                      publish_error: newError
                    })
                  });
                  
                  // ✅ v6.0.0: Actualizar item en memoria para preservar flag en errores posteriores
                  item.publish_error = newError;
                  
                } else {
                  const errorText = await deleteRes.text().catch(() => 'no body');
                  console.error('[SW] ❌ DELETE falló:', { status: deleteRes.status, body: errorText });
                  await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                    method: 'PATCH',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ publish_status: 'error_retry', publish_error: `Error eliminando de Wallapop: HTTP ${deleteRes.status}` })
                  });
                  sendResponse({ ok: false, error: `Error eliminando de Wallapop: HTTP ${deleteRes.status}` });
                  return;
                }
              } catch (deleteErr) {
                console.error('[SW] ❌ Excepción eliminando anuncio:', deleteErr);
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ publish_status: 'error_retry', publish_error: `Excepción eliminando anuncio: ${deleteErr.message}` })
                });
                sendResponse({ ok: false, error: `Excepción eliminando anuncio: ${deleteErr.message}` });
                return;
              }
            } else {
            }

              // 18c. ⏳ DELAY CONFIGURABLE PARA REVISAR FORMULARIO (solo si no se saltó DELETE)
              if (!skipDelete) {
                
                // ✅ v2.2.0: Verificar pausa antes del delay
                await checkPauseAndWait();
                
                // ✅ v10.5.36: Leer delay de BD en tiempo real (por si el usuario lo cambió)
                let delaySeconds = userDelaySeconds; // Fallback al valor inicial
                try {
                  const prefsRes = await apiFetch('/api/user/preferences', { method: 'GET' });
                  if (prefsRes.ok) {
                    const prefsData = await prefsRes.json();
                    delaySeconds = prefsData?.ui_preferences?.publish_delay_seconds ?? delaySeconds;
                  }
                } catch (e) {
                  // Si falla, usar valor inicial
                }
                
                // ✅ v2.6.0: Notificar al backend que inicia delay (BD + WebSocket)
                try {
                  await apiFetch('/api/walla/publish2/session/start-delay', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                      account_id: accountId,
                      delay_seconds: delaySeconds,
                      item_id: item.id
                    })
                  });
                } catch (delayStartErr) {
                  console.error('[SW] ⚠️ Error notificando inicio de delay:', delayStartErr);
                  // No es error crítico, continuar
                }
                
                // ✅ v7.0.7: Aplicar variación humana al delay (±15%)
                const delayWithVariance = humanDelay(delaySeconds * 1000, 0.15);
                const finalDelaySeconds = Math.round(delayWithVariance / 1000);
                
                
                // ✅ v7.0.7: Pausa larga cada 10±2 anuncios (comportamiento humano)
                publishState.processedCount = (publishState.processedCount || 0) + 1;
                
                // ✅ v10.5.46: Guardar timestamp de inicio si es el primer anuncio
                if (publishState.processedCount === 1) {
                  publishState.cycleStartTime = Date.now();
                  dlog(`[PUBLISH] 🚀 Inicio de ciclo de publicación a las ${new Date().toLocaleTimeString()}`);
                }
                
                // ✅ v10.5.60: Log de progreso con edad REAL del token
                const cycleElapsed = publishState.cycleStartTime 
                  ? Math.round((Date.now() - publishState.cycleStartTime) / 1000) 
                  : 0;
                const diagProgress = getTokenDiagnostics();
                dlog(`[PUBLISH] ✅ Anuncio #${publishState.processedCount} | Ciclo: ${cycleElapsed}s | Token real: ${diagProgress.realAge.formatted} | Restante: ${diagProgress.realAge.remainingFormatted}`);
                
                // Determinar intervalo de pausa al inicio (8-12 anuncios)
                if (!publishState.breakInterval) {
                  publishState.breakInterval = 8 + Math.floor(Math.random() * 5); // 8-12
                }
                
                const shouldTakeLongBreak = publishState.processedCount > 0 && publishState.processedCount % publishState.breakInterval === 0;
                
                if (shouldTakeLongBreak) {
                  const longBreak = humanDelay(45 * 1000, 0.3); // 31-58 segundos
                  const longBreakSec = Math.round(longBreak / 1000);
                  
                  // ✅ v10.5.130: Lanzar refresh en background durante la pausa
                  const backgroundRefreshPromise = startBackgroundRefresh(tab.id, longBreakSec);
                  
                  // ✅ v10.5.34: Añadir flag isLongBreak para mostrar mensaje diferente en UI
                  chrome.runtime.sendMessage({
                    type: 'PUBLISH.COUNTDOWN_START',
                    delaySeconds: longBreakSec,
                    isLongBreak: true  // Pausa anti-detección
                  }).catch(() => {});
                  
                  await new Promise(r => setTimeout(r, longBreak));
                  
                  // ✅ v10.5.130: Esperar a que termine el refresh background (si lo hubo)
                  await backgroundRefreshPromise;
                  
                  // ✅ v2.6.0: Notificar fin de pausa larga
                  try {
                    await apiFetch('/api/walla/publish2/session/end-delay', {
                      method: 'POST',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify({
                        account_id: accountId,
                        reason: 'completed'
                      })
                    });
                  } catch (endErr) {
                    console.error('[SW] ⚠️ Error notificando fin de pausa larga:', endErr);
                  }
                }
                
                // ✅ v7.2.26: Sistema de alarms para delays largos (>60s)
                // ✅ v10.5.35: Guardar estado para continuar después de que la alarm despierte el SW
                if (finalDelaySeconds > 60) {
                  // ✅ v10.5.41: Margen de 5s (suficiente para PCs lentos/red inestable)
                  const SAFETY_MARGIN_SECONDS = 5;
                  const totalDelaySeconds = finalDelaySeconds + SAFETY_MARGIN_SECONDS;
                  const alarmTime = Date.now() + (totalDelaySeconds * 1000);
                  
                  // ✅ v10.5.41: Mostrar tiempo REAL al usuario (incluye margen)
                  chrome.runtime.sendMessage({
                    type: 'PUBLISH.COUNTDOWN_START',
                    delaySeconds: totalDelaySeconds,  // Usuario ve el tiempo real
                    isLongBreak: false
                  }).catch(() => {});
                  
                  // Guardar estado del item para continuar después de la alarm
                  // ✅ v10.5.130: Añadir flag para indicar que necesitará refresh al despertar
                  await savePendingPublish({
                    itemId: item.id,
                    accountId: accountId,
                    tabId: tab.id,
                    skipDelete: skipDelete,
                    oldId: oldId,
                    snapshot: snapshot,
                    title: extractTitle(snapshot?.title),
                    needsTokenRefresh: true  // ✅ v10.5.130: El token habrá expirado
                  });
                  
                  // Crear alarm con timestamp exacto (más preciso que delayInMinutes)
                  const alarmName = `publish-next-${accountId}`;
                  await chrome.alarms.create(alarmName, { when: alarmTime });
                  
                  console.log(`[SW] ⏰ Alarm creada para ${totalDelaySeconds}s (${finalDelaySeconds}s delay + ${SAFETY_MARGIN_SECONDS}s margen). Estado guardado para item ${item.id}`);
                  
                  // Enviar respuesta y permitir que SW se duerma
                  sendResponse({ ok: true, waiting_alarm: true, delay_seconds: totalDelaySeconds });
                  return; // SW puede dormirse, alarm lo despertará
                } else {
                  // Delay corto (≤60s): usar setTimeout tradicional, sin margen
                  
                  // ✅ v10.5.130: Lanzar refresh en background durante el countdown
                  const backgroundRefreshPromise = startBackgroundRefresh(tab.id, finalDelaySeconds);
                  
                  // Notificar al panel para mostrar countdown
                  chrome.runtime.sendMessage({
                    type: 'PUBLISH.COUNTDOWN_START',
                    delaySeconds: finalDelaySeconds,
                    isLongBreak: false
                  }).catch(() => {});
                  
                  await new Promise(r => setTimeout(r, delayWithVariance));
                  
                  // ✅ v10.5.130: Esperar a que termine el refresh background (si lo hubo)
                  await backgroundRefreshPromise;
                  
                  // ✅ v2.6.0: Notificar fin de delay
                  try {
                    await apiFetch('/api/walla/publish2/session/end-delay', {
                      method: 'POST',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify({
                        account_id: accountId,
                        reason: 'completed'
                      })
                    });
                  } catch (endErr) {
                    console.error('[SW] ⚠️ Error notificando fin de delay:', endErr);
                  }
                }
              } else {
                // ✅ v6.5.4: Varianza ±30% para delay de estabilidad
                await new Promise(r => setTimeout(r, humanDelay(2000, 0.3)));
              }

              // 19. Publicar y capturar ID
              sendStepMessage('Publicando...', 9);
              if (!(await ensureCS())) {
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ 
                    publish_status: 'error_retry', 
                    publish_error: createPublishError(
                      'content_script_error',
                      'CS no responde antes de publicar',
                      { step: 'before_publish' },
                      { itemId: item.id, wallapopId: oldId, title: extractTitle(snapshot?.title), accountId },
                      item.publish_error // ← Preservar flags
                    )
                  })
                });
                sendResponse({ ok: false, error: 'CS no responde antes de publicar' });
                return;
              }
              
              // ✅ v2.2.0: Verificar pausa antes de publicar (punto crítico)
              await checkPauseAndWait();
              
              // ✅ v10.5.130: Verificar token - solo refresh si necesario (ya se hizo en background)
              // Si el background refresh funcionó, esto será instantáneo
              // Si no, hará refresh con feedback
              const refreshBeforePublish = await smartRefreshToken(tab.id, 60, 'Verificando sesión...', 7);
              if (!refreshBeforePublish.ok) {
                console.error('[SW] ❌ Token expirado y no se pudo refrescar');
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ 
                    publish_status: 'error_retry', 
                    publish_error: 'Token expirado y no se pudo refrescar'
                  })
                });
                // ✅ v10.5.131: Incluir item_id para que panel pueda actualizar UI
                sendResponse({ ok: false, error: 'Token expirado', item_id: item.id });
                return;
              }
              
              // ✅ v2.4.6: Habilitar botón Wallapop justo antes del click automático
              try {
                await chrome.tabs.sendMessage(tab.id, { type: 'WALLAPOP.ENABLE_BUTTON' });
                dlog('[SW] ✅ Botón Wallapop habilitado antes del click');
              } catch (e) {
                console.warn('[SW] ⚠️ No se pudo habilitar botón:', e.message);
              }
              
              const publishRes = await sendToTab({
                type: 'PUBLISH.CLICK_AND_CAPTURE_ID',
                snapshot,
                maxWaitMs: 60000
              }, 65000);
              
              if (!publishRes?.ok) {
                // Si el anuncio ya fue eliminado (ready_publish), mantener ese estado
                // Si no, marcar como error_retry
                const errorMsg = publishRes?.code || 'Error publicando';
                const updatePayload = skipDelete 
                  ? { error: errorMsg }  // Mantener ready_publish
                  : { 
                      publish_status: 'error_retry', 
                      publish_error: createPublishError(
                        'publish_error',
                        errorMsg,
                        { step: 'click_publish', code: publishRes?.code },
                        { itemId: item.id, wallapopId: oldId, title: extractTitle(snapshot?.title), accountId },
                        item.publish_error // ← Preservar flags
                      )
                    };
                
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify(updatePayload)
                });
                sendResponse({ ok: false, error: errorMsg });
                return;
              }
              
              const newId = publishRes.itemId;
              const newSlug = publishRes.itemSlug;
              

              // 20. Validar newId
              if (!newId) {
                console.error('[SW] ❌ No se capturó newId');
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ publish_status: 'error_retry', publish_error: 'No se capturó ID nuevo' })
                });
                sendResponse({ ok: false, error: 'No se capturó ID nuevo' });
                return;
              }
              
              if (newId === oldId) {
                console.error('[SW] ❌ Mismo ID', { newId, oldId });
                await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                  method: 'PATCH',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({ publish_status: 'error_retry', publish_error: 'ID nuevo igual al viejo' })
                });
                sendResponse({ ok: false, error: 'ID nuevo igual al viejo' });
                return;
              }

              // 21. Actualizar BD con nuevo ID y marcar como 'published'
              await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ 
                  publish_status: 'published',  // ✅ v4.84.0: Marcar como published
                  new_id_wallapop: newId,
                  new_slug: newSlug,
                  publish_error: null
                })
              });
              
              
              // ✅ v6.6.0: Enviar status success al panel para el monitor
              try {
                chrome.runtime.sendMessage({
                  type: 'PUBLISH_ITEM_UPDATE',
                  data: {
                    id: item.id,
                    status: 'success',
                    error: null
                  }
                }).catch(() => {});
              } catch (e) {
              }
              
              sendResponse({ ok: true, done: false, has_next: true, item_id: item.id, new_id: newId });

            } catch (err) {
              console.error('[SW] ❌ Error crítico en PUBLISH.PROCESS_NEXT:', err);
              
              // ✅ v2.6.0: Notificar fin de delay por error
              try {
                await apiFetch('/api/walla/publish2/session/end-delay', {
                  method: 'POST',
                  headers: { 'Content-Type': 'application/json' },
                  body: JSON.stringify({
                    account_id: accountId,
                    reason: 'error'
                  })
                });
              } catch (endErr) {
                console.error('[SW] ⚠️ Error notificando fin de delay por error:', endErr);
              }
              
              // ✅ v5.x: Detectar tipo de error para mensaje apropiado con contexto completo
              let errorMessage = String(err.message || err);
              let errorTypeCode = 'unknown_error';
              let errorStep = 'process_next';
              
              if (errorMessage.includes('cerró') || errorMessage.includes('closed')) {
                errorMessage = 'La pestaña de Wallapop se cerró durante la publicación';
                errorTypeCode = 'tab_closed';
                errorStep = 'communication';
              } else if (errorMessage.includes('Timeout')) {
                errorTypeCode = 'timeout';
                errorStep = 'communication';
              } else if (errorMessage.includes('network') || errorMessage.includes('internet')) {
                errorMessage = 'Error de conexión a internet';
                errorTypeCode = 'network_error';
                errorStep = 'communication';
              } else if (errorMessage.includes('JSON')) {
                errorTypeCode = 'json_parse_error';
                errorStep = 'data_processing';
              }
              
              try {
                if (item?.id) {
                  const snapshot = item.snapshot || (item.json_data ? JSON.parse(item.json_data) : {});
                  const oldId = item.id_wallapop || snapshot.id;
                  const accountId = msg.account_id;
                  
                  await apiFetch(`/api/walla/publish2/update-status/${item.id}`, {
                    method: 'PATCH',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                      publish_status: 'error_retry',
                      publish_error: createPublishError(
                        errorTypeCode,
                        errorMessage,
                        { 
                          step: errorStep,
                          original_error: String(err.message || err).substring(0, 500),
                          stack: err.stack ? err.stack.substring(0, 1000) : null
                        },
                        { 
                          itemId: item.id, 
                          wallapopId: oldId, 
                          title: extractTitle(snapshot?.title, item.title || 'Sin título'),
                          accountId 
                        },
                        item.publish_error // ← Preservar flags del error anterior
                      )
                    })
                  });
                  
                  // ✅ v6.6.8: Enviar status error_retry al panel para el monitor (sync con BD)
                  try {
                    chrome.runtime.sendMessage({
                      type: 'PUBLISH_ITEM_UPDATE',
                      data: {
                        id: item.id,
                        status: 'error_retry',
                        error: errorMessage
                      }
                    }).catch(() => {});
                  } catch (e) {
                  }
                }
              } catch (updateErr) {
                console.error('[SW] ❌ No se pudo actualizar estado de error:', updateErr);
              }
              
              sendResponse({ ok: false, error: errorMessage });
            }
}

// =============================================================================
// ✅ v10.5.35: Continuar publicación después de alarm larga
// =============================================================================

/**
 * Continúa la publicación de un item después de que la alarm despierte el SW
 * Se llama desde keepalive.js cuando hay estado pendiente guardado
 * 
 * @param {Object} pendingState - Estado guardado antes de la alarm
 * @param {Function} sendResponse - Callback para responder
 */
async function continueAfterDelay(pendingState, sendResponse) {
  const { itemId, accountId, tabId, skipDelete, oldId, snapshot, title, needsTokenRefresh } = pendingState;
  
  console.log(`[SW] ⏰ Continuando publicación del item ${itemId} después de alarm`);
  
  // ✅ v2.6.0: Notificar fin de delay (alarm completada)
  try {
    await apiFetch('/api/walla/publish2/session/end-delay', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        account_id: accountId,
        reason: 'completed'
      })
    });
  } catch (endErr) {
    console.error('[SW] ⚠️ Error notificando fin de delay después de alarm:', endErr);
  }
  
  try {
    // 1. Verificar que la pestaña sigue activa
    let tab;
    try {
      tab = await chrome.tabs.get(tabId);
    } catch (e) {
      // La pestaña se cerró, buscar otra de Wallapop
      const tabs = await chrome.tabs.query({ url: '*://*.wallapop.com/*' });
      if (tabs.length === 0) {
        await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
          method: 'PATCH',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ 
            publish_status: 'error_retry', 
            publish_error: 'La pestaña de Wallapop se cerró durante el delay' 
          })
        });
        await clearPendingPublish();
        sendResponse({ ok: false, error: 'La pestaña de Wallapop se cerró durante el delay' });
        return;
      }
      tab = tabs[0];
    }
    
    // 2. Helper para enviar mensajes al tab con timeout
    const sendToTab = async (message, timeout = 65000) => {
      return new Promise((resolve, reject) => {
        const timer = setTimeout(() => {
          reject(new Error(`Timeout después de ${timeout}ms`));
        }, timeout);
        
        chrome.tabs.sendMessage(tab.id, message, (response) => {
          clearTimeout(timer);
          if (chrome.runtime.lastError) {
            reject(new Error(chrome.runtime.lastError.message));
          } else {
            resolve(response);
          }
        });
      });
    };
    
    // 3. Verificar que el CS responde
    const ensureCS = async () => {
      try {
        const response = await sendToTab({ type: 'SYS.PING' }, 5000);
        return response?.ok === true;
      } catch {
        return false;
      }
    };
    
    if (!(await ensureCS())) {
      // Intentar reinyectar CS
      try {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          files: ['content_script.js']
        });
        await new Promise(r => setTimeout(r, 1000));
      } catch (e) {
        console.error('[SW] Error reinyectando CS:', e);
      }
      
      if (!(await ensureCS())) {
        await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
          method: 'PATCH',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ 
            publish_status: 'error_retry', 
            publish_error: 'Content script no responde después del delay' 
          })
        });
        await clearPendingPublish();
        sendResponse({ ok: false, error: 'Content script no responde después del delay' });
        return;
      }
    }
    
    // ✅ v10.5.130: Refrescar token después de alarm larga (SIEMPRE necesario)
    // El token habrá expirado durante delays de minutos/horas
    sendStepMessage('Retomando sesión...', 9);
    console.log(`[SW] 🔄 Refrescando token después de alarm...`);
    
    const refreshResult = await smartRefreshToken(tab.id, 60, null, 7); // Sin mensaje extra, ya mostramos uno
    
    if (!refreshResult.ok) {
      console.error('[SW] ❌ No se pudo refrescar token después de alarm');
      await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          publish_status: 'error_retry', 
          publish_error: 'Token expirado después del delay y no se pudo refrescar' 
        })
      });
      await clearPendingPublish();
      sendResponse({ ok: false, error: 'Token expirado después del delay' });
      return;
    }
    
    // 4. Publicar y capturar ID
    sendStepMessage('Publicando...', 9);
    console.log(`[SW] ▶️ Enviando PUBLISH.CLICK_AND_CAPTURE_ID para item ${itemId}`);
    
    const publishRes = await sendToTab({
      type: 'PUBLISH.CLICK_AND_CAPTURE_ID',
      snapshot,
      maxWaitMs: 60000
    }, 65000);
    
    if (!publishRes?.ok) {
      const errorMsg = publishRes?.code || 'Error publicando después del delay';
      await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          publish_status: 'error_retry', 
          publish_error: errorMsg 
        })
      });
      await clearPendingPublish();
      sendResponse({ ok: false, error: errorMsg });
      return;
    }
    
    const newId = publishRes.itemId;
    const newSlug = publishRes.itemSlug;
    
    // 5. Validar newId
    if (!newId) {
      console.error('[SW] ❌ No se capturó newId después del delay');
      await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          publish_status: 'error_retry', 
          publish_error: 'No se capturó ID nuevo después del delay' 
        })
      });
      await clearPendingPublish();
      sendResponse({ ok: false, error: 'No se capturó ID nuevo' });
      return;
    }
    
    // 6. Actualizar BD con éxito
    console.log(`[SW] ✅ Item ${itemId} publicado exitosamente. Nuevo ID: ${newId}`);
    
    const newUrl = newSlug 
      ? `https://es.wallapop.com/item/${newSlug}` 
      : `https://es.wallapop.com/item/${newId}`;
    
    await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
      method: 'PATCH',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        publish_status: 'success',
        publish_error: null,
        id_wallapop: newId,
        slug: newSlug || null,
        url: newUrl
      })
    });
    
    // 7. Notificar al panel
    try {
      chrome.runtime.sendMessage({
        type: 'PUBLISH_ITEM_UPDATE',
        data: {
          id: itemId,
          status: 'success',
          id_wallapop: newId,
          slug: newSlug,
          url: newUrl
        }
      }).catch(() => {});
    } catch (e) {}
    
    // 8. Limpiar estado pendiente
    await clearPendingPublish();
    
    // 9. Notificar al panel que puede continuar
    console.log(`[SW] ✅ Item ${itemId} publicado después de alarm. Notificando al panel...`);
    sendResponse({ ok: true, published: itemId, newId });
    
    // ✅ v10.5.43: Notificar al panel que la alarm terminó
    // El panel está esperando este mensaje para continuar con el siguiente item
    try {
      await chrome.runtime.sendMessage({
        type: 'PUBLISH.ALARM_COMPLETED',
        ok: true,
        itemId: itemId,
        newId: newId
      });
      console.log(`[SW] 📤 Notificación PUBLISH.ALARM_COMPLETED enviada al panel`);
    } catch (e) {
      // Panel puede estar cerrado, no es error crítico
      console.log(`[SW] ℹ️ Panel no recibió notificación (puede estar cerrado)`);
    }
    
  } catch (error) {
    console.error('[SW] ❌ Error continuando publicación después del delay:', error);
    
    // ✅ v2.6.0: Notificar fin de delay por error
    try {
      await apiFetch('/api/walla/publish2/session/end-delay', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          account_id: accountId,
          reason: 'error'
        })
      });
    } catch (endErr) {
      console.error('[SW] ⚠️ Error notificando fin de delay por error:', endErr);
    }
    
    try {
      await apiFetch(`/api/walla/publish2/update-status/${itemId}`, {
        method: 'PATCH',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          publish_status: 'error_retry', 
          publish_error: `Error después del delay: ${error.message}` 
        })
      });
    } catch (e) {}
    
    await clearPendingPublish();
    sendResponse({ ok: false, error: error.message });
    
    // ✅ v10.5.43: Notificar error al panel
    try {
      await chrome.runtime.sendMessage({
        type: 'PUBLISH.ALARM_COMPLETED',
        ok: false,
        error: error.message
      });
    } catch (e) {}
  }
}

// ✅ v7.2.26: Export para poder llamar desde keepalive.js (alarms)
export const handlePublishProcessNext = handleProcessNext;

// ✅ v10.5.35: Export para continuar publicación después de alarm larga
export { continueAfterDelay };
