/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/constants.js — Rol: Constantes centralizadas para el Service Worker
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
  
  ✅ v6.5.3: Archivo creado para centralizar timeouts y delays del Service Worker
*/

/**
 * ✅ v6.5.3: Constantes centralizadas para el Service Worker
 * 
 * Este archivo centraliza todos los timeouts, delays y configuración
 * usados en el orquestador de publicación del SW.
 * 
 * @module sw/constants
 */

// =============================================================================
// TIMEOUTS DE COMUNICACIÓN (SW → CS)
// =============================================================================

export const SW_TIMEOUTS = {
  // Comunicación con Content Script
  CS_DEFAULT: 20000,              // Timeout por defecto para mensajes al CS
  CS_SELECT_VERTICAL: 15000,      // Selección de tipo de anuncio
  CS_FILL_FORM: 25000,            // Llenado de formulario completo
  CS_UPLOAD_IMAGES: 35000,        // Subida de imágenes (más lento)
  CS_CLICK_CONTINUE: 12000,       // Click en continuar
  CS_SELECT_CATEGORY: 20000,      // Selección de categoría
  CS_PUBLISH_CAPTURE: 30000,      // Publicar y capturar ID
  CS_CHECK_BUTTON: 8000,          // Verificar botón habilitado
  CS_PING: 5000,                  // Ping de verificación
  
  // Operaciones de API
  API_FETCH: 10000,               // Llamadas al backend
  WALLA_CREDS: 8000,              // Obtener credenciales Wallapop
  
  // Navegación
  TAB_LOAD: 15000,                // Esperar carga de pestaña
  TAB_RELOAD: 20000,              // Recargar pestaña
  URL_CHANGE: 10000,              // Esperar cambio de URL
  
  // Reintentos
  MAX_RETRY_WAIT: 60000,          // Máximo tiempo entre reintentos
};

// =============================================================================
// DELAYS ENTRE OPERACIONES
// =============================================================================

export const SW_DELAYS = {
  // Entre pasos del proceso
  BETWEEN_STEPS: 1500,            // Delay entre pasos principales
  BEFORE_FILL_FORM: 500,          // Antes de rellenar formulario
  AFTER_CATEGORY: 800,            // Después de seleccionar categoría
  BEFORE_PUBLISH: 3000,           // Delay humano antes de publicar (countdown)
  
  // Entre items
  BETWEEN_ITEMS: 2000,            // Entre publicación de items
  AFTER_ERROR: 3000,              // Después de un error antes de continuar
  
  // Verificaciones
  CS_VERIFY_READY: 1000,          // Verificar CS listo
  TAB_STABILIZE: 2000,            // Esperar que pestaña se estabilice
  
  // Anti-detección
  HUMAN_MIN: 500,                 // Mínimo delay humano
  HUMAN_MAX: 3000,                // Máximo delay humano
};

// =============================================================================
// CONFIGURACIÓN DE REINTENTOS
// =============================================================================

export const SW_RETRY = {
  // Número de intentos
  MAX_CS_RETRIES: 3,              // Reintentos de comunicación con CS
  MAX_API_RETRIES: 2,             // Reintentos de API
  MAX_STEP_RETRIES: 2,            // Reintentos por paso
  
  // Backoff exponencial
  INITIAL_DELAY: 1000,            // Delay inicial
  MULTIPLIER: 1.5,                // Multiplicador
  MAX_DELAY: 10000,               // Delay máximo
  
  // Condiciones de reintento
  RETRY_ON_TIMEOUT: true,         // Reintentar en timeout
  RETRY_ON_CS_NOT_READY: true,    // Reintentar si CS no está listo
  RETRY_ON_NETWORK_ERROR: false,  // No reintentar errores de red
};

// =============================================================================
// MENSAJES AL CONTENT SCRIPT
// =============================================================================

export const CS_MESSAGES = {
  // Publicación
  SELECT_VERTICAL: 'PUBLISH.SELECT_VERTICAL',
  FILL_BASIC_FIELDS: 'PUBLISH.FILL_BASIC_FIELDS',
  CLICK_CONTINUE: 'PUBLISH.CLICK_CONTINUE',
  UPLOAD_IMAGES: 'PUBLISH.UPLOAD_IMAGES',
  SELECT_CATEGORY: 'PUBLISH.SELECT_CATEGORY',
  FILL_FORM: 'PUBLISH.FILL_FORM',
  CHECK_BUTTON: 'PUBLISH.CHECK_BUTTON',
  CLICK_AND_CAPTURE_ID: 'PUBLISH.CLICK_AND_CAPTURE_ID',
  
  // Sistema
  PING: 'SYS.PING',
  SCAN_WIDGETS: 'FORM.SCAN_WIDGETS',
  
  // Notificaciones al panel
  COUNTDOWN_START: 'PUBLISH.COUNTDOWN_START',
  SHOW_SUCCESS_MODAL: 'PUBLISH.SHOW_SUCCESS_MODAL',
  COMPLETED: 'PUBLISH.COMPLETED',
};

// =============================================================================
// CÓDIGOS DE ERROR DEL SW
// =============================================================================

export const SW_ERROR_CODES = {
  // Comunicación
  CS_NOT_READY: 'content_script_not_ready',
  CS_TIMEOUT: 'content_script_timeout',
  CS_ERROR: 'content_script_error',
  TAB_NOT_FOUND: 'wallapop_tab_not_found',
  TAB_CLOSED: 'tab_closed_during_process',
  
  // API
  API_ERROR: 'api_request_failed',
  NO_NEXT_ITEM: 'no_pending_items',
  INVALID_ITEM: 'invalid_item_data',
  
  // Publicación
  UNSUPPORTED_TYPE: 'unsupported_ad_type',
  FORM_INCOMPLETE: 'form_validation_failed',
  PUBLISH_FAILED: 'publish_button_failed',
  ID_CAPTURE_FAILED: 'id_capture_failed',
  
  // Sistema
  NETWORK_ERROR: 'network_error',
  AUTH_EXPIRED: 'authentication_expired',
  WALLAPOP_OFFLINE: 'wallapop_service_unavailable',
};

// =============================================================================
// PASOS DEL PROCESO DE PUBLICACIÓN
// =============================================================================

export const PUBLISH_STEPS = {
  GET_NEXT_ITEM: 1,
  MARK_PROCESSING: 3,
  NAVIGATE_TO_UPLOAD: 4,
  VERIFY_CS: 5,
  SELECT_VERTICAL: 6,
  VERIFY_AFTER_VERTICAL: 7,
  FILL_SUMMARY: 8,
  CLICK_CONTINUE_TITLE: 9,
  // paso 10 saltado
  VERIFY_BEFORE_IMAGES: 11,
  UPLOAD_IMAGES: 12,
  CLICK_CONTINUE_PHOTO: 13,
  SELECT_CATEGORY: 14,
  SCAN_FORM: 15,
  FILL_FORM: 16,
  RESCAN_VERIFY: 17,
  CHECK_BUTTON: 18,
  PUBLISH_CAPTURE: 19,
  VALIDATE_ID: 20,
  UPDATE_DB: 21,
};

// =============================================================================
// UTILIDADES
// =============================================================================

/**
 * Calcula delay con backoff exponencial y jitter
 * @param {number} attempt - Número de intento (1-based)
 * @param {number} initial - Delay inicial en ms
 * @param {number} multiplier - Factor multiplicador
 * @param {number} maxDelay - Delay máximo en ms
 * @returns {number} Delay en ms
 */
export function calculateBackoffDelay(
  attempt,
  initial = SW_RETRY.INITIAL_DELAY,
  multiplier = SW_RETRY.MULTIPLIER,
  maxDelay = SW_RETRY.MAX_DELAY
) {
  const baseDelay = initial * Math.pow(multiplier, attempt - 1);
  // Agregar jitter ±20%
  const jitter = 0.8 + Math.random() * 0.4;
  return Math.min(Math.floor(baseDelay * jitter), maxDelay);
}

/**
 * Genera delay humano aleatorio
 * @param {number} base - Tiempo base en ms
 * @param {number} variance - Variación (0-1)
 * @returns {number} Delay en ms
 */
export function humanDelay(base = SW_DELAYS.HUMAN_MIN, variance = 0.3) {
  const min = base * (1 - variance);
  const max = base * (1 + variance);
  return Math.floor(min + Math.random() * (max - min));
}

/**
 * Obtiene timeout específico para un mensaje al CS
 * @param {string} messageType - Tipo de mensaje
 * @returns {number} Timeout en ms
 */
export function getCSTimeout(messageType) {
  const timeoutMap = {
    [CS_MESSAGES.SELECT_VERTICAL]: SW_TIMEOUTS.CS_SELECT_VERTICAL,
    [CS_MESSAGES.FILL_BASIC_FIELDS]: SW_TIMEOUTS.CS_DEFAULT,
    [CS_MESSAGES.CLICK_CONTINUE]: SW_TIMEOUTS.CS_CLICK_CONTINUE,
    [CS_MESSAGES.UPLOAD_IMAGES]: SW_TIMEOUTS.CS_UPLOAD_IMAGES,
    [CS_MESSAGES.SELECT_CATEGORY]: SW_TIMEOUTS.CS_SELECT_CATEGORY,
    [CS_MESSAGES.FILL_FORM]: SW_TIMEOUTS.CS_FILL_FORM,
    [CS_MESSAGES.CHECK_BUTTON]: SW_TIMEOUTS.CS_CHECK_BUTTON,
    [CS_MESSAGES.CLICK_AND_CAPTURE_ID]: SW_TIMEOUTS.CS_PUBLISH_CAPTURE,
    [CS_MESSAGES.PING]: SW_TIMEOUTS.CS_PING,
    [CS_MESSAGES.SCAN_WIDGETS]: SW_TIMEOUTS.CS_DEFAULT,
  };
  
  return timeoutMap[messageType] || SW_TIMEOUTS.CS_DEFAULT;
}
