/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/boot.js — Rol: Router principal + listeners + bootstrap minimal.
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/boot.js — Rol: Router principal + listeners + bootstrap minimal.
  ... resto de cabecera ...
*/

import { 
  CONFIG, 
  loadConfigOverrides,
  fetchAndCacheRemoteConfig,
  AUTH_STATES, 
  setAuthState, 
  loadSession, 
  saveSession, 
  session, 
  apiFetch, 
  withDeviceHeaders, 
  getAuthState, 
  setActivity,
  ensureDeviceId,
  isAccessNearExpiry,
  doRefresh
} from './api_client.js';
import { setupKeepAlive } from './keepalive.js';
import { initWallaCredsFromStorage, setupWallaInterceptors, ensureWallaCreds, fullExportByProfileUrl, exportAll, importIntoAPI } from './walla.js';
import { startOutboxDrainInterval, outboxDrainOnce } from './outbox.js';
import { singleFlight } from './utils.js';
import { initRealtimeClient, disconnectRealtimeClient, getRealtimeClient } from './realtime.js';
import { startAutoRefresh, stopAutoRefresh } from './token_refresher.js';  // ✅ Auto-refresh proactivo
import { handleProcessNext } from './handlers/publish-process-next.js';  // ✅ v4.87.0: Handler PUBLISH.PROCESS_NEXT modularizado
import { publishState } from './state.js';  // ✅ v6.7.0: Estado global de publicación

// ============================================
// WEBSOCKET HANDLERS (función reutilizable)
// ============================================

/**
 * Registra los handlers de eventos WebSocket
 * Se llama después de initRealtimeClient tanto en login como en bootstrap
 */
function registerWebSocketHandlers() {
  const rtClient = getRealtimeClient();
  if (!rtClient) return;
  
  // 🎯 BACKUP PROGRESS → ACTIVITY.UPDATE + REENVIAR AL PANEL
  rtClient.on('backup_progress', (data, account_id) => {
    setActivity('backup', {
      state: 'running',
      seen: data.items_processed || 0,
      total: data.items_total || 0,
      run_id: data.run_id
    });
    
    chrome.runtime.sendMessage({
      type: 'backup_progress',
      data: data,
      account_id: account_id
    }).catch(() => {});
  });
  
  // ✅ BACKUP ITEM SAVED → REENVIAR AL PANEL
  rtClient.on('backup_item_saved', (data, account_id) => {
    chrome.runtime.sendMessage({
      type: 'backup_item_saved',
      data: data,
      account_id: account_id
    }).catch(() => {});
  });
  
  // 🎯 BACKUP COMPLETE → ACTIVITY.UPDATE + REENVIAR AL PANEL
  rtClient.on('backup_complete', (data, account_id) => {
    setActivity('backup', {
      state: data.status === 'ok' ? 'ok' : 'error',
      seen: data.items_processed || 0,
      total: data.items_total || 0,
      run_id: data.run_id
    });
    
    chrome.runtime.sendMessage({
      type: 'backup_complete',
      data: data,
      account_id: account_id
    }).catch(() => {});
  });
  
  // 🎯 PUBLISH PROGRESS → ACTIVITY.UPDATE + REENVIAR AL PANEL
  rtClient.on('publish_progress', (data, account_id) => {
    setActivity('publish', {
      state: 'running',
      seen: data.processed || 0,
      total: data.total || 0
    });
    
    chrome.runtime.sendMessage({
      type: 'publish_progress',
      data: data,
      account_id: account_id
    }).catch(() => {});
  });
  
  // 🎯 PUBLISH COMPLETE → ACTIVITY.UPDATE + REENVIAR AL PANEL
  rtClient.on('publish_complete', (data, account_id) => {
    setActivity('publish', {
      state: 'ok',
      seen: data.processed || 0,
      total: data.total || 0
    });
    
    chrome.runtime.sendMessage({
      type: 'publish_complete',
      data: data,
      account_id: account_id
    }).catch(() => {});
  });
}

// ============================================
// FEATURE DETECTION
// ============================================

/**
 * Detecta qué features soporta la API actual
 * 
 * @returns {Promise<Object>} Features soportados { websocket, preferences, etc }
 */
async function detectApiFeatures() {
  try {
    const response = await fetch(`${CONFIG.API_BASE}/api/version`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        'X-Extension-Version': chrome.runtime.getManifest().version
      }
    });
    
    if (!response.ok) {
      return {
        websocket: false,
        preferences: true,
        multi_account: true
      };
    }
    
    const data = await response.json();
    const features = data.features || {};
    
    return features;
    
  } catch (error) {
    // Fallback: asumir API básica sin WebSocket
    return {
      websocket: false,
      preferences: true,
      multi_account: true
    };
  }
}

// ============================================
// 🔄 RENOVACIÓN AUTOMÁTICA DE TOKEN
// ============================================

/**
 * Intenta renovar el access token usando el refresh token
 * Con singleFlight: múltiples llamadas simultáneas comparten la misma promesa
 * @returns {Promise<boolean>} true si se renovó exitosamente
 */
let refreshTokenPromise = null;
async function tryRefreshToken() {
  // Si ya hay un refresh en curso, retornar la misma promesa
  if (refreshTokenPromise) {
    return refreshTokenPromise;
  }
  
  refreshTokenPromise = (async () => {
    try {
      
      // Verificar que tenemos refresh token
      if (!session?.refresh || !session?.device_id) {
        return false;
      }
      
      // Llamar al endpoint de refresh
      const response = await fetch(`${CONFIG.API_BASE}/api/auth/refresh`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          refresh: session.refresh,
          device_id: session.device_id
        })
      });
      
      if (!response.ok) {
        return false;
      }
      
      const data = await response.json();
      
      // Actualizar sesión con nuevo access token
      session.access = data.access;
      session.exp = data.exp || 0;
      
      // Guardar sesión actualizada
      await saveSession();
      
      return true;
      
    } catch (err) {
      console.error('[SW] ❌ Error en tryRefreshToken:', err);
      return false;
    } finally {
      // Limpiar promesa después de 1 segundo
      setTimeout(() => { refreshTokenPromise = null; }, 1000);
    }
  })();
  
  return refreshTokenPromise;
}

// ============================================
// MESSAGE LISTENERS
// ============================================

export function setupListeners() {
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    // ✅ CRÍTICO: Envolver en async y retornar true
    (async () => {
      try {
        if (msg?.type === 'SYS.PING') { 
          sendResponse({ ok: true, data: { now: Date.now() } }); 
          return; 
        }

        // === API.FETCH_JSON (sin import() dinámico) ===
        if (msg?.type === 'API.FETCH_JSON') {
          try {
            const { url, method = 'GET', body = null, timeout = 10000 } = msg;

            // ⚠️ Cargar config/sesión por si el SW acababa de “despertar”
            await loadConfigOverrides();
            await loadSession();

            if (!CONFIG.API_BASE) { sendResponse({ ok:false, error:'no_api_base' }); return; }
            if (!session?.access) { sendResponse({ ok:false, error:'no_session' }); return; }

            const fullUrl = /^https?:/.test(url) ? url : `${CONFIG.API_BASE}${url}`;
            
            // Función interna para hacer la petición
            const makeRequest = async (accessToken) => {
              const headers = withDeviceHeaders({
                'Authorization': `Bearer ${accessToken}`,
                'Content-Type': 'application/json',
                'X-Extension-Version': chrome.runtime.getManifest().version
              });

              const opts = { method, headers };
              if (body && method !== 'GET') opts.body = (typeof body === 'string') ? body : JSON.stringify(body);

              const ctl = new AbortController();
              const to = setTimeout(() => ctl.abort(), timeout);
              
              let res;
              try {
                res = await fetch(fullUrl, { ...opts, signal: ctl.signal });
                clearTimeout(to);
              } catch (fetchError) {
                clearTimeout(to);
                
                // Detectar tipo de error
                if (fetchError.name === 'AbortError') {
                  throw new Error('TIMEOUT');
                } else if (fetchError.message.includes('Failed to fetch') || fetchError.message.includes('NetworkError')) {
                  throw new Error('CONNECTION_ERROR');
                }
                throw fetchError;
              }
              
              return res;
            };

            // Intento 1: Con token actual
            let res = await makeRequest(session.access);
            
            // 🔄 Si es 401, intentar renovar token y reintentar
            if (res.status === 401) {
              
              const refreshed = await tryRefreshToken();
              
              if (refreshed && session.access) {
                // Intento 2: Con token renovado
                res = await makeRequest(session.access);
              } else {
                sendResponse({ ok:false, error:'HTTP 401', needsReauth: true }); 
                return;
              }
            }
            
            // 🔄 Si es 426, extensión desactualizada (BLOQUEO FORZADO)
            if (res.status === 426) {
              
              try {
                const errorData = await res.json();
                
                // Notificar al panel para mostrar modal FORZADA
                chrome.runtime.sendMessage({
                  type: 'FORCE_UPDATE_MODAL',
                  data: {
                    minimum_version: errorData.minimum_version,
                    current_version: errorData.current_version,
                    download_url: errorData.download_url || 'https://www.mitiklive.com/wallapop/assets/downloads/extension.zip'
                  }
                });
                
                sendResponse({ ok: false, error: 'extension_outdated', status: 426, data: errorData });
                return;
              } catch (e) {
                console.error('[SW] Error parseando respuesta 426:', e);
                sendResponse({ ok: false, error: 'extension_outdated', status: 426 });
                return;
              }
            }

            if (!res.ok) { 
              // Intentar obtener mensaje de error del backend
              let errorMsg = `HTTP ${res.status}`;
              try {
                const errorData = await res.json();
                if (errorData.detail) {
                  errorMsg = errorData.detail;
                }
              } catch {}
              
              sendResponse({ ok:false, error: errorMsg, status: res.status }); 
              return; 
            }
            
            // Validar que la respuesta es JSON
            const contentType = res.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
              sendResponse({ ok: false, error: 'SERVER_ERROR', message: 'El servidor devolvió una respuesta inválida' });
              return;
            }
            
            // Parse JSON con mejor manejo de errores
            let data;
            try {
              const text = await res.text();
              
              data = JSON.parse(text);
              
              // Si la respuesta es {ok: true, data: {...}}, extraer el .data
              if (data.ok && data.data) {
                data = data.data;
              }
            } catch (e) {
              console.error('[SW] ❌ Error parseando JSON:', e);
              console.error('[SW] ❌ URL:', fullUrl);
              sendResponse({ ok: false, error: 'JSON_PARSE_ERROR', message: e.message });
              return;
            }
            
            sendResponse({ ok:true, data });
          } catch (err) {
            console.error('[SW] Error en API.FETCH_JSON:', err, { url: msg?.url, method: msg?.method });
            
            // Mensajes de error amigables
            let errorMessage = String(err?.message || err);
            if (err.message === 'TIMEOUT') {
              errorMessage = 'La petición tardó demasiado tiempo. El servidor podría estar lento.';
            } else if (err.message === 'CONNECTION_ERROR') {
              errorMessage = 'No se pudo conectar con el servidor. Verifica tu conexión.';
            }
            
            sendResponse({ ok:false, error: errorMessage, type: err.message });
          }
          return;
        }

        await loadConfigOverrides();
        await loadSession();

        // === SESSION.* ===
        if (msg?.type === 'SESSION.LOGIN_EMAIL') {
          try {
            setAuthState(AUTH_STATES.AUTHENTICATING);
            const device_id = session.device_id || await ensureDeviceId();
            
            const loginUrl = `${CONFIG.API_BASE}/api/auth/login`;
            
            let r, data;
            try {
              r = await fetch(loginUrl, {
                method: 'POST',
                headers: { 'Content-Type':'application/json' },
                body: JSON.stringify({ 
                  email: msg.email, 
                  password: msg.password, 
                  device_id, 
                  user_agent: navigator.userAgent 
                })
              });
              
              
              // Intentar parsear JSON
              const contentType = r.headers.get('content-type');
              if (contentType && contentType.includes('application/json')) {
                data = await r.json();
              } else {
                // Respuesta no es JSON (probablemente HTML de error 502/503)
                throw new Error('SERVER_DOWN');
              }
              
            } catch (fetchError) {
              console.error('[LOGIN] ❌ Error de conexión:', fetchError);
              
              // Determinar tipo de error
              let errorMessage = 'No se pudo conectar con el servidor';
              let errorType = 'connection_error';
              
              if (fetchError.message === 'SERVER_DOWN') {
                errorMessage = '🔴 El servidor está caído o en mantenimiento. Por favor, inténtalo más tarde.';
                errorType = 'server_down';
              } else if (fetchError.message.includes('Failed to fetch')) {
                errorMessage = '🔴 No se pudo conectar con el servidor. Verifica tu conexión a internet.';
                errorType = 'network_error';
              } else if (fetchError.message.includes('NetworkError')) {
                errorMessage = '🔴 Error de red. El servidor podría estar caído.';
                errorType = 'network_error';
              }
              
              setAuthState(AUTH_STATES.UNAUTHENTICATED);
              sendResponse({ 
                ok: false, 
                error: errorType,
                message: errorMessage
              });
              return;
            }
            
            // Manejar respuesta con detalles de brute force
            if (!r.ok) {
              // Error 429: Bloqueado por brute force
              if (r.status === 429 && data.detail?.error === 'too_many_attempts') {
                setAuthState(AUTH_STATES.UNAUTHENTICATED);
                sendResponse({ 
                  ok: false, 
                  error: 'too_many_attempts',
                  message: data.detail.message || 'Demasiados intentos',
                  retry_after: data.detail.retry_after || 120,
                  blocked_until: data.detail.blocked_until
                });
                return;
              }
              
              // Error 401: Credenciales incorrectas
              if (r.status === 401) {
                setAuthState(AUTH_STATES.UNAUTHENTICATED);
                sendResponse({ 
                  ok: false, 
                  error: 'invalid_credentials',
                  message: data.detail?.message || 'Email o contraseña incorrectos',
                  attempts_left: data.detail?.attempts_left || 0,
                  will_block: data.detail?.will_block || false
                });
                return;
              }
              
              // ✅ v10.5.127: Error 403: Cuenta suspendida/baneada
              if (r.status === 403) {
                setAuthState(AUTH_STATES.UNAUTHENTICATED);
                const detail = data.detail || '';
                sendResponse({ 
                  ok: false, 
                  status: 403,
                  error: detail.includes('account_suspended') ? detail : 'account_suspended',
                  message: detail.replace('account_suspended:', '').trim() || 'Cuenta suspendida'
                });
                return;
              }
              
              // Otros errores
              throw new Error('login_'+r.status);
            }
            
            // Login exitoso
            session.access = data.access;
            session.refresh = data.refresh;
            session.exp = data.exp || 0;
            session.user = data.user || null;
            session.device_id = device_id;
            session.is_new_user = data.is_new_user || false;  // 🆕 Guardar si es usuario nuevo
            
            // Guardar debug_mode si viene del backend
            if (data.debug_mode !== undefined) {
              session.debug_mode = data.debug_mode;
            }
            
            await saveSession();
            setAuthState(AUTH_STATES.AUTHENTICATED);
            
            // Notificar a todos los listeners que hubo login exitoso
            chrome.runtime.sendMessage({
              type: 'SESSION.LOGGED_IN',
              user: session.user
            }).catch(() => {});
            
            // Iniciar WebSocket después de login exitoso (con feature detection)
            try {
              if (session.access) {
                // Detectar si API soporta WebSocket
                const features = await detectApiFeatures();
                
                if (features.websocket) {
                  await initRealtimeClient(session.access);
                  registerWebSocketHandlers();
                  
                } else {
                }
              }
            } catch (wsError) {
              // No fallar el login por esto
            }
            
            sendResponse({ 
              ok: true, 
              data: { 
                user: session.user, 
                exp: session.exp,
                debug_mode: session.debug_mode || false,
                is_new_user: session.is_new_user || false  // 🆕 Indicar si es usuario nuevo
              } 
            });
          } catch(e) {
            console.error('[LOGIN] ❌ Error en login:', e);
            setAuthState(AUTH_STATES.UNAUTHENTICATED);
            sendResponse({ ok: false, error: String(e?.message || e) });
          }
          return;
        }

        if (msg?.type === 'SESSION.GET') {
          try {
            const currentState = getAuthState();
            const response = {
              ok: true,
              data: {
                state: currentState,
                access: session.access,
                refresh: session.refresh,
                exp: session.exp,
                user: session.user,
                device_id: session.device_id,
                config: { API_BASE: CONFIG.API_BASE }
              }
            };
            sendResponse(response);
          } catch (e) {
            console.error('[SW] Error en SESSION.GET:', e);
            sendResponse({ ok: false, error: String(e?.message || e) });
          }
          return;
        }

        // 🆕 AUTH.GET_TOKEN - Para obtener token desde panel
        if (msg?.type === 'AUTH.GET_TOKEN') {
          try {
            if (!session?.access) {
              sendResponse({ ok: false, error: 'No hay token disponible' });
              return;
            }
            sendResponse({
              ok: true,
              data: {
                accessToken: session.access
              }
            });
          } catch (e) {
            console.error('[SW] Error en AUTH.GET_TOKEN:', e);
            sendResponse({ ok: false, error: String(e?.message || e) });
          }
          return;
        }

        // ✅ v6.4.0: API.CALL - Llamadas genéricas al backend
        if (msg?.type === 'API.CALL') {
          (async () => {
            try {
              if (!session?.access) {
                sendResponse({ ok: false, error: 'No hay sesión activa' });
                return;
              }
              
              const { method = 'GET', endpoint, body } = msg;
              const url = `${CONFIG.API_BASE}${endpoint}`;
              
              const options = {
                method: method,
                headers: {
                  'Authorization': `Bearer ${session.access}`,
                  'Content-Type': 'application/json'
                }
              };
              
              if (body && (method === 'POST' || method === 'PUT' || method === 'PATCH')) {
                options.body = JSON.stringify(body);
              }
              
              const response = await fetch(url, options);
              
              if (!response.ok) {
                const errorText = await response.text();
                console.error(`[SW] ❌ API.CALL error: ${response.status}`, errorText);
                sendResponse({ ok: false, error: `HTTP ${response.status}: ${errorText}` });
                return;
              }
              
              const data = await response.json();
              sendResponse({ ok: true, data });
              
            } catch (e) {
              console.error('[SW] ❌ Error en API.CALL:', e);
              sendResponse({ ok: false, error: String(e?.message || e) });
            }
          })();
          return;
        }

        // 🆕 WS.GET_STATUS - Consultar estado del WebSocket
        if (msg?.type === 'WS.GET_STATUS') {
          try {
            const rtClient = getRealtimeClient();
            if (!rtClient) {
              sendResponse({ ok: true, connected: false });
              return;
            }
            
            const connected = typeof rtClient.isConnected === 'function' 
              ? rtClient.isConnected() 
              : false;
              
            sendResponse({
              ok: true,
              connected: connected
            });
          } catch (e) {
            console.error('[SW] Error en WS.GET_STATUS:', e);
            sendResponse({ ok: false, connected: false });
          }
          return;
        }

        if (msg?.type === 'SESSION.LOGIN_GOOGLE_TOKENS') {
          session.access = msg.access;
          session.refresh = msg.refresh || session.refresh;
          session.exp = msg.exp || 0;
          session.device_id = session.device_id || await ensureDeviceId();
          await saveSession();
          try {
            const r = await fetch(`${CONFIG.API_BASE}/api/me`, { method:'GET', headers: { Authorization: `Bearer ${session.access}` } });
            session.user = r.ok ? await r.json() : null;
          } catch { session.user = null; }
          await saveSession();
          
          // Notificar a todos los listeners que hubo login exitoso
          chrome.runtime.sendMessage({
            type: 'SESSION.LOGGED_IN',
            user: session.user
          }).catch(() => {});
          
          // Iniciar WebSocket después de login Google exitoso (con feature detection)
          try {
            if (session.access) {
              // Detectar si API soporta WebSocket
              const features = await detectApiFeatures();
              
              if (features.websocket) {
                await initRealtimeClient(session.access);
                registerWebSocketHandlers();
                
              } else {
              }
            }
          } catch (wsError) {
          }
          
          sendResponse({ ok: true, data: { user: session.user, exp: session.exp } });
          return;
        }

        if (msg?.type === 'SESSION.LOGOUT') {
          try {
            // Detener auto-refresh proactivo
            stopAutoRefresh();
            
            // Desconectar WebSocket antes de logout
            disconnectRealtimeClient();
            
            if (session.access && session.device_id) {
              await fetch(`${CONFIG.API_BASE}/api/auth/revoke_device`, {
                method:'POST',
                headers: { 'Content-Type':'application/json', 'Authorization': `Bearer ${session.access}` },
                body: JSON.stringify({ device_id: session.device_id })
              });
            }
          } catch {}
          session.access = null; session.refresh = null; session.exp = 0; session.user = null;
          await saveSession();
          setAuthState(AUTH_STATES.UNAUTHENTICATED);
          sendResponse({ ok:true });
          return;
        }

        // === WALLA CREDS ===
        if (msg?.type === 'WALLA.CREDS') {
          const creds = await ensureWallaCreds({ timeoutMs: 8000, kick: true });
          if (!creds) { sendResponse({ ok:false }); return; }
          sendResponse({ 
            ok: !!(creds && creds.token && creds.deviceId), 
            data: { deviceId: creds.deviceId || null, token_masked: creds.token || null } 
          }); 
          return;
        }
        
        // === WALLA CREDS GET (para uso interno - token completo) ===
        if (msg?.type === 'WALLA.CREDS.GET') {
          const creds = await ensureWallaCreds({ timeoutMs: 8000, kick: true });
          if (!creds || !creds.token || !creds.deviceId) {
            sendResponse({ ok: false, error: 'no_walla_creds' });
            return;
          }
          
          sendResponse({ 
            ok: true, 
            bearer: creds.token,
            deviceId: creds.deviceId,
            appVersion: '8.177.0', // Versión actual de Wallapop
            deviceOs: 'web'
          }); 
          return;
        }
        
        // === WALLA CREDS INVALIDATE (limpiar para forzar recaptura) ===
        if (msg?.type === 'WALLA.CREDS.INVALIDATE') {
          // Limpiar credenciales en memoria (walla.js las volverá a capturar)
          const { invalidateWallaCredsInSW } = await import('./walla.js');
          await invalidateWallaCredsInSW();
          sendResponse({ ok: true });
          return;
        }

        // === WALLA.CREDS_STATUS === v10.5.71: Añadido handler faltante
        if (msg?.type === 'WALLA.CREDS_STATUS') {
          const { getTokenDiagnostics } = await import('./walla.js');
          const status = getTokenDiagnostics();
          sendResponse({ ok: true, ...status });
          return;
        }

        // === WALLA.GET_USER_ID === v10.5.76: Obtener ID alfanumérico del __NEXT_DATA__
        if (msg?.type === 'WALLA.GET_USER_ID') {
          try {
            const { getUserIdFromProfileUrl } = await import('./walla.js');
            const profileUrl = msg.profileUrl;
            if (!profileUrl) {
              sendResponse({ ok: false, error: 'no_profile_url' });
              return;
            }
            const wallaUserId = await getUserIdFromProfileUrl(profileUrl);
            sendResponse({ ok: true, walla_user_id: wallaUserId });
          } catch (e) {
            sendResponse({ ok: false, error: String(e?.message || e) });
          }
          return;
        }

        // === ACCOUNTS.LIST ===
        if (msg?.type === 'ACCOUNTS.LIST') {
          const r = await apiFetch('/api/walla/accounts?with_counts=1', { method:'GET' });
          const j = await r.json().catch(()=> ([]));
          const list = Array.isArray(j) ? j
                    : Array.isArray(j?.accounts) ? j.accounts
                    : Array.isArray(j?.items) ? j.items
                    : [];
          sendResponse({ ok:r.ok, data:list }); 
          return;
        }

        // === ACCOUNT.CREATE === v4.40.0
        if (msg?.type === 'ACCOUNT.CREATE') {
          try {
            const r = await apiFetch('/api/walla/accounts', {
              method: 'POST',
              headers: withDeviceHeaders({ 'Content-Type': 'application/json' }),
              body: JSON.stringify({
                walla_user_id: msg.data?.walla_user_id || null,
                profile_url: msg.data?.profile_url || null,
                alias: msg.data?.alias || null
              })
            });
            const json = await r.json().catch(() => ({}));
            sendResponse({ ok: r.ok, data: json });
          } catch (error) {
            sendResponse({ ok: false, error: String(error) });
          }
          return;
        }

        // === BACKUP RUN ===
        if (msg?.type === 'BACKUP.RUN') {
          const r = await apiFetch(`/api/ads/backup`, {
            method:'POST',
            headers: withDeviceHeaders({ 'Content-Type': 'application/json' }),
            body: JSON.stringify({ scope: msg.scope || 'default' })
          });
          const json = await r.json().catch(()=> ({}));
          sendResponse({ ok: r.ok && (json?.ok !== false), data: json });
          return;
        }

        // === EXPORT + IMPORT ===
        if (msg?.type === 'EXPORT.PROFILE_AND_IMPORT') {
          try {
            const data = await singleFlight('EXPORT.PROFILE_AND_IMPORT', async () => {
              const href = msg.profileUrl || msg.href;
              const exp  = await exportAll(href);
              
              // 🔧 v5.10.0: SIEMPRE llamar a importIntoAPI, incluso si exp.items está vacío
              // Esto permite que el backend sincronice y marque listings como 'deleted'
              // cuando el usuario eliminó todos sus anuncios de Wallapop
              const items = exp.items || [];
              
              const imp = await importIntoAPI({
                wallaUserId: exp.wallaUserId,
                profileUrl: href,
                items: items  // ✅ Envía [], no importa - el backend sincronizará
              });
              const runId = Number(imp?.data?.backup_run_id) || null;
              if (runId) setActivity('backup', { state: 'running', run_id: runId });
              const importOk = !!imp?.ok;
              const topError = importOk ? null : (imp?.data?.error || imp?.data?.detail || 'import_failed');
              return {
                ok: importOk,
                error: topError,
                data: {
                  export:        exp?.stats || null,
                  import:        imp?.data  || null,
                  backup_run_id: runId
                }
              };
            });
            sendResponse(data);
          } catch(e) {
            sendResponse({ ok:false, error:String(e?.message || e) });
          }
          return;
        }

        // === RUN STATUS/LAST/CANCEL ===
        if (msg?.type === 'RUN.STATUS') {
          const r = await apiFetch(`/api/walla/runs/${encodeURIComponent(msg.runId)}/status`, { method:'GET' });
          const j = await r.json().catch(()=> ({}));
          if (j?.status !== 'running') setActivity('backup', { state: (j?.status || 'none'), run_id: j?.run_id || null });
          else setActivity('backup', { state: 'running', run_id: j?.run_id || msg.runId || null });
          sendResponse({ ok:r.ok, data:j });
          return;
        }

        if (msg?.type === 'RUN.LAST') {
          const r = await apiFetch(`/api/walla/runs/last`, { method:'GET' });
          const j = await r.json().catch(()=> ({}));
          setActivity('backup', { state: (j?.status || (j?.exists ? 'ok' : 'none')), run_id: j?.run_id || null });
          sendResponse({ ok:r.ok, data:j });
          return;
        }

        if (msg?.type === 'RUN.RESUME') {
          const r = await apiFetch(`/api/walla/runs/${encodeURIComponent(msg.runId)}/resume`, {
            method:'POST',
            headers: withDeviceHeaders({ 'Content-Type':'application/json' })
          });
          const j = await r.json().catch(()=> ({}));
          setActivity('backup', { state:'running', run_id: msg.runId || null });
          sendResponse({ ok:r.ok, data:j });
          return;
        }

        if (msg?.type === 'RUN.CANCEL') {
          const r = await apiFetch(`/api/walla/runs/${encodeURIComponent(msg.runId)}/cancel`, {
            method:'POST',
            headers: withDeviceHeaders({ 'Content-Type':'application/json' })
          });
          const j = await r.json().catch(()=> ({}));
          setActivity('backup', { state:'cancelled', run_id: j?.run_id || msg.runId || null });
          sendResponse({ ok:r.ok, data:j });
          return;
        }

        // === PUBLISH PENDING ===
        if (msg?.type === 'PUBLISH.PENDING') {
          try {
            const accountId = msg.account_id || null;
            if (!accountId) {
              sendResponse({ ok:false, pending:0, errors:0, resumable:false });
              return;
            }
            
            
            const r = await apiFetch(`/api/walla/publish2/status?account_id=${accountId}`, { 
              method:'GET', 
              cache:'no-store' 
            });
            const j = r.ok ? (await r.json().catch(()=> ({}))) : ({});
            
            
            const pending = Number(j.pending || 0);
            const errors  = Number(j.errors  || 0);
            const resumable = pending > 0; // Si hay pendientes, es reanudable
            
            
            if (pending === 0 && errors === 0) {
            }
            
            const data = { ok:r.ok, pending, errors, resumable, run_id: null };
            
            
            sendResponse(data);
          } catch (err) {
            console.error('[SW] ❌ Error en PUBLISH.PENDING:', err);
            sendResponse({ ok:false, pending:0, errors:0, resumable:false, run_id:null }); 
          }
          return;
        }

        // === PUBLISH.PROCESS_NEXT (Sistema nuevo sin run_id) ===
        // ✅ v4.87.0: Handler extraído a sw/handlers/publish-process-next.js (867 líneas)
        if (msg?.type === 'PUBLISH.PROCESS_NEXT') {
          handleProcessNext(msg, sender, sendResponse);
          return true; // Async response
        }



        if (msg?.type === 'PUBLISH.START') {
          try {
            const accountId = msg.account_id;
            const listingIds = msg.listing_ids || [];  // Array de IDs seleccionados
            const excludeIds = msg.exclude_ids || [];  // ✅ v2.0.0: IDs a excluir
            const isResume = msg.is_resume !== undefined ? msg.is_resume : true; // Default true (reanudar)
            
            if (!accountId) { sendResponse({ ok:false, error:'account_id required' }); return; }
            
            // ✅ v6.7.0: Guardar modo resume en estado global
            publishState.isResumeMode = isResume;
            publishState.resetAt = Date.now();
            
            // ✅ v2.0.0: Construir payload con exclude_ids si hay
            const payload = { 
              account_id: accountId,
              listing_ids: listingIds
            };
            if (excludeIds.length > 0) {
              payload.exclude_ids = excludeIds;
            }
            
            const r = await apiFetch('/api/walla/publish2/start', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify(payload)
            });
            
            if (!r.ok) {
              const errorText = await r.text().catch(() => '');
              sendResponse({ ok: false, error: errorText || `HTTP ${r.status}` });
              return;
            }
            
            const data = await r.json();
            sendResponse({ ok: true, ...data });
          } catch (err) {
            console.error('[SW] Error en PUBLISH.START:', err);
            sendResponse({ ok: false, error: String(err?.message || err) });
          }
          return;
        }


        // === PUBLISH.SET_PAUSED === ✅ v2.2.0
        // ✅ v2.2.2: Broadcast a todas las vistas para sincronizar UI
        if (msg?.type === 'PUBLISH.SET_PAUSED') {
          const paused = !!msg.paused;
          await chrome.storage.local.set({ 'publish.paused': paused });
          console.log(`[SW] Publicación ${paused ? 'PAUSADA' : 'REANUDADA'}`);
          
          // Broadcast a todas las vistas (panel puede estar en múltiples ventanas)
          chrome.runtime.sendMessage({ 
            type: 'PUBLISH.PAUSED_CHANGED', 
            paused 
          }).catch(() => {});
          
          sendResponse({ ok: true, paused });
          return;
        }


        // === PUBLISH.RESET ===
        if (msg?.type === 'PUBLISH.RESET') {
          try {
            const accountId = msg.account_id;
            if (!accountId) { sendResponse({ ok:false, error:'account_id required' }); return; }
            
            
            // ✅ v6.7.0: Resetear estado de publicación
            publishState.isResumeMode = false;
            publishState.resetAt = Date.now();
            
            const r = await apiFetch('/api/walla/publish2/reset', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify({ account_id: accountId })
            });
            
            if (!r.ok) {
              const errorText = await r.text().catch(() => '');
              sendResponse({ ok: false, error: errorText || `HTTP ${r.status}` });
              return;
            }
            
            const data = await r.json();
            sendResponse({ ok: true, ...data });
          } catch (err) {
            console.error('[SW] Error en PUBLISH.RESET:', err);
            sendResponse({ ok: false, error: String(err?.message || err) });
          }
          return;
        }


        // === USER PREFERENCES GET ===
        if (msg?.type === 'USER.PREFERENCES.GET') {
          try {
            const r = await apiFetch('/api/user/preferences', { method: 'GET', cache: 'no-store' });
            if (!r.ok) {
              sendResponse({ ok: false, error: `HTTP ${r.status}` });
              return;
            }
            const data = await r.json();
            sendResponse({ ok: true, data });
          } catch (err) {
            // NO_AUTH es esperado antes del login - no mostrar como error
            const errorMsg = String(err?.message || err);
            if (errorMsg.includes('NO_AUTH') || errorMsg.includes('UNAUTHENTICATED')) {
            } else {
              console.error('[SW] Error en USER.PREFERENCES.GET:', err);
            }
            sendResponse({ ok: false, error: errorMsg });
          }
          return;
        }

        // === USER PREFERENCES UPDATE ===
        if (msg?.type === 'USER.PREFERENCES.UPDATE') {
          try {
            const payload = msg.payload || {};
            const r = await apiFetch('/api/user/preferences', {
              method: 'PATCH',
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify(payload)
            });
            
            if (!r.ok) {
              sendResponse({ ok: false, error: `HTTP ${r.status}` });
              return;
            }
            
            const data = await r.json();
            sendResponse({ ok: true, data });
          } catch (err) {
            // NO_AUTH es esperado antes del login - no mostrar como error
            const errorMsg = String(err?.message || err);
            if (errorMsg.includes('NO_AUTH') || errorMsg.includes('UNAUTHENTICATED')) {
            } else {
              console.error('[SW] Error en USER.PREFERENCES.UPDATE:', err);
            }
            sendResponse({ ok: false, error: errorMsg });
          }
          return;
        }

        // === LISTING.EDITOR.GET - Obtener datos para edición ===
        if (msg?.type === 'LISTING.EDITOR.GET') {
          try {
            
            const { listing_id } = msg.payload;
            
            const response = await apiFetch(
              `/api/walla/listing-editor/${listing_id}`,
              { method: 'GET' }
            );
            
            if (!response.ok) {
              const errorText = await response.text();
              console.error('[SW] ❌ Error HTTP:', response.status, errorText);
              sendResponse({ ok: false, error: `HTTP ${response.status}: ${errorText}` });
              return;
            }
            
            const data = await response.json();
            sendResponse(data);
            
          } catch (error) {
            console.error('[SW] ❌ Error obteniendo listing:', error);
            sendResponse({ ok: false, error: error.message });
          }
          return;
        }

        // === LISTING.EDITOR.SAVE - Guardar cambios ===
        if (msg?.type === 'LISTING.EDITOR.SAVE') {
          try {
            
            const { listing_id, title, description, price, images_order, images_to_delete, inventory_note } = msg.payload;
            
            const response = await apiFetch(
              `/api/walla/listing-editor/${listing_id}`,
              {
                method: 'PATCH',
                body: JSON.stringify({ 
                  title, 
                  description, 
                  price,
                  images_order: images_order || [],
                  images_to_delete: images_to_delete || [],
                  inventory_note: inventory_note || ''  // ✅ v6.8.3
                })
              }
            );
            
            if (!response.ok) {
              const errorText = await response.text();
              console.error('[SW] ❌ Error HTTP:', response.status, errorText);
              sendResponse({ ok: false, error: `HTTP ${response.status}: ${errorText}` });
              return;
            }
            
            const data = await response.json();
            sendResponse(data);
            
          } catch (error) {
            console.error('[SW] ❌ Error guardando listing:', error);
            sendResponse({ ok: false, error: error.message });
          }
          return;
        }

        // === LISTING.EDITOR.UPLOAD_IMAGE - Subir imagen nueva ===
        if (msg?.type === 'LISTING.EDITOR.UPLOAD_IMAGE') {
          try {
            
            const { listing_id, filename, content_type, base64_data } = msg.payload;
            
            const response = await apiFetch(
              `/api/walla/listing-editor/${listing_id}/upload-image`,
              {
                method: 'POST',
                body: JSON.stringify({ 
                  filename, 
                  content_type, 
                  base64_data 
                })
              }
            );
            
            if (!response.ok) {
              const errorText = await response.text();
              console.error('[SW] ❌ Error HTTP:', response.status, errorText);
              sendResponse({ ok: false, error: `HTTP ${response.status}: ${errorText}` });
              return;
            }
            
            const data = await response.json();
            sendResponse(data);
            
          } catch (error) {
            console.error('[SW] ❌ Error subiendo imagen:', error);
            sendResponse({ ok: false, error: error.message });
          }
          return;
        }

        // === LISTING.INVENTORY.UPDATE ===
        if (msg?.type === 'LISTING.INVENTORY.UPDATE') {
          try {
            
            const { listing_web_id, inventory_note } = msg.payload;
            
            // Usar apiFetch que gestiona automáticamente tokens y renovación
            const response = await apiFetch(
              `/api/listings/${listing_web_id}/inventory`,
              {
                method: 'POST',
                body: JSON.stringify({ inventory_note })
              }
            );
            
            if (!response.ok) {
              const errorText = await response.text();
              console.error('[SW] ❌ Error HTTP:', response.status, errorText);
              sendResponse({ success: false, error: `HTTP ${response.status}` });
              return;
            }
            
            const data = await response.json();
            sendResponse({ success: true, data });
            
          } catch (error) {
            console.error('[SW] ❌ Error guardando inventario:', error);
            sendResponse({ success: false, error: error.message });
          }
          return;
        }

        // === No manejado ===
        sendResponse({ ok:false, code:'unhandled_in_modular_sw', type: msg?.type });
      } catch (e) {
        sendResponse({ ok:false, code:'exception', error: String(e?.message || e) });
      }
    })();
    
    // ✅ CRÍTICO: Mantener canal abierto para async
    return true;
  });
}

export function bootstrap() {
  (async () => {
    try {
      await loadConfigOverrides();
      
      // ✅ v10.5.33: Obtener config del backend y cachearla
      // Esto permite que MEDIA_URL y API_BASE vengan del .env del servidor
      await fetchAndCacheRemoteConfig();
      
      await loadSession();
      await initWallaCredsFromStorage();
      setupWallaInterceptors();
      startOutboxDrainInterval();
      await outboxDrainOnce().catch(err => { console.error(`[${filepath.split('/').pop()}] Error:`, err); });
      setupKeepAlive();
      
      // ✅ v7.2.27: #4 - Al abrir extensión, el panel verificará procesos pendientes
      // No creamos alarms aquí porque no sabemos qué cuenta está activa
      // El panel lo manejará cuando el usuario abra el panel
      
      // Iniciar WebSocket si hay sesión válida (con feature detection)
      if (getAuthState() === AUTH_STATES.AUTHENTICATED) {
        try {
          if (session.access) {
            if (!isAccessNearExpiry(60)) {
              // ✅ Iniciar auto-refresh proactivo (renueva cada 5 min)
              startAutoRefresh();
              
              // Detectar si API soporta WebSocket
              const features = await detectApiFeatures();
              
              if (features.websocket) {
                await initRealtimeClient(session.access);
                registerWebSocketHandlers();
              } else {
              }
            } else {
              console.log('[SW] ⚠️ Token cerca de expirar, no iniciando WebSocket al despertar');
              // Intentar refrescar token primero
              try {
                const refreshed = await doRefresh();
                if (refreshed && session.access) {
                  startAutoRefresh();
                  const features = await detectApiFeatures();
                  if (features.websocket) {
                    await initRealtimeClient(session.access);
                    registerWebSocketHandlers();
                  }
                }
              } catch (refreshErr) {
                console.log('[SW] ⚠️ No se pudo refrescar token al despertar');
              }
            }
          }
        } catch (wsError) {
        }
      }
    } catch {}
  })();

  // Proxy SW → backend
  self.addEventListener('fetch', (event) => {
    const url = new URL(event.request.url);
    const isMark =
      (url.origin === self.location.origin) &&
      (url.pathname.endsWith('/listings/mark') || url.pathname.endsWith('/ml/listings/mark'));
    if (isMark && event.request.method === 'POST') {
      event.respondWith((async () => {
        try {
          await loadConfigOverrides();
          await loadSession();
          if (getAuthState() !== AUTH_STATES.AUTHENTICATED) {
            return new Response(JSON.stringify({ detail: 'unauthenticated' }), { status: 401, headers: { 'content-type': 'application/json' } });
          }
          const headers = withDeviceHeaders({
            'content-type': 'application/json',
            'authorization': `Bearer ${session.access}`,
          });
          const body = await event.request.clone().text();
          return await fetch(`${CONFIG.API_BASE}/api/walla/listings/mark`, { method: 'POST', headers, body });
        } catch (e) {
          return new Response(JSON.stringify({ detail: String(e) }), { status: 500, headers: { 'content-type': 'application/json' } });
        }
      })());
    }
  });
}
