/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: sw/api_client.js — Rol: Configuración, sesión JWT, refresh, apiFetch y headers.
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

import { stGet, stSet, stRemove } from './storage.js';
import { fetchWithTimeout } from './utils.js';
import { safeSend } from './bus.js';

/* =========================================================
   CONFIG (única fuente) con overrides en chrome.storage
   ========================================================= */
export const CONFIG_DEFAULTS = {
  // ⚠️ PRODUCCIÓN: URL por defecto (puede ser override en chrome.storage)
  // Para cambiar: Abrir panel → Configuración → Backend URL
  API_BASE: 'https://www.mitiklive.com/fa',
  
  // ⚠️ DESARROLLO: Descomentar para usar localhost
  // API_BASE: 'http://localhost:8000',
  
  AUTH_FREE_PREFIXES: [
    '/api/auth/login',
    '/api/auth/refresh',
    '/api/auth/google',
    '/health',
    '/public'
  ],
};

export const CFG_KEY_PREFIX = 'cfg.';
export let CONFIG = { ...CONFIG_DEFAULTS };

export async function loadConfigOverrides() {
  try {
    const all = await stGet(null);
    const entries = Object.entries(all || {}).filter(([k]) => k.startsWith(CFG_KEY_PREFIX));
    for (const [k, v] of entries) {
      const plain = k.slice(CFG_KEY_PREFIX.length);
      if (plain in CONFIG_DEFAULTS && v != null) CONFIG[plain] = v;
    }
  } catch {}
}

/**
 * ✅ v10.5.33: Obtiene configuración del backend y la cachea en storage
 * 
 * Llama a GET /api/config para obtener MEDIA_URL, API_BASE, etc.
 * Guarda los valores en chrome.storage para que otros módulos los usen.
 * Falla silenciosamente si no hay conexión (usa defaults).
 * 
 * @returns {Promise<boolean>} true si se obtuvo config, false si usó fallback
 */
export async function fetchAndCacheRemoteConfig() {
  try {
    const response = await fetch(`${CONFIG.API_BASE}/api/config`, {
      method: 'GET',
      headers: { 'Accept': 'application/json' }
    });
    
    if (!response.ok) {
      return false;
    }
    
    const data = await response.json();
    
    // Construir API_BASE completo si viene relativo (ej: "/fa")
    let apiBase = data.API_BASE || '/fa';
    if (apiBase.startsWith('/')) {
      // Extraer dominio de CONFIG.API_BASE actual
      const url = new URL(CONFIG.API_BASE);
      apiBase = `${url.origin}${apiBase}`;
    }
    
    // Guardar en storage con prefijo cfg.
    const toSave = {
      [`${CFG_KEY_PREFIX}API_BASE`]: apiBase,
      [`${CFG_KEY_PREFIX}MEDIA_URL`]: data.MEDIA_URL || CONFIG.API_BASE.replace('/fa', ''),
      [`${CFG_KEY_PREFIX}APP_ENV`]: data.APP_ENV || 'production'
    };
    
    await stSet(toSave);
    
    // Aplicar inmediatamente a CONFIG
    CONFIG.API_BASE = apiBase;
    CONFIG.MEDIA_URL = toSave[`${CFG_KEY_PREFIX}MEDIA_URL`];
    CONFIG.APP_ENV = toSave[`${CFG_KEY_PREFIX}APP_ENV`];
    
    return true;
    
  } catch (err) {
    // Silencioso - usa fallback hardcodeado
    return false;
  }
}

function cfgPathIsAuthFree(path) {
  const p = String(path || '').replace(/^https?:\/\/[^/]+/, '');
  return CONFIG.AUTH_FREE_PREFIXES.some(pref => p.startsWith(pref));
}
function isProtectedPath(path) { return !cfgPathIsAuthFree(path); }

/* =========================================================
   Sesión JWT + estados
   ========================================================= */
export const AUTH_STATES = {
  UNAUTHENTICATED: 'UNAUTHENTICATED',
  AUTHENTICATING:  'AUTHENTICATING',
  AUTHENTICATED:   'AUTHENTICATED'
};

let authState = AUTH_STATES.UNAUTHENTICATED;
let authStartedAt = 0;

export function getAuthState() { return authState; }
export function setAuthState(next) {
  if (authState === next) return;
  authState = next;
  if (next === AUTH_STATES.AUTHENTICATING) authStartedAt = Date.now();
  safeSend({ type: 'SESSION.STATE', state: authState });
}

// watchdog anti-loader infinito
setInterval(() => {
  if (authState === AUTH_STATES.AUTHENTICATING && Date.now() - authStartedAt > 15000) {
    authState = AUTH_STATES.UNAUTHENTICATED;
    safeSend({ type:'SESSION.STATE', state: authState, reason:'watchdog_timeout' });
  }
}, 3000);

export const session = {
  access: null, refresh: null, exp: 0, user: null, device_id: null, refreshing: null
};


export function isAccessNearExpiry(skewSec=120){
  return (session.exp || 0) - Math.floor(Date.now()/1000) <= skewSec;
}

export async function ensureDeviceId(){
  const s = await stGet(['session.device_id']);
  if (s['session.device_id']) return s['session.device_id'];
  const did = (crypto.randomUUID ? crypto.randomUUID() : Math.random().toString(36).slice(2) + Date.now().toString(36));
  await stSet({ 'session.device_id': did });
  return did;
}

export async function loadSession() {
  const s = await stGet(['session.access','session.refresh','session.exp','session.user','session.device_id']);
  session.access    = s['session.access']    || null;
  session.refresh   = s['session.refresh']   || null;
  session.exp       = s['session.exp']       || 0;
  session.user      = s['session.user']      || null;
  session.device_id = s['session.device_id'] || await ensureDeviceId();

  const now = Math.floor(Date.now()/1000);
  const valid = !!session.access && (session.exp ? session.exp > now : true);
  setAuthState(valid ? AUTH_STATES.AUTHENTICATED : AUTH_STATES.UNAUTHENTICATED);
}

export async function saveSession() {
  await stSet({
    'session.access': session.access,
    'session.refresh': session.refresh,
    'session.exp': session.exp,
    'session.user': session.user,
    'session.device_id': session.device_id
  });
  safeSend({
    type: 'SESSION.UPDATED',
    access: session.access,
    refresh: session.refresh,
    exp: session.exp,
    user: session.user,
    device_id: session.device_id
  });

  const now = Math.floor(Date.now()/1000);
  const valid = !!session.access && (session.exp ? session.exp > now : true);
  setAuthState(valid ? AUTH_STATES.AUTHENTICATED : AUTH_STATES.UNAUTHENTICATED);
}

let refreshCooldownUntil = 0;
// ✅ CORRECCIÓN: Contador de fallos consecutivos para evitar loop infinito
let refreshFailCount = 0;
const MAX_REFRESH_FAILS = 3;

export async function doRefresh() {
  console.log('[AUTH] 🔄 doRefresh llamado | refresh:', !!session.refresh, '| device_id:', !!session.device_id);
  
  if (!session.refresh || !session.device_id) throw new Error('no_refresh_or_device');

  const now = Date.now();
  if (now < refreshCooldownUntil) throw new Error('refresh_cooldown');
  if (session.refreshing) return session.refreshing;

  // ✅ Si ya falló 3 veces consecutivas → logout limpio
  if (refreshFailCount >= MAX_REFRESH_FAILS) {
    console.error('[AUTH] 🚨 Máximo de fallos de refresh alcanzado. Cerrando sesión...');
    await stRemove(['session.access','session.refresh','session.exp','session.user']);
    session.access = null; session.refresh = null; session.exp = 0; session.user = null;
    refreshFailCount = 0;
    setAuthState(AUTH_STATES.UNAUTHENTICATED);
    throw new Error('max_refresh_fails_exceeded');
  }

  session.refreshing = (async () => {
    try {
      const res = await fetchWithTimeout(`${CONFIG.API_BASE}/api/auth/refresh`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ refresh: session.refresh, device_id: session.device_id })
      }, 10000);

      if (res.status === 429) { 
        refreshCooldownUntil = Date.now() + 120000; 
        refreshFailCount++;
        throw new Error('refresh_429'); 
      }
      
      if (!res.ok) {
        let detail = ''; try { detail = (await res.json()).detail || ''; } catch {}
        if (res.status === 401 && (detail === 'refresh_revoked_or_expired' || detail === 'refresh_reuse_detected')) {
          await stRemove(['session.access','session.refresh','session.exp','session.user']);
          session.access = null; session.refresh = null; session.exp = 0; session.user = null;
          refreshFailCount = 0;
          setAuthState(AUTH_STATES.UNAUTHENTICATED);
        } else {
          refreshFailCount++;
          // ✅ Backoff exponencial: 2s, 4s, 8s
          refreshCooldownUntil = Date.now() + (2000 * Math.pow(2, refreshFailCount - 1));
        }
        throw new Error('refresh_' + (detail || res.status));
      }

      const data = await res.json();
      session.access  = data.access;
      session.refresh = data.refresh || session.refresh;
      session.exp     = data.exp || session.exp;
      await saveSession();

      // ✅ Éxito → resetear contador
      refreshFailCount = 0;
      refreshCooldownUntil = Date.now() + 15000;
      console.log('[AUTH] ✅ Token refrescado OK | exp:', new Date(session.exp * 1000).toLocaleTimeString());
      return session.access;
      
    } catch (err) {
      console.error('[AUTH] ❌ Error en refresh:', err.message || err);
      throw err;
    }
  })();

  try { return await session.refreshing; }
  finally { session.refreshing = null; }
}

export function withDeviceHeaders(headers={}) {
  const h = Object.assign({}, headers);
  if (session.device_id) h['X-Device-Id'] = session.device_id;
  return h;
}

export async function apiFetch(path, init={}) {
  const authedNeeded = isProtectedPath(path);
  
  // ✅ v2.2.4: Si no está autenticado pero hay refresh_token, intentar refrescar
  if (authedNeeded && getAuthState() !== AUTH_STATES.AUTHENTICATED) {
    if (session.refresh) {
      console.log('[AUTH] 🔄 Estado no autenticado pero hay refresh_token, intentando refrescar...');
      try {
        await doRefresh();
        console.log('[AUTH] ✅ Refresh recuperó la sesión');
      } catch (refreshErr) {
        console.error('[AUTH] ❌ Refresh falló, lanzando NO_AUTH:', refreshErr.message);
        throw new Error('NO_AUTH');
      }
    } else {
      throw new Error('NO_AUTH');
    }
  }

  const url = `${CONFIG.API_BASE}${path}`;
  init.headers = Object.assign({}, init.headers || {});
  if (init.body && !init.headers['Content-Type']) init.headers['Content-Type'] = 'application/json';

  // 🔒 SEGURIDAD: Agregar versión de extensión para validación en backend
  if (!init.headers['X-Extension-Version']) {
    try {
      init.headers['X-Extension-Version'] = chrome.runtime.getManifest().version;
    } catch (e) {
    }
  }

  if (session.device_id && !('X-Device-Id' in init.headers)) init.headers['X-Device-Id'] = session.device_id;

  // ✅ v1.8.0: Refresh silencioso - NO cambiar estado para evitar disparar cascada de llamadas
  if (session.access && authedNeeded && isAccessNearExpiry(120)) {
    try { await doRefresh(); } catch {}
  }

  if (session.access && authedNeeded) init.headers.Authorization = `Bearer ${session.access}`;

  let res = await fetch(url, init);
  
  // ✅ v10.5.125: Manejar cuenta suspendida/baneada (403)
  if (res.status === 403) {
    try {
      const cloned = res.clone();
      const data = await cloned.json();
      const detail = data?.detail || '';
      if (detail.includes('account_suspended')) {
        // Notificar al panel que la cuenta está suspendida
        safeSend({ 
          type: 'ACCOUNT.SUSPENDED', 
          reason: detail.replace('account_suspended:', '').trim() || 'Cuenta suspendida'
        });
      }
    } catch {}
    return res;
  }
  
  if (res.status !== 401) return res;
  if (!session.refresh)  return res;

  try {
    setAuthState(AUTH_STATES.AUTHENTICATING);
    await doRefresh();
    if (authedNeeded) init.headers.Authorization = `Bearer ${session.access}`;
    else delete init.headers.Authorization;
    res = await fetch(url, init);
  } catch {}
  return res;
}

/* =========================================================
   Actividad canónica (backup/publish) para broadcast de UI
   ========================================================= */
let activity = { backup:  { state: 'none', run_id: null }, publish: { state: 'none', run_id: null } };
export function setActivity(part, next) {
  activity = { ...activity, [part]: { ...activity[part], ...next } };
  safeSend({ type: 'ACTIVITY.UPDATE', activity });
}
export function getActivity() { return activity; }
