/**
 * SHARED UTILITIES - MitikLive
 * Funciones reutilizables por todas las plataformas
 * NO incluir código específico de plataforma aquí
 */

// ==================== NOTIFICACIONES (TOAST) ====================

/**
 * Muestra notificación toast
 * @param {string} message - Mensaje a mostrar
 * @param {string} type - Tipo: 'info'|'success'|'warning'|'error'
 * @param {number} duration - Duración en ms (default: 5000)
 */
export function showToast(message, type = 'info', duration = 5000) {
  const toast = document.getElementById('ml-toast') || 
                document.getElementById('mil-toast');
  
  if (!toast) {
    console.warn('[Shared] Toast element not found');
    return;
  }
  
  const iconEl = toast.querySelector('.toast-icon');
  const messageEl = toast.querySelector('.toast-message');
  
  const icons = {
    info: 'ℹ️',
    warning: '⚠️',
    error: '❌',
    success: '✅'
  };
  
  toast.className = `ml-toast ${type}`;
  if (iconEl) iconEl.textContent = icons[type] || 'ℹ️';
  if (messageEl) messageEl.textContent = message;
  
  toast.style.display = 'flex';
  
  setTimeout(() => {
    hideToast();
  }, duration);
}

/**
 * Oculta notificación toast
 */
export function hideToast() {
  const toast = document.getElementById('ml-toast') || 
                document.getElementById('mil-toast');
  if (toast) {
    toast.style.display = 'none';
  }
}

// ==================== DELAY / SLEEP ====================

/**
 * Delay asíncrono
 * @param {number} ms - Milisegundos a esperar
 */
export function delay(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

// ==================== STORAGE HELPERS ====================

/**
 * Obtiene valor del storage (SYNC para persistencia)
 * @param {string|string[]} keys - Key o array de keys
 */
export async function getStorage(keys) {
  return await chrome.storage.local.get(keys);
}

/**
 * Guarda valor en storage (SYNC para persistencia)
 * @param {object} data - Objeto con keys y valores
 */
export async function setStorage(data) {
  return await chrome.storage.local.set(data);
}

/**
 * Elimina keys del storage
 * @param {string|string[]} keys - Key o array de keys
 */
export async function removeStorage(keys) {
  return await chrome.storage.local.remove(keys);
}

/**
 * Limpia todo el storage
 */
export async function clearStorage() {
  return await chrome.storage.local.clear();
}

// ==================== TABS / COMUNICACIÓN ====================

/**
 * Envía mensaje a content script de la tab activa
 * @param {object} message - Mensaje a enviar
 * @returns {Promise<any>} Respuesta del content script
 */
export async function sendMessageToActiveTab(message) {
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (!tabs || tabs.length === 0) {
    throw new Error('No active tab found');
  }
  
  return new Promise((resolve, reject) => {
    chrome.tabs.sendMessage(tabs[0].id, message, (response) => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else {
        resolve(response);
      }
    });
  });
}

/**
 * Obtiene la tab activa
 * @returns {Promise<chrome.tabs.Tab>} Tab activa
 */
export async function getActiveTab() {
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (!tabs || tabs.length === 0) {
    throw new Error('No active tab found');
  }
  
  return tabs[0];
}

// ==================== VALIDACIÓN ====================

/**
 * Valida que un ID sea numérico y válido
 * @param {string|number} id - ID a validar
 * @returns {boolean}
 */
export function isValidId(id) {
  if (!id) return false;
  const numId = typeof id === 'string' ? parseInt(id) : id;
  return !isNaN(numId) && numId > 0;
}

/**
 * Valida email básico
 * @param {string} email
 * @returns {boolean}
 */
export function isValidEmail(email) {
  const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return re.test(email);
}

// ==================== FORMATEO ====================

/**
 * Formatea fecha a string legible
 * @param {Date|string} date
 * @returns {string}
 */
export function formatDate(date) {
  const d = date instanceof Date ? date : new Date(date);
  return d.toLocaleDateString('es-ES', {
    year: 'numeric',
    month: 'short',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit'
  });
}

/**
 * Trunca texto a longitud máxima
 * @param {string} text
 * @param {number} maxLength
 * @returns {string}
 */
export function truncate(text, maxLength = 50) {
  if (!text || text.length <= maxLength) return text;
  return text.substring(0, maxLength) + '...';
}

// ==================== LOGGING ====================

/**
 * Logger con prefijo de plataforma
 * @param {string} platform - 'Wallapop'|'Milanuncios'|etc
 * @param {string} module - 'Panel'|'CS'|'API'|etc
 */
export function createLogger(platform, module) {
  const prefix = `[${platform} ${module}]`;
  
  return {
    log: (...args) => console.log(prefix, ...args),
    error: (...args) => console.error(prefix, ...args),
    warn: (...args) => console.warn(prefix, ...args),
    info: (...args) => console.info(prefix, ...args)
  };
}

// ==================== EXPORT DEFAULT ====================

export default {
  showToast,
  hideToast,
  delay,
  getStorage,
  setStorage,
  removeStorage,
  clearStorage,
  sendMessageToActiveTab,
  getActiveTab,
  isValidId,
  isValidEmail,
  formatDate,
  truncate,
  createLogger
};
