/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: service_worker.js — Rol: MV3 SW (entrada ligera que delega en sw/boot.js).
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
  En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa liena de rol  
*/

import { setupListeners, bootstrap } from './sw/boot.js';
import { loadSession, doRefresh, AUTH_STATES, setAuthState } from './sw/api_client.js';
import { getPendingPublish } from './sw/state.js';
// ✅ v10.5.39: Imports ESTÁTICOS (no dinámicos) - requerido por MV3
import { handlePublishProcessNext, continueAfterDelay } from './sw/handlers/publish-process-next.js';
// ✅ v10.5.54: Import estático para refresh de token
import { refreshWallaCredsIfOld } from './sw/walla.js';

// Arranque súper-ligero: solo registra listeners y hace bootstrap mínimo.
setupListeners();
bootstrap();

// ============================================================
// ⏰ ALARMS HANDLER - NIVEL SUPERIOR (crítico para MV3)
// ============================================================
// ✅ v10.5.38: Movido al nivel superior para que funcione cuando el SW se despierta
// ✅ v10.5.39: Usa imports estáticos (imports dinámicos prohibidos en SW de MV3)
// ✅ v10.5.43: Callback robusto que no falla si el canal está cerrado
chrome.alarms.onAlarm.addListener(async (alarm) => {
  console.log(`[SW] ⏰ Alarm recibida: ${alarm.name}`);
  
  // Keep-alive periódico (MitikLive token refresh)
  if (alarm.name === 'mitiklive.refresh') {
    await loadSession();
    try { 
      setAuthState(AUTH_STATES.AUTHENTICATING); 
      const result = await doRefresh(); 
      console.log('[SW] ✅ Token MitikLive refrescado via alarm');
    } catch (e) {
      console.warn('[SW] ⚠️ Error refrescando token MitikLive:', e.message || e);
    }
    return;
  }
  
  // Alarms de publicación (formato: publish-next-{accountId})
  if (alarm.name?.startsWith('publish-next-')) {
    const accountId = parseInt(alarm.name.replace('publish-next-', ''));
    
    console.log(`[SW] ⏰ Alarm de publicación para cuenta ${accountId}`);
    
    // Verificar si hay un item pendiente de publicar (delay largo)
    const pendingState = await getPendingPublish();
    console.log(`[SW] 📦 Estado pendiente:`, pendingState ? `item ${pendingState.itemId}` : 'ninguno');
    
    // ✅ v10.5.43: Callback robusto que nunca falla
    // Cuando venimos de una alarm, no hay canal de mensaje válido
    const safeCallback = (response) => {
      console.log(`[SW] 📤 Respuesta (desde alarm):`, response?.ok ? 'OK' : response?.error || 'unknown');
    };
    
    if (pendingState && pendingState.accountId === accountId) {
      // Hay un item esperando ser publicado después del delay
      console.log(`[SW] ✅ Continuando publicación del item ${pendingState.itemId}`);
      
      try {
        await continueAfterDelay(pendingState, safeCallback);
      } catch (err) {
        console.error('[SW] ❌ Error en continueAfterDelay:', err);
      }
      
    } else {
      // No hay estado pendiente, procesar siguiente item normalmente
      console.log(`[SW] ➡️ Procesando siguiente item (sin estado pendiente)...`);
      
      try {
        // ✅ v10.5.43: Pasar sender=null indica que viene de alarm
        await handlePublishProcessNext({ account_id: accountId }, null, safeCallback);
      } catch (err) {
        console.error('[SW] ❌ Error procesando alarm de publicación:', err);
      }
    }
    
    return;
  }
});


// ============================================================
// 🖱️ CLICK EN ICONO → Abrir panel
// ============================================================

chrome.action.onClicked.addListener((tab) => {
  
  if (!tab || !tab.windowId) {
    console.error('[MitikLive] ❌ Tab inválido');
    return;
  }
  
  chrome.sidePanel.open({ windowId: tab.windowId })
    .then(() => {
    })
    .catch((err) => {
      console.error('[MitikLive] ❌ Error:', err.message);
    });
});

// ============================================================
// 🔘 MENSAJE DESDE CONTENT SCRIPT (botón flotante)
// ============================================================

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // Handler: Abrir panel
  if (message.type === 'OPEN_PANEL') {
    
    if (!sender.tab) {
      sendResponse({ ok: false, error: 'No tab' });
      return true;
    }
    
    chrome.sidePanel.open({ windowId: sender.tab.windowId })
      .then(() => {
        sendResponse({ ok: true });
      })
      .catch((err) => {
        console.error('[MitikLive] ❌ Error:', err.message);
        sendResponse({ ok: false, error: err.message });
      });
    
    return true;
  }
  
  // ✅ v10.5.44: Handler para reenviar mensajes a todas las pestañas de Wallapop
  if (message.type === 'PUBLISH.BROADCAST_TO_WALLAPOP') {
    (async () => {
      try {
        const tabs = await chrome.tabs.query({ url: '*://*.wallapop.com/*' });
        for (const tab of tabs) {
          chrome.tabs.sendMessage(tab.id, message.payload).catch(() => {});
        }
        sendResponse({ ok: true, tabCount: tabs.length });
      } catch (e) {
        sendResponse({ ok: false, error: e.message });
      }
    })();
    return true;
  }
  
  // ✅ v10.5.54: Handler para forzar refresh de token desde CS
  if (message.type === 'TOKEN.FORCE_REFRESH') {
    (async () => {
      try {
        // Obtener tab de Wallapop activa
        const tabs = await chrome.tabs.query({ url: '*://*.wallapop.com/*', active: true });
        const tabId = tabs[0]?.id || sender.tab?.id;
        
        if (!tabId) {
          sendResponse({ ok: false, error: 'No se encontró tab de Wallapop' });
          return;
        }
        
        // Forzar refresh (forceRefresh = true) - usando import estático
        const result = await refreshWallaCredsIfOld(tabId, 90000, true);
        console.log(`[SW] 🔄 TOKEN.FORCE_REFRESH resultado:`, result.ok ? '✅' : '❌');
        sendResponse({ ok: result.ok, refreshed: result.refreshed });
      } catch (e) {
        console.error('[SW] ❌ Error en TOKEN.FORCE_REFRESH:', e.message);
        sendResponse({ ok: false, error: e.message });
      }
    })();
    return true;
  }
  
  // Handler: Actualizar inventario
  if (message.type === 'LISTING.INVENTORY.UPDATE') {
    
    (async () => {
      try {
        // Intentar obtener de chrome.storage.session primero
        let session = null;
        
        try {
          const result = await chrome.storage.session.get('session');
          session = result?.session;
        } catch (e) {
        }
        
        // Si no hay en chrome.storage.session, intentar desde el mensaje
        if (!session && message.session) {
          session = message.session;
        }
        
        if (!session || !session.access) {
          console.error('[SW] ❌ No hay sesión activa. Session:', session);
          sendResponse({ success: false, error: 'No hay sesión activa' });
          return;
        }
        
        // ✅ v10.5.33: Obtener API_BASE de storage (fuente única de verdad)
        const storageData = await chrome.storage.local.get(['cfg.API_BASE']);
        const apiBase = storageData['cfg.API_BASE'] || 'https://www.mitiklive.com/fa';
        const { listing_web_id, inventory_note } = message.payload;
        
        // 🔒 Token NO se loguea por seguridad
        
        // Hacer request al backend
        const url = `${apiBase}/api/listings/${listing_web_id}/inventory`;
        
        const response = await fetch(url, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${session.access}`
          },
          body: JSON.stringify({ inventory_note })
        });
        
        
        if (!response.ok) {
          const errorText = await response.text();
          console.error('[SW] ❌ Error HTTP:', response.status, errorText);
          sendResponse({ success: false, error: `HTTP ${response.status}: ${errorText}` });
          return;
        }
        
        const data = await response.json();
        sendResponse({ success: true, data });
        
      } catch (error) {
        console.error('[SW] ❌ Error:', error);
        sendResponse({ success: false, error: error.message });
      }
    })();
    
    return true; // Respuesta asíncrona
  }
});

// ============================================================
// 📊 INFO AL INICIAR
// ============================================================

chrome.runtime.onInstalled.addListener((details) => {
});

// ============================================================
// 🔄 KEEP-ALIVE: Evitar suspensión del Service Worker
// ============================================================

let keepAliveInterval = null;

// Iniciar keep-alive cuando se recibe RUN.RESUME
chrome.runtime.onMessage.addListener((message) => {
  if (message.type === 'RUN.RESUME' && !keepAliveInterval) {
    
    // Ping cada 20 segundos para mantener el SW activo
    keepAliveInterval = setInterval(() => {
    }, 20000);
  }
  
  if (message.type === 'PUBLISH.COMPLETED' && keepAliveInterval) {
    clearInterval(keepAliveInterval);
    keepAliveInterval = null;
  }
});
