/*
  Archivo: utils.js — Utilidades compartidas (UI + helpers puros)
  Proyecto: MitikLive Wallapop (MV3)
*/

/* ===== UI: Toasts, Modales, Loader ===== */
const _uStyle = `
.mlui-root{position:fixed;inset:0;pointer-events:none;z-index:2147483647;font-family:system-ui,Segoe UI,Roboto,Arial}
.mlui-toasts{position:fixed;top:16px;left:50%;transform:translateX(-50%);display:flex;flex-direction:column;gap:8px;pointer-events:none}
.mlui-toast{min-width:260px;max-width:420px;padding:10px 12px;border-radius:10px;border:1px solid #2b3658;background:#0b1220f2;color:#e6eaf2;box-shadow:0 8px 30px rgba(0,0,0,.35);pointer-events:auto}
.mlui-toast.ok{border-color:#1b8f5a}
.mlui-toast.warn{border-color:#caa014}
.mlui-toast.err{border-color:#b04848}
.mlui-toast .ttl{font-weight:600;margin-bottom:4px}
.mlui-modal-wrap{position:fixed;inset:0;display:flex;align-items:center;justify-content:center;background:rgba(0,0,0,.45);pointer-events:auto}
.mlui-modal{width:min(92vw,460px);background:#0b1220;color:#e6eaf2;border:1px solid #1f2a44;border-radius:14px;box-shadow:0 12px 42px rgba(0,0,0,.5);padding:14px}
.mlui-modal h3{margin:0 0 6px 0}
.mlui-modal .body{margin:8px 0 12px 0}
.mlui-modal .btns{display:flex;justify-content:flex-end;gap:8px}
.mlui-btn{padding:8px 10px;border-radius:8px;border:1px solid #2b3658;background:#172241;color:#e6eaf2;cursor:pointer}
.mlui-btn.primary{background:#1b8f5a;border-color:#1b8f5a}
.mlui-loader-wrap{position:fixed;inset:0;background:rgba(11,18,32,0.55);display:flex;align-items:center;justify-content:center;pointer-events:auto;opacity:0;transition:opacity .15s ease}
.mlui-loader-wrap.on{opacity:1}
.mlui-loader{display:flex;flex-direction:column;align-items:center;gap:10px;padding:16px 18px;border-radius:12px;border:1px solid #1f2a44;background:#0b1220cc;box-shadow:0 8px 30px rgba(0,0,0,.35)}
.mlui-spinner{width:36px;height:36px;border-radius:50%;border:3px solid #2b3658;border-top-color:#7aa2ff;animation:mlui-spin .9s linear infinite}
@keyframes mlui-spin{to{transform:rotate(360deg)}}
.mlui-loader .msg{color:#e6eaf2;font-weight:600;opacity:.9}
.mlui-loader-cancel{margin-top:8px;padding:6px 14px;border-radius:8px;border:1px solid #2b3658;background:#172241;color:#e6eaf2;cursor:pointer;font-size:13px;transition:all .15s}
.mlui-loader-cancel:hover{background:#1f2a44;border-color:#3a4a6a}
`;

let _root, _toasts;

function ensureUI(){
  if (_root && document.documentElement.contains(_root)) return;
  if (!_root){
    _root = document.createElement('div');
    _root.className = 'mlui-root';
    const style = document.createElement('style'); style.textContent = _uStyle; _root.appendChild(style);
    _toasts = document.createElement('div'); _toasts.className = 'mlui-toasts'; _root.appendChild(_toasts);
  }
  document.documentElement.appendChild(_root);
}

/**
 * Muestra una notificación toast temporal
 * @param {string} msg - Mensaje a mostrar
 * @param {string} type - Tipo: 'ok'|'error'|'warning'|'info'
 * @param {Object} options - Opciones
 * @param {string} options.title - Título opcional
 * @param {number} options.ms - Duración en ms (default: 3000)
 */
export function toast(msg, type='ok', {title=null, ms=3000} = {}){
  ensureUI();
  const el = document.createElement('div');
  el.className = `mlui-toast ${type}`;
  el.innerHTML = `${title?`<div class="ttl">${title}</div>`:''}<div class="msg">${msg}</div>`;
  _toasts.appendChild(el);
  const t = setTimeout(()=>{el.remove();}, ms);
  el.addEventListener('click', ()=>{ clearTimeout(t); el.remove(); }, {once:true});
}

/**
 * Muestra un modal con botones personalizables
 * @param {Object} options - Configuración del modal
 * @param {string} options.title - Título del modal
 * @param {string} options.html - Contenido HTML del modal
 * @param {Array} options.buttons - Array de botones: [{text, value, primary}]
 * @returns {Promise<string>} - Valor del botón pulsado
 */
export function showModal({title='Confirmar', html='', buttons=[{text:'OK', value:'ok', primary:true}]}={}){
  return new Promise(res=>{
    ensureUI();
    const wrap = document.createElement('div'); wrap.className='mlui-modal-wrap';
    wrap.innerHTML = `
      <div class="mlui-modal" role="dialog" aria-modal="true" aria-label="${title}">
        <h3>${title}</h3>
        <div class="body">${html}</div>
        <div class="btns"></div>
      </div>`;
    const btns = wrap.querySelector('.btns');
    for(const b of buttons){
      const btn = document.createElement('button');
      btn.className = 'mlui-btn' + (b.primary?' primary':'');
      btn.textContent = b.text;
      btn.addEventListener('click', ()=>{ wrap.remove(); res(b.value); });
      btns.appendChild(btn);
    }
    _root.appendChild(wrap);
  });
}

let _loaderEl = null;
let _loaderTimer = null;
let _loaderShownAt = 0;
let _pendingText = null;
let _loaderTimeoutTimer = null;
let _loaderCancelCallback = null;

const LOADER_DELAY_MS = 250;
const LOADER_MIN_MS   = 250;

/**
 * Muestra un loader (spinner) con mensaje
 * @param {string} text - Mensaje a mostrar (default: 'Cargando…')
 * @param {Object} options - Opciones
 * @param {number} options.timeout - Timeout en ms (0 = sin timeout)
 * @param {Function} options.onCancel - Callback al cancelar
 * @param {Function} options.onTimeout - Callback al timeout
 */
export function showLoader(text='Cargando…', { timeout = 0, onCancel = null, onTimeout = null } = {}){
  ensureUI();
  
  // Cancelar timeout anterior si existe
  if (_loaderTimeoutTimer) {
    clearTimeout(_loaderTimeoutTimer);
    _loaderTimeoutTimer = null;
  }
  
  _loaderCancelCallback = onCancel;
  
  if (_loaderEl){
    const m = _loaderEl.querySelector('.msg'); 
    if (m) m.textContent = text;
    
    // Actualizar/añadir botón cancelar
    const cancelBtn = _loaderEl.querySelector('.mlui-loader-cancel');
    if (onCancel && !cancelBtn) {
      const loader = _loaderEl.querySelector('.mlui-loader');
      const btn = document.createElement('button');
      btn.className = 'mlui-loader-cancel';
      btn.textContent = 'Cancelar';
      btn.onclick = () => {
        hideLoader();
        if (_loaderCancelCallback) _loaderCancelCallback();
      };
      loader.appendChild(btn);
    } else if (!onCancel && cancelBtn) {
      cancelBtn.remove();
    }
    return;
  }
  
  _pendingText = text;
  if (_loaderTimer) return;

  _loaderTimer = setTimeout(() => {
    _loaderTimer = null;
    if (_loaderEl) return;

    const wrap = document.createElement('div');
    wrap.className = 'mlui-loader-wrap';
    wrap.innerHTML = `
      <div class="mlui-loader" role="status" aria-live="polite">
        <div class="mlui-spinner"></div>
        <div class="msg">${_pendingText ?? text}</div>
        ${onCancel ? '<button class="mlui-loader-cancel">Cancelar</button>' : ''}
      </div>`;
    _root.appendChild(wrap);
    
    // Listener para botón cancelar
    if (onCancel) {
      const btn = wrap.querySelector('.mlui-loader-cancel');
      btn.onclick = () => {
        hideLoader();
        if (_loaderCancelCallback) _loaderCancelCallback();
        _loaderCancelCallback = null;
      };
    }

    requestAnimationFrame(()=> wrap.classList.add('on'));

    _loaderEl = wrap;
    _loaderShownAt = performance.now();
    _pendingText = null;
    
    // Configurar timeout si se especificó
    if (timeout > 0) {
      _loaderTimeoutTimer = setTimeout(() => {
        hideLoader();
        if (onTimeout) {
          onTimeout();
        } else {
          toast('⏱️ La operación está tardando demasiado', 'warning', 5000);
        }
      }, timeout);
    }
  }, LOADER_DELAY_MS);
}

/**
 * Oculta el loader activo
 * Respeta el tiempo mínimo de visualización para evitar flicker
 */
export function hideLoader(){
  // Limpiar timeout
  if (_loaderTimeoutTimer) {
    clearTimeout(_loaderTimeoutTimer);
    _loaderTimeoutTimer = null;
  }
  
  _loaderCancelCallback = null;
  
  if (_loaderTimer){
    clearTimeout(_loaderTimer);
    _loaderTimer = null;
    _pendingText = null;
    return;
  }

  if (!_loaderEl) return;

  const elapsed = performance.now() - _loaderShownAt;
  const wait = Math.max(0, LOADER_MIN_MS - elapsed);

  const doRemove = () => {
    if (_loaderEl){
      _loaderEl.classList.remove('on');
      setTimeout(() => { _loaderEl?.remove?.(); _loaderEl = null; }, 150);
    }
  };

  if (wait) setTimeout(doRemove, wait);
  else doRemove();
}

/**
 * Actualiza el texto del loader activo
 * @param {string} text - Nuevo texto a mostrar
 */
export function setLoaderText(text){
  if (_loaderEl) {
    const m = _loaderEl.querySelector('.msg');
    if (m) m.textContent = text;
  } else {
    if (_loaderTimer) _pendingText = text;
  }
}

/* ===== HELPERS PUROS (antes duplicados) ===== */

/**
 * Sleep universal
 */
export const sleep = (ms) => new Promise(r => setTimeout(r, ms));

/**
 * Normalización de strings (quita diacríticos, lowercase, trim)
 */
export function normalize(str) {
  return String(str || '')
    .toLowerCase()
    .normalize('NFD')
    .replace(/\p{Diacritic}+/gu, '')
    .trim();
}

/**
 * Normalización simple (espacios → 1, quita zero-width)
 */
export function normalizeSimple(str) {
  return String(str || '')
    .replace(/\s+/g, ' ')
    .replace(/\u200B/g, '')
    .trim();
}

/**
 * Trunca string por code points (para maxLength)
 */
export function truncateCP(str, max) {
  if (!max || max <= 0) return String(str || '');
  const it = str[Symbol.iterator]();
  let out = '', n = 0;
  for (let s = it.next(); !s.done; s = it.next()) {
    if (++n > max) break;
    out += s.value;
  }
  return out;
}

/**
 * Retry con backoff exponencial
 */
export async function retryWithBackoff(fn, {
  maxRetries = 3,
  baseDelay = 250,
  maxDelay = 5000,
  timeoutMs = null,
  onRetry = null
} = {}) {
  let lastError = null;

  for (let attempt = 0; attempt < maxRetries; attempt++) {
    try {
      if (timeoutMs) {
        return await Promise.race([
          fn(),
          new Promise((_, reject) =>
            setTimeout(() => reject(new Error('timeout')), timeoutMs)
          )
        ]);
      }

      return await fn();
    } catch (err) {
      lastError = err;

      if (attempt === maxRetries - 1) {
        throw err;
      }

      const delay = Math.min(baseDelay * Math.pow(2, attempt), maxDelay);

      if (onRetry) {
        onRetry(attempt + 1, delay, err);
      }

      await sleep(delay);
    }
  }

  throw lastError;
}

/**
 * Genera UUID v4
 */
export function uuidv4() {
  return ([1e7] + -1e3 + -4e3 + -8e3 + -1e11)
    .replace(/[018]/g, c => (c ^ crypto.getRandomValues(new Uint8Array(1))[0] & 15 >> c / 4).toString(16));
}

/**
 * CSS.escape seguro (fallback manual)
 */
export function cssEscape(s) {
  return window.CSS?.escape ? CSS.escape(String(s)) : String(s).replace(/"/g, '\\"');
}

/**
 * Extrae alias de URL de perfil (/user/alias-123 → alias)
 */
export function aliasFromProfileUrl(url) {
  try {
    const last = new URL(url).pathname.split('/').filter(Boolean).pop() || '';
    return last.replace(/-\d+$/, '');
  } catch {
    return null;
  }
}

/**
 * Detección de error cerca de un input (aria-invalid, .error, etc)
 */
export function nearError(el) {
  const wrap = el.closest('.inputWrapper, .form-group, label, form') || document;
  const msg = wrap.querySelector('[role="alert"], .inputWrapper--error, .error, .input-error');
  if (!msg) return false;
  const st = getComputedStyle(msg), r = msg.getBoundingClientRect();
  return st.display !== 'none' && st.visibility !== 'hidden' && (st.opacity !== '0' || (r.width > 0 && r.height > 0));
}

/**
 * ✅ v10.5.27: Escapa HTML para prevenir XSS
 * @param {string} text - Texto a escapar
 * @returns {string} Texto escapado
 */
export function escapeHtml(text) {
  if (typeof text !== 'string') return '';
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

/**
 * ✅ v4.0.8: Debounce - Retrasa ejecución hasta que dejen de llamar
 * @param {Function} func - Función a ejecutar
 * @param {number} wait - Milisegundos de espera
 * @returns {Function} Función con debounce
 */
export function debounce(func, wait = 300) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

// ==================== FUNCIONES ADICIONALES (usadas por Milanuncios) ====================
// Estas funciones se añaden sin modificar las existentes de Wallapop

/**
 * Alias de toast para compatibilidad con Milanuncios
 */
export function showToast(message, type = 'info', duration = 5000) {
  // Mapear tipos de Milanuncios a tipos de Wallapop
  const typeMap = { info: 'ok', success: 'ok', warning: 'warn', error: 'err' };
  toast(message, typeMap[type] || 'ok', { ms: duration });
}

/**
 * Oculta todos los toasts
 */
export function hideToast() {
  const toasts = document.querySelectorAll('.mlui-toast');
  toasts.forEach(t => t.remove());
}

/**
 * Alias de sleep para compatibilidad
 */
export const delay = sleep;

/**
 * Obtiene la tab activa
 */
export async function getActiveTab() {
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tabs || tabs.length === 0) throw new Error('No active tab found');
  return tabs[0];
}

/**
 * Envía mensaje a la tab activa
 */
export async function sendMessageToActiveTab(message) {
  const tab = await getActiveTab();
  return new Promise((resolve, reject) => {
    chrome.tabs.sendMessage(tab.id, message, (response) => {
      if (chrome.runtime.lastError) reject(new Error(chrome.runtime.lastError.message));
      else resolve(response);
    });
  });
}

/**
 * Storage helpers
 */
export async function getStorage(keys) {
  return await chrome.storage.local.get(keys);
}

export async function setStorage(data) {
  return await chrome.storage.local.set(data);
}

export async function removeStorage(keys) {
  return await chrome.storage.local.remove(keys);
}

export async function clearStorage() {
  return await chrome.storage.local.clear();
}

/**
 * Logger con prefijo de plataforma
 */
export function createLogger(platform, module) {
  const prefix = `[${platform} ${module}]`;
  return {
    log: (...args) => console.log(prefix, ...args),
    error: (...args) => console.error(prefix, ...args),
    warn: (...args) => console.warn(prefix, ...args),
    info: (...args) => console.info(prefix, ...args)
  };
}

/**
 * Validación
 */
export function isValidId(id) {
  if (!id) return false;
  const numId = typeof id === 'string' ? parseInt(id) : id;
  return !isNaN(numId) && numId > 0;
}

export function isValidEmail(email) {
  return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
}

/**
 * Formateo
 */
export function formatDate(date) {
  const d = date instanceof Date ? date : new Date(date);
  return d.toLocaleDateString('es-ES', {
    year: 'numeric', month: 'short', day: 'numeric',
    hour: '2-digit', minute: '2-digit'
  });
}

export function truncate(text, maxLength = 50) {
  if (!text || text.length <= maxLength) return text;
  return text.substring(0, maxLength) + '...';
}
