/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: publish/shared.js — Rol: Funciones compartidas entre módulos de publicación
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
  
  ✅ v6.5.4: Módulo creado para centralizar funciones compartidas y evitar duplicación
*/

import { sleep } from '../utils.js';
import { TIMEOUTS, DELAYS } from './constants.js';

// =============================================================================
// CARGA DINÁMICA DE MÓDULOS
// =============================================================================

/**
 * Cache para módulos cargados dinámicamente
 */
let domModule = null;
let loggerModule = null;

/**
 * Carga dinámicamente el módulo dom.js
 * @returns {Promise<Object|null>} Módulo dom.js o null si falla
 */
export async function ensureDom() {
  if (domModule) return domModule;
  
  try {
    const mod = await import('../dom.js');
    domModule = mod;
    return mod;
  } catch (e) {
    return null;
  }
}

/**
 * Carga dinámicamente el logger centralizado con fallback
 * @returns {Promise<Object>} Logger con métodos debug, info, warn, error
 */
export async function getLogger() {
  if (loggerModule) return loggerModule;
  
  try {
    const mod = await import('../panel/logger.js');
    loggerModule = mod.logger;
    return loggerModule;
  } catch (e) {
    // Fallback si no está disponible (ej: en content script)
    // ✅ v10.5.109: Silenciados info/debug para no ensuciar consola de Wallapop
    loggerModule = {
      info: () => {},  // Silenciado
      warn: (...args) => console.warn('[publish]', ...args),
      error: (...args) => console.error('[publish]', ...args),
      debug: () => {},  // Silenciado
      log: () => {},  // Silenciado
    };
    return loggerModule;
  }
}

// =============================================================================
// ESPERA INTELIGENTE DEL DOM
// =============================================================================

/**
 * @deprecated ✅ v10.5.31: Usar waitFor de dom.js en su lugar
 * Esta función se mantiene por compatibilidad pero no se recomienda su uso.
 * 
 * Espera hasta que una función getter retorne un valor truthy
 * @param {Function} getter - Función que retorna el elemento/valor a esperar
 * @param {Object} options - Opciones
 * @param {number} options.timeout - Timeout máximo en ms
 * @param {number} options.interval - Intervalo de polling en ms
 * @returns {Promise<any|null>} Valor retornado o null si timeout
 */
export async function waitFor(getter, { timeout = TIMEOUTS.DEFAULT, interval = DELAYS.POLL_FAST } = {}) {
  const t0 = Date.now();
  while (Date.now() - t0 < timeout) {
    try {
      const value = getter();
      if (value) return value;
    } catch (e) {
      // Ignorar errores, seguir intentando
    }
    await sleep(interval);
  }
  return null;
}

/**
 * Versión mejorada de waitFor que usa dom.js si está disponible
 * @param {Function} getter - Función que retorna el elemento/valor a esperar
 * @param {Object} options - Opciones
 * @returns {Promise<any|null>} Valor retornado o null si timeout
 */
export async function waitForWithDom(getter, options = {}) {
  const dom = await ensureDom();
  if (dom?.waitFor) {
    return dom.waitFor(getter, options);
  }
  return waitFor(getter, options);
}

// =============================================================================
// DELAYS HUMANOS
// =============================================================================

/**
 * Genera un delay aleatorio con varianza para simular comportamiento humano
 * @param {number} base - Tiempo base en ms
 * @param {number} variance - Variación como decimal (0.3 = ±30%)
 * @returns {number} Delay en ms con varianza aplicada
 * 
 * @example
 * humanDelay(300)       → 210-390ms (±30%)
 * humanDelay(500, 0.2)  → 400-600ms (±20%)
 * humanDelay(1000, 0.4) → 600-1400ms (±40%)
 */
export function humanDelay(base, variance = 0.3) {
  const min = base * (1 - variance);
  const max = base * (1 + variance);
  return Math.floor(min + Math.random() * (max - min));
}

/**
 * Ejecuta un delay humano aleatorio
 * @param {number} base - Tiempo base en ms
 * @param {number} variance - Variación como decimal
 * @returns {Promise<void>}
 */
export async function humanSleep(base, variance = 0.3) {
  await sleep(humanDelay(base, variance));
}

// =============================================================================
// UTILIDADES DE NORMALIZACIÓN
// =============================================================================

/**
 * Normaliza texto para comparaciones (lowercase, sin acentos, espacios)
 * @param {string} s - Texto a normalizar
 * @returns {string} Texto normalizado
 */
export function normText(s) {
  return String(s || '')
    .normalize('NFD')
    .replace(/[\u0300-\u036f]/g, '')
    .toLowerCase()
    .replace(/\s+/g, ' ')
    .trim();
}

/**
 * Limpia espacios múltiples de un texto
 * @param {string} s - Texto a limpiar
 * @returns {string} Texto limpio
 */
export function cleanText(s) {
  return String(s || '').replace(/\s+/g, ' ').trim();
}

// =============================================================================
// CSS ESCAPE
// =============================================================================

/**
 * Escapa caracteres especiales para selectores CSS
 * @param {string} s - String a escapar
 * @returns {string} String escapado para CSS
 */
export function cssEscape(s) {
  if (typeof CSS !== 'undefined' && CSS.escape) {
    return CSS.escape(s);
  }
  return String(s).replace(/([!"#$%&'()*+,.\/:;<=>?@[\\\]^`{|}~])/g, '\\$1');
}

// =============================================================================
// DEBUG LOGGING
// =============================================================================

// Flag global para debug (puede activarse desde consola)
export const DEBUG = false;

/**
 * Log de debug condicional
 * @param  {...any} args - Argumentos a loggear
 */
export function dlog(...args) {
}

/**
 * Crea un logger con prefijo específico
 * @param {string} prefix - Prefijo para los logs
 * @param {boolean} enabled - Si está habilitado
 * @returns {Function} Función de log
 */
export function createLogger(prefix, enabled = true) {
  return (...args) => {
  };
}
