/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: publish/form-filler.js — Rol: Rellenar formularios de publicación en Wallapop (extraído de publish.js)
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
  
  ✅ v6.5.3: Módulo extraído de publish.js (originalmente 1143 líneas dentro de una función monolítica)
*/

/* ============================================================
   Imports
   ============================================================ */
import { sleep, normalize, truncateCP, nearError } from '../utils.js';
import { kgToTier, norm } from '../dom.js';
import { TIMEOUTS, DELAYS, SELECTORS, RETRY, ERROR_CODES } from './constants.js';
import { 
  retryWithBackoff, 
  isRetryable, 
  classifyError,
  RECOVERY_STRATEGIES,
  createStructuredError,
  serializeError
} from './error-handler.js';
import {
  ensureDom,
  getLogger,
  humanDelay,
  dlog as sharedDlog,
  DEBUG as SHARED_DEBUG
} from './shared.js';

// ✅ v6.10.0: Import dinámico de dom.js (waitFor optimizado) solo si estamos en content script
// (Service Worker no tiene acceso al DOM y causará error)
let waitForOptimized = null;

/* ============================================================
   Utils locales - ✅ v6.5.4: Ahora importados de shared.js
   ============================================================ */
const DEBUG = SHARED_DEBUG;
const dlog = sharedDlog;

// ✅ v6.5.4: Funciones centralizadas importadas de shared.js
// - humanDelay()
// - getLogger()
// - ensureDom()
// - dlog()

/* ============================================================
   Funciones auxiliares de combos (Brand, Model, Storage)
   ============================================================ */

// ✅ v10.4.2: Restaurada declaración de función que faltaba
async function fillBrandCombo(brandValue, { log = console.log } = {}) {
  if (!brandValue) return { ok: false, error: 'no_brand_value' };
  
  // ✅ v6.10.0: Cargar dom.js (waitFor optimizado) dinámicamente
  if (!waitForOptimized) {
    try {
      const module = await import('../dom.js');
      waitForOptimized = module.waitFor;
    } catch (err) {
      // Fallback a polling tradicional
      waitForOptimized = async (fn, {timeout, interval}) => {
        const t0 = Date.now();
        while (Date.now()-t0 < timeout) { 
          const v = fn(); 
          if (v) return v; 
          await sleep(interval); 
        }
        return null;
      };
    }
  }
  
  const logger = await getLogger();
  log(`[Brand] Buscando combo marca para: "${brandValue}"`);
  
  try {
    // 1. Localizar combo
    const combo = document.querySelector('wallapop-combo-box[data-testid="brand"]');
    if (!combo) {
      log('[Brand] ⚠️ Combo de marca no encontrado en DOM');
      return { ok: false, error: 'combo_not_found' };
    }
    
    // 2. Buscar input dentro del combo
    const input = combo.querySelector('walla-text-input')?.querySelector('input');
    if (!input) {
      log('[Brand] ❌ Input de marca no encontrado');
      return { ok: false, error: 'input_not_found' };
    }
    
    // 3. Verificar si ya está seleccionado el valor correcto (case-insensitive)
    const valorActual = input.value || '';
    log(`[Brand] Valor actual: "${valorActual}" | Valor deseado: "${brandValue}"`);
    
    if (valorActual.toLowerCase() === brandValue.toLowerCase()) {
      log(`[Brand] ✅ Marca "${brandValue}" ya está seleccionada, skip`);
      return { ok: true };
    }
    
    // 4. Valor diferente o vacío → Limpiar y escribir
    log(`[Brand] 🔄 Cambiando de "${valorActual}" a "${brandValue}"`);
    
    // Limpiar input (desmarca pre-seleccionado)
    input.focus();
    await sleep(humanDelay(100, 0.2)); // Mínimo delay humano
    input.value = '';
    input.dispatchEvent(new InputEvent('input', { bubbles: true, composed: true }));
    
    // ✅ v6.8.9: Esperar ACTIVAMENTE a que el input se limpie
    await sleep(humanDelay(150, 0.2)); // Delay mínimo para limpieza
    
    // Escribir nuevo valor para filtrar
    log(`[Brand] Escribiendo "${brandValue}" para filtrar...`);
    input.value = brandValue;
    input.dispatchEvent(new InputEvent('input', { bubbles: true, composed: true }));
    
    // 5. ✅ v6.10.0: Espera ACTIVA con waitFor optimizado (MutationObserver)
    log('[Brand] Esperando filtrado de opciones...');
    
    const opciones = await waitForOptimized(() => {
      const listbox = combo.querySelector('[role="listbox"]');
      if (!listbox) return null;
      
      const opts = listbox.querySelectorAll('wallapop-combo-box-item');
      if (opts.length > 0) {
        log(`[Brand] ✅ Opciones filtradas: ${opts.length}`);
        return opts;
      }
      return null;
    }, { timeout: 8000, interval: 200, useMutationObserver: true });
    
    // ✅ v10.5.64: Si no hay opciones, intentar botón "Crear marca" ANTES de fallar
    if (!opciones || opciones.length === 0) {
      log('[Brand] ⚠️ Sin opciones, buscando botón "Crear marca"...');
      
      try {
        // Buscar en múltiples lugares (el listbox puede estar en un slot/portal)
        let emptyState = null;
        let listbox = combo.querySelector('[role="listbox"]');
        
        if (listbox) {
          emptyState = listbox.querySelector('.wallapop-combo-box__empty-state-section');
        }
        
        // Si no está en el combo, buscar en floating-area-content global
        if (!emptyState) {
          const floatingContent = document.querySelector('.wallapop-combo-box__floating-area-content');
          if (floatingContent) {
            listbox = floatingContent.querySelector('[role="listbox"]');
            emptyState = floatingContent.querySelector('.wallapop-combo-box__empty-state-section');
            log(`[Brand] ℹ️ Buscando en floating-area-content...`);
          }
        }
        
        const wallaButton = emptyState?.querySelector('walla-button');
        const shadowButton = wallaButton?.shadowRoot?.querySelector('button');
        
        log(`[Brand] Debug: emptyState=${!!emptyState}, wallaButton=${!!wallaButton}, shadowButton=${!!shadowButton}`);
        
        if (shadowButton) {
          log(`[Brand] 🆕 Encontrado botón "Crear", haciendo click...`);
          
          shadowButton.scrollIntoView?.({ block: 'nearest' });
          await sleep(humanDelay(100, 0.2));
          shadowButton.click();
          await sleep(humanDelay(300, 0.2));
          
          const valorFinal = input.value || '';
          if (valorFinal) {
            log(`[Brand] ✅ Marca "${valorFinal}" creada exitosamente`);
            return { ok: true };
          }
        }
      } catch (e) {
        log(`[Brand] ⚠️ Error buscando botón crear: ${e.message}`);
      }
      
      // Si tampoco funcionó el botón crear, ahora sí fallar
      log('[Brand] ❌ No se encontraron opciones tras filtrar');
      return { 
        ok: false, 
        error: 'no_options_after_filter',
        error_type: 'combo_no_options',
        details: {
          field_name: 'brand',
          selector: 'wallapop-combo-box[data-testid="brand"]',
          json_value: brandValue,
          available_options: []
        }
      };
    }
    
    // 6. Buscar opción - usamos waitForOptimized para buscar de forma fresca
    // (La lista 'opciones' puede estar desactualizada por re-renders de Wallapop)
    
    // 7. Verificar si ya está seleccionada (evitar toggle/desmarcar)
    const brandValueLower = brandValue.toLowerCase();
    const opcionPrevia = Array.from(opciones).find(
      opt => opt.getAttribute('aria-label')?.toLowerCase() === brandValueLower
    );
    if (opcionPrevia?.getAttribute('aria-selected') === 'true') {
      log(`[Brand] ℹ️ Opción "${brandValue}" ya tiene aria-selected=true, skip click`);
      return { ok: true };
    }
    
    // 8. Seleccionar con comportamiento humano
    log(`[Brand] Seleccionando "${brandValue}"...`);
    
    // Esperar a que el filtrado termine y la opción sea visible
    const opcionFinal = await waitForOptimized(() => {
      const listboxFresh = combo.querySelector('[role="listbox"]');
      if (!listboxFresh) return null;
      
      const opcionesFresh = listboxFresh.querySelectorAll('wallapop-combo-box-item');
      // Buscar la opción exacta
      const target = Array.from(opcionesFresh).find(
        opt => opt.getAttribute('aria-label')?.toLowerCase() === brandValue.toLowerCase()
      );
      
      return target || null;
    }, { timeout: 3000, interval: 100 });
    
    // ✅ v10.5.64: Si no encuentra la opción exacta, intentar crear
    if (!opcionFinal) {
      log(`[Brand] ⚠️ Opción "${brandValue}" no encontrada, intentando crear...`);
      
      try {
        // Buscar botón crear en cualquier lugar visible
        const floatingContent = document.querySelector('.wallapop-combo-box__floating-area-content');
        const emptyState = floatingContent?.querySelector('.wallapop-combo-box__empty-state-section');
        const wallaButton = emptyState?.querySelector('walla-button');
        const shadowButton = wallaButton?.shadowRoot?.querySelector('button');
        
        log(`[Brand] Debug fallback2: floating=${!!floatingContent}, empty=${!!emptyState}, btn=${!!shadowButton}`);
        
        if (shadowButton) {
          log(`[Brand] 🆕 Usando botón "Crear" como fallback...`);
          shadowButton.click();
          await sleep(humanDelay(300, 0.2));
          
          const valorFinal = input.value || '';
          if (valorFinal) {
            log(`[Brand] ✅ Marca "${valorFinal}" creada exitosamente`);
            return { ok: true };
          }
        }
      } catch (e) {
        log(`[Brand] ⚠️ Error en fallback crear: ${e.message}`);
      }
      
      log(`[Brand] ❌ Opción "${brandValue}" no encontrada tras esperar filtrado`);
      return { ok: false, error: 'option_not_found_after_filter' };
    }
    
    // Click con eventos completos para mayor compatibilidad
    opcionFinal.scrollIntoView?.({ block: 'nearest' });
    await sleep(humanDelay(50, 0.2));
    
    const r = opcionFinal.getBoundingClientRect();
    const x = r.left + r.width/2;
    const y = r.top + r.height/2;
    opcionFinal.dispatchEvent(new PointerEvent('pointerdown', {bubbles:true, composed:true, clientX:x, clientY:y}));
    opcionFinal.dispatchEvent(new MouseEvent('mousedown', {bubbles:true, composed:true, clientX:x, clientY:y}));
    opcionFinal.dispatchEvent(new MouseEvent('mouseup', {bubbles:true, composed:true, clientX:x, clientY:y}));
    opcionFinal.dispatchEvent(new PointerEvent('pointerup', {bubbles:true, composed:true, clientX:x, clientY:y}));
    opcionFinal.click();
    
    // 9. ✅ v6.8.9: Verificar ACTIVAMENTE que se seleccionó
    const seleccionExitosa = await waitForOptimized(() => {
      const valorActualizado = input.value || '';
      return valorActualizado === brandValue ? true : null;
    }, { timeout: 2000, interval: 100 });
    
    if (seleccionExitosa) {
      log(`[Brand] ✅ Marca "${brandValue}" confirmada`);
      return { ok: true };
    }
    
    log(`[Brand] ⚠️ Verificación: valor final="${valorFinal}" vs esperado="${brandValue}"`);
    return { ok: true }; // Asumimos éxito si el click se ejecutó
    
  } catch (error) {
    logger.error('❌ Error al rellenar marca:', error);
    log(`[Brand] ❌ Exception: ${error.message}`);
    return { ok: false, error: error.message || 'unknown_error' };
  }
}

/**
 * Rellena el combo de Modelo (model) en formularios de tecnología
 * Usa estrategia de escritura + filtrado AJAX para evitar scroll infinito
 * @param {string} modelValue - Valor del modelo desde JSON (ej: "Galaxy S25", "iPhone 14 Pro")
 * @param {object} options - Opciones: { log: function }
 * @returns {Promise<{ok: boolean, error?: string}>}
 */
async function fillModelCombo(modelValue, { log = console.log } = {}) {
  if (!modelValue) return { ok: false, error: 'no_model_value' };
  
  // ✅ v6.10.0: Cargar dom.js (waitFor optimizado) dinámicamente
  if (!waitForOptimized) {
    try {
      const module = await import('../dom.js');
      waitForOptimized = module.waitFor;
    } catch (err) {
      waitForOptimized = async (fn, {timeout, interval}) => {
        const t0 = Date.now();
        while (Date.now()-t0 < timeout) { 
          const v = fn(); 
          if (v) return v; 
          await sleep(interval); 
        }
        return null;
      };
    }
  }
  
  const logger = await getLogger();
  log(`[Model] Buscando combo modelo para: "${modelValue}"`);
  
  try {
    // 1. Localizar combo
    const combo = document.querySelector('wallapop-combo-box[data-testid="model"]');
    if (!combo) {
      log('[Model] ⚠️ Combo de modelo no encontrado en DOM');
      return { ok: false, error: 'combo_not_found' };
    }
    
    // 2. Buscar input dentro del combo
    const input = combo.querySelector('walla-text-input')?.querySelector('input');
    if (!input) {
      log('[Model] ❌ Input de modelo no encontrado');
      return { ok: false, error: 'input_not_found' };
    }
    
    // 3. Verificar si ya está seleccionado el valor correcto (case-insensitive)
    const valorActual = input.value || '';
    log(`[Model] Valor actual: "${valorActual}" | Valor deseado: "${modelValue}"`);
    
    if (valorActual.toLowerCase() === modelValue.toLowerCase()) {
      log(`[Model] ✅ Modelo "${modelValue}" ya está seleccionado, skip`);
      return { ok: true };
    }
    
    // 4. Valor diferente o vacío → Limpiar y escribir
    log(`[Model] 🔄 Cambiando de "${valorActual}" a "${modelValue}"`);
    
    // Focus en el input
    input.focus();
    await sleep(humanDelay(100, 0.2)); // Delay mínimo humano
    
    // Limpiar input (desmarca pre-seleccionado)
    input.value = '';
    input.dispatchEvent(new InputEvent('input', { bubbles: true, composed: true }));
    await sleep(humanDelay(150, 0.2)); // Delay mínimo para limpieza
    
    // 5. Escribir para filtrar (evita scroll infinito)
    log(`[Model] Escribiendo "${modelValue}" para filtrar...`);
    input.value = modelValue;
    input.dispatchEvent(new InputEvent('input', { bubbles: true, composed: true }));
    input.dispatchEvent(new Event('change', { bubbles: true, composed: true }));
    
    // 6. ✅ v6.10.0: Espera ACTIVA con waitFor optimizado (MutationObserver)
    log('[Model] Esperando filtrado AJAX...');
    
    const opciones = await waitForOptimized(() => {
      const listbox = combo.querySelector('[role="listbox"]');
      if (!listbox) return null;
      
      const opts = listbox.querySelectorAll('wallapop-combo-box-item');
      if (opts.length > 0) {
        log(`[Model] ✅ Filtrado completado: ${opts.length} opciones`);
        return opts;
      }
      return null;
    }, { timeout: 10000, interval: 200, useMutationObserver: true });
    
    // Verificación final
    if (!opciones || opciones.length === 0) {
      log('[Model] ❌ No se encontraron opciones tras filtrar');
      return { 
        ok: false, 
        error: 'no_options_after_filter',
        error_type: 'combo_no_options',
        details: {
          field_name: 'model',
          selector: 'wallapop-combo-box[data-testid="model"]',
          json_value: modelValue,
          available_options: []
        }
      };
    }
    
    // 7. Buscar coincidencia case-insensitive por aria-label
    log(`[Model] Buscando coincidencia para "${modelValue}"...`);
    const availableModels = Array.from(opciones)
      .map(o => o.getAttribute('aria-label'))
      .filter(Boolean);
    
    const modelValueLower = modelValue.toLowerCase();
    const opcionExacta = Array.from(opciones).find(
      opt => opt.getAttribute('aria-label')?.toLowerCase() === modelValueLower
    );
    
    if (opcionExacta) {
      // Verificar que NO esté ya seleccionada (evitar toggle/desmarcar)
      const yaSeleccionada = opcionExacta.getAttribute('aria-selected') === 'true';
      if (yaSeleccionada) {
        log(`[Model] ℹ️ Opción "${modelValue}" ya tiene aria-selected=true, skip click`);
        return { ok: true };
      }
      
      // Coincidencia exacta encontrada y no seleccionada
      log(`[Model] ✅ Coincidencia exacta encontrada, seleccionando...`);
      
      // Esperar a que el filtrado termine y el modelo sea visible
      const opcionFinal = await waitForOptimized(() => {
        const listboxFresh = combo.querySelector('[role="listbox"]');
        if (!listboxFresh) return null;
        
        const opcionesFresh = listboxFresh.querySelectorAll('wallapop-combo-box-item');
        const target = Array.from(opcionesFresh).find(
          opt => opt.getAttribute('aria-label')?.toLowerCase() === modelValue.toLowerCase()
        );
        
        if (target && (opcionesFresh.length <= 10 || opcionesFresh[0] === target)) {
          return target;
        }
        return null;
      }, { timeout: 3000, interval: 100 });
      
      if (!opcionFinal) {
        log(`[Model] ❌ Opción "${modelValue}" no encontrada tras esperar filtrado`);
        return { ok: false, error: 'option_not_found_after_filter' };
      }
      
      // Click con eventos completos
      opcionFinal.scrollIntoView?.({ block: 'nearest' });
      await sleep(humanDelay(50, 0.2));
      
      const r = opcionFinal.getBoundingClientRect();
      const x = r.left + r.width/2;
      const y = r.top + r.height/2;
      opcionFinal.dispatchEvent(new PointerEvent('pointerdown', {bubbles:true, composed:true, clientX:x, clientY:y}));
      opcionFinal.dispatchEvent(new MouseEvent('mousedown', {bubbles:true, composed:true, clientX:x, clientY:y}));
      opcionFinal.dispatchEvent(new MouseEvent('mouseup', {bubbles:true, composed:true, clientX:x, clientY:y}));
      opcionFinal.dispatchEvent(new PointerEvent('pointerup', {bubbles:true, composed:true, clientX:x, clientY:y}));
      opcionFinal.click();
      
      // ✅ v6.9.0: Verificar ACTIVAMENTE que se seleccionó
      const seleccionExitosa = await waitForOptimized(() => {
        const valorActualizado = input.value || '';
        return valorActualizado === modelValue ? true : null;
      }, { timeout: 2000, interval: 100 });
      
      if (seleccionExitosa) {
        log(`[Model] ✅ Modelo "${modelValue}" confirmado`);
        return { ok: true };
      }
    }
    
    // 8. Si no hay exacta, listar opciones y seleccionar primera (tolerante)
    log(`[Model] ⚠️ No hay coincidencia exacta, opciones disponibles:`);
    Array.from(opciones).forEach((opt, i) => {
      log(`  ${i}: "${opt.getAttribute('aria-label')}" | selected="${opt.getAttribute('aria-selected')}"`);
    });
    
    // Verificar si la primera ya está seleccionada
    const primeraYaSeleccionada = opciones[0].getAttribute('aria-selected') === 'true';
    if (primeraYaSeleccionada) {
      const labelPrimera = opciones[0].getAttribute('aria-label');
      log(`[Model] ℹ️ Primera opción "${labelPrimera}" ya seleccionada, skip click`);
      return { 
        ok: true, 
        warning: 'already_selected_first',
        warning_details: {
          expected: modelValue,
          actual: labelPrimera,
          available_options: availableModels.slice(0, 20)
        }
      };
    }
    
    // Seleccionar primera opción como fallback
    log(`[Model] Seleccionando primera opción como fallback...`);
    opciones[0].scrollIntoView?.({ block: 'nearest' });
    await sleep(humanDelay(150, 0.2));
    opciones[0].click();
    await sleep(humanDelay(400, 0.2));
    
    const selectedLabel = opciones[0].getAttribute('aria-label');
    log(`[Model] ⚠️ Seleccionado: "${selectedLabel}" (no coincide exactamente)`);
    
    return { 
      ok: true, 
      warning: 'fallback_first_option',
      warning_details: {
        expected: modelValue,
        actual: selectedLabel,
        available_options: availableModels.slice(0, 20)
      }
    };
    
  } catch (error) {
    logger.error('❌ Error al rellenar modelo:', error);
    log(`[Model] ❌ Exception: ${error.message}`);
    return { ok: false, error: error.message || 'unknown_error' };
  }
}

/**
 * Normaliza valores de capacidad de almacenamiento para coincidencia flexible
 * Ejemplos: "128gb" → "128 GB", "1tb" → "1 TB", "256 GB" → "256 GB"
 * @param {string} raw - Valor crudo (ej: "128gb", "1TB", "256 GB")
 * @returns {string} Valor normalizado (ej: "128 GB", "1 TB")
 */
function normalizeStorageCapacity(raw) {
  if (!raw) return '';
  
  // Normalizar a string, limpiar y pasar a lowercase para comparaciones
  let str = String(raw).trim().toLowerCase();
  
  // Casos especiales: "más de", "menos de"
  // Mapeo de variantes a formato estándar
  const specialCases = {
    // Menos de 4 GB
    'menos de 4gb': 'Menos de 4 GB',
    'menos de 4 gb': 'Menos de 4 GB',
    'menosde4gb': 'Menos de 4 GB',
    '<4gb': 'Menos de 4 GB',
    '< 4gb': 'Menos de 4 GB',
    
    // Más de 1 TB
    'mas de 1tb': 'Más de 1 TB',
    'más de 1tb': 'Más de 1 TB',
    'mas de 1 tb': 'Más de 1 TB',
    'más de 1 tb': 'Más de 1 TB',
    'masde1tb': 'Más de 1 TB',
    '>1tb': 'Más de 1 TB',
    '> 1tb': 'Más de 1 TB',
  };
  
  // Buscar caso especial
  if (specialCases[str]) {
    return specialCases[str];
  }
  
  // Patrón: captura número (con decimales opcionales) + unidad (gb/tb/mb)
  const match = str.match(/^(\d+(?:\.\d+)?)\s*(gb|tb|mb)$/i);
  
  if (match) {
    const num = match[1];
    const unit = match[2].toUpperCase();
    return `${num} ${unit}`;
  }
  
  // Si ya tiene el formato correcto, devolverlo tal cual
  return raw.trim();
}

/**
 * Rellena el dropdown de Capacidad de Almacenamiento (storage_capacity)
 * @param {string} storageValue - Valor desde JSON (ej: "128gb", "1tb")
 * @param {object} options - Opciones: { log: function }
 * @returns {Promise<{ok: boolean, error?: string}>}
 */
async function fillStorageCapacityCombo(storageValue, { log = console.log } = {}) {
  if (!storageValue) return { ok: false, error: 'no_storage_value' };
  
  const logger = await getLogger();
  const normalized = normalizeStorageCapacity(storageValue);
  log(`[Storage] Valor del JSON: "${storageValue}" → Normalizado: "${normalized}"`);
  
  try {
    // 1. Localizar dropdown
    const dropdown = document.querySelector('walla-dropdown[data-testid="storage_capacity"]');
    if (!dropdown) {
      log('[Storage] ⚠️ Dropdown de capacidad no encontrado en DOM');
      return { ok: false, error: 'dropdown_not_found' };
    }
    
    // 2. Buscar hidden input (valor actual)
    const hidden = dropdown.querySelector('input[type="hidden"][name="storage_capacity"]');
    if (!hidden) {
      log('[Storage] ❌ Hidden input no encontrado');
      return { ok: false, error: 'hidden_input_not_found' };
    }
    
    // 3. Verificar si ya está seleccionado el valor correcto
    const valorActual = hidden.value || '';
    log(`[Storage] Valor actual (hidden): "${valorActual}"`);
    
    // Normalizar valor actual también para comparar
    const valorActualNorm = normalizeStorageCapacity(valorActual);
    
    if (valorActualNorm === normalized) {
      log(`[Storage] ✅ Capacidad "${normalized}" ya está seleccionada, skip`);
      return { ok: true };
    }
    
    // 4. Buscar trigger para abrir dropdown
    const trigger = dropdown.querySelector('[role="button"]');
    if (!trigger) {
      log('[Storage] ❌ Trigger button no encontrado');
      return { ok: false, error: 'trigger_not_found' };
    }
    
    // 5. Abrir dropdown si no está abierto
    log(`[Storage] Abriendo dropdown...`);
    const isExpanded = trigger.getAttribute('aria-expanded') === 'true';
    if (!isExpanded) {
      trigger.click();
      await sleep(humanDelay(400, 0.2)); // Esperar apertura
    }
    
    // 6. Buscar listbox con opciones
    const listbox = dropdown.querySelector('[role="listbox"]');
    if (!listbox) {
      log('[Storage] ❌ Listbox no se desplegó');
      return { ok: false, error: 'listbox_not_found' };
    }
    
    const opciones = listbox.querySelectorAll('walla-dropdown-item[role="option"]:not([aria-disabled="true"])');
    log(`[Storage] ✅ Dropdown abierto con ${opciones.length} opciones`);
    
    // 7. Buscar coincidencia por aria-label (case-insensitive y normalizado)
    log(`[Storage] Buscando coincidencia para "${normalized}"...`);
    
    const normalizedLower = normalized.toLowerCase();
    
    // Intentar coincidencia case-insensitive primero
    let opcionTarget = Array.from(opciones).find(
      opt => opt.getAttribute('aria-label')?.toLowerCase() === normalizedLower
    );
    
    // Si no hay exacta, intentar con el valor original (case-insensitive)
    if (!opcionTarget && storageValue !== normalized) {
      const storageValueLower = storageValue.toLowerCase();
      opcionTarget = Array.from(opciones).find(
        opt => opt.getAttribute('aria-label')?.toLowerCase() === storageValueLower
      );
    }
    
    // Si aún no hay match, intentar comparación flexible (normalizar ambos + case-insensitive)
    if (!opcionTarget) {
      opcionTarget = Array.from(opciones).find(opt => {
        const label = opt.getAttribute('aria-label') || '';
        return normalizeStorageCapacity(label).toLowerCase() === normalizedLower;
      });
    }
    
    if (!opcionTarget) {
      log(`[Storage] ❌ Capacidad "${normalized}" no encontrada en opciones`);
      const availableOptions = Array.from(opciones)
        .map(o => o.getAttribute('aria-label'))
        .filter(Boolean);
      log(`[Storage] Opciones disponibles: ${availableOptions.join(', ')}`);
      return { 
        ok: false, 
        error: 'storage_not_in_options',
        error_type: 'dropdown_value_not_found',
        details: {
          field_name: 'storage_capacity',
          selector: 'walla-dropdown[data-testid="storage_capacity"]',
          json_value: storageValue,
          normalized_value: normalized,
          available_options: availableOptions.slice(0, 20)
        }
      };
    }
    
    // 8. Verificar que NO esté ya seleccionada (evitar toggle)
    const yaSeleccionada = opcionTarget.getAttribute('aria-selected') === 'true';
    if (yaSeleccionada) {
      log(`[Storage] ℹ️ Opción "${normalized}" ya tiene aria-selected=true, skip click`);
      return { ok: true };
    }
    
    // 9. Seleccionar con comportamiento humano
    const label = opcionTarget.getAttribute('aria-label');
    log(`[Storage] Seleccionando opción: "${label}"...`);
    opcionTarget.scrollIntoView?.({ block: 'nearest' });
    await sleep(humanDelay(100, 0.2));
    opcionTarget.click();
    await sleep(humanDelay(400, 0.2));
    
    // 10. Verificar selección final
    await sleep(200);
    const valorFinal = hidden.value || '';
    log(`[Storage] Valor final en hidden: "${valorFinal}"`);
    
    return { ok: true };
    
  } catch (error) {
    logger.error('❌ Error al rellenar capacidad:', error);
    log(`[Storage] ❌ Exception: ${error.message}`);
    return { ok: false, error: error.message || 'unknown_error' };
  }
}

/* ============================================================
   inspectListingForm - Inspecciona el formulario para detectar campos
   ============================================================ */
async function inspectListingForm() {
  // ✅ v6.10.0: Cargar dom.js (waitFor optimizado) dinámicamente
  if (!waitForOptimized) {
    try {
      const module = await import('../dom.js');
      waitForOptimized = module.waitFor;
    } catch (err) {
      waitForOptimized = async (fn, {timeout, interval}) => {
        const t0 = Date.now();
        while (Date.now()-t0 < timeout) { 
          const v = fn(); 
          if (v) return v; 
          await sleep(interval); 
        }
        return null;
      };
    }
  }
  
  const dom = await ensureDom();
  
  // ✅ v6.10.0: Usar waitFor optimizado
  const waitFor = dom?.waitFor || (async (getter, { timeout=TIMEOUTS.ELEMENT_APPEAR, interval=DELAYS.POLL_FAST } = {}) => {
    return await waitForOptimized(getter, { timeout, interval, useMutationObserver: true });
  });

  const root = await waitFor(() =>
    document.querySelector('#step-listing, tsl-upload-step-form') ||
    document.querySelector('tsl-upload tsl-upload-step-form'),
    { timeout: TIMEOUTS.BUTTON_ENABLED }
  );
  const out = [];
  if (!root) return { ok:false, code:'step_listing_not_found', items:[] };

  const normTxt = (s)=>String(s||'').replace(/\s+/g,' ').trim();

  // Texto: Título
  (function(){
    const el = root.querySelector('input#title[name="title"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Título';
    out.push({ kind:'text', id:el.id||null, name:el.name||'title',
      label:normTxt(label), required:/\*/.test(label), maxlength: el.maxLength>0?el.maxLength:null,
      value: el.value || '' });
  })();

  // Textarea: Descripción
  (function(){
    const el = root.querySelector('textarea#description[name="description"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Descripción';
    out.push({ kind:'textarea', id:el.id||null, name:el.name||'description',
      label:normTxt(label), required:/\*/.test(label), maxlength: el.maxLength>0?el.maxLength:null,
      value: el.value || '' });
  })();

  // Dropdowns genéricos
  (function collectDropdowns(){
    const hosts = root.querySelectorAll('walla-dropdown');
    hosts.forEach(host => {
      const trigger = host.querySelector('.walla-dropdown__inner-input[role="button"]');
      const hidden  = host.querySelector('.walla-dropdown__inner-input__hidden-input');
      if (!trigger || !hidden) return;

      const label = trigger.querySelector('label')?.textContent || host.getAttribute('aria-label') || '';
      const expanded = (trigger.getAttribute('aria-expanded') === 'true');
      const value = hidden.value || '';

      const listbox = host.querySelector('[role="listbox"]');
      const items = listbox ? Array.from(listbox.querySelectorAll('walla-dropdown-item'))
        .map(li => li.getAttribute('aria-label') || li.textContent || '')
        .map(normTxt).filter(Boolean) : [];

      out.push({
        kind:'dropdown',
        label: normTxt(label),
        name: hidden.name || hidden.id || null,
        id: hidden.id || null,
        required:/\*/.test(label),
        expanded,
        value,
        options: items.length ? items : null
      });
    });
  })();

  // Precio
  (function(){
    const el = root.querySelector('input#sale_price[name="sale_price"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Precio';
    out.push({ kind:'text', id:el.id||null, name:el.name||'sale_price',
      label:normTxt(label), required:/\*/.test(label),
      min: el.min?Number(el.min):null, max: el.max?Number(el.max):null,
      value: el.value || '' });
  })();

  // Toggle envío
  (function(){
    const input = root.querySelector('wallapop-toggle input[type="checkbox"]');
    if (!input) return;
    const label = root.querySelector('.ShippingSection__topBottomBorder span')?.textContent || 'Activar envío';
    out.push({ kind:'toggle', label:normTxt(label),
      name: input.name || 'supports_shipping', id: input.id || null, checked: !!input.checked });
  })();

  // Radio peso (solo requerido si el envío está activado)
  (function () {
    const shipOn = !!root.querySelector('wallapop-toggle input[type="checkbox"]')?.checked;
    const radios = Array.from(root.querySelectorAll('tsl-delivery-radio-selector input[type="radio"]'));

    // ✅ v6.6.0: Si no hay radios de peso, NO es un error
    // Puede que la categoría no requiera envío o que Wallapop cambió la estructura
    if (!radios.length) {
      // Solo reportar como informativo, NO como requerido
      if (shipOn) {
        out.push({
          kind: 'radio-group',
          label: '¿Cuánto pesa?',
          name: 'weight_tier',
          required: false,  // ← Cambiado a false para evitar falso positivo
          value: '(no disponible)',
          options: []
        });
      }
      return;
    }

    const label = root.querySelector('#itemWeightTitle')?.textContent || '¿Cuánto pesa?';
    const options = Array.from(root.querySelectorAll('#weightTierLabel')).map(n => normTxt(n.textContent));
    const checked = radios.find(r => r.checked);

    out.push({
      kind: 'radio-group',
      label: normTxt(label),
      name: 'weight_tier',
      required: shipOn,
      value: checked ? checked.value : '',
      options: options.filter(Boolean)
    });
  })();

  // Localización
  (function(){
    const el = document.querySelector('input#location[name="location"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Marca la localización';
    out.push({ kind:'location', label:normTxt(label), name: el.name || 'location', id: el.id || null, value: el.value || '' });
  })();
  
  const missingRequired = out
    .filter(w => w.required && !String(w.value||'').trim())
    .map(w => ({
      name: w.name || w.label || w.kind,
      reason: `Campo requerido vacío: ${w.label || w.name || w.kind}`
    }));

  return { ok: true, items: out, missingRequired };
}

/* ============================================================
   fillListingForm - Función principal exportada
   ============================================================ */

export async function fillListingForm(snapshot = {}, { log = true } = {}) {
  // ✅ v6.10.0: Cargar dom.js (waitFor optimizado) dinámicamente
  if (!waitForOptimized) {
    try {
      const module = await import('../dom.js');
      waitForOptimized = module.waitFor;
    } catch (err) {
      waitForOptimized = async (fn, {timeout, interval}) => {
        const t0 = Date.now();
        while (Date.now()-t0 < timeout) { 
          const v = fn(); 
          if (v) return v; 
          await sleep(interval); 
        }
        return null;
      };
    }
  }
  
  const logger = await getLogger();
  
  const _log = () => {};
  // ✅ Usa norm importado desde dom.js

  const dom = await ensureDom();
  
  // ✅ v6.10.0: Usar waitFor optimizado con MutationObserver
  const waitForLocal = async (fn, {timeout=TIMEOUTS.DEFAULT, interval=DELAYS.POLL_FAST}={})=>{
    return await waitForOptimized(fn, { timeout, interval, useMutationObserver: true });
  };
  
  const waitFor = dom?.waitFor || waitForLocal;

  // 1) Espera formulario (SPA) + campos clave visibles
  
  const root = await waitFor(() => {
    const container = document.querySelector('#step-listing') ||
                     document.querySelector('tsl-upload-step-form') ||
                     document.querySelector('tsl-upload tsl-upload-step-form');
    if (!container) return null;
    
        const titleField = container.querySelector('input#title[name="title"]') ||
                      container.querySelector('input#titulo[name="titulo"]') ||
                      container.querySelector('input[name="title"]');
    
    if (!titleField) return null;
    
    // Verificar que sea visible
    const rect = titleField.getBoundingClientRect();
    const style = getComputedStyle(titleField);
    const isVisible = style.display !== 'none' &&
                     style.visibility !== 'hidden' &&
                     rect.width > 0 && rect.height > 0;
    
    if (!isVisible) return null;
    
    
    return container;
  }, { timeout: TIMEOUTS.FORM_LOAD }); // ✅ Centralizado

  if (!root) return { ok:false, code:'no_form', filled:[], missingRequired:[], skipped:[], lastScan:null };

  const filled = [], skipped = [];
  let missingRequired = [];
  
  // ✅ v6.5.3: tryDD mejorado con retry inteligente
  const tryDD = async (...trys) => {
    
    for (let i = 0; i < trys.length; i++) {
      const t = trys[i];
      
      
      await sleep(220);
      
      try {
        const startTime = Date.now();
        
        const r = await dom.selectWallaDropdown?.(t);
        const elapsed = Date.now() - startTime;
        
        
        
        
        if (r?.ok) { 
          await sleep(250);
          return r;
        } else {
        }
      } catch (e) { 
        console.error(`[tryDD] ❌ Excepción en intento ${i+1}:`, e.message);
        
      }
    }
    
    return null;
  };
  
  // ✅ v6.5.3: Versión con reintentos automáticos
  const tryDDWithRetry = async (fieldName, attempts, recoveryFn = null) => {
    return await retryWithBackoff(
      async (attempt) => {
        
        
        for (const t of attempts) {
          
          await sleep(humanDelay(200, 0.2));
          
          try {
            const r = await dom.selectWallaDropdown?.(t);
            
            if (r?.ok) { 
              await sleep(humanDelay(250, 0.2)); 
              return { ok: true, result: r };
            }
          } catch (e) { 
            
          }
        }
        
        // No se logró, aplicar recovery si existe
        if (recoveryFn && attempt < RETRY.MAX_ATTEMPTS) {
          await recoveryFn();
        }
        
        return { ok: false, code: ERROR_CODES.DROPDOWN_TIMEOUT };
      },
      {
        maxAttempts: RETRY.MAX_ATTEMPTS,
        initialDelay: RETRY.INITIAL_BACKOFF,
        multiplier: RETRY.BACKOFF_MULTIPLIER,
        context: fieldName,
        log: log,
        shouldRetry: (err) => isRetryable(err.code),
      }
    );
  };
  
  const toSymbol = (c) => ({ EUR:'€', USD:'$', GBP:'£' }[String(c).toUpperCase()] || c);

  // 🔍 v6.8.8: Helper para verificar existencia rápida de campos opcionales
  // Evita timeouts largos en campos que no existen en la categoría actual
  const fieldExistsQuick = (selectors) => {
    if (!Array.isArray(selectors)) selectors = [selectors];
    for (const sel of selectors) {
      if (document.querySelector(sel)) return true;
    }
    return false;
  };

  const fieldExists = (selector) => {
    const el = root.querySelector(selector);
    if (!el) return null;
    // Verificar visibilidad básica
    const rect = el.getBoundingClientRect();
    const style = getComputedStyle(el);
    const visible = style.display !== 'none' && style.visibility !== 'hidden';
    return visible ? el : null;
  };
  
  const tryFillOptional = async (fieldName, selector, fillFn) => {
    try {
      const el = fieldExists(selector);
      if (!el) {
        
        skipped.push(fieldName);
        return false;
      }
      await fillFn(el);
      filled.push(fieldName);
      return true;
    } catch (e) {
      
      skipped.push(fieldName);
      return false;
    }
  };

  // 2) Título — limpiar IA + set robusto
  try {
    const v = snapshot?.title?.original ?? snapshot?.title ?? '';
    const el = root.querySelector('input#title[name="title"]');
    
    if (el) {
      let val = String(v || '');
      const max = (el.maxLength > 0) ? el.maxLength : 0;
      if (max) val = truncateCP(val, max);
      if (typeof dom.clearThenSetText === 'function') {
        await dom.clearThenSetText(el, val);
      } else {
        const { clearThenSetText } = await import('../dom.js');
        await clearThenSetText(el, val);
      }
      (val ? filled : skipped).push('title');
    } else { skipped.push('title'); }
  } catch { skipped.push('title'); }

  // 3) Descripción — limpiar IA + set robusto
  try {
    const v = snapshot?.description?.original ?? snapshot?.description ?? '';
    const el = root.querySelector('textarea#description[name="description"]');
    
    if (el) {
      let val = String(v || '');
      const max = (el.maxLength > 0) ? el.maxLength : 0;
      if (max) val = truncateCP(val, max);
      if (typeof dom.clearThenSetText === 'function') {
        await dom.clearThenSetText(el, val);
      } else {
        const { clearThenSetText } = await import('../dom.js');
        await clearThenSetText(el, val);
      }
      // 🤖 Delay humano después de escribir (±20% variación)
      await sleep(humanDelay(150, 0.2)); // 120-180ms
      (val ? filled : skipped).push('description');
    } else { skipped.push('description'); }
  } catch { skipped.push('description'); }

  // 4) Talla (size)
  const resolveSize = (snap)=>{
    const ta = snap?.type_attributes?.size;
    const fromTA = ta?.text || ta?.icon_text || ta?.label || ta?.name;
    const fromDetails = snap?.characteristics_details?.find(d=>/talla/i.test(d.attribute))?.value;
    const fromChars = (snap?.characteristics?.text||'').match(/\b(\d{2}(?:[.,]\d)?)\b/)?.[1];
    let s = fromTA || fromDetails || fromChars || snap?.size || '';
    s = String(s).replace(',', '.').trim();
    if (/unica|única|one/i.test(s)) return 'Talla única';
    const m = s.match(/\d{2}(?:\.\d)?/);
    return m ? m[0] : s;
  };

  try {
    const want = resolveSize(snapshot);
    
    if (want) {
      // 🔍 v6.8.7: Verificar primero si el campo existe antes de esperar timeout largo
      const sizeExists = !!document.querySelector('input#size[name="size"]');
      if (!sizeExists) {
        
        skipped.push('size');
      } else {
        // ✅ v6.9.1: Size NO necesita AJAX, timeout corto
        let r =
          await dom.selectWallaDropdown?.({ hiddenName:'size', value: want, contains:false, timeout:TIMEOUTS.DROPDOWN_SELECT, log:true }) ||
          await dom.selectWallaDropdown?.({ label:'Talla',  value: want, contains:true,  timeout:TIMEOUTS.DROPDOWN_SELECT, log:true });

        if (!r?.ok) {
          
          const host = document.querySelector('#step-listing input#size[name="size"]')
                       ?.closest('walla-floating-area, walla-dropdown');
          const trigger = host?.querySelector('.walla-dropdown__inner-input[role="button"]');
          if (trigger) {
            trigger.scrollIntoView({ block:'center' });
            trigger.click();
            await new Promise(r=>setTimeout(r, DELAYS.AFTER_SELECT));
            const list = host.querySelector('[role="listbox"]');
            if (list) {
              const wantN = want.toLowerCase();
              const opt = [...list.querySelectorAll('[role="option"]')]
                .find(o => (o.getAttribute('aria-label')||o.textContent||'').toLowerCase().includes(wantN));
              if (opt) {
                (dom?.dispatchPointerClick ? dom.dispatchPointerClick(opt) : opt.click());
                await new Promise(r=>setTimeout(r, DELAYS.AFTER_CLICK));
              }
            }
            r = { ok: !!host.querySelector('input#size[name="size"]')?.value };
          }
        }
        if (r?.ok) filled.push('size'); else skipped.push('size');
      }
    } else {
      
      skipped.push('size');
    }
  } catch { skipped.push('size'); }

  // 5) Moneda
  try {
    const want = (snapshot?.price?.cash?.currency || snapshot?.price?.currency || 'EUR').toUpperCase();
    const hidden = root.querySelector('input#currency[name="currency"]');
    
    // 🔍 v6.9.4: Verificación rápida - si el campo no existe, skip inmediato
    if (!hidden) {
      
      skipped.push('currency');
    } else if (hidden.value && hidden.value.toUpperCase() === want) {
      
      filled.push('currency');
    } else {
      
      // ✅ v6.9.1: Currency NO necesita AJAX, timeout corto
      const r = await tryDD(
        { hiddenName:'currency', value: want,          contains:false, timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
        { hiddenName:'currency', value: toSymbol(want),contains:true,  timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
        { label:'Moneda',        value: want,          contains:false, timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
        { label:'Currency',      value: want,          contains:false, timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
      );
      if (r) filled.push('currency'); else skipped.push('currency');
    }
  } catch { skipped.push('currency'); }

  // 6) Estado / Condición (sinónimos + opciones reales)
    try {
    const conditionStartTime = Date.now();
    const raw = snapshot?.type_attributes?.condition?.text
             || snapshot?.type_attributes?.condition?.value
             || snapshot?.condition || '';
    
    // 🔍 DIAGNÓSTICO: Log COMPLETO para identificar timeout
    
    if (!raw) {
      
      skipped.push('condition');
    } else {
      // ✅ v6.5.5: Mapeo de sinónimos sin depender del escaneo
      
      
      
      const syn = {
        'sin estrenar':'Sin estrenar','a estrenar':'Sin estrenar','brand new':'Sin estrenar',
        'nuevo':'Nuevo','new':'Nuevo',
        'como nuevo':'Como nuevo','like new':'Como nuevo',
        'en buen estado':'En buen estado','good':'En buen estado',
        'aceptable':'En condiciones aceptables','fair':'En condiciones aceptables'
      };
      
      const pick = (() => {
        const n = norm(raw);
        const wanted = syn[n] || Object.entries(syn).find(([k]) => n.includes(k))?.[1] || raw;
        return wanted;
      })();

      

      // 🔍 DIAGNÓSTICO: Llamar tryDD con logs activados
      
      const tryDDStartTime = Date.now();
      // ✅ v6.9.1: Condition NO necesita AJAX, usar timeout corto
      const r = await tryDD(
        { testId:'condition',        value: pick, contains:true,  timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
        { hiddenName:'condition',    value: pick, contains:true,  timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
        { label:'Estado',            value: pick, contains:true,  timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
        { label:'Condition',         value: pick, contains:true,  timeout:TIMEOUTS.DROPDOWN_SELECT, log:true },
      );
      const tryDDElapsed = Date.now() - tryDDStartTime;
      
      // 🔍 DIAGNÓSTICO: Log del resultado
      
      if (r) {
        filled.push('condition');
        
      } else {
        
        skipped.push('condition');
      }
    }
    
    const conditionElapsed = Date.now() - conditionStartTime;
    
  } catch (e) { 
    
    console.error('%c[PUB][fill] ❌ ERROR EN CONDITION', 'color: white; background: #dc3545; font-weight: bold; padding: 4px;');
    console.error('[PUB][fill] ❌ Error:', e.message);
    console.error('[PUB][fill] ❌ Stack:', e.stack);
    skipped.push('condition');
  }

  // 6.3) MARCA (Brand) - Solo para productos tecnológicos
  try {
    const brandValue = snapshot?.type_attributes?.brand?.value || null;
    
    
    if (brandValue) {
      // Si hay valor en JSON, el combo DEBE existir
      const combo = document.querySelector('wallapop-combo-box[data-testid="brand"]');
      
      if (!combo) {
        // ❌ CRÍTICO: JSON tiene marca pero combo no existe
        const errorMsg = `Campo "Marca" requerido en JSON ("${brandValue}") pero combo no encontrado en formulario`;
        
        logger.error('❌ CRÍTICO: Marca en JSON pero combo ausente', { brandValue });
        
        missingRequired.push({
          name: 'brand',
          reason: errorMsg
        });
        
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Combo existe, intentar rellenar
      
      
      // ✅ v6.8.9: OPTIMIZADO - Esperar ACTIVAMENTE que combo marca tenga opciones cargadas
      
      
      const brandCombo = document.querySelector('wallapop-combo-box[data-testid="brand"]');
      if (!brandCombo) {
        const errorMsg = 'Combo marca desapareció del DOM';
        
        missingRequired.push({ name: 'brand', reason: errorMsg });
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Esperar activamente hasta que el combo tenga opciones disponibles
      // ✅ v10.5.48: Usar CLICK para forzar apertura SOLO si el combo está vacío
      const brandInput = brandCombo.querySelector('input');
      if (brandInput) {
        const valorActual = brandInput.value || '';
        // Solo hacer click si está vacío (Wallapop no pre-seleccionó nada)
        if (!valorActual.trim()) {
          brandInput.click(); // Forzar apertura del combo vacío
          await sleep(humanDelay(300, 0.2)); // Esperar que se abra
        }
      }
      
      const brandReady = await waitFor(() => {
        const inp = brandCombo.querySelector('input');
        if (!inp) return null;
        
        // Hacer focus para activar carga de opciones
        inp.focus();
        
        // Verificar si hay opciones disponibles
        const listbox = brandCombo.querySelector('[role="listbox"]');
        if (!listbox) return null;
        
        const items = listbox.querySelectorAll('wallapop-combo-box-item');
        if (items.length > 0) {
          
          return true;
        }
        
        return null;
      }, { timeout: 8000, interval: 200 }); // Poll cada 200ms
      
      if (!brandReady) {
        const errorMsg = 'Combo marca no cargó opciones después de 8s';
        
        logger.error('❌ CRÍTICO: Timeout esperando opciones de marca');
        missingRequired.push({ name: 'brand', reason: errorMsg });
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Cerrar listbox antes de llamar fillBrandCombo (para que la abra limpio)
      if (brandInput) {
        brandInput.blur();
        await sleep(humanDelay(150, 0.2)); // Delay humano mínimo
      }
      
      const result = await fillBrandCombo(brandValue, { log: _log });
      
      if (result.ok) {
        filled.push('brand');
        
      } else {
        // ❌ CRÍTICO: No se pudo seleccionar la marca
        const errorMsg = `No se pudo seleccionar marca "${brandValue}": ${result.error}`;
        
        logger.error('❌ CRÍTICO: Error al seleccionar marca', { brandValue, error: result.error });
        
        missingRequired.push({
          name: 'brand',
          reason: errorMsg
        });
        
        return { ok: false, filled, skipped, missingRequired };
      }
    } else {
      // No hay marca en JSON, es normal para otras categorías
      skipped.push('brand');
    }
  } catch (e) {
    
    logger.error('❌ Error inesperado en marca:', e);
    
    // Si había valor en JSON, es crítico
    const brandValue = snapshot?.type_attributes?.brand?.value;
    if (brandValue) {
      missingRequired.push({
        name: 'brand',
        reason: `Error al procesar marca "${brandValue}": ${e.message}`
      });
      return { ok: false, filled, skipped, missingRequired };
    }
    
    skipped.push('brand');
  }

  // 6.4) MODELO (Model) - Depende de marca seleccionada
  // Esperar un momento para que Wallapop recargue los modelos según la marca seleccionada
  await sleep(humanDelay(500, 0.3));
  
  try {
    const modelValue = snapshot?.type_attributes?.model?.value || null;
    
    
    if (modelValue) {
      // Si hay valor en JSON, el combo DEBE existir
      const combo = document.querySelector('wallapop-combo-box[data-testid="model"]');
      
      if (!combo) {
        // ❌ CRÍTICO: JSON tiene modelo pero combo no existe
        const errorMsg = `Campo "Modelo" requerido en JSON ("${modelValue}") pero combo no encontrado en formulario`;
        
        logger.error('❌ CRÍTICO: Modelo en JSON pero combo ausente', { modelValue });
        
        missingRequired.push({
          name: 'model',
          reason: errorMsg
        });
        
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Combo existe, intentar rellenar
      
      
      // ✅ v6.8.9: OPTIMIZADO - Esperar ACTIVAMENTE que combo modelo tenga opciones cargadas
      
      
      const modelCombo = document.querySelector('wallapop-combo-box[data-testid="model"]');
      if (!modelCombo) {
        const errorMsg = 'Combo modelo desapareció del DOM';
        
        missingRequired.push({ name: 'model', reason: errorMsg });
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Esperar activamente hasta que el combo tenga opciones disponibles
      // ✅ v10.5.48: Usar CLICK para forzar apertura SOLO si el combo está vacío
      const modelInput = modelCombo.querySelector('input');
      if (modelInput) {
        const valorActual = modelInput.value || '';
        // Solo hacer click si está vacío (Wallapop no pre-seleccionó nada)
        if (!valorActual.trim()) {
          modelInput.click(); // Forzar apertura del combo vacío
          await sleep(humanDelay(300, 0.2)); // Esperar que se abra
        }
      }
      
      const modelReady = await waitFor(() => {
        const inp = modelCombo.querySelector('input');
        if (!inp) return null;
        
        // Hacer focus para activar carga de opciones
        inp.focus();
        
        // Verificar si hay opciones disponibles
        const listbox = modelCombo.querySelector('[role="listbox"]');
        if (!listbox) return null;
        
        const items = listbox.querySelectorAll('wallapop-combo-box-item');
        if (items.length > 0) {
          
          return true;
        }
        
        return null;
      }, { timeout: 8000, interval: 200 }); // Poll cada 200ms
      
      if (!modelReady) {
        const errorMsg = 'Combo modelo no cargó opciones después de 8s';
        
        logger.error('❌ CRÍTICO: Timeout esperando opciones de modelo');
        missingRequired.push({ name: 'model', reason: errorMsg });
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Cerrar listbox antes de llamar fillModelCombo (para que la abra limpio)
      if (modelInput) {
        modelInput.blur();
        await sleep(humanDelay(150, 0.2)); // Delay humano mínimo
      }
      
      const result = await fillModelCombo(modelValue, { log: _log });
      
      if (result.ok) {
        filled.push('model');
        if (result.warning === 'fallback_first_option') {
        } else {
          
        }
      } else {
        // ❌ CRÍTICO: No se pudo seleccionar el modelo
        const errorMsg = `No se pudo seleccionar modelo "${modelValue}": ${result.error}`;
        
        logger.error('❌ CRÍTICO: Error al seleccionar modelo', { modelValue, error: result.error });
        
        missingRequired.push({
          name: 'model',
          reason: errorMsg
        });
        
        return { ok: false, filled, skipped, missingRequired };
      }
    } else {
      // No hay modelo en JSON
      
      skipped.push('model');
    }
  } catch (e) {
    
    logger.error('❌ Error inesperado en modelo:', e);
    
    // Si había valor en JSON, es crítico
    const modelValue = snapshot?.type_attributes?.model?.value;
    if (modelValue) {
      missingRequired.push({
        name: 'model',
        reason: `Error al procesar modelo "${modelValue}": ${e.message}`
      });
      return { ok: false, filled, skipped, missingRequired };
    }
    
    skipped.push('model');
  }

  // 6.5) CAPACIDAD DE ALMACENAMIENTO (Storage Capacity)
  try {
    const storageValue = snapshot?.type_attributes?.storage_capacity?.value || null;
    
    
    if (storageValue) {
      // Si hay valor en JSON, el dropdown DEBE existir
      const dropdown = document.querySelector('walla-dropdown[data-testid="storage_capacity"]');
      
      if (!dropdown) {
        // ❌ CRÍTICO: JSON tiene capacidad pero dropdown no existe
        const errorMsg = `Campo "Capacidad de almacenamiento" requerido en JSON ("${storageValue}") pero dropdown no encontrado en formulario`;
        
        logger.error('❌ CRÍTICO: Capacidad en JSON pero dropdown ausente', { storageValue });
        
        missingRequired.push({
          name: 'storage_capacity',
          reason: errorMsg
        });
        
        return { ok: false, filled, skipped, missingRequired };
      }
      
      // Dropdown existe, intentar rellenar
      
      const result = await fillStorageCapacityCombo(storageValue, { log: _log });
      
      if (result.ok) {
        filled.push('storage_capacity');
        
      } else {
        // ❌ CRÍTICO: No se pudo seleccionar la capacidad
        const errorMsg = `No se pudo seleccionar capacidad "${storageValue}": ${result.error}`;
        
        logger.error('❌ CRÍTICO: Error al seleccionar capacidad', { storageValue, error: result.error });
        
        missingRequired.push({
          name: 'storage_capacity',
          reason: errorMsg
        });
        
        return { ok: false, filled, skipped, missingRequired };
      }
    } else {
      // No hay capacidad en JSON
      
      skipped.push('storage_capacity');
    }
  } catch (e) {
    
    logger.error('❌ Error inesperado en capacidad:', e);
    
    // Si había valor en JSON, es crítico
    const storageValue = snapshot?.type_attributes?.storage_capacity?.value;
    if (storageValue) {
      missingRequired.push({
        name: 'storage_capacity',
        reason: `Error al procesar capacidad "${storageValue}": ${e.message}`
      });
      return { ok: false, filled, skipped, missingRequired };
    }
    
    skipped.push('storage_capacity');
  }

  // 6.6) PRECIO
  try {
    const priceValue = snapshot?.price?.cash?.amount 
                    ?? snapshot?.price?.amount 
                    ?? snapshot?.sale_price 
                    ?? null;
    
    
    
    // ✅ v10.5.80: Permitir precio 0 (antes if(priceValue) ignoraba el 0)
    if (priceValue !== null && priceValue !== undefined) {
            
      await waitFor(() => {
        const loaders = root.querySelectorAll('[class*="loader"], [class*="loading"], [class*="spinner"]');
        const hasVisibleLoader = Array.from(loaders).some(l => {
          const style = getComputedStyle(l);
          return style.display !== 'none' && style.visibility !== 'hidden' && style.opacity !== '0';
        });
        return !hasVisibleLoader;
      }, { timeout: TIMEOUTS.BRAND_COMBO, interval: DELAYS.POLL_SLOW });
      
      
      
      // ✅ Múltiples selectores para mayor robustez
      const el = await waitFor(() => {
        // Intentar múltiples selectores
        return root.querySelector('input#sale_price[name="sale_price"]') ||
               root.querySelector('input[name="sale_price"]') ||
               root.querySelector('input#sale_price') ||
               root.querySelector('walla-text-input input[name="sale_price"]') ||
               Array.from(root.querySelectorAll('input[type="text"]')).find(i => 
                 i.name === 'sale_price' || i.id === 'sale_price'
               );
      }, { timeout: TIMEOUTS.DEFAULT, interval: DELAYS.POLL_SLOW });
      
      if (el) {
        
        
        // Scroll al campo para asegurar visibilidad
        try {
          el.scrollIntoView({ block: 'center', behavior: 'smooth' });
          await sleep(300);
        } catch {}
        
        const price = String(Math.round(Number(priceValue)));
        
        
        if (typeof dom.clearThenSetText === 'function') {
          await dom.clearThenSetText(el, price);
        } else {
          const { clearThenSetText } = await import('../dom.js');
          await clearThenSetText(el, price);
        }
        await sleep(400);
        filled.push('sale_price');
        
      } else {
        
        skipped.push('sale_price');
      }
    } else {
      
      skipped.push('sale_price');
    }
  } catch (e) {
    
    skipped.push('sale_price');
  }
  
  // ✅ v10.5.80: STOCK (Nº de unidades)
  try {
    const stockValue = snapshot?.stock ?? null;
    
    // Solo rellenar si viene stock y es diferente de 1 (1 es el default)
    if (stockValue !== null && stockValue !== undefined && stockValue !== 1) {
      const stockEl = await waitFor(() => {
        return root.querySelector('input#stock[name="stock"]') ||
               root.querySelector('input#stock') ||
               root.querySelector('input[name="stock"]');
      }, { timeout: 3000, interval: DELAYS.POLL_SLOW });
      
      if (stockEl) {
        try {
          stockEl.scrollIntoView({ block: 'center', behavior: 'smooth' });
          await sleep(200);
        } catch {}
        
        const stock = String(Math.round(Number(stockValue)));
        
        if (typeof dom.clearThenSetText === 'function') {
          await dom.clearThenSetText(stockEl, stock);
        } else {
          const { clearThenSetText } = await import('../dom.js');
          await clearThenSetText(stockEl, stock);
        }
        await sleep(300);
        filled.push('stock');
      } else {
        skipped.push('stock');
      }
    }
  } catch (e) {
    skipped.push('stock');
  }
  
  // 6 bis) Envío (toggle) + Peso (radio) + Color (multi)
  try {
    const parseKg = (snap) => {
      const raw = snap?.type_attributes?.up_to_kg?.value ?? snap?.type_attributes?.kg?.value ?? snap?.shipping?.up_to_kg ?? null;
      const s = String(raw ?? '').replace(',', '.');
      const n = Number((s.match(/[\d.]+/) || [])[0]);
      return Number.isFinite(n) ? n : null;
    };
    // ✅ Usa kgToTier importado desde dom.js

    // ✅ CRÍTICO: Detectar explícitamente si supports_shipping es false
    const supportsShippingFlag = snapshot?.supports_shipping?.flag;
    const shippingUserAllows = snapshot?.shipping?.user_allows_shipping;
    
    // Si alguno es explícitamente false, wantShip = false
    const wantShip = supportsShippingFlag === false || shippingUserAllows === false 
      ? false 
      : !!(supportsShippingFlag ?? shippingUserAllows ?? false);
    
    const kg = parseKg(snapshot);
    

    // Color (multi)
    try {
      
      
      
      
      // 🤖 Configuración con delays humanos para anti-detección
      const CFG = { 
        openWait: humanDelay(600, 0.25),        // 450-750ms
        afterClearWait: humanDelay(800, 0.25),  // 600-1000ms
        betweenClicks: humanDelay(240, 0.30),   // 168-312ms
        waitTimeout: 10000,                      // Timeout fijo (no aleatorio)
        pollStep: 140,                           // Poll fijo (no aleatorio)
        retries: 4                               // Reintentos fijos
      };
      const normC  = s => String(s||'').toLowerCase().normalize('NFD').replace(/\p{Diacritic}+/gu,'').trim();

      const MAP = new Map(Object.entries({
        white:'white',  blanco:'white',
        black:'black',  negro:'black',
        gray:'gray',    grey:'gray', gris:'gray',
        blue:'blue',    azul:'blue',
        red:'red',      rojo:'red',
        yellow:'yellow',amarillo:'yellow',
        orange:'orange',naranja:'orange',
        brown:'brown',  marron:'brown','marrón':'brown',
        beige:'beige',
        purple:'purple',morado:'purple',
        pink:'pink',    rosa:'pink',
        green:'green',  verde:'green',
        turquoise:'turquoise', turquesa:'turquoise',
        teal:'teal','verde azulado':'teal',
        olive_green:'olive_green', oliva:'olive_green','verde oliva':'olive_green',
        silver:'silver', plateado:'silver',
        gold:'gold', dorado:'gold', 'oro rosa':'rose_gold', 'rose_gold':'rose_gold',
        multicolor:'multicolor',
        other:'other',   otro:'other',
        // Colores adicionales para smartphones
        'gris espacial':'space_gray', 'space_gray':'space_gray',
        coral:'coral',
        'azul marino':'navy_blue', 'navy_blue':'navy_blue',
        violeta:'violet', violet:'violet',
        'verde noche':'night_green', 'night_green':'night_green'
      }));
      
      // ✅ Mapeo INVERSO: ID interno → nombre español (para aria-label)
      const ID_TO_SPANISH = {
        white: 'Blanco', black: 'Negro', gray: 'Gris', blue: 'Azul',
        red: 'Rojo', yellow: 'Amarillo', orange: 'Naranja', brown: 'Marrón',
        beige: 'Beige', purple: 'Morado', pink: 'Rosa', green: 'Verde',
        turquoise: 'Turquesa', teal: 'Verde azulado', olive_green: 'Verde oliva',
        silver: 'Plateado', gold: 'Dorado', rose_gold: 'Oro rosa',
        multicolor: 'Multicolor', other: 'Otro',
        space_gray: 'Gris espacial', coral: 'Coral', navy_blue: 'Azul marino',
        violet: 'Violeta', night_green: 'Verde noche'
      };

      const parseWant = (snap) => {
        const colorAttr = snap?.type_attributes?.color || snap?.type_attributes?.colors;
        const rawValue = colorAttr?.value || snap?.color || '';
        const rawText = colorAttr?.text || colorAttr?.icon_text || 
          snap?.characteristics_details?.find(d => /color/i.test(d.attribute))?.value || '';
        
        let tokens = [];
        let textTokens = [];
        
        // Extraer IDs (values)
        if (rawValue && String(rawValue).trim()) {
          tokens = String(rawValue).split(/[,\s/|]+/).filter(Boolean);
        }
        
        // Extraer textos en español
        if (rawText && String(rawText).trim()) {
          const t = String(rawText).replace(/\s+y\s+/gi, ",").replace(/\s*[·/|]\s*/g, ",");
          textTokens = t.split(",").map(s => s.trim()).filter(Boolean);
        }
        
        // Crear array de objetos {id, text}
        const result = [];
        for (let i = 0; i < Math.max(tokens.length, textTokens.length); i++) {
          const id = tokens[i] ? tokens[i].toLowerCase() : (textTokens[i] ? normC(textTokens[i]) : null);
          const text = textTokens[i] || (tokens[i] ? ID_TO_SPANISH[tokens[i].toLowerCase()] || tokens[i] : null);
          
          if (id && text && !result.some(r => r.id === id)) {
            result.push({ id, text });
          }
        }
        
        return result.slice(0, 2); // Máximo 2 colores
      };

      const wantColors = parseWant(snapshot);
      const want = wantColors.map(c => c.id); // Para compatibilidad con código existente
      
      if (!want.length) { skipped.push('color'); throw 'no_wanted_colors'; }

      const waitFor = async (fn, to=CFG.waitTimeout) => {
        const end = Date.now()+to;
        while (Date.now()<end) { const v = fn(); if (v) return v; await sleep(CFG.pollStep); }
        return null;
      };
      const clickPtr = (el) => {
        if (!el) return;
        const E = window.PointerEvent || window.MouseEvent;
        const r = el.getBoundingClientRect();
        const x = Math.round(r.left + Math.max(4, r.width/2));
        const y = Math.round(r.top  + Math.max(4, r.height/2));
        const ev=(t)=>el.dispatchEvent(new E(t,{bubbles:true,composed:true,cancelable:true,clientX:x,clientY:y,button:0,pointerId:1}));
        try { el.scrollIntoView({block:'center'}); } catch {}
        ev('pointerdown'); ev('mousedown'); ev('mouseup'); ev('pointerup'); el.click?.();
      };
      const findBtnByText = (scope, re) => {
        for (const wb of scope?.querySelectorAll?.('walla-button') || []) {
          const b = wb.shadowRoot?.querySelector('button');
          const t = (b?.textContent || '').trim().toLowerCase();
          if (re.test(t)) return b;
        }
        return null;
      };

      const rootSel = '#step-listing';
      const rootF = document.querySelector(rootSel) || document;
      const inputHidden = rootF.querySelector('input#color[name="color"]');
      if (!inputHidden) { skipped.push('color'); throw 'hidden_not_found'; }

      // ✅ VERIFICACIÓN PREVIA (ANTES de abrir dropdown)
      const currentValue = String(inputHidden.value || '').trim().toLowerCase();
      
      // Verificar si ya tiene los colores correctos
      const currentColors = currentValue ? currentValue.split(',').map(c => c.trim().toLowerCase()).sort() : [];
      const wantedColors = want.map(w => w.toLowerCase()).sort();
      
      const arraysMatch = currentColors.length === wantedColors.length && 
                         currentColors.every((val, idx) => val === wantedColors[idx]);
      
      if (arraysMatch && currentColors.length > 0) {
        filled.push('color');
        throw 'already_selected'; // Salir sin abrir nada
      }
      

      const getHost   = () => inputHidden.closest('walla-floating-area, walla-dropdown');
      const getTrig   = (h) => h?.querySelector('.walla-dropdown__inner-input[role="button"]');
            const getPanel  = (h) => h?.querySelector('[role="listbox"]'); // Sin filtrar aria-multiselectable
      const getHidden = (h) => h?.querySelector('input[type=hidden][name="color"]');
      
            const isMultiSelect = (panel) => panel?.getAttribute('aria-multiselectable') === 'true';

      const ensureOpen = async () => {
        let host = getHost(); if (!host) return null;
        let trigger = getTrig(host); if (!trigger) return null;
        if (trigger.getAttribute('aria-expanded') !== 'true') { clickPtr(trigger); await sleep(CFG.openWait); }
        let panel = getPanel(host);
        if (!panel) { clickPtr(trigger); await sleep(CFG.openWait); panel = getPanel(host); }
        const hidden = getHidden(host);
        if (!panel || !hidden) return null;
        return { host, trigger, panel, hidden, isMulti: isMultiSelect(panel) };
      };

      // abrir (con reintentos)
      
      let ctx = null;
      for (let i=0;i<CFG.retries && !ctx;i++){ 
        
        ctx = await ensureOpen(); 
      }
      if (!ctx) { 
        
        skipped.push('color'); 
        throw 'cannot_open_dropdown'; 
      }
      
      

      // limpiar si hay colores
      const clearIfAny = async () => {
        const { host, hidden, isMulti } = ctx;
        const cur = String(hidden.value||'').trim();
        if (!cur) return; // No hay valor previo, no hay que limpiar
        
        const clear = findBtnByText(host, /borrar/);
        if (clear) { 
          clickPtr(clear); 
          await sleep(CFG.afterClearWait); 
        } else {
          ctx.panel.querySelectorAll('walla-dropdown-item[aria-selected="true"]').forEach(el=>clickPtr(el));
          await sleep(CFG.afterClearWait);
        }
        
        // ✅ Solo reabrir en modo MULTI (en single se cierra y queda limpio)
        if (isMulti) {
          ctx = null;
          for (let i=0;i<CFG.retries && !ctx;i++){ ctx = await ensureOpen(); }
        }
        // En modo single, el dropdown ya se cerró, no reabrir
      };

      await clearIfAny();
      
      // ✅ Si es single y se limpió, necesitamos reabrir para seleccionar
      if (!ctx.isMulti && !ctx.panel) {
        ctx = await ensureOpen();
      }
      
      if (!ctx) { skipped.push('color'); throw 'reopen_after_clear_failed'; }

      // ✅ ESPERA ACTIVA: Asegurar que las opciones estén cargadas
      
      const maxWait = Date.now() + 5000; // máx 5s
      while (Date.now() < maxWait) {
        const opts = ctx.panel.querySelectorAll('walla-dropdown-item[role="option"], input.walla-checkbox__input');
        if (opts.length > 0) {
          
          break;
        }
        await sleep(200);
      }

      // seleccionar
      const pickOne = async (colorId) => {
        const { panel, isMulti } = ctx;
        
        // Obtener el texto español del JSON
        const colorObj = wantColors.find(c => c.id === colorId);
        const spanishText = colorObj?.text || ID_TO_SPANISH[colorId.toLowerCase()] || colorId;
        
        
        
        // ✅ Modo MULTI: Buscar checkbox por ID
        if (isMulti) {
          try {
            const byId = panel.querySelector(`input.walla-checkbox__input#${cssId(colorId)}`);
            if (byId && !byId.disabled) { 
              byId.click(); // ✅ v10.5.71: Click simple en checkbox
              
              return true; 
            }
          } catch {}
        }
        
        // ✅ Modo SINGLE o fallback: Buscar por aria-label usando texto español del JSON
        const wantText = normC(spanishText);
        const opt = Array.from(panel.querySelectorAll('walla-dropdown-item[role="option"]:not([aria-disabled="true"])'))
          .find(o => normC(o.getAttribute('aria-label')||'') === wantText);
        if (opt) { 
          // ✅ v10.5.71: En modo multi, click en checkbox interno; en single, click en item
          if (isMulti) {
            const checkbox = opt.querySelector('input.walla-checkbox__input');
            if (checkbox) {
              checkbox.click();
              return true;
            }
          }
          clickPtr(opt); 
          // ✅ En modo single, el dropdown se cierra automáticamente
          if (!isMulti) await sleep(CFG.betweenClicks);
          return true; 
        }
        
        // Fallback: intentar buscar directamente por el ID
        const optFallback = Array.from(panel.querySelectorAll('walla-dropdown-item[role="option"]:not([aria-disabled="true"])'))
          .find(o => normC(o.getAttribute('aria-label')||'') === normC(colorId));
        if (optFallback) {
          // ✅ v10.5.71: En modo multi, click en checkbox interno
          if (isMulti) {
            const checkbox = optFallback.querySelector('input.walla-checkbox__input');
            if (checkbox) {
              checkbox.click();
              return true;
            }
          }
          clickPtr(optFallback);
          if (!isMulti) await sleep(CFG.betweenClicks);
          return true;
        }
        
        
        return false;
      };

      for (const id of want) {
        
        const ok = await pickOne(id)
          || await pickOne([...MAP.entries()].find(([,v])=>v===id)?.[0] || id);
        if (!ok) {
          
          // Recopilar opciones disponibles para error detallado
          const availableColors = Array.from(ctx.panel.querySelectorAll('walla-dropdown-item[role="option"], input.walla-checkbox__input'))
            .map(el => el.getAttribute('aria-label') || el.id)
            .filter(Boolean)
            .slice(0, 20);
          
          missingRequired.push({
            name: 'color',
            reason: `Color "${id}" no encontrado en opciones del dropdown`,
            error_type: 'color_value_not_found',
            details: {
              field_name: 'color',
              selector: 'walla-dropdown[data-testid="color"]',
              json_value: id,
              available_options: availableColors
            }
          });
          // Cerrar dropdown si está abierto
          if (ctx.trigger.getAttribute('aria-expanded') === 'true') {
            clickPtr(ctx.trigger);
          }
          throw `color_not_found_${id}`;
        }
        else 
        // ✅ v10.5.70: Más delay después de seleccionar color en modo multi
        await sleep(ctx.isMulti ? 800 : CFG.betweenClicks);
      }

      // ✅ v10.5.70: Espera adicional antes de buscar botón Seleccionar
      if (ctx.isMulti) {
        await sleep(500);
        console.log('[Color] Esperando antes de buscar botón Seleccionar...');
      }

      // confirmar y verificar (solo en modo multi)
      {
                if (ctx.isMulti) {
          // ✅ v10.5.69: Buscar sticky-button DIRECTAMENTE (hay múltiples floating-areas)
          let okBtn = null;
          
          // Intentar hasta 3 veces con espera
          for (let attempt = 0; attempt < 3 && !okBtn; attempt++) {
            // Verificar si dropdown está cerrado y reabrirlo
            const trigger = ctx.trigger || getTrig(getHost());
            if (trigger?.getAttribute('aria-expanded') !== 'true') {
              clickPtr(trigger); 
              await sleep(CFG.openWait);
            }
            
            // Esperar a que aparezca el botón
            await sleep(200);
            
            // Buscar DIRECTAMENTE el sticky-button (no dentro de floating-area)
            const stickyContainer = document.querySelector('.walla-dropdown__sticky-button');
            const wallaBtn = stickyContainer?.querySelector('walla-button');
            
            if (wallaBtn) {
              okBtn = wallaBtn.shadowRoot?.querySelector('button');
            }
            
            if (!okBtn) {
              console.log(`[Color] Intento ${attempt + 1}: sticky=${!!stickyContainer}, wallaBtn=${!!wallaBtn}`);
            }
          }
          
          if (okBtn) { 
            // Click con eventos completos
            const r = okBtn.getBoundingClientRect();
            const x = r.left + r.width/2;
            const y = r.top + r.height/2;
            okBtn.dispatchEvent(new PointerEvent('pointerdown', {bubbles:true, composed:true, clientX:x, clientY:y}));
            okBtn.dispatchEvent(new MouseEvent('mousedown', {bubbles:true, composed:true, clientX:x, clientY:y}));
            okBtn.dispatchEvent(new MouseEvent('mouseup', {bubbles:true, composed:true, clientX:x, clientY:y}));
            okBtn.dispatchEvent(new PointerEvent('pointerup', {bubbles:true, composed:true, clientX:x, clientY:y}));
            okBtn.click();
            console.log('[Color] ✅ Botón "Seleccionar" pulsado');
            
            // ✅ v10.5.70: Espera larga después del click
            await sleep(800);
            
            // Forzar cierre: blur y click en body
            document.activeElement?.blur();
            document.body.click();
            
            // Disparar eventos en el hidden por si acaso
            const hiddenAfter = document.querySelector('input#color[name="color"]');
            if (hiddenAfter) {
              hiddenAfter.dispatchEvent(new Event('change', {bubbles: true}));
              hiddenAfter.dispatchEvent(new Event('input', {bubbles: true}));
            }
            
            await sleep(500);
            console.log('[Color] ✅ Confirmación completada');
          } else {
            console.warn('[Color] ⚠️ Botón "Seleccionar" no encontrado en modo multi después de 3 intentos');
          }
        }
        // ✅ En modo single, NO hacer nada - el dropdown ya se cerró automáticamente
      }

      // check final
      {
        const hidden = getHidden(getHost());
        const val = String(hidden?.value||'');
        
        if (val) filled.push('color'); else skipped.push('color');
      }
    } catch (e) {
      
      if (!skipped.includes('color') && !filled.includes('color')) skipped.push('color');
    }

    // ✅ v6.9.4: NUEVA LÓGICA DE ENVÍO con orden correcto
    // PASO 1: Extraer dimensiones del JSON
    const measures = {
      width: snapshot?.type_attributes?.width_cm?.value ?? snapshot?.type_attributes?.width?.value ?? snapshot?.width_cm ?? snapshot?.width ?? null,
      length: snapshot?.type_attributes?.length_cm?.value ?? snapshot?.type_attributes?.length?.value ?? snapshot?.length_cm ?? snapshot?.length ?? null,
      height: snapshot?.type_attributes?.height_cm?.value ?? snapshot?.type_attributes?.height?.value ?? snapshot?.height_cm ?? snapshot?.height ?? null
    };
    
    const hasMeasures = measures.width && measures.length && measures.height;
    
    // PASO 2: Decidir tipo de envío basado en DIMENSIONES (no en is_bulky)
    // - Si tiene las 3 dimensiones → VOLUMINOSO
    // - Si solo tiene peso → ESTÁNDAR
    // - Si no tiene ni dimensiones ni peso → SIN ENVÍO
    const isBulky = hasMeasures;
    const needsShipping = wantShip || hasMeasures || (kg != null);
    
    // PASO 3: Si tiene dimensiones, rellenarlas ANTES de tocar el toggle
    if (hasMeasures) {
      const measuresResult = await dom.fillBulkyMeasures(measures);
      if (measuresResult?.ok) {
        filled.push('product_measures');
        
      } else {
        
        missingRequired.push({
          name: 'bulky_measures',
          reason: `Error rellenando medidas: ${measuresResult?.errors?.join(', ')}`
        });
        return { ok: false, filled, skipped, missingRequired };
      }
    }
    
    // PASO 4: Ajustar toggle según necesidad de envío
    const toggle = root.querySelector('wallapop-toggle input[type="checkbox"]');
    if (toggle) {
      const currentToggleState = !!toggle.checked;
      
      
      if (currentToggleState !== needsShipping) {
        
        
        const wallaToggleComponent = toggle.closest('wallapop-toggle');
        const clickTarget = wallaToggleComponent?.querySelector('.wallapop-toggle__switch') 
                         || wallaToggleComponent?.querySelector('[class*="switch"]')
                         || wallaToggleComponent?.querySelector('label')
                         || wallaToggleComponent
                         || toggle;
        
        if (dom?.dispatchPointerClick) {
          dom.dispatchPointerClick(clickTarget);
        } else {
          clickTarget.click?.();
          toggle.checked = needsShipping;
          toggle.dispatchEvent(new Event('change', { bubbles: true }));
          toggle.dispatchEvent(new Event('input', { bubbles: true }));
        }
        
        // Esperar a que el toggle cambie
        const toggleChanged = await waitFor(() => {
          return toggle.checked === needsShipping ? true : null;
        }, { timeout: 2000, interval: 100 });
        
        if (!toggleChanged) {
          
          toggle.checked = needsShipping;
          toggle.dispatchEvent(new Event('change', { bubbles: true }));
          toggle.dispatchEvent(new Event('input', { bubbles: true }));
        }
        
        
      } else {
        
      }
    }
    
    // PASO 5: Si NO necesita envío, terminar aquí
    if (!needsShipping) {
      
      filled.push('shipping_disabled');
      
      // Esperar a que desaparezcan los radios
      if (toggle?.checked === false) {
        
        await waitFor(() => {
          const radios = document.querySelectorAll('tsl-delivery-radio-selector input[type="radio"]');
          return radios.length === 0 ? true : null;
        }, { timeout: 3000, interval: 150 });
      }
    } else {
      // PASO 6: Configurar tipo de envío (VOLUMINOSO o ESTÁNDAR)
      filled.push(`shipping_type=${isBulky ? 'bulky' : 'standard'}`);
      
      if (isBulky) {
        // 📦 VOLUMINOSO: Seleccionar radiobutton "Voluminoso"
        
        const bulkyRadio = document.querySelector('input[type="radio"][name="bulky"]')
                        || document.querySelector('input[type="radio"][aria-label="bulky"]')
                        || document.querySelector('input[type="radio"][id="bulky"]')
                        || Array.from(document.querySelectorAll('input[type="radio"]'))
                            .find(r => r.closest('label')?.textContent?.toLowerCase().includes('voluminoso'));
        
        if (bulkyRadio && !bulkyRadio.checked) {
          // ✅ v6.11.2: Scroll para que sea visible
          bulkyRadio.scrollIntoView({ block: 'center', behavior: 'smooth' });
          await sleep(humanDelay(300, 0.2)); // Esperar scroll
          
          bulkyRadio.click();
          await sleep(humanDelay(150, 0.2));
          
        } else if (bulkyRadio?.checked) {
          
        } else {
          
        }
        
        filled.push('bulky_shipping');
        
      } else {
        // 📦 ESTÁNDAR: Seleccionar peso
        if (kg == null) {
          missingRequired.push({
            name: 'weight_tier',
            reason: 'Envío estándar activado pero no se encontró peso en JSON (up_to_kg, kg o shipping.up_to_kg)'
          });
          
          return { ok: false, filled, skipped, missingRequired };
        }
        
        // ✅ v6.11.4: Buscar por name/aria-label (no por value)
        
        const standardRadio = document.querySelector('input[type="radio"][name="delivery"]')
                           || document.querySelector('input[type="radio"][aria-label="delivery"]')
                           || document.querySelector('input[type="radio"][id="delivery"]')
                           || Array.from(document.querySelectorAll('input[type="radio"]'))
                               .find(r => r.closest('label')?.textContent?.toLowerCase().includes('estándar'));
        
        if (standardRadio && !standardRadio.checked) {
          // ✅ v6.11.2: Scroll para que sea visible
          standardRadio.scrollIntoView({ block: 'center', behavior: 'smooth' });
          await sleep(humanDelay(300, 0.2)); // Esperar scroll
          
          standardRadio.click();
          // ✅ v6.11.5: NO sleep fijo, dejar que waitFor siguiente espere activamente
          
        } else if (standardRadio?.checked) {
          
        } else {
          
        }
        
        
        
        const radios = await waitFor(() => {
          const r = document.querySelectorAll('tsl-delivery-radio-selector input[type="radio"]');
          if (r.length < 6) return null;
          
          const hasInteractive = Array.from(r).some(radio => 
            !radio.disabled && radio.offsetParent !== null
          );
          
          if (!hasInteractive) {
            
            return null;
          }
          
          return Array.from(r);
        }, { timeout: TIMEOUTS.DROPDOWN_OPTIONS, interval: DELAYS.POLL_NORMAL });

        if (!radios || radios.length === 0) {
          missingRequired.push({
            name: 'weight_tier',
            reason: 'Radios de peso no aparecieron después del timeout'
          });
          
          return { ok: false, filled, skipped, missingRequired };
        }
        
        
        
        const weightResult = await dom.selectWeightTier(kg);
        if (!weightResult?.ok) {
          missingRequired.push({
            name: 'weight_tier',
            reason: `No se pudo seleccionar peso: ${weightResult?.error}`
          });
          
          return { ok: false, filled, skipped, missingRequired };
        }
        
        filled.push(`weight_tier=${weightResult.tier}`);
        
      }
    }
  } catch (e) {
    
    skipped.push('supports_shipping'); 
    skipped.push('weight_tier');
  }

  // 7) Re-scan (requeridos sin valor)
  const scan = await inspectListingForm().catch(()=>({ items:[], missingRequired:[] }));

  // Combinar missingRequired del scan con los acumulados
  const scanMissing = scan.missingRequired ?? [];
  missingRequired.push(...scanMissing);
  
  // ✅ v6.6.0: FILTRAR falsos positivos de weight_tier
  // Si wantShip=false, ignorar errores de weight_tier porque no lo necesitamos
  const supportsShippingFlag = snapshot?.supports_shipping?.flag;
  const shippingUserAllows = snapshot?.shipping?.user_allows_shipping;
  const wantShipForValidation = supportsShippingFlag === false || shippingUserAllows === false 
    ? false 
    : !!(supportsShippingFlag ?? shippingUserAllows ?? false);
  
  if (!wantShipForValidation) {
    const beforeFilter = missingRequired.length;
    missingRequired = missingRequired.filter(field => {
      const fieldName = typeof field === 'string' ? field : field?.name;
      return fieldName !== 'weight_tier';
    });
    if (missingRequired.length !== beforeFilter) {
    }
  }
  
  // ✅ CRÍTICO: Verificar si algún campo en 'skipped' era requerido
  const requiredFieldsSkipped = scan.items
    .filter(item => item.required && skipped.includes(item.name))
    .map(item => ({
      name: item.name,
      reason: `Campo requerido no se pudo rellenar (marcado en skipped)`
    }));
  
  if (requiredFieldsSkipped.length > 0) {
    missingRequired = [...missingRequired, ...requiredFieldsSkipped];
    
  }
  
  const ok = missingRequired.length === 0;

  
  
    
  
  const formErrors = [];
  
  // 1. Buscar campos con clase de error
  const errorWrappers = root.querySelectorAll('.inputWrapper--error, [class*="error"]');
  errorWrappers.forEach(wrapper => {
    const label = wrapper.querySelector('label')?.textContent || 
                  wrapper.querySelector('[class*="label"]')?.textContent ||
                  'Campo sin nombre';
    const input = wrapper.querySelector('input, textarea, select');
    const inputName = input?.name || input?.id || label;
    
    formErrors.push({
      field: inputName,
      label: label.trim(),
      reason: 'Campo con error visual'
    });
  });
  
  // 2. Buscar campos obligatorios vacíos
  const requiredInputs = root.querySelectorAll('input[required], textarea[required], select[required]');
  requiredInputs.forEach(input => {
    const value = input.value?.trim();
    if (!value || value === '') {
      const label = input.closest('label')?.textContent ||
                    input.closest('.inputWrapper')?.querySelector('label')?.textContent ||
                    input.name || input.id || 'Campo sin nombre';
      
      formErrors.push({
        field: input.name || input.id,
        label: label.trim(),
        reason: 'Campo obligatorio vacío'
      });
    }
  });
  
  // 3. Verificar campo precio específicamente (crítico)
  const priceInput = root.querySelector('input#sale_price[name="sale_price"]');
  if (priceInput) {
    const priceValue = priceInput.value?.trim();
    const priceWrapper = priceInput.closest('.inputWrapper');
    const hasError = priceWrapper?.classList.contains('inputWrapper--error');
    
    if (!priceValue || priceValue === '' || hasError) {
      formErrors.push({
        field: 'sale_price',
        label: 'Precio',
        reason: !priceValue ? 'Campo precio vacío' : 'Campo precio con error'
      });
    }
  }
  
  // 4. Si hay errores, marcar como incompleto
  if (formErrors.length > 0) {
    
    logger.error('❌ Formulario tiene errores, publicación DETENIDA', { errors: formErrors });
    
    // Agregar a missingRequired
    formErrors.forEach(err => {
      if (!missingRequired.some(m => m.name === err.field)) {
        missingRequired.push({
          name: err.field,
          reason: `${err.label}: ${err.reason}`
        });
      }
    });
    
    ok = false;
  } else {
    
  }
  
  if (ok) {
  } else {
  }
  
  return { ok, filled, missingRequired, skipped, lastScan: scan.items || [] };
}

