/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: publish/error-handler.js — Rol: Sistema de manejo de errores con reintentos inteligentes
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
  
  ✅ v6.5.3: Módulo creado para clasificación de errores, reintentos con backoff y circuit breaker
*/

/**
 * ✅ v6.5.3: Sistema de manejo de errores con reintentos inteligentes
 * 
 * Este módulo proporciona:
 * - Clasificación de errores (recuperables vs fatales)
 * - Reintentos con backoff exponencial
 * - Circuit breaker para evitar cascadas de fallos
 * - Logging estructurado de errores
 * 
 * @module publish/error-handler
 */

import { TIMEOUTS, DELAYS, ERROR_CODES, RETRY } from './constants.js';

// =============================================================================
// CLASIFICACIÓN DE ERRORES
// =============================================================================

/**
 * Tipos de errores según su recuperabilidad
 */
export const ERROR_TYPES = {
  TRANSIENT: 'transient',    // Puede recuperarse con reintento
  PERSISTENT: 'persistent',   // Requiere intervención del usuario
  FATAL: 'fatal',             // No se puede recuperar
  TIMEOUT: 'timeout',         // Específicamente timeout (reintentable)
  NETWORK: 'network',         // Error de red (puede reintentar)
  VALIDATION: 'validation',   // Error de validación (fatal)
};

/**
 * Clasifica un error según su código
 * @param {string} errorCode - Código de error
 * @returns {string} Tipo de error
 */
export function classifyError(errorCode) {
  const classifications = {
    // Errores transitorios (reintentar)
    [ERROR_CODES.CS_TIMEOUT]: ERROR_TYPES.TIMEOUT,
    [ERROR_CODES.DROPDOWN_TIMEOUT]: ERROR_TYPES.TIMEOUT,
    [ERROR_CODES.ELEMENT_NOT_FOUND]: ERROR_TYPES.TRANSIENT,
    [ERROR_CODES.BUTTON_DISABLED]: ERROR_TYPES.TRANSIENT,
    [ERROR_CODES.AJAX_PENDING]: ERROR_TYPES.TRANSIENT,
    
    // Errores persistentes (requieren atención)
    [ERROR_CODES.FORM_INCOMPLETE]: ERROR_TYPES.PERSISTENT,
    [ERROR_CODES.MISSING_DATA]: ERROR_TYPES.PERSISTENT,
    [ERROR_CODES.IMAGE_UPLOAD_FAILED]: ERROR_TYPES.PERSISTENT,
    
    // Errores fatales (no reintentar)
    [ERROR_CODES.UNSUPPORTED_TYPE]: ERROR_TYPES.FATAL,
    [ERROR_CODES.NO_FORM]: ERROR_TYPES.FATAL,
    [ERROR_CODES.VALIDATION_FAILED]: ERROR_TYPES.VALIDATION,
    [ERROR_CODES.REQUIRED_FIELD_MISSING]: ERROR_TYPES.VALIDATION,
  };
  
  return classifications[errorCode] || ERROR_TYPES.TRANSIENT;
}

/**
 * Determina si un error es reintentable
 * @param {string} errorCode - Código de error
 * @param {number} currentAttempt - Intento actual
 * @param {number} maxAttempts - Máximo de intentos
 * @returns {boolean}
 */
export function isRetryable(errorCode, currentAttempt = 1, maxAttempts = RETRY.MAX_ATTEMPTS) {
  const errorType = classifyError(errorCode);
  
  if (currentAttempt >= maxAttempts) {
    return false;
  }
  
  return [
    ERROR_TYPES.TRANSIENT,
    ERROR_TYPES.TIMEOUT,
    ERROR_TYPES.NETWORK
  ].includes(errorType);
}

// =============================================================================
// SISTEMA DE REINTENTOS CON BACKOFF
// =============================================================================

/**
 * Ejecuta una función con reintentos automáticos y backoff exponencial
 * @param {Function} fn - Función async a ejecutar
 * @param {Object} options - Opciones
 * @returns {Promise<Object>} Resultado o error
 */
export async function retryWithBackoff(fn, options = {}) {
  const {
    maxAttempts = RETRY.MAX_ATTEMPTS,
    initialDelay = RETRY.INITIAL_BACKOFF,
    multiplier = RETRY.BACKOFF_MULTIPLIER,
    maxDelay = RETRY.MAX_BACKOFF,
    shouldRetry = (error) => true,
    onRetry = null,
    context = 'operation',
    log = false,
  } = options;
  
  
  let lastError = null;
  
  for (let attempt = 1; attempt <= maxAttempts; attempt++) {
    try {
      const result = await fn(attempt);
      
      if (result?.ok !== false) {
        return { ok: true, result, attempts: attempt };
      }
      
      // La función retornó ok: false
      lastError = result;
      
      if (!shouldRetry(result)) {
        return { ok: false, error: result, attempts: attempt, reason: 'not_retryable' };
      }
      
    } catch (error) {
      lastError = error;
      
      if (!shouldRetry({ error: error.message, code: 'exception' })) {
        return { ok: false, error: error.message, attempts: attempt, reason: 'exception_not_retryable' };
      }
    }
    
    // Si no es el último intento, esperar con backoff
    if (attempt < maxAttempts) {
      const delay = calculateBackoffDelay(attempt, initialDelay, multiplier, maxDelay);
      
      if (onRetry) {
        await onRetry(attempt, delay, lastError);
      }
      
      await sleep(delay);
    }
  }
  
  return { 
    ok: false, 
    error: lastError, 
    attempts: maxAttempts, 
    reason: 'max_attempts_exceeded' 
  };
}

/**
 * Calcula delay con backoff exponencial y jitter
 * @param {number} attempt - Número de intento
 * @param {number} initial - Delay inicial
 * @param {number} multiplier - Multiplicador
 * @param {number} maxDelay - Delay máximo
 * @returns {number} Delay en ms
 */
function calculateBackoffDelay(attempt, initial, multiplier, maxDelay) {
  const baseDelay = initial * Math.pow(multiplier, attempt - 1);
  // Jitter ±20%
  const jitter = 0.8 + Math.random() * 0.4;
  return Math.min(Math.floor(baseDelay * jitter), maxDelay);
}

/**
 * Sleep helper
 * @param {number} ms - Milisegundos
 */
async function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

// =============================================================================
// CIRCUIT BREAKER
// =============================================================================

/**
 * Circuit breaker para evitar cascadas de fallos
 */
export class CircuitBreaker {
  constructor(options = {}) {
    this.failureThreshold = options.failureThreshold || 5;
    this.recoveryTimeout = options.recoveryTimeout || 60000; // 1 minuto
    this.halfOpenRequests = options.halfOpenRequests || 1;
    
    this.state = 'closed'; // closed, open, half-open
    this.failureCount = 0;
    this.successCount = 0;
    this.lastFailureTime = null;
    this.halfOpenAttempts = 0;
  }
  
  /**
   * Ejecuta una función protegida por el circuit breaker
   * @param {Function} fn - Función async a ejecutar
   * @returns {Promise<any>}
   */
  async execute(fn) {
    if (this.state === 'open') {
      if (this._shouldTransitionToHalfOpen()) {
        this.state = 'half-open';
        this.halfOpenAttempts = 0;
      } else {
        throw new Error('Circuit breaker is OPEN - service unavailable');
      }
    }
    
    try {
      const result = await fn();
      this._onSuccess();
      return result;
    } catch (error) {
      this._onFailure();
      throw error;
    }
  }
  
  _onSuccess() {
    if (this.state === 'half-open') {
      this.halfOpenAttempts++;
      if (this.halfOpenAttempts >= this.halfOpenRequests) {
        this.state = 'closed';
        this.failureCount = 0;
      }
    } else {
      this.failureCount = Math.max(0, this.failureCount - 1);
    }
  }
  
  _onFailure() {
    this.failureCount++;
    this.lastFailureTime = Date.now();
    
    if (this.state === 'half-open') {
      this.state = 'open';
    } else if (this.failureCount >= this.failureThreshold) {
      this.state = 'open';
    }
  }
  
  _shouldTransitionToHalfOpen() {
    return Date.now() - this.lastFailureTime >= this.recoveryTimeout;
  }
  
  getState() {
    return {
      state: this.state,
      failureCount: this.failureCount,
      lastFailure: this.lastFailureTime,
    };
  }
  
  reset() {
    this.state = 'closed';
    this.failureCount = 0;
    this.successCount = 0;
    this.lastFailureTime = null;
    this.halfOpenAttempts = 0;
  }
}

// =============================================================================
// ERROR RECOVERY STRATEGIES
// =============================================================================

/**
 * Estrategias de recuperación para diferentes tipos de errores
 */
export const RECOVERY_STRATEGIES = {
  /**
   * Estrategia para errores de timeout en dropdowns
   */
  async dropdownTimeout(context) {
    const { dom, fieldName, snapshot, log = false } = context;
    
    
    // 1. Cerrar cualquier dropdown abierto
    const openDropdowns = document.querySelectorAll('[aria-expanded="true"]');
    for (const dd of openDropdowns) {
      try { dd.click(); await sleep(200); } catch {}
    }
    
    // 2. Scroll al campo
    const field = document.querySelector(`[name="${fieldName}"], #${fieldName}`);
    if (field) {
      field.scrollIntoView({ block: 'center', behavior: 'smooth' });
      await sleep(500);
    }
    
    // 3. Reintentar la operación
    return { recoveryApplied: true, shouldRetry: true };
  },
  
  /**
   * Estrategia para errores de formulario incompleto
   */
  async formIncomplete(context) {
    const { missingFields = [], log = false } = context;
    
    
    // Si faltan campos requeridos, no podemos continuar
    const criticalFields = ['title', 'category_leaf_id', 'sale_price'];
    const hasCriticalMissing = missingFields.some(f => criticalFields.includes(f.name));
    
    if (hasCriticalMissing) {
      return { recoveryApplied: false, shouldRetry: false, fatal: true };
    }
    
    // Si solo faltan campos opcionales, podemos continuar
    return { recoveryApplied: true, shouldRetry: true };
  },
  
  /**
   * Estrategia para errores de conexión
   */
  async networkError(context) {
    const { log = false } = context;
    
    
    // Verificar si estamos online
    if (!navigator.onLine) {
      return { recoveryApplied: false, shouldRetry: false, reason: 'offline' };
    }
    
    // Esperar un poco y reintentar
    await sleep(2000);
    return { recoveryApplied: true, shouldRetry: true };
  },
  
  /**
   * Estrategia para botón deshabilitado
   */
  async buttonDisabled(context) {
    const { buttonSelector, log = false } = context;
    
    
    // Esperar un poco más
    await sleep(DELAYS.AFTER_CLICK * 2);
    
    // Verificar si el botón ahora está habilitado
    const btn = document.querySelector(buttonSelector);
    if (btn && !btn.disabled) {
      return { recoveryApplied: true, shouldRetry: true };
    }
    
    return { recoveryApplied: false, shouldRetry: false, reason: 'still_disabled' };
  },
};

// =============================================================================
// WRAPPER DE ERROR HANDLING
// =============================================================================

/**
 * Ejecuta una operación de publicación con manejo de errores completo
 * @param {string} operationName - Nombre de la operación
 * @param {Function} operation - Función async a ejecutar
 * @param {Object} options - Opciones
 * @returns {Promise<Object>}
 */
export async function executeWithErrorHandling(operationName, operation, options = {}) {
  const {
    maxRetries = RETRY.MAX_ATTEMPTS,
    recoveryStrategy = null,
    context = {},
    log = false,
    onError = null,
    circuitBreaker = null,
  } = options;
  
  
  const executeOp = async () => {
    if (circuitBreaker) {
      return await circuitBreaker.execute(operation);
    }
    return await operation();
  };
  
  const result = await retryWithBackoff(
    async (attempt) => {
      try {
        return await executeOp();
      } catch (error) {
        
        // Aplicar estrategia de recuperación si existe
        if (recoveryStrategy && attempt < maxRetries) {
          const recovery = await recoveryStrategy({ ...context, error, log });
          
          if (recovery.fatal) {
            throw new Error(`Fatal: ${recovery.reason || 'unrecoverable'}`);
          }
          
          if (recovery.shouldRetry) {
            // Dejar que el retry loop maneje el siguiente intento
          }
        }
        
        throw error;
      }
    },
    {
      maxAttempts: maxRetries,
      context: operationName,
      log,
      shouldRetry: (error) => {
        const code = error.code || 'unknown';
        return isRetryable(code);
      },
      onRetry: async (attempt, delay, error) => {
        if (onError) {
          await onError(attempt, delay, error);
        }
      },
    }
  );
  
  return result;
}

// =============================================================================
// LOGGING ESTRUCTURADO
// =============================================================================

/**
 * Crea un error estructurado para persistencia
 * @param {string} errorType - Tipo de error
 * @param {string} message - Mensaje
 * @param {Object} details - Detalles adicionales
 * @param {Object} context - Contexto
 * @returns {Object}
 */
export function createStructuredError(errorType, message, details = {}, context = {}) {
  return {
    timestamp: new Date().toISOString(),
    error_type: errorType,
    error_message: message,
    classification: classifyError(errorType),
    retryable: isRetryable(errorType),
    details: {
      ...details,
      step: details.step || 'unknown',
      attempt: details.attempt || 1,
    },
    context: {
      itemId: context.itemId || null,
      accountId: context.accountId || null,
      field: context.field || null,
    },
    metadata: {
      userAgent: typeof navigator !== 'undefined' ? navigator.userAgent : null,
      url: typeof location !== 'undefined' ? location.href : null,
    },
  };
}

/**
 * Serializa error para almacenamiento
 * @param {Object} errorObj - Objeto de error
 * @returns {string} JSON string
 */
export function serializeError(errorObj) {
  return JSON.stringify(errorObj);
}
