/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: publish/constants.js — Rol: Constantes centralizadas para módulos de publicación
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
  
  ✅ v6.5.3: Archivo creado para centralizar timeouts, delays, selectores y códigos de error
*/

/**
 * ✅ v6.10.0: Constantes centralizadas para el sistema de publicación
 * 
 * CAMBIOS v6.10.0:
 * - Integrado MutationObserver para esperas activas (90% menos CPU)
 * - Polling de backup reducido a 500ms (vs 200ms antes)
 * - Reacción instantánea a cambios del DOM (<5ms vs 200ms)
 * 
 * Este archivo centraliza todos los timeouts, delays y selectores usados
 * en el proceso de publicación para facilitar el mantenimiento y testing.
 * 
 * @module publish/constants
 */

// =============================================================================
// TIMEOUTS (en milisegundos)
// =============================================================================

export const TIMEOUTS = {
  // Esperas de carga de página/formulario
  FORM_LOAD: 15000,              // Esperar formulario visible
  PAGE_READY: 12000,             // Página de upload lista
  STEP_VERTICAL: 18000,          // Contenedor de selección de vertical
  
  // Operaciones de DOM
  ELEMENT_APPEAR: 8000,          // Elemento general aparezca
  SHADOW_BUTTON: 12000,          // Botón en shadow DOM
  BUTTON_ENABLED: 12000,         // Botón habilitado (no disabled)
  
  // Dropdowns y combos
  DROPDOWN_OPEN: 4000,           // Dropdown se abra
  DROPDOWN_OPTIONS: 5000,        // Opciones carguen
  DROPDOWN_SELECT: 8000,         // ✅ v6.9.3: Aumentado de 5s a 8s (PCs lentos, conexiones 3G)
  BRAND_COMBO: 20000,            // ✅ v6.9.3: Aumentado de 15s a 20s (AJAX + conexión lenta)
  MODEL_COMBO: 15000,            // ✅ v6.9.3: Aumentado de 10s a 15s (dependiente + lento)
  STORAGE_DROPDOWN: 12000,       // ✅ v6.9.3: Aumentado de 8s a 12s (AJAX + lento)
  
  // Clicks y navegación
  CLICK_CONTINUE: 12000,         // Click en continuar
  CLICK_PUBLISH: 16000,          // Click en publicar
  
  // Subida de imágenes
  IMAGE_UPLOAD: 35000,           // Subida de todas las imágenes
  IMAGE_SINGLE: 8000,            // Una imagen individual
  
  // Captura de ID
  PUBLISH_CAPTURE: 25000,        // Captura de ID después de publicar
  SW_RESOLVE: 12000,             // Resolución via Service Worker
  
  // Categorías
  CATEGORY_SELECT: 20000,        // Selección completa de taxonomía
  CATEGORY_LEVEL: 5000,          // Un nivel del árbol
  
  // Operaciones generales
  DEFAULT: 10000,                // Timeout por defecto
  QUICK_CHECK: 2000,             // Verificación rápida (ej: botón habilitado)
  SHORT: 4000,                   // Operaciones rápidas
  LONG: 20000,                   // Operaciones lentas
};

// =============================================================================
// DELAYS (en milisegundos) - Para comportamiento humano
// =============================================================================

export const DELAYS = {
  // Entre acciones (simular usuario)
  HUMAN_MIN: 200,                // Mínimo entre clicks
  HUMAN_BASE: 500,               // Base para humanDelay()
  HUMAN_VARIANCE: 0.3,           // Varianza (±30%)
  
  // Después de acciones específicas
  AFTER_CLICK: 350,              // Después de click
  AFTER_INPUT: 250,              // Después de escribir
  AFTER_SELECT: 400,             // Después de seleccionar dropdown
  AFTER_TOGGLE: 350,             // Después de toggle
  
  // Entre pasos del proceso
  BETWEEN_STEPS: 600,            // Entre pasos del formulario
  BEFORE_PUBLISH: 3000,          // Antes de publicar (prudencia)
  
  // Polling intervals
  POLL_FAST: 200,                // ✅ v6.9.3: Aumentado de 120ms a 200ms (menos agresivo en PCs lentos)
  POLL_NORMAL: 300,              // ✅ v6.9.3: Aumentado de 200ms a 300ms
  POLL_SLOW: 500,                // ✅ v6.9.3: Aumentado de 300ms a 500ms
  
  // Combos AJAX
  BRAND_SEARCH: 800,             // Delay de búsqueda de marca
  MODEL_WAIT: 600,               // Espera carga de modelos
  
  // Anti-detección
  COLOR_OPEN: 600,               // Abrir selector de colores
  COLOR_CLEAR: 800,              // Limpiar colores previos
  COLOR_SELECT: 240,             // Entre selecciones de color
};

// =============================================================================
// SELECTORES CSS
// =============================================================================

export const SELECTORS = {
  // Contenedores principales
  FORM_ROOT: '#step-listing',
  FORM_ALT: 'tsl-upload-step-form',
  FORM_CONTAINER: 'tsl-upload tsl-upload-step-form',
  
  // Campos de texto
  TITLE: 'input#title[name="title"]',
  TITLE_ALT: 'input#titulo[name="titulo"]',
  DESCRIPTION: 'textarea#description[name="description"]',
  PRICE: 'input#sale_price[name="sale_price"]',
  
  // Botones
  CONTINUE_BUTTON: 'walla-button[data-testid="continue-action-button"]',
  PUBLISH_BUTTON: 'walla-button[data-testid="continue-action-button"]',
  
  // Dropdowns
  SIZE_HIDDEN: 'input[type="hidden"][name="size"]',
  CURRENCY_HIDDEN: 'input[type="hidden"][name="currency"]',
  CONDITION_HIDDEN: 'input[type="hidden"][name="condition"]',
  
  // Combos específicos
  BRAND_INPUT: 'input[name="brand"]',
  MODEL_INPUT: 'input[name="model"]',
  STORAGE_SELECT: 'select[name="storage_capacity"]',
  
  // Envío
  SHIPPING_TOGGLE: 'wallapop-toggle input[type="checkbox"]',
  WEIGHT_RADIO: 'input[type="radio"][name="up_to_kg"]',
  
  // Colores
  COLOR_BUTTON: '[data-testid="colors-button"]',
  COLOR_CHIP: '[role="option"][aria-selected]',
  
  // Estados de error
  INPUT_ERROR: '.inputWrapper--error',
  FIELD_ERROR: '[class*="error"]',
  
  // Vertical (tipo de anuncio)
  VERTICAL_CONTAINER: '#vertical-selector',
  VERTICAL_BUTTON: 'button[data-testid="vertical-option"]',
};

// =============================================================================
// MAPEOS DE VALORES
// =============================================================================

export const MAPPINGS = {
  // Vertical por taxonomía
  TAXONOMY_TO_VERTICAL: {
    empleo: 'jobs',
    servicios: 'services',
    // Todo lo demás es consumer_goods
  },
  
  // Condición
  CONDITION: {
    nuevo: 'as_good_as_new',
    new: 'as_good_as_new',
    'como nuevo': 'as_good_as_new',
    'as good as new': 'as_good_as_new',
    'buen estado': 'good',
    good: 'good',
    aceptable: 'fair',
    fair: 'fair',
    'ha dado guerra': 'has_given_it_all',
    used: 'has_given_it_all',
  },
  
  // Moneda
  CURRENCY_SYMBOLS: {
    EUR: '€',
    USD: '$',
    GBP: '£',
  },
  
  // Peso (kg → tier)
  WEIGHT_TIERS: {
    1: 'tier_1',
    2: 'tier_2',
    5: 'tier_3',
    10: 'tier_4',
    20: 'tier_5',
    30: 'tier_6',
  },
};

// =============================================================================
// CONFIGURACIÓN DE REINTENTOS
// =============================================================================

export const RETRY = {
  // Número de intentos
  MAX_ATTEMPTS: 5,               // ✅ v6.9.3: Aumentado de 3 a 5 (conexiones inestables)
  DROPDOWN_ATTEMPTS: 6,          // ✅ v6.9.3: Aumentado de 4 a 6
  COLOR_ATTEMPTS: 6,             // ✅ v6.9.3: Aumentado de 4 a 6
  
  // Backoff exponencial
  INITIAL_DELAY: 1000,
  MULTIPLIER: 1.5,
  MAX_DELAY: 10000,
  
  // Jitter (variación aleatoria)
  JITTER_MIN: 0.8,
  JITTER_MAX: 1.2,
};

// =============================================================================
// MENSAJES DE ERROR
// =============================================================================

export const ERROR_CODES = {
  // Formulario
  NO_FORM: 'no_form',
  FORM_TIMEOUT: 'form_timeout',
  FIELD_NOT_FOUND: 'field_not_found',
  FIELD_NOT_VISIBLE: 'field_not_visible',
  
  // Validación
  REQUIRED_EMPTY: 'required_field_empty',
  INVALID_VALUE: 'invalid_value',
  
  // Dropdowns
  DROPDOWN_CLOSED: 'dropdown_closed',
  OPTION_NOT_FOUND: 'option_not_found',
  AJAX_TIMEOUT: 'ajax_timeout',
  
  // Publicación
  BUTTON_DISABLED: 'button_disabled',
  BUTTON_NOT_FOUND: 'button_not_found',
  CAPTURE_FAILED: 'id_capture_failed',
  
  // Categorías
  CATEGORY_NOT_FOUND: 'category_not_found',
  TAXONOMY_INVALID: 'taxonomy_invalid',
  
  // Red/Sistema
  NETWORK_ERROR: 'network_error',
  WALLAPOP_OFFLINE: 'wallapop_offline',
  CS_NOT_READY: 'content_script_not_ready',
};

// =============================================================================
// CONFIGURACIÓN DE LOGGING
// =============================================================================

export const LOG = {
  // Niveles
  DEBUG: true,
  INFO: true,
  WARN: true,
  ERROR: true,
  
  // Prefijos
  PREFIX_PUBLISH: '[PUB]',
  PREFIX_FORM: '[PUB][fill]',
  PREFIX_BRAND: '[PUB][brand]',
  PREFIX_MODEL: '[PUB][model]',
  PREFIX_COLOR: '[Color]',
  PREFIX_CAPTURE: '[id-capture]',
};

// =============================================================================
// UTILIDADES
// =============================================================================

/**
 * Obtiene timeout con jitter para evitar patrones detectables
 * @param {number} base - Timeout base en ms
 * @param {number} variance - Varianza (0-1)
 * @returns {number} Timeout con jitter
 */
export function getTimeoutWithJitter(base, variance = 0.1) {
  const min = base * (1 - variance);
  const max = base * (1 + variance);
  return Math.floor(min + Math.random() * (max - min));
}

/**
 * Calcula delay con backoff exponencial
 * @param {number} attempt - Número de intento (1-based)
 * @param {number} initial - Delay inicial
 * @param {number} multiplier - Multiplicador
 * @param {number} max - Máximo delay
 * @returns {number} Delay en ms
 */
export function getBackoffDelay(attempt, initial = RETRY.INITIAL_DELAY, multiplier = RETRY.MULTIPLIER, max = RETRY.MAX_DELAY) {
  const delay = initial * Math.pow(multiplier, attempt - 1);
  const jitter = RETRY.JITTER_MIN + Math.random() * (RETRY.JITTER_MAX - RETRY.JITTER_MIN);
  return Math.min(delay * jitter, max);
}
