/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: publish.js — Rol: Publicación en Wallapop (DOM): selección de vertical y, después, formulario.
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  En Rol: si esta vacio o el fichero se modifica o reestructura si hacer cambio en Rol
*/

/* ============================================================
   Imports de utilidades reutilizables
   ============================================================ */
import { sleep, normalize, truncateCP, nearError } from './utils.js';
import { norm, kgToTier, pick as pickElement } from './dom.js';
import { TIMEOUTS, DELAYS, SELECTORS, ERROR_CODES } from './publish/constants.js';
import { 
  ensureDom, 
  getLogger, 
  humanDelay, 
  cssEscape,
  dlog as sharedDlog,
  DEBUG as SHARED_DEBUG
} from './publish/shared.js';
import { createStructuredError, serializeError } from './publish/error-handler.js';

/* ============================================================
   Utils locales específicos de publish
   ============================================================ */
const DEBUG = SHARED_DEBUG;
const dlog = sharedDlog;

// ✅ v6.5.4: createDetailedError ahora usa createStructuredError del módulo centralizado
const createDetailedError = (errorType, message, details = {}, context = {}) => {
  return serializeError(createStructuredError(errorType, message, details, context));
};

// ✅ v6.5.4: Funciones centralizadas importadas de shared.js
// - ensureDom()
// - getLogger()
// - humanDelay()
// - cssEscape()
// - dlog()

// ✅ v6.5.3: Funciones duplicadas ELIMINADAS
// Las siguientes funciones se importan ahora de utils.js/dom.js:
// - normalize() de utils.js (antes _normalize)
// - truncateCP() de utils.js (antes _truncateCP)  
// - nearError() de utils.js (antes _nearError)
// - clearThenSetText() de dom.js (antes clearThenSetTextLocal)
// Mantiene retrocompatibilidad mediante re-export al final del archivo

/* ============================================================
   1) Mapeo NUEVO: taxonomy → vertical
   ============================================================ */
/**
 * ✅ v4.80.2: Nueva lógica basada en taxonomy[0].name
 * Acepta snapshot completo
 * 
 * LÓGICA:
 * - Si taxonomy[0].name === "Empleo" → jobs (Jobs Vertical)
 * - Si taxonomy[0].name === "Servicios" → services (Services Vertical)
 * - Si taxonomy[0].name === "Coches" o "Inmobiliaria" → ERROR (no soportado)
 * - Todo lo demás → consumer_goods (Consumer Goods Vertical)
 * 
 * @param {object} src - Snapshot completo con campo taxonomy
 * @returns {string|object} - Vertical string o {error: true, vertical: string, message: string}
 */
export function mapTypeToVertical(src = {}) {
  // Extraer primer taxonomy name
  const firstTaxonomyName = src?.taxonomy?.[0]?.name || '';
  const taxonomyNorm = norm(firstTaxonomyName);
  
  // ⚠️ EXCEPCIONES NO SOPORTADAS
  if (taxonomyNorm === 'coches' || taxonomyNorm === 'inmobiliaria') {
    return {
      error: true,
      vertical: taxonomyNorm === 'coches' ? 'cars' : 'real_estate',
      message: `El tipo de anuncio "${firstTaxonomyName}" no está soportado actualmente. Este anuncio será omitido.`
    };
  }
  
  // ✅ MAPEO POR TAXONOMY
  if (taxonomyNorm === 'empleo') {
    return 'jobs';
  }
  
  if (taxonomyNorm === 'servicios') {
    return 'services';
  }
  
  // ✅ DEFAULT: Consumer Goods (todo lo demás)
  return 'consumer_goods';
}

// Etiquetas/Selectores por vertical (aria, data-testid y textos visibles ES/EN)
function verticalLabels(vertical){
  const map = {
    consumer_goods: {
      aria:   ['Consumer Goods Vertical'],
      texts:  ['Algo que ya no necesito', 'Consumer Goods', 'Cosas', 'Artículos'],
      testid: ['UploadStepVerticalGoods']
    },
    jobs: {
      aria:   ['Jobs Vertical'],
      texts:  ['Empleo', 'Jobs', 'Trabajo'],
      testid: ['UploadStepVerticalJobs']
    },
    services: {
      aria:   ['Services Vertical'],
      texts:  ['Mis servicios', 'Servicios', 'Services'],
      testid: ['UploadStepVerticalServices']
    },
    cars: {
      aria:   ['Cars Vertical', 'Vehicles Vertical'],
      texts:  ['Un vehículo', 'Vehículos', 'Vehicle', 'Coches'],
      testid: ['UploadStepVerticalCars']
    },
    real_estate: {
      aria:   ['Real Estate Vertical'],
      texts:  ['Una propiedad', 'Inmuebles', 'Real Estate'],
      testid: ['UploadStepVerticalRealEstate']
    },
  };
  return map[vertical] || map.consumer_goods;
}

/* ============================================================
   2) Espera del step inicial de vertical - ✅ v6.6.4: Simplificado
   ============================================================ */
// ✅ v6.6.4: Esta función ya no es necesaria porque WAIT_UPLOAD_READY
// hace todas las verificaciones, pero la mantenemos por compatibilidad
async function waitStepVertical() {
  const dom = await ensureDom();
  const waitFor = dom?.waitFor || (async (getter, { timeout=TIMEOUTS.STEP_VERTICAL, interval=DELAYS.POLL_FAST } = {}) => {
    const t0 = Date.now();
    while (Date.now() - t0 < timeout) {
      try { const v = getter(); if (v) return v; } catch {}
      await sleep(interval);
    }
    return null;
  });

  const getter = () => {
    const step = document.querySelector('.UploadStepVertical__container, tsl-upload-step-vertical, [data-testid="UploadStepVertical"]');
    if (step && step.offsetParent !== null) return step;
    return null;
  };

  return await waitFor(getter, { timeout: TIMEOUTS.STEP_VERTICAL });
}

/* ============================================================
   3) Click helpers para el step de vertical
   ============================================================ */
function _click(el){
  try { el.scrollIntoView({ block:'center' }); } catch {}
  try { el.click(); } catch {}
  try { el.dispatchEvent(new MouseEvent('click', { bubbles:true, composed:true })); } catch {}
}
function _qIn(root, sel){
  return root?.querySelector?.(sel) || root?.shadowRoot?.querySelector?.(sel) || null;
}
function clickByAria(root, labels = []){
  for (const a of labels || []){
    const sel1 = `button[aria-label="${cssEscape(a)}"]`;
    const sel2 = `button[aria-label*="${cssEscape(a)}"]`;
    const el = _qIn(root, sel1) || _qIn(root, sel2) || document.querySelector(sel1) || document.querySelector(sel2);
    if (el){ _click(el); return true; }
  }
  return false;
}
function clickByTestId(root, testids = []){
  for (const t of testids || []){
    const host = _qIn(root, `[data-testid="${cssEscape(t)}"]`) || document.querySelector(`[data-testid="${cssEscape(t)}"]`);
    const el = host?.querySelector?.('button') || host;
    if (el){ _click(el); return true; }
  }
  return false;
}
function clickByExactText(root, texts = []){
  if (!texts?.length) return false;
  const candidates = [
    ...root.querySelectorAll('button, [role="button"], .UploadStepVertical__title, .UploadStepVertical__multiIcons__title')
  ];
  for (const t of texts) {
    const target = candidates.find(el => norm(el.textContent) === norm(t));
    if (target){ _click(target.closest?.('button,[role="button"]') || target); return true; }
  }
  return false;
}
function clickByContainsText(root, texts = []){
  if (!texts?.length) return false;
  const candidates = [
    ...root.querySelectorAll('button, [role="button"], .UploadStepVertical__title, .UploadStepVertical__multiIcons__title')
  ];
  for (const t of texts) {
    const target = candidates.find(el => norm(el.textContent).includes(norm(t)));
    if (target){ _click(target.closest?.('button,[role="button"]') || target); return true; }
  }
  return false;
}
async function waitNextStepOrForm(stepEl) {
  const t0 = Date.now();
  while (Date.now() - t0 < 15000) {
    // Verificar si el step de vertical ya no está visible
    const stillHere = document.body.contains(stepEl) && stepEl.offsetParent !== null;
    
    // Detectar formulario por múltiples selectores
    const formReady =
      document.querySelector('tsl-upload tsl-upload-step-form') ||
      document.querySelector('tsl-upload-step-form') ||
      document.querySelector('form [name="title"], input[name="title"], textarea[name="description"]') ||
      document.querySelector('#summary, input[name="summary"], input#summary') ||  // ✅ v10.4.0: Input summary
      document.querySelector('[data-testid*="upload-form"], [data-testid*="UploadForm"]');  // ✅ v10.4.0: TestIds
    
    if (!stillHere || formReady) return true;
    await sleep(120);
  }
  return false;
}

/* ============================================================
   3.5) Modal para anuncios no soportados
   ============================================================ */
/**
 * Muestra modal al usuario cuando encuentra un anuncio no soportado
 * @param {string} message - Mensaje a mostrar
 * @returns {Promise<void>}
 */
async function showUnsupportedAdModal(message) {
  return new Promise((resolve) => {
    // Crear overlay
    const overlay = document.createElement('div');
    overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.8);
      backdrop-filter: blur(8px);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 999999;
      animation: fadeIn 0.3s ease;
    `;
    
    // Crear modal
    const modal = document.createElement('div');
    modal.style.cssText = `
      background: linear-gradient(135deg, #1a1d29 0%, #0f1419 100%);
      border: 2px solid rgba(239, 68, 68, 0.3);
      border-radius: 20px;
      padding: 40px;
      max-width: 500px;
      width: 90%;
      box-shadow: 0 20px 60px rgba(239, 68, 68, 0.2);
      text-align: center;
    `;
    
    modal.innerHTML = `
      <div style="font-size: 60px; margin-bottom: 20px;">⚠️</div>
      <h2 style="font-size: 24px; font-weight: 700; color: #ef4444; margin-bottom: 15px;">
        Anuncio no soportado
      </h2>
      <p style="font-size: 16px; color: rgba(255, 255, 255, 0.9); margin-bottom: 30px; line-height: 1.6;">
        ${message}
      </p>
      <button id="unsupported-accept-btn" style="
        background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        color: white;
        border: none;
        padding: 15px 40px;
        border-radius: 12px;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(239, 68, 68, 0.4);
      ">
        Aceptar
      </button>
    `;
    
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    
    // Handler del botón
    const btn = modal.querySelector('#unsupported-accept-btn');
    btn.addEventListener('click', () => {
      overlay.remove();
      resolve();
    });
    
    // Hover effect
    btn.addEventListener('mouseenter', () => {
      btn.style.transform = 'translateY(-2px)';
      btn.style.boxShadow = '0 6px 20px rgba(239, 68, 68, 0.6)';
    });
    btn.addEventListener('mouseleave', () => {
      btn.style.transform = 'translateY(0)';
      btn.style.boxShadow = '0 4px 15px rgba(239, 68, 68, 0.4)';
    });
  });
}

/* ============================================================
   4) API: seleccionar vertical del upload
   ============================================================ */
/**
 * Selecciona la vertical del primer paso en /app/catalog/upload
 * ✅ v4.80.2: Ahora maneja excepciones de anuncios no soportados
 * ✅ v6.6.4: Simplificado - las verificaciones previas las hace WAIT_UPLOAD_READY
 * @param {object|string} snapshotOrType  snapshot completo
 * @returns {Promise<{ok:boolean, picked?:string, code?:string, note?:string, skipped?:boolean}>}
 */
export async function selectVerticalForUpload(snapshotOrType = {}) {
  const log = await getLogger();
  
  const verticalResult = mapTypeToVertical(snapshotOrType);
  
  // ⚠️ MANEJAR ANUNCIOS NO SOPORTADOS
  if (verticalResult?.error) {
    log.warn(`⚠️ Anuncio no soportado: ${verticalResult.message}`, { 
      vertical: verticalResult.vertical,
      taxonomy: snapshotOrType?.taxonomy?.[0]?.name 
    });
    
    // Mostrar modal al usuario
    await showUnsupportedAdModal(verticalResult.message);
    
    return { 
      ok: false, 
      code: 'unsupported_ad_type', 
      note: verticalResult.message,
      skipped: true,
      vertical: verticalResult.vertical
    };
  }
  
  const vertical = verticalResult;
  const labels   = verticalLabels(vertical);
  
  log.info(`📤 Seleccionando vertical: ${vertical}`, { 
    vertical,
    taxonomy: snapshotOrType?.taxonomy?.[0]?.name 
  });

  const dom = await ensureDom();

  // 1) Estado de conexión
  if (dom?.isOnline) {
    const online = await dom.isOnline({ pingPath:'/' });
    if (!online) return { ok:false, code:'offline', note:'Sin conexión' };
  }

  // ✅ v6.6.4: Verificación mínima de URL (WAIT_UPLOAD_READY ya validó todo lo demás)
  if (!/\/app\/catalog\/upload(?:\/|$)/.test(location.pathname)) {
    log.error('❌ URL incorrecta:', location.pathname);
    return { ok:false, code:'not_on_upload_page', note:'URL incorrecta' };
  }

  // 2) Obtener step vertical (ya debe estar presente y visible)
  const step = document.querySelector('.UploadStepVertical__container, tsl-upload-step-vertical, [data-testid="UploadStepVertical"]');
  if (!step || step.offsetParent === null) {
    log.error('❌ Contenedor de vertical no encontrado o no visible');
    return { ok:false, code:'step_vertical_not_found' };
  }
  
  log.info('✅ Contenedor de vertical encontrado:', step.className || step.tagName);

  // 3) Click con retry/backoff
  const attemptClick = () =>
    clickByAria(step, labels.aria) ||
    clickByTestId(step, labels.testid) ||
    clickByExactText(step, labels.texts) ||
    clickByContainsText(step, labels.texts);

  let clicked = false;
  if (dom?.retry) {
    await dom.retry(async () => {
      if (!attemptClick()) throw new Error('click_failed');
      clicked = true;
    }, { retries: 3, baseDelay: 220, factor: 1.8, jitter: true });
  } else {
    for (let i = 0; i < 4 && !clicked; i++) {
      clicked = attemptClick();
      if (!clicked) await sleep(220);
    }
  }
  if (!clicked) {
    log.error('❌ No se encontró el botón de vertical', { vertical });
    return { ok:false, code:'vertical_button_not_found', picked: vertical };
  }

  // ✅ v10.4.0: CRÍTICO - Esperar a que el formulario aparezca después del click
  // Esto verifica que el click realmente funcionó
  log.info('⏳ Esperando formulario después de click en vertical...');
  const formAppeared = await waitNextStepOrForm(step);
  
  if (!formAppeared) {
    log.error('❌ El formulario no apareció después de hacer click en la vertical');
    return { ok:false, code:'form_not_appeared_after_vertical_click', picked: vertical };
  }

  // 🤖 Delay humano después de seleccionar vertical (±30% variación)
  await sleep(humanDelay(300));

  log.info(`✅ Vertical seleccionada y formulario cargado: ${vertical}`);
  return { ok:true, picked: vertical };
}

/* ============================================================
   5) Rellenar "Resumen del producto" (#summary)
   ============================================================ */
export async function fillSummary(snapshot = {}) {
  const dom = await ensureDom();

  // 5.1) Extraer título
  const pickTitle = (s) => {
    const take = (v) => {
      if (!v) return '';
      if (typeof v === 'string') return v;
      if (typeof v === 'object') return (v.original ?? v.text ?? v.value ?? v.label ?? v.name ?? '');
      return String(v || '');
    };
    const cands = [s?.title, s?.product?.title, s?.item?.title, s?.raw?.title];
    for (const c of cands) {
      const v = take(c).trim();
      if (v) return v;
    }
    return '';
  };

  let title = normalize(pickTitle(snapshot));
  dlog('[fillSummary] resolved title:', title);
  if (!title) return { ok:false, code:'summary_missing_title' };

  // 5.2) Esperar input visible
  const SEL = '#summary[name="summary"], input#summary, input[name="summary"], input[aria-label*="Resumen" i]';
  let el = null;
  if (typeof dom.waitFor === 'function') {
    el = await dom.waitFor(() => document.querySelector(SEL), { timeout: TIMEOUTS.FORM_LOAD });
  } else {
    const t0 = Date.now();
    while (Date.now() - t0 < TIMEOUTS.FORM_LOAD) {
      const cand = document.querySelector(SEL);
      if (cand && cand.offsetParent !== null) { el = cand; break; }
      await sleep(DELAYS.POLL_FAST);
    }
  }
  if (!el) return { ok:false, code:'summary_input_not_found' };

  // 5.3) Truncar por maxLength (code points)
  const max = (el.maxLength > 0) ? el.maxLength : 0;
  if (max) title = truncateCP(title, max);

  // 5.4) Set robusto (anti-IA): limpiar y escribir con eventos reales
  if (typeof dom.clearThenSetText === 'function') {
    await dom.clearThenSetText(el, title);
  } else {
    // Fallback: importar directamente desde dom.js
    const { clearThenSetText } = await import('./dom.js');
    await clearThenSetText(el, title);
  }
  
  // 🤖 Delay humano después de escribir (±20% variación)
  await sleep(humanDelay(150, 0.2)); // 120-180ms

  // 5.5) Validación básica
  const invalid =
    el.getAttribute('aria-invalid') === 'true' ||
    (typeof el.checkValidity === 'function' && !el.checkValidity());
  if (invalid) return { ok:false, code:'summary_invalid' };

  return { ok:true, value: String(el.value || title) };
}

/* ============================================================
   6) Click en "Continuar" (title/photo/auto)
   ============================================================ */
/**
 * @param {'title'|'photo'|null} which
 */
export async function clickContinue({
  which = null,          // 'title' | 'photo' | null (auto)
  delay = 0,             // p.ej. 600ms antes del primer click; 250ms en fotos
  timeout = TIMEOUTS.CLICK_CONTINUE,
  log = false  // ✅ v10.5.109: Silenciado por defecto
} = {}) {

  // ✅ v10.4.4: Restaurada definición de logg que faltaba

  const hasPhotoInput = () => {
    // Buscar múltiples señales de que estamos en la pantalla de fotos
    const zone = document.querySelector('tsl-drop-area-zone, tsl-drop-area-v2');
    const preview = document.querySelector('tsl-drop-area-preview');
    const fileInput = document.querySelector('tsl-drop-area-preview input[type="file"], tsl-drop-area-zone input[type="file"]');
    const photosSection = document.querySelector('#step-photo, section.UploadStepTemplate h1');
    const subirBtn = document.querySelector('walla-button[buttontype="secondary"]');
    
    // Si encontramos cualquiera de estas señales, estamos en fotos
    return !!(zone || preview || fileInput || photosSection || subirBtn);
  };

  if (delay > 0) await new Promise(r => setTimeout(r, delay));

  const dom = await ensureDom();

  // 1) Scope y botón
  const scopeSel = which === 'photo' ? '#step-photo'
                : which === 'title' ? '#step-title'
                : null;

  const hostSel = scopeSel
    ? `${scopeSel} walla-button[data-testid="continue-button"]`
    : `walla-button[data-testid="continue-button"]`;

  const hadPhotoBefore = which ? null : hasPhotoInput();

  const { host, btn } = dom.waitShadowButton
    ? await dom.waitShadowButton(hostSel, 'button', { timeout: Math.max(1000, timeout - 800) })
    : (() => {
        const h = document.querySelector(hostSel);
        const b = h?.shadowRoot?.querySelector('button') || null;
        return { host: h, btn: b };
      })();

  if (!host) return { ok:false, code:'host_not_found', which, hostSel };
  if (!btn)  return { ok:false, code:'inner_button_not_found', which, hostSel };

  // 2) Habilitación
  if (dom.waitEnabledAttr) {
    
    const ok = await dom.waitEnabledAttr([host, btn], { timeout: TIMEOUTS.BUTTON_ENABLED });
    
    if (!ok) { return { ok:false, code:'disabled', which }; }
  } else {
    
    if (btn.disabled || host.getAttribute('aria-disabled') === 'true' || host.hasAttribute('disabled')) {
      
      return { ok:false, code:'disabled', which };
    }
  }

  // 3) Click realista
  const doClickOnce = () => {
    if (dom.dispatchPointerClick) dom.dispatchPointerClick(btn, { host });
    else {
      try { btn.scrollIntoView({ block:'center' }); btn.focus?.({ preventScroll:true }); } catch {}
      btn.dispatchEvent(new MouseEvent('click', { bubbles:true, composed:true, cancelable:true }));
      try { btn.click(); host.click(); } catch {}
    }
  };

  // 4) Verificar avance INTELIGENTE: confirmar que apareció la siguiente pantalla
  const verifyAdvance = async () => {
    const start = Date.now();
    const max = 15000; // Aumentado para dar más tiempo a la transición Angular
    while (Date.now() - start < max) {
      const still = document.querySelector(hostSel);
      const vis = !!(still && (still.offsetParent !== null || (still.getBoundingClientRect?.().width || 0) > 0));
      
      // ✅ Verificación: apareció la SIGUIENTE pantalla
      if (which === 'title') {
        // Esperamos que aparezca la pantalla de fotos
        
        const hasIt = hasPhotoInput();
        
        
        if (hasIt) {
          
          await new Promise(r => setTimeout(r, 500)); // Respiro post-transición
          return true;
        }
      } else if (which === 'photo') {
        // Esperamos que aparezca el combo de categorías
        
        const categoryCombo = document.querySelector('#step-category walla-dropdown [role="button"][aria-haspopup="listbox"]');
        
        if (categoryCombo && categoryCombo.offsetParent !== null) {
          
          return true;
        }
        
        // O que simplemente salimos de fotos
        if (!hasPhotoInput()) {
          
          return true;
        }
      } else {
        // Auto: verificar toggle de estado
        const now = hasPhotoInput();
        if (hadPhotoBefore === false && now === true) {
          
          return true;
        }
        if (hadPhotoBefore === true && now === false) {
          
          return true;
        }
      }

      // Continuar esperando
      await new Promise(r => setTimeout(r, 150));
    }
    
    return false;
  };

  // 5) Ejecutar click UNA VEZ (los reintentos los maneja boot.js)
  
  doClickOnce();
  await new Promise(r => setTimeout(r, 800)); // Respiro post-click
  
  
  const ok = await verifyAdvance();
  
  if (ok) {
    
    return { ok:true, which: which ?? (hadPhotoBefore ? 'photo->next' : 'title->photo') };
  } else {
    
    return { ok:false, code: 'no_advance', which };
  }
}

/* ============================================================
   7) Click en "Publicar" y confirmar diálogo/redirección
   ============================================================ */
export async function clickPublishAndConfirm({
  timeout = TIMEOUTS.CLICK_PUBLISH,
  closeDialog = true,
  log = false,  // ✅ v10.5.109: Silenciado por defecto
} = {}) {
  const dom = await ensureDom();
  const waitFor = dom?.waitFor || (async (getter, { timeout=TIMEOUTS.DEFAULT, interval=DELAYS.POLL_FAST } = {}) => {
    const t0 = Date.now();
    while (Date.now() - t0 < timeout) {
      const v = getter();
      if (v) return v;
      await sleep(interval);
    }
    return null;
  });

  const hostSel = 'walla-button[data-testid="continue-action-button"]';
  const host = await waitFor(()=>document.querySelector(hostSel), { timeout: Math.max(TIMEOUTS.SHORT, timeout-6000) });
  if (!host) return { ok:false, code:'publish_button_not_found' };
  const btn = host.shadowRoot?.querySelector('button');
  if (!btn)  return { ok:false, code:'inner_button_not_found' };

  if (dom.waitEnabledAttr) {
    const ok = await dom.waitEnabledAttr([host, btn], { timeout: TIMEOUTS.QUICK_CHECK });
    if (!ok) return { ok:false, code:'disabled' };
  } else if (btn.disabled || host.getAttribute('aria-disabled')==='true') {
    return { ok:false, code:'disabled' };
  }

  dom?.dispatchPointerClick ? dom.dispatchPointerClick(btn, { host }) : btn.click();
  if (log) console.log('[PUB][publish] clicked');

  // SIMPLIFICADO: Solo esperar para que los interceptores capturen
  // 🤖 Delay humano con alta variación (±40%) para simular lectura/revisión
  if (log) console.log('[PUB][publish] esperando ~3s para que interceptores capturen ID...');
  await sleep(humanDelay(3000, 0.4)); // 1800-4200ms
  
  // Los interceptores 5 capas ya capturaron el ID en background
  if (log) console.log('[PUB][publish] asumiendo OK - interceptores deberían haber capturado');
  return { ok:true, via: 'no-modal-check' };
}

/* ============================================================
   8) Subida de imágenes desde snapshot
   ============================================================ */
export async function uploadImagesFromSnapshot(snapshot = {}, opts = {}) {
  const logger = await getLogger();
  
  const {
    apiBase = '',          // ej: "https://www.mitiklive.com/fa"
    log = false,
    timeout = TIMEOUTS.BRAND_COMBO  // ✅ Usa constante
  } = opts;

  // ✅ v10.4.4: Restaurada definición de logg que faltaba
  const t0 = Date.now();

  const ABS = (u) => {
    if (!u) return '';
    if (/^https?:\/\//i.test(u)) return u;
    const base = String(apiBase || '').replace(/\/$/, '');
    return base + (u.startsWith('/') ? u : '/' + u);
  };

  // 1) Normaliza imágenes
  const imgs = (snapshot.images || [])
    .map(x => ({ seq: Number(x.seq) || 0, url: ABS(x.local_url || x.file_url || x.url) }))
    .filter(x => x.url)
    .sort((a, b) => a.seq - b.seq)
    .slice(0, 10);

  if (!imgs.length) {
    return { ok: false, code: 'no_images' };
  }
  

  // 2) Localiza input
  const findInput = () =>
    document.querySelector('tsl-drop-area-preview input[type="file"][multiple]') ||
    document.querySelector('tsl-drop-area-zone input[type="file"][accept*="image" i]');

  let input = null;
  while (!(input = findInput()) && Date.now() - t0 < timeout) {
    await new Promise(r => setTimeout(r, 120));
  }
  if (!input) return { ok: false, code: 'file_input_not_found' };

  // 3) Descarga y construye FileList
  const dt = new DataTransfer();
  
  for (let i = 0; i < imgs.length; i++) {
    const it = imgs[i];
    const seq = it.seq || (i + 1);
    try {
      
      // Fetch simple sin autenticación (imágenes en /public)
      const fetchOptions = { 
        cache: 'no-store', 
        credentials: 'omit', 
        mode: 'cors'
      };
      
      const res = await fetch(it.url, fetchOptions);
      
      if (!res.ok) {
        logger.error(`❌ HTTP ${res.status} al descargar imagen`, { url: it.url, status: res.status });
        throw new Error('http_' + res.status);
      }
      
      const b = await res.blob();
      const mime = b.type || 'image/jpeg';
      const ext = /png/i.test(mime) ? 'png' : /webp/i.test(mime) ? 'webp' : 'jpg';
      const name = String(seq).padStart(2, '0') + '.' + ext;
      dt.items.add(new File([b], name, { type: mime, lastModified: Date.now() }));
      
      
    } catch (e) {
      logger.error('❌ Error al descargar imagen', { url: it.url, error: e.message, seq });
      return { ok: false, code: 'fetch_fail', error: String(e?.message || e), url: it.url };
    }
  }

  // 4) Asigna y dispara eventos
  try { input.scrollIntoView({ block: 'center' }); input.focus?.(); } catch {}
  try { input.files = dt.files; } catch (e) {
    logger.error('❌ Error al asignar archivos al input', { error: e.message });
    return { ok: false, code: 'assign_files_failed', error: String(e?.message || e) };
  }
  input.dispatchEvent(new Event('input',  { bubbles: true, composed: true }));
  input.dispatchEvent(new Event('change', { bubbles: true, composed: true }));

  // 5) Verifica previews o count
  const deadline = Date.now() + 12000;
  let ok = false;
  while (Date.now() < deadline) {
    const thumbs = document.querySelectorAll('.DropAreaPreview__button img, .DropAreaPreview__button picture img');
    if (thumbs.length >= imgs.length || (input.files?.length || 0) >= imgs.length) { ok = true; break; }
    await new Promise(r => setTimeout(r, 180));
  }
  
  
  if (ok) {
  } else {
  }
  
  return { ok, added: imgs.length, code: ok ? 'uploaded' : 'verify_timeout' };
}

/* ============================================================
   9) Selección de categoría (path/taxonomía)
   ============================================================ */
export async function selectCategoryPath({ taxonomy = [] } = {}) {
  const path = taxonomy.map(t => t.name).filter(Boolean);
  if (!path.length) return { ok:false, code:'empty_taxonomy' };
  
  // ✅ Usa norm importado desde dom.js
  const ROOT = '#step-category walla-dropdown';
  const TRIGGER_SEL = `${ROOT} [role="button"][aria-haspopup="listbox"]`;
  const LISTBOX_SEL  = `${ROOT} [role="listbox"]`;
  const OPTION_SEL   = `${LISTBOX_SEL} [role="option"]`;
  const HIDDEN_SEL   = '#category_leaf_id';

  const fireAtPoint = (x, y) => {
    const E = window.PointerEvent || window.MouseEvent;
    const t = document.elementFromPoint(x, y);
    if (!t) return false;
    const ev = (type)=>t.dispatchEvent(new E(type,{bubbles:true,composed:true,cancelable:true,clientX:x,clientY:y,button:0,pointerId:1}));
    ev('pointerdown'); ev('mousedown'); ev('mouseup'); ev('pointerup'); ev('click');
    return true;
  };

  const waitFor = async (fn, {timeout=TIMEOUTS.ELEMENT_APPEAR, interval=DELAYS.POLL_FAST}={})=>{
    const t0=Date.now(); let v=null;
    while(Date.now()-t0<timeout){
      v = await Promise.resolve().then(fn).catch(()=>null);
      if(v) return v;
      await sleep(interval);
    }
    return null;
  };

  const openDropdown = async () => {
    const trigger = await waitFor(()=>document.querySelector(TRIGGER_SEL));
    if (!trigger) throw new Error('trigger no encontrado');
    
    // ✅ v6.6.0: Reintentos inteligentes para abrir dropdown
    const MAX_RETRIES = 3;
    let lastError = null;
    
    for (let attempt = 1; attempt <= MAX_RETRIES; attempt++) {
      
      trigger.scrollIntoView({block:'center'});
      await sleep(200); // Esperar a que termine el scroll
      
      const r = trigger.getBoundingClientRect();
      fireAtPoint(Math.round(r.left+r.width/2), Math.round(r.top+r.height/2));
      
      // Esperar a que se abra
      const opened = await waitFor(
        ()=>document.querySelector(TRIGGER_SEL)?.getAttribute('aria-expanded')==='true',
        {timeout: 2000} // 2 segundos por intento
      );
      
      if (opened) {
        const lb = await waitFor(()=>document.querySelector(LISTBOX_SEL),{timeout:TIMEOUTS.DROPDOWN_OPEN});
        await sleep(800);
        return; // ¡Éxito!
      }
      
      lastError = `Intento ${attempt}: dropdown no se abrió (aria-expanded != true)`;
      
      if (attempt < MAX_RETRIES) {
        // Esperar antes del siguiente intento con backoff
        const waitTime = attempt * 500;
        await sleep(waitTime);
      }
    }
    
    throw new Error(`No se pudo abrir el dropdown después de ${MAX_RETRIES} intentos. ${lastError}`);
  };

  const clickLabel = async (label, {isLast=false}={}) => {
    const normLabel = norm(label);
    
    // Esperar a que el listbox esté visible
    await waitFor(() => document.querySelector(LISTBOX_SEL), { timeout: TIMEOUTS.DROPDOWN_OPTIONS });
    await sleep(300);
    
    // Función auxiliar: buscar en shadow DOM
    const buscarEnShadow = (buscarTexto, debeContener = null) => {
      const contenedores = Array.from(document.querySelectorAll('walla-list-item.sc-walla-dropdown-item'));
      
      
      const resultados = [];
      
      for (const contenedor of contenedores) {
        const shadowRoot = contenedor.shadowRoot;
        if (!shadowRoot) continue;
        
        const spans = shadowRoot.querySelectorAll('span');
        
        for (const span of spans) {
          const text = (span.textContent || '').trim().toLowerCase();
          const textNorm = norm(text);
          
          const coincide = textNorm === buscarTexto || textNorm.includes(buscarTexto);
          const contieneExtra = !debeContener || text.includes(debeContener);
          
          if (coincide && contieneExtra) {
            resultados.push({ span, contenedor, text });
          }
        }
      }
      
      return resultados[0] || null;
    };
    
    // PASO 1: Buscar y clickear la categoría normal (NO "Sólo seleccionar")
    
    const resultado = buscarEnShadow(normLabel);
    
    if (!resultado) {
      throw new Error(`no encontrado: ${label}`);
    }
    
    const { span, contenedor, text } = resultado;
    
    // Scroll y click
    try {
      span.scrollIntoView({ block: 'center', behavior: 'auto' });
      await sleep(300);
    } catch (e) {
    }
    
    // Intentar click
    let clickeado = false;
    
    try {
      span.click();
      clickeado = true;
    } catch (e) {}
    
    if (!clickeado) {
      try {
        span.dispatchEvent(new MouseEvent('click', {bubbles: true, cancelable: true, view: window}));
        clickeado = true;
      } catch (e) {}
    }
    
    if (!clickeado) {
      try {
        contenedor.click();
        clickeado = true;
      } catch (e) {}
    }
    
    if (!clickeado) {
      try {
        const rect = span.getBoundingClientRect();
        fireAtPoint(Math.round(rect.left + rect.width / 2), Math.round(rect.top + rect.height / 2));
        clickeado = true;
      } catch (e) {}
    }
    
    if (!clickeado) {
      throw new Error(`no se pudo clickear: ${label}`);
    }
    
    // 🤖 Delay humano después de click (±30% variación)
    await sleep(humanDelay(800)); // 560-1040ms
    
    // PASO 2: Si es último nivel, verificar si el dropdown sigue abierto
    if (isLast) {
      
      const trigger = document.querySelector(TRIGGER_SEL);
      const dropdownAbierto = trigger?.getAttribute('aria-expanded') === 'true';
      
      if (dropdownAbierto) {
        
        // 🤖 Espera humana para que se abra el submenu (±25% variación)
        await sleep(humanDelay(1200, 0.25)); // 900-1500ms
        
        // PASO 2: Scroll forzado para cargar elementos lazy
        const listbox = document.querySelector(LISTBOX_SEL);
        if (listbox) {
          listbox.scrollTop = 0;
          await sleep(100);
          listbox.scrollTop = listbox.scrollHeight; // Scroll al final
          await sleep(100);
          listbox.scrollTop = 0; // Volver arriba
          await sleep(200);
        }
        
        // PASO 3: Esperar activamente hasta que aparezca "Sólo seleccionar"
        const foundSolo = await waitFor(() => {
          const items = document.querySelectorAll('walla-list-item.sc-walla-dropdown-item');
          for (const item of items) {
            const shadow = item.shadowRoot;
            if (!shadow) continue;
            const spans = shadow.querySelectorAll('span');
            for (const span of spans) {
              const text = (span.textContent || '').toLowerCase();
              if (text.includes('sólo seleccionar') || text.includes('solo seleccionar')) {
                return true;
              }
            }
          }
          return false;
        }, { timeout: TIMEOUTS.DROPDOWN_OPTIONS, interval: DELAYS.POLL_NORMAL });
        
        if (foundSolo) {
        } else {
        }
        
        // PASO 4: Buscar "Sólo seleccionar [nombre]"
        // Buscar TODAS las opciones con "sólo seleccionar"
        const contenedores = Array.from(document.querySelectorAll('walla-list-item.sc-walla-dropdown-item'));
        let soloSel = null;
        
        for (const contenedor of contenedores) {
          const shadowRoot = contenedor.shadowRoot;
          if (!shadowRoot) continue;
          
          const spans = shadowRoot.querySelectorAll('span');
          
          for (const span of spans) {
            const text = (span.textContent || '').trim().toLowerCase();
            const textNorm = norm(text);
            
            // Verificar si contiene "sólo seleccionar" Y el nombre de la categoría
            if ((text.includes('sólo seleccionar') || text.includes('solo seleccionar')) && 
                textNorm.includes(normLabel)) {
              soloSel = { span, contenedor, text };
              break;
            } else if (text.includes('sólo seleccionar') || text.includes('solo seleccionar')) {
            }
          }
          
          if (soloSel) break;
        }
        
        if (soloSel) {
          
          // Scroll y click
          try {
            soloSel.span.scrollIntoView({ block: 'center', behavior: 'auto' });
            await sleep(300);
          } catch (e) {}
          
          // Click en "Sólo seleccionar"
          try {
            soloSel.span.click();
          } catch (e) {
            try {
              soloSel.contenedor.click();
            } catch (e2) {
              const rect = soloSel.span.getBoundingClientRect();
              fireAtPoint(Math.round(rect.left + rect.width / 2), Math.round(rect.top + rect.height / 2));
            }
          }
          
          await sleep(600);
        } else {
        }
      } else {
      }
      
      // Verificación final
      
      const dropdownClosed = await waitFor(() => {
        const trigger = document.querySelector(TRIGGER_SEL);
        return trigger?.getAttribute('aria-expanded') === 'false';
      }, {timeout: TIMEOUTS.DROPDOWN_OPEN});
      
      if (dropdownClosed) {
      } else {
      }
      
      const v = await waitFor(()=>document.querySelector(HIDDEN_SEL)?.value,{timeout: TIMEOUTS.DROPDOWN_OPTIONS});
      
      const triggerText = document.querySelector(TRIGGER_SEL)?.textContent?.trim();
      
      if (!v) {
        if (triggerText && triggerText !== 'Selecciona una categoría' && triggerText.length > 0) {
        } else {
          throw new Error('sin leaf id final y sin texto en trigger');
        }
      }
    }
  };

  await openDropdown();
  for (let i=0; i<path.length; i++) {
    const isLast = i === path.length - 1;
    await clickLabel(path[i], { isLast });
    await sleep(500);
  }

  return { ok:true, path };
}

/* ============================================================
   10) Inspector del formulario de publicación (step-listing)
   ============================================================ */
export async function inspectListingForm() {
  const dom = await ensureDom();
  const waitFor = dom?.waitFor || (async (getter, { timeout=TIMEOUTS.ELEMENT_APPEAR, interval=DELAYS.POLL_FAST } = {}) => {
    const t0 = Date.now();
    while (Date.now() - t0 < timeout) {
      const v = getter();
      if (v) return v;
      await new Promise(r => setTimeout(r, interval));
    }
    return null;
  });

  const root = await waitFor(() =>
    document.querySelector('#step-listing, tsl-upload-step-form') ||
    document.querySelector('tsl-upload tsl-upload-step-form'),
    { timeout: TIMEOUTS.BUTTON_ENABLED }
  );
  const out = [];
  if (!root) return { ok:false, code:'step_listing_not_found', items:[] };

  const normTxt = (s)=>String(s||'').replace(/\s+/g,' ').trim();

  // Texto: Título
  (function(){
    const el = root.querySelector('input#title[name="title"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Título';
    out.push({ kind:'text', id:el.id||null, name:el.name||'title',
      label:normTxt(label), required:/\*/.test(label), maxlength: el.maxLength>0?el.maxLength:null,
      value: el.value || '' });
  })();

  // Textarea: Descripción
  (function(){
    const el = root.querySelector('textarea#description[name="description"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Descripción';
    out.push({ kind:'textarea', id:el.id||null, name:el.name||'description',
      label:normTxt(label), required:/\*/.test(label), maxlength: el.maxLength>0?el.maxLength:null,
      value: el.value || '' });
  })();

  // Dropdowns genéricos
  (function collectDropdowns(){
    const hosts = root.querySelectorAll('walla-dropdown');
    hosts.forEach(host => {
      const trigger = host.querySelector('.walla-dropdown__inner-input[role="button"]');
      const hidden  = host.querySelector('.walla-dropdown__inner-input__hidden-input');
      if (!trigger || !hidden) return;

      const label = trigger.querySelector('label')?.textContent || host.getAttribute('aria-label') || '';
      const expanded = (trigger.getAttribute('aria-expanded') === 'true');
      const value = hidden.value || '';

      const listbox = host.querySelector('[role="listbox"]');
      const items = listbox ? Array.from(listbox.querySelectorAll('walla-dropdown-item'))
        .map(li => li.getAttribute('aria-label') || li.textContent || '')
        .map(normTxt).filter(Boolean) : [];

      out.push({
        kind:'dropdown',
        label: normTxt(label),
        name: hidden.name || hidden.id || null,
        id: hidden.id || null,
        required:/\*/.test(label),
        expanded,
        value,
        options: items.length ? items : null
      });
    });
  })();

  // Precio
  (function(){
    const el = root.querySelector('input#sale_price[name="sale_price"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Precio';
    out.push({ kind:'text', id:el.id||null, name:el.name||'sale_price',
      label:normTxt(label), required:/\*/.test(label),
      min: el.min?Number(el.min):null, max: el.max?Number(el.max):null,
      value: el.value || '' });
  })();

  // Toggle envío
  (function(){
    const input = root.querySelector('wallapop-toggle input[type="checkbox"]');
    if (!input) return;
    const label = root.querySelector('.ShippingSection__topBottomBorder span')?.textContent || 'Activar envío';
    out.push({ kind:'toggle', label:normTxt(label),
      name: input.name || 'supports_shipping', id: input.id || null, checked: !!input.checked });
  })();

  // Radio peso (solo requerido si el envío está activado)
  (function () {
    const shipOn = !!root.querySelector('wallapop-toggle input[type="checkbox"]')?.checked;
    const radios = Array.from(root.querySelectorAll('tsl-delivery-radio-selector input[type="radio"]'));

    // Fallback: envío activo pero aún no montaron los radios
    if (!radios.length) {
      if (shipOn) {
        out.push({
          kind: 'radio-group',
          label: '¿Cuánto pesa?',
          name: 'weight_tier',
          required: true,
          value: '',
          options: []
        });
      }
      return;
    }

    const label = root.querySelector('#itemWeightTitle')?.textContent || '¿Cuánto pesa?';
    const options = Array.from(root.querySelectorAll('#weightTierLabel')).map(n => normTxt(n.textContent));
    const checked = radios.find(r => r.checked);

    out.push({
      kind: 'radio-group',
      label: normTxt(label),
      name: 'weight_tier',
      required: shipOn,
      value: checked ? checked.value : '',
      options: options.filter(Boolean)
    });
  })();

  // Localización
  (function(){
    const el = document.querySelector('input#location[name="location"]');
    if (!el) return;
    const label = el.closest('.inputWrapper')?.querySelector('label')?.textContent || 'Marca la localización';
    out.push({ kind:'location', label:normTxt(label), name: el.name || 'location', id: el.id || null, value: el.value || '' });
  })();
  
  const missingRequired = out
    .filter(w => w.required && !String(w.value||'').trim())
    .map(w => w.name || w.label || w.kind);

  return { ok: true, items: out, missingRequired };
}

/* ============================================================
   11) Rellenar el formulario de publicación (step-listing)
   ============================================================ */
/* ============================================================
   11) Rellenar el formulario de publicación (step-listing)
   ✅ v4.81.0: REFACTORIZADO COMPLETAMENTE
   - Llena EN ORDEN del JSON (no hardcoded)
   - Valida cada campo inline
   - Se detiene si falla campo CRÍTICO
   - Errores detallados para debugging
   - Modular y mantenible (710 líneas → 150 líneas)
   ============================================================ */
/**
 * Llenar formulario de publicación (step-listing)
 * 
 * @param {object} snapshot - JSON del anuncio de Wallapop
 * @param {object} options
 * @param {boolean} options.log - Si mostrar logs en consola
 * @returns {Promise<{ok: boolean, filled: string[], errors: object[], skipped: string[], missingRequired: object[], lastScan: object[]}>}
 */

// ✅ v6.5.3: fillListingForm EXTRAÍDO a publish/form-filler.js (1143 líneas → módulo separado)
// Se re-exporta al final del archivo para mantener compatibilidad

/* ============================================================
   12) Captura robusta de ID al publicar (Braze + API + URL + SW)
   ============================================================ */
const __ic_sleep = (ms) => new Promise(r => setTimeout(r, ms));
const __ic_dlog  = (...a) => { /* cambia a true si quieres logs */ if (false) console.debug('[id-capture]', ...a); };

function __ic_installBrazeSniffer(onFound) {
  const isBraze = (u) => /https:\/\/[^/]*braze\.com\/api\/v3\/data\/?$/i.test(u || '');
  const extract = (bodyStr) => {
    try {
      const j  = JSON.parse(bodyStr);
      const ev = (j.events || []).find(e => e?.data?.p?.itemId || e?.data?.p?.itemSlug);
      if (!ev) return;
      const p = ev.data.p || {};
      const m = String(p.itemSlug || '').match(/(.+)-(\d+)$/);
      onFound?.({
        source: 'braze',
        itemId:   p.itemId || null,
        itemSlug: p.itemSlug || null,
        legacyId: m ? m[2] : null,
        url: p.itemSlug ? `https://es.wallapop.com/item/${p.itemSlug}` : null,
        title: p.title ?? null,
        price: p.salePrice ?? null,
      });
    } catch {}
  };

  const _fetch = window.fetch;
  window.fetch = function(input, init) {
    try {
      const url = typeof input === 'string' ? input : input?.url;
      if (isBraze(url) && init?.body) {
        if (typeof init.body === 'string') extract(init.body);
        else if (init.body instanceof Blob) init.body.text().then(extract);
      }
    } catch {}
    return _fetch.apply(this, arguments);
  };

  const _open = XMLHttpRequest.prototype.open;
  const _send = XMLHttpRequest.prototype.send;
  XMLHttpRequest.prototype.open = function(m, u) { this.__ml_url = u; this.__ml_m = m; return _open.apply(this, arguments); };
  XMLHttpRequest.prototype.send = function(body) {
    try {
      if (this.__ml_url && isBraze(this.__ml_url) && typeof body === 'string') extract(body);
    } catch {}
    return _send.call(this, body);
  };

  return () => { window.fetch = _fetch; XMLHttpRequest.prototype.open = _open; XMLHttpRequest.prototype.send = _send; };
}

function __ic_installApiCreateSniffer(onFound) {
  const isApi = (u) => /https:\/\/api\.wallapop\.(?:com|es)\//i.test(u || '');

  const parseJSON = (t) => { try { return JSON.parse(t); } catch { return null; } };
  const tryExtract = (j) => {
    if (!j || typeof j !== 'object') return null;
    let id = null, slug = null;
    if (j.id && typeof j.id === 'string') id = j.id;
    if (!slug && j.slug && typeof j.slug === 'string') slug = j.slug;
    if (!slug && j.itemSlug && typeof j.itemSlug === 'string') slug = j.itemSlug;
    if (!id && j.itemId && typeof j.itemId === 'string') id = j.itemId;
    if (!id || id.length < 8) return null;
    const m = String(slug || '').match(/(.+)-(\d+)$/);
    return { source: 'api', itemId: id, itemSlug: slug || null, legacyId: m ? m[2] : null, url: slug ? `https://es.wallapop.com/item/${slug}` : null };
  };

  const _fetch = window.fetch;
  window.fetch = function(input, init) {
    const url = typeof input === 'string' ? input : input?.url;
    const method = (init?.method || 'GET').toUpperCase();
    const p = _fetch.apply(this, arguments);
    try {
      if (isApi(url) && /\/items/i.test(url) && /^(POST|PUT|PATCH)$/i.test(method)) {
        p.then(r => { try { return r.clone().text(); } catch { return ''; } })
         .then(txt => { const d = tryExtract(parseJSON(txt)); if (d) onFound?.(d); });
      }
    } catch {}
    return p;
  };

  const _open = XMLHttpRequest.prototype.open;
  const _send = XMLHttpRequest.prototype.send;
  XMLHttpRequest.prototype.open = function(m, u) { this.__ml_url = u; this.__ml_m = (m||'GET').toUpperCase(); return _open.apply(this, arguments); };
  XMLHttpRequest.prototype.send = function(body) {
    const xhr = this;
    const url = xhr.__ml_url || '';
    const method = xhr.__ml_m || 'GET';
    const watch = /https:\/\/api\.wallapop\.(?:com|es)\//i.test(url) && /\/items/i.test(url) && /^(POST|PUT|PATCH)$/i.test(method);
    if (watch) {
      const _onload = xhr.onload;
      xhr.onload = function(ev) {
        try {
          const d = tryExtract(parseJSON(xhr.responseText || ''));
          if (d) onFound?.(d);
        } catch {}
        if (_onload) _onload.call(xhr, ev);
      };
    }
    return _send.call(this, body);
  };

  return () => { window.fetch = _fetch; XMLHttpRequest.prototype.open = _open; XMLHttpRequest.prototype.send = _send; };
}

function __ic_installUrlWatcher(onFound) {
  const extractFromDom = () => {
    try {
      const candidates = Array.from(document.querySelectorAll('script[type^="application/json"], script#__NEXT_DATA__'));
      for (const s of candidates) {
        const t = s.textContent || s.innerText || '';
        if (!t) continue;
        const j = JSON.parse(t);
        const stack = [j];
        while (stack.length) {
          const cur = stack.pop();
          if (!cur || typeof cur !== 'object') continue;
          for (const [k, v] of Object.entries(cur)) {
            if (typeof v === 'string' && /\b[a-z0-9]{10,16}\b/i.test(v) && /id$/i.test(k)) return { itemId: v };
            if (typeof v === 'string' && /-\d{6,}$/.test(v) && /slug/i.test(k)) return { itemSlug: v };
            if (v && typeof v === 'object') stack.push(v);
          }
        }
      }
    } catch {}
    return null;
  };

  const check = () => {
    const m = location.pathname.match(/\/item\/([^/?#]+)/i);
    if (!m) return;
    const slug = m[1];
    const info = extractFromDom() || {};
    const m2 = String(slug).match(/(.+)-(\d+)$/);
    onFound?.({
      source: 'url',
      itemId: info.itemId || null,
      itemSlug: slug,
      legacyId: m2 ? m2[2] : null,
      url: `https://es.wallapop.com/item/${slug}`,
    });
  };

  const _push = history.pushState;
  const _repl = history.replaceState;
  window.addEventListener('popstate', check);
  history.pushState    = function(){ const r = _push.apply(this, arguments); setTimeout(check, 0); return r; };
  history.replaceState = function(){ const r = _repl.apply(this, arguments); setTimeout(check, 0); return r; };
  setTimeout(check, 0);

  return () => { history.pushState = _push; history.replaceState = _repl; window.removeEventListener('popstate', check); };
}

async function __ic_resolveViaSW(slug, timeoutMs = TIMEOUTS.SW_RESOLVE) {
  if (!slug) return null;
  return new Promise((resolve) => {
    let done = false;
    const to = setTimeout(() => { if (!done) resolve(null); }, timeoutMs);
    try {
      chrome.runtime?.sendMessage?.({ type: 'WALLA.RESOLVE.FROM_SLUG', slug }, (resp) => {
        done = true; clearTimeout(to);
        if (resp && resp.ok && (resp.itemId || resp.itemSlug)) {
          resolve({ source: 'sw', itemId: resp.itemId || null, itemSlug: resp.itemSlug || slug, url: resp.url || `https://es.wallapop.com/item/${slug}` });
        } else {
          resolve(null);
        }
      });
    } catch { resolve(null); }
  });
}

/* ============================================================
   13) Plan B por token: listar recientes del usuario y casar
   ============================================================ */
async function __ic_resolveByToken({ slug, title, createdAt }, timeoutMs = TIMEOUTS.SW_RESOLVE) {
  // ✅ Usa normalize importado desde utils.js
  const normTitle = normalize(title);
  const legacyId  = (String(slug || '').match(/(\d+)$/) || [])[1] || null;

  const creds = await new Promise((resolve) => {
    try {
      chrome.runtime?.sendMessage?.({ type: 'WALLA.CREDS.GET' }, (resp) => resolve(resp || null));
    } catch { resolve(null); }
    setTimeout(() => resolve(null), 2000);
  });
  if (!creds?.bearer) return null;

  const uid  = creds.userId || creds.user_id || 'me';
  const urls = [
    `https://api.wallapop.com/api/v3/users/${encodeURIComponent(uid)}/items?order_by=created_at&order=desc&limit=40`,
    `https://api.wallapop.com/api/v3/users/me/items?order_by=created_at&order=desc&limit=40`,
  ];

  const headers = new Headers();
  headers.set('authorization', `Bearer ${creds.bearer}`);
  if (creds.deviceId)   headers.set('X-DeviceId', creds.deviceId);
  if (creds.deviceOs)   headers.set('X-DeviceOS', String(creds.deviceOs));
  if (creds.appVersion) headers.set('X-AppVersion', String(creds.appVersion));
  headers.set('Accept-Language', 'es-ES,es;q=0.9');

  const deadline = Date.now() + timeoutMs;
  for (const url of urls) {
    try {
      const r = await fetch(url, { method: 'GET', headers, credentials: 'omit' });
      if (!r.ok) continue;
      const j = await r.json().catch(() => null);
      const list = Array.isArray(j?.items) ? j.items : (Array.isArray(j) ? j : []);
      if (!list.length) continue;

      // Buscamos por prioridad: slug → legacyId → título ~similar → más reciente
      let found =
        list.find(it => String(it.slug || it.itemSlug || '').toLowerCase() === String(slug || '').toLowerCase()) ||
        (legacyId ? list.find(it => String(it.slug || it.itemSlug || '').endsWith(legacyId)) : null) ||
        (normTitle ? list.find(it => normalize(it.title || '') === normTitle) : null) ||
        list[0];

      if (found) {
        return {
          source: 'token',
          itemId:   found.id || found.itemId || null,
          itemSlug: found.slug || found.itemSlug || null,
          url: (found.slug || found.itemSlug) ? `https://es.wallapop.com/item/${found.slug || found.itemSlug}` : null
        };
      }
    } catch {}
    if (Date.now() > deadline) break;
    await __ic_sleep(250);
  }
  return null;
}

/* ============================================================
   14) API pública: esperar ID publicado con degradaciones
   ============================================================ */
async function waitForPublishedIdRobust({ maxWaitMs = 60000, toast, onLog } = {}) {
  let resolved = null;
  const log = (...a) => { onLog?.(...a); __ic_dlog(...a); };

  const unhooks = [];
  const once = (data) => {
    if (resolved) return;
    if (data && (data.itemId || data.itemSlug)) resolved = data;
  };

  try { unhooks.push(__ic_installBrazeSniffer(once)); } catch {}
  try { unhooks.push(__ic_installApiCreateSniffer(once)); } catch {}
  try { unhooks.push(__ic_installUrlWatcher(once)); } catch {}

  const start = Date.now();
  const deadline = start + maxWaitMs;
  let backoff = 250;

  // Espera activa con backoff
  while (!resolved && Date.now() < deadline) {
    await __ic_sleep(backoff);
    backoff = Math.min(backoff * 1.6, 1500);
  }

  // Si solo hay slug → intentar resolver por SW (HTML)
  if (!resolved?.itemId && resolved?.itemSlug) {
    const viaSw = await __ic_resolveViaSW(resolved.itemSlug, 12000);
    if (viaSw?.itemId) resolved = { ...resolved, ...viaSw };
  }

  // Limpieza hooks
  for (const u of unhooks) { try { u && u(); } catch {} }

  if (resolved) {
    try { toast?.(`Publicado: ${resolved.itemId || '(sin id)'} — ${resolved.itemSlug || ''}`); } catch {}
    return resolved;
  }

  // Última oportunidad: si hay /item/<slug> visible en URL, pedir al SW
  const m = location.pathname.match(/\/item\/([^/?#]+)/i);
  if (m) {
    const slug = m[1];
    const viaSw = await __ic_resolveViaSW(slug, 12000);
    if (viaSw?.itemId || viaSw?.itemSlug) {
      const out = { source: 'sw-last', itemId: viaSw.itemId || null, itemSlug: viaSw.itemSlug || slug, url: viaSw.url };
      try { toast?.(`Publicado (fallback): ${out.itemId || '(sin id)'} — ${out.itemSlug}`); } catch {}
      return out;
    }
  }
  return null;
}

/* ============================================================
   15) Wrapper: publicar + capturar ID (con plan B por token)
   ============================================================ */
export async function publishAndCaptureId({ snapshot = {}, maxWaitMs = 60000, toast, onLog } = {}) {
  const logger = await getLogger();
  const title = snapshot?.title?.original || snapshot?.title;
  
  // 🚨 Importar showLoader desde utils.js
  const { showLoader, hideLoader } = await import('./utils.js');
  
  // Mostrar loader con mensaje de no cambiar de pestaña
  showLoader('⚠️ No cambies de pestaña durante la publicación');
  
  try {
    // instala captura antes del click y lanza la publicación
    const waitIdPromise = waitForPublishedIdRobust({ maxWaitMs, toast, onLog });

  const pub = await clickPublishAndConfirm({ timeout: Math.max(16000, maxWaitMs), log: true });
  if (!pub?.ok) {
    logger.error('❌ Error al hacer click en publicar', { code: pub?.code });
    return { ok:false, code: pub?.code || 'publish_failed' };
  }

  // espera ID
  let res = await waitIdPromise;

  // si no llega → token (listar recientes y casar)
  if (!res || !res.itemId) {
    const title = snapshot?.title?.original ?? snapshot?.title ?? snapshot?.product?.title ?? '';
    const slug  = res?.itemSlug || (location.pathname.match(/\/item\/([^/?#]+)/i) || [])[1] || null;
    res = await __ic_resolveByToken({ slug, title, createdAt: Date.now() }, 15000) || res;
  }

  if (res?.itemId || res?.itemSlug) {
    
    // notifica al SW/backend
    try {
      chrome.runtime?.sendMessage?.({
        type: 'LISTING.PUBLISHED',
        payload: { ...res, at: new Date().toISOString(), previousDeletedId: window.__ml_prev_deleted_id || null }
      });
    } catch {}

    try { toast?.(`✅ ID capturado: ${res.itemId || '(sin id)'} — ${res.itemSlug || ''}`); } catch {}
    return { ok:true, via: pub.via, ...res };
  }

  try { toast?.('⚠️ Publicado, pero no se pudo capturar el ID'); } catch {}
  logger.error('❌ No se pudo capturar el ID del anuncio publicado');
  return { ok:false, code:'id_not_captured', via: pub.via || null };
  } finally {
    // 🚨 Ocultar loader siempre (éxito o error)
    hideLoader();
  }
}
/* ============================================================
   Verificar si el botón Publicar está visible
   ============================================================ */
export function checkPublishButton() {
  // Buscar walla-button en shadow DOM
  const wallaButton = document.querySelector('walla-button[data-testid="continue-action-button"]');
  
  if (!wallaButton) {
    dlog('[PUB][check] ❌ No se encontró walla-button');
    return { ok: false, visible: false, reason: 'walla_button_not_found' };
  }
  
  const shadowRoot = wallaButton.shadowRoot;
  if (!shadowRoot) {
    dlog('[PUB][check] ❌ walla-button sin shadowRoot');
    return { ok: false, visible: false, reason: 'no_shadow_root' };
  }
  
  const btn = shadowRoot.querySelector('button');
  if (!btn) {
    dlog('[PUB][check] ❌ No se encontró button en shadowRoot');
    return { ok: false, visible: false, reason: 'button_not_found_in_shadow' };
  }
  
  const rect = btn.getBoundingClientRect();
  const visible = rect.width > 0 && rect.height > 0;
  const disabled = btn.disabled || btn.hasAttribute('disabled') || btn.getAttribute('aria-disabled') === 'true';
  
  dlog('[PUB][check] ✅ Botón encontrado:', {
    text: btn.textContent?.trim(),
    visible,
    disabled,
    ariaDisabled: btn.getAttribute('aria-disabled')
  });
  
  // ✅ v2.5.3: Solo verificar que existe y es visible
  // El estado disabled lo controla MitikLive (se habilita antes del click)
  return { 
    ok: visible, 
    visible, 
    disabled,
    text: btn.textContent?.trim()
  };
}
/* ============================================================
   ✅ v6.5.3: RE-EXPORTS para mantener compatibilidad
   ============================================================ */
// Re-exportar fillListingForm desde el módulo extraído
// Esto permite que content_script.js siga usando:
// const { fillListingForm } = await import('scripts/publish.js');
export { fillListingForm } from './publish/form-filler.js';
