/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/ui.js – Rol: Pintado de UI (badges, botones, run box, legales) - Renderizado desde UIState
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
    
    ✅ v10.5.29: btnBackup movido a ButtonManager (publish-button-manager.js)
*/

import DOMUtils from './dom-utils.js';
import { PUBLISH_ASKED_KEY } from './constants.js';
import { logger } from './logger.js';
import * as ButtonManager from './publish-button-manager.js'; // ✅ v10.5.29

import { stGet, stSet, stRemove } from './storage.js';
import { Bus } from './bus.js';
import {
  AUTH, RUN_ACTIVE, LAST_GOOD, pollTimer, currentRunId, lastNotifiedTerminal, lastWarnedRunId,
  NOLOGIN_HOLD, NOLOGIN_HOLD_MS, LAST_PENDING_RUN_ID, API_STATE, _es, _sseRetryTimer, _hbTimer,
  pollingIntervalId, CHECK_INTERVAL_MS, CHECKING_GRACE_MS, OFFLINE_MS, RECONNECT_MS,
  verEl, pillIP, pillUser, blockLogin, blockApp, userEmail, inEmail, inPass, remember, listEmails,
  btnLogin, btnGoogle, btnBackup,  // ✅ btnBackup solo para referencia, lógica en ButtonManager
  warningBadges,
  btnMenu, menuPop, miConfig, miLogout, btnPopout, selAccount, accBadge, accTxt, avatarEl, LEGAL,
  _POST_BK_PROMPTED_RUN_ID, UIState, setRenderCallback, setUIState, BACKUP_STATE,
  set_AUTH, set_RUN_ACTIVE, set_LAST_GOOD, set_API_STATE, set_LAST_PENDING_RUN_ID,
  set_lastNotifiedTerminal, set__POST_BK_PROMPTED_RUN_ID
} from './state.js';
import { toast, showModal, showLoader as _showLoader, hideLoader as _hideLoader, retryWithBackoff, sleep } from '../utils.js';
import { sw } from './runs.js';
import { stopPolling } from './polling.js';

/* ======================
   Helpers básicos de UI
   ====================== */
export function ensureBadge(parent, cls) {
  if (!parent) return null;
  let b = parent.querySelector(`.${cls}`);
  if (!b) {
    b = document.createElement('span');
    b.className = cls;
    parent.appendChild(b);
  }
  return b;
}

/* =========================================================
   ✅ RENDERIZADO CENTRALIZADO DESDE UIState
   ========================================================= */
export function renderUIState() {
  const { publish, validation, loading } = UIState;
  
  // ✅ Backup usa BACKUP_STATE (fuente única de verdad)
  // NO usar UIState.backup porque puede estar desincronizado
  updateBackupButtonUI(BACKUP_STATE);
  
  // ✅ v4.98.0: ButtonManager gestiona automáticamente según estado
  // updatePublishButtonsState(BACKUP_STATE);
  
  // ✅ v5.1.0: setPublishResumeAffordance eliminado - ButtonManager gestiona automáticamente
  
  showValidationBadges(validation);
  
  // Loading: deshabilitar selector de cuentas
  if (selAccount) {
    selAccount.disabled = loading;
  }
}

/* =========================================================
   ✅ Mostrar badges de validation warnings
   ========================================================= */
export function showValidationBadges(warnings) {
  if (!warningBadges) return;
  
  warningBadges.innerHTML = '';
  
  if (warnings?.duplicates?.length > 0) {
    const badge = document.createElement('span');
    badge.className = 'badge badge-warning';
    badge.textContent = `⚠️ ${warnings.duplicates.length} duplicados`;
    badge.title = 'Items duplicados detectados';
    warningBadges.appendChild(badge);
  }
  
  if (warnings?.orphans?.length > 0) {
    const badge = document.createElement('span');
    badge.className = 'badge badge-error';
    badge.textContent = `❌ ${warnings.orphans.length} huérfanos`;
    badge.title = 'Items huérfanos (sin listing)';
    warningBadges.appendChild(badge);
  }
  
  if (warnings?.inconsistent_states?.length > 0) {
    const badge = document.createElement('span');
    badge.className = 'badge badge-info';
    badge.textContent = `⏳ ${warnings.inconsistent_states.length} colgados`;
    badge.title = 'Items en estados inconsistentes';
    warningBadges.appendChild(badge);
  }
}

/* ==============================
   Botón Backup (estado + badges)
   ============================== */
/**
 * ✅ v10.5.29: DELEGADO A ButtonManager
 * Mantiene compatibilidad con código existente
 * @param {Object} state - Estado del backup (opcional)
 */
export function updateBackupButtonUI(state = null) {
  // Usar state pasado o el global importado
  const s = state || BACKUP_STATE;
  
  // ✅ v10.5.29: Delegar a ButtonManager
  ButtonManager.updateBackupState(s);
  ButtonManager.setAuthenticated(AUTH.on);
}

// ✅ v10.5.29: Escuchar evento de backup-viewed para marcar como visto
window.addEventListener('backup-viewed', (e) => {
  if (e.detail?.runId) {
    markBackupAsViewed(e.detail.runId);
  }
});

/* =========================================================
   ✅ ACTUALIZAR BOTONES DE PUBLICACIÓN según estado del backup
   ========================================================= */
/**
 * Deshabilita botones de publicación cuando backup está activo/incompleto/error
 * Solo permite publicar cuando backup está 'ok' o 'idle'
 */
export function updatePublishButtonsState(backupState = null) {
  const state = backupState || BACKUP_STATE;
  const status = String(state.status || 'idle').toLowerCase();
  
  // ✅ Solo permitir publicar si backup OK o idle
  const canPublish = status === 'ok' || status === 'idle';
  const isBackupActive = status === 'running';
  
  // Botón "Publicar Todos" / "Reanudar (X)"
  const btnPublishAll = document.getElementById('btn-publish-all');
  if (btnPublishAll) {
    btnPublishAll.disabled = !canPublish || !AUTH.on;
    btnPublishAll.setAttribute('aria-disabled', (!canPublish || !AUTH.on) ? 'true' : 'false');
    
    if (!canPublish) {
      if (isBackupActive) {
        btnPublishAll.title = '⏳ Espera a que termine el backup';
      } else if (status === 'incomplete' || status === 'cancelled' || status === 'canceled') {
        btnPublishAll.title = '⚠️ Completa o cancela el backup incompleto primero';
      } else if (status === 'error') {
        btnPublishAll.title = '❌ Hay errores en el backup, resuélvelos primero';
      } else {
        btnPublishAll.title = 'Completa el backup antes de publicar';
      }
    } else {
      btnPublishAll.title = '';
    }
  }
  
  // Botón "Publicar (X) seleccionados"
  const btnPublishSelected = document.getElementById('btn-publish-selected');
  if (btnPublishSelected) {
    btnPublishSelected.disabled = !canPublish || !AUTH.on;
    btnPublishSelected.setAttribute('aria-disabled', (!canPublish || !AUTH.on) ? 'true' : 'false');
    
    if (!canPublish) {
      if (isBackupActive) {
        btnPublishSelected.title = '⏳ Espera a que termine el backup';
      } else if (status === 'incomplete' || status === 'cancelled' || status === 'canceled') {
        btnPublishSelected.title = '⚠️ Completa o cancela el backup incompleto primero';
      } else if (status === 'error') {
        btnPublishSelected.title = '❌ Hay errores en el backup, resuélvelos primero';
      } else {
        btnPublishSelected.title = 'Completa el backup antes de publicar';
      }
    } else {
      btnPublishSelected.title = '';
    }
  }
  
}

/* ==========================================
   Reanudar publicación (botón + contadores)
   ========================================== */
/* ==============================================
   Reanudar backup (botón dentro de la run box)
   ============================================== */
// ✅ v5.1.0: Funciones deprecated eliminadas:
// - setPublishResumeAffordance() → usar ButtonManager
// - updatePublishResumeButton() → usar ButtonManager  
// - setResumeAffordance() → no-op eliminado
// - setBar/setBarIndeterminate/showRunUI/hideRunUI → barra progreso eliminada

/* =======================
   Pill estado API (UI)
   ======================= */
export function paintApiStatus(kind = 'checking') {
  if (!pillIP) return;
  if (API_STATE === kind) return;
  set_API_STATE(kind);

  const dot = kind === 'online' ? 'ok' : (kind === 'offline' ? 'err' : 'warn');
  
  // ✅ Solo dot, sin texto
  pillIP.innerHTML = `<span class="ml-dot ${dot}" aria-hidden="true"></span>`;

  const online = kind === 'online';
  document.querySelectorAll('.needs-api').forEach(el => {
    if (online) { el.removeAttribute('disabled'); el.setAttribute('aria-disabled','false'); }
    else        { el.setAttribute('disabled','');  el.setAttribute('aria-disabled','true'); }
  });
}

/* =======================
   Controles generales
   ======================= */
// ✅ v4.97.3: Migrado a ButtonManager
export function setRunControls(running) {
  set_RUN_ACTIVE(!!running);
  
  // ✅ Informar a ButtonManager
  // ButtonManager gestionará el estado de los botones automáticamente
  // (La lógica de deshabilitar botones durante backup se maneja en ButtonManager)
}

export async function setAuthUI(isOn, email='') {
  set_AUTH({ ...AUTH, on: !!isOn });
  
  // ✅ v5.8.1: Verificar plataforma ANTES de mostrar block-app
  // Evita flash del panel Wallapop cuando el usuario seleccionó Milanuncios
  if (isOn) {
    const { selectedPlatform } = await chrome.storage.local.get(['selectedPlatform']);
    if (selectedPlatform === 'milanuncios') {
      logger.info('[UI] Plataforma Milanuncios detectada, redirigiendo sin mostrar Wallapop...');
      if (blockLogin) blockLogin.style.display = 'none';
      window.location.href = 'milanuncios/panel-milanuncios.html';
      return; // No continuar, evitar flash
    }
  }
  
  if (blockLogin) blockLogin.style.display = isOn ? 'none' : 'flex';
  if (blockApp)   blockApp.style.display   = isOn ? 'flex' : 'none';
  if (userEmail) userEmail.textContent = isOn ? (email || '') : '';
  // ✅ v10.5.144: Usar span interno para el texto del email
  const pillUserText = document.querySelector('.ml-user-email-text');
  if (pillUserText) pillUserText.textContent = isOn ? (email || '') : '';

  // Añadir clase al body para controlar [data-auth] con CSS
  if (isOn) {
    document.body.classList.add('authenticated');
  } else {
    document.body.classList.remove('authenticated');
  }

  document.querySelectorAll('[data-auth]').forEach(el => {
    el.setAttribute('aria-disabled', isOn ? 'false' : 'true');
    if (!isOn) el.classList.add('needs-auth'); else el.classList.remove('needs-auth');
  });

  // ✅ v10.5.29: Delegar btnBackup a ButtonManager
  ButtonManager.setAuthenticated(isOn);

  if (!isOn) {
    try { _hideLoader(); } catch {}
    // ✅ v5.1.0: hideRunUI eliminado (era no-op)
    stopPolling();
  }
}

/* ===========
   Badge Wallapop
   =========== */
export function setBadgeText(txt){
  if (accTxt) accTxt.textContent = txt || '';
}
export function setBadgeTitle(txt){
  if (accBadge) accBadge.title = txt || '';
}
export function setBadgeLoading(on){
  accBadge?.classList.toggle('is-loading', !!on);
}

export function setAvatar(url){
  if (!avatarEl) return;
  if (url){
    avatarEl.src = url;
    avatarEl.referrerPolicy = 'no-referrer';
    avatarEl.hidden = false;
  } else {
    avatarEl.hidden = true;
    avatarEl.removeAttribute('src');
  }
}

export function paintConnected(alias, avatarUrl){
  setBadgeText(alias ? `Conectada: ${alias}` : 'Conectada');
  setAvatar(avatarUrl || '');
  setBadgeTitle('Wallapop abierto y con sesión.');
  accBadge?.classList.remove('state-connected','state-nologin','state-nodetect');
  accBadge?.classList.add('state-connected');
}

export function paintNoLogin(){
  setBadgeText('No login en Wallapop');
  setAvatar('');
  setBadgeTitle('Wallapop abierto, sin sesión.');
  accBadge?.classList.remove('state-connected','state-nodetect');
  accBadge?.classList.add('state-nologin');
}

export function paintNotDetected(){
  setBadgeText('No detectado Wallapop');
  setAvatar('');
  setBadgeTitle('No hay pestañas de Wallapop abiertas.');
  accBadge?.classList.remove('state-connected','state-nologin');
  accBadge?.classList.add('state-nodetect');
}

/* ==========================
   Selector de cuentas (UI)
   ========================== */
export function readOptionDatasets(opt){
  if (!opt) return null;
  return {
    alias:   opt.dataset.alias   || '',
    avatar:  opt.dataset.avatar  || '',
    profile: opt.dataset.profile || ''
  };
}

export function uiFromOption(opt){
  const d = readOptionDatasets(opt);
  if (!d) return;
  const alias = d.alias || '';
  const avatar = d.avatar || '';
  if (alias) setBadgeTitle(`Seleccionada: ${alias}`);
  if (avatar) setAvatar(avatar);
}

export async function renderAccounts(list){
  if (!selAccount) return;
  
  // ✅ v6.5.2: Guard contra renderizado duplicado
  // Si ya hay opciones y la lista es la misma, no re-renderizar
  const currentOptionsCount = selAccount.options.length;
  const newListCount = (list || []).length;
  
  // Si hay cuentas y el número coincide (placeholder + cuentas), verificar si es el mismo
  if (currentOptionsCount > 1 && newListCount > 0 && currentOptionsCount === newListCount + 1) {
    const firstExistingValue = selAccount.options[1]?.dataset?.accountId;
    const firstNewValue = list[0]?.id;
    if (firstExistingValue && firstNewValue && String(firstExistingValue) === String(firstNewValue)) {
      
      // ✅ v6.6.0: FIX - Aunque saltemos el re-render, disparar evento change
      // para que handleAccountChange() cargue los datos de la cuenta
      if (selAccount.value && selAccount.value !== '') {
        const changeEvent = new Event('change', { bubbles: true });
        selAccount.dispatchEvent(changeEvent);
      }
      
      return;
    }
  }
  
  selAccount.innerHTML = '';

  // ✅ v6.2.0: Limpiar localStorage si el usuario cambió (seguridad)
  // Verificar que el usuario actual es el mismo que guardó el localStorage
  try {
    const sessionResponse = await new Promise((resolve) => {
      chrome.runtime.sendMessage({ type: 'SESSION.GET' }, resolve);
    });
    const currentUserId = sessionResponse?.data?.user?.id;
    
    if (currentUserId) {
      const { last_user_id } = await stGet(['last_user_id']);
      
      if (last_user_id && last_user_id !== currentUserId) {
        // Usuario diferente al anterior → limpiar TODO el localStorage
        await stRemove(['selected_account_id', 'account_id', 'last_user_id']);
        
        // 🔥 CRÍTICO: Limpiar también la tabla de anuncios residuales
        const tbody = document.querySelector('#listings-table tbody, #listings-tbody');
        if (tbody) {
          tbody.innerHTML = '';
        }
        
        // Resetear stats a cero
        const statsWrap = document.querySelector('.ml-listings-stats');
        if (statsWrap) {
          statsWrap.querySelectorAll('.stat-value').forEach(el => el.textContent = '0');
        }
      }
      
      // Guardar usuario actual para próxima comparación
      await stSet({ last_user_id: currentUserId });
    }
  } catch (err) {
  }

  // 🔴 Si no hay cuentas, mostrar mensaje apropiado
  if (!list || list.length === 0) {
    const ph = document.createElement('option');
    ph.value = '';
    ph.textContent = '— No tienes cuentas vinculadas —';
    selAccount.appendChild(ph);
    
    
    // Limpiar storage y UI
    await stRemove(['selected_account_id', 'account_id']).catch(err => { console.error(`[${filepath.split('/').pop()}] Error:`, err); });
    
    // 🔥 Ocultar TODA la sección de listings
    const statsWrap = document.querySelector('.ml-listings-stats');
    if (statsWrap) {
      statsWrap.querySelectorAll('.stat-value').forEach(el => el.textContent = '0');
      DOMUtils.hide(statsWrap);
    }
    
    // 🔥 Ocultar controles (buscador, contador, selección)
    const controls = document.querySelector('.ml-listings-controls');
    if (controls) DOMUtils.hide(controls);
    
    // 🔥 Ocultar tabla completa
    const tableWrapper = document.querySelector('.ml-listings-table-wrapper');
    if (tableWrapper) DOMUtils.hide(tableWrapper);
    
    // 🔥 Limpiar tbody por si acaso
    const tbody = document.querySelector('#listings-table tbody, #listings-tbody');
    if (tbody) tbody.innerHTML = '';
    
    // 🔥 Ocultar sección completa de listings
    const listingsSection = document.getElementById('listings-section');
    if (listingsSection) DOMUtils.hide(listingsSection);
    
    // 🔥 Ocultar también las acciones de listings
    const listingsActions = document.querySelector('.ml-listings-actions');
    if (listingsActions) DOMUtils.hide(listingsActions);
    
    // ✅ Obtener créditos con sistema de reintentos
    let credits = 0;
    
    if (window.CreditsManager && typeof window.CreditsManager.fetchCredits === 'function') {
      
      // Sistema de reintentos: máximo 5 intentos con delay incremental
      const MAX_RETRIES = 5;
      const BASE_DELAY = 300; // ms
      
      for (let attempt = 1; attempt <= MAX_RETRIES; attempt++) {
        
        // Forzar actualización desde API
        await window.CreditsManager.fetchCredits(true);
        
        // Esperar delay progresivo (300ms, 500ms, 800ms, 1200ms, 1700ms)
        const delay = BASE_DELAY + (attempt - 1) * 200;
        await new Promise(resolve => setTimeout(resolve, delay));
        
        // Verificar si ya tenemos créditos
        const creditsState = window.CreditsManager.getState();
        credits = creditsState.credits || 0;
        
        
        // Si tenemos créditos O es el último intento, salir del loop
        if (credits > 0 || attempt === MAX_RETRIES) {
          if (credits > 0) {
          } else {
          }
          break;
        }
      }
    } else {
      // Fallback: leer del DOM
      const creditCountEl = document.getElementById('credit-count');
      const creditText = creditCountEl?.textContent || '0';
      credits = parseInt(creditText, 10) || 0;
    }
    
    // ⚠️ COMENTADO: La modal ahora se muestra desde auth.js cuando is_new_user = true
    // await showWelcomeModal(credits);
    // Razón: Evitar duplicación de modales. La nueva función showWelcomeModalForNewUser()
    // se llama desde auth.js después del login de Google y maneja mejor los reintentos.
    
    // 🔥 MOSTRAR mensaje bonito centrado PRIMERO
    showWallapopConnectionMessage();
    
    // ✅ Renderizar botones según estado (sin cuentas) CON DELAY
    // Necesitamos esperar a que el DOM esté completamente listo
    setTimeout(() => {
      renderActionButtons(USER_STATES.NEW_USER_NO_ACCOUNTS, credits);
    }, 100);
    
    return;
  }
  
  // 🔥 Si HAY cuentas, ocultar mensaje y mostrar UI normal
  hideWallapopConnectionMessage();
  
  // Mostrar stats, controles, tabla
  const statsWrap = document.querySelector('.ml-listings-stats');
  if (statsWrap) DOMUtils.show(statsWrap);
  
  const controls = document.querySelector('.ml-listings-controls');
  if (controls) DOMUtils.show(controls);
  
  const tableWrapper = document.querySelector('.ml-listings-table-wrapper');
  if (tableWrapper) DOMUtils.show(tableWrapper);
  
  const listingsSection = document.getElementById('listings-section');
  if (listingsSection) DOMUtils.show(listingsSection);
  
  const listingsActions = document.querySelector('.ml-listings-actions');
  if (listingsActions) DOMUtils.show(listingsActions);

  const ph = document.createElement('option');
  ph.value = '';
  ph.textContent = '— selecciona cuenta —';
  selAccount.appendChild(ph);

  for (const a of (list || [])) {
    const opt = document.createElement('option');

    opt.value = a.id || a.walla_user_id || a.profile_url || (a.alias || '');
    opt.dataset.alias   = a.alias || '';
    opt.dataset.profile = a.profile_url || '';
    opt.dataset.avatar  = a.avatar_url || a.photo_url || a.picture || '';

    if (a.id) opt.dataset.accountId = String(a.id);
    
    // ✅ v6.3.0: Guardar fecha de último backup
    if (a.last_backup_at) opt.dataset.lastBackupAt = String(a.last_backup_at);

    const active =
      a.active_count ?? a.count_active ?? a.counts?.active ??
      a.listings_active ?? a.stats?.active;

    const total =
      a.total_count ?? a.count_total ?? a.counts?.total ??
      a.listings_total ?? a.stats?.total;

    const labelBase = a.alias || a.walla_user_id || a.profile_url || `Cuenta #${a.id}`;
    
    // ✅ v10.5.142: Mostrar pendientes de publicar si hay
    const pendingPublish = a.pending_publish_count || 0;
    let label = labelBase;
    
    if (pendingPublish > 0) {
      label = `• ${labelBase} ⏳${pendingPublish}`;  // ✅ v10.5.144: Añadir bullet
    } else if ((active ?? total) != null) {
      label = `• ${labelBase} (${active ?? total})`;  // ✅ v10.5.144: Añadir bullet
    } else {
      label = `• ${labelBase}`;  // ✅ v10.5.144: Añadir bullet
    }

    opt.textContent = label;
    opt.dataset.activeCount = active ?? '';
    opt.dataset.totalCount  = total ?? '';
    opt.dataset.pendingPublishCount = pendingPublish;

    selAccount.appendChild(opt);
  }
  
  // 🔥 MEJORADO v6.2.0: Intentar restaurar la última cuenta seleccionada desde BD
  try {
    // Primero: intentar obtener desde la sesión del usuario (viene del backend)
    let lastSelectedAccountId = null;
    
    // Obtener sesión del SW que tiene user.last_selected_account_id
    const sessionResponse = await new Promise((resolve) => {
      chrome.runtime.sendMessage({ type: 'SESSION.GET' }, resolve);
    });
    
    if (sessionResponse?.ok && sessionResponse?.data?.user?.last_selected_account_id) {
      lastSelectedAccountId = sessionResponse.data.user.last_selected_account_id;
    }
    
    // Fallback: intentar desde localStorage (compatibilidad hacia atrás)
    if (!lastSelectedAccountId) {
      const { selected_account_id } = await stGet(['selected_account_id']);
      if (selected_account_id) {
        lastSelectedAccountId = selected_account_id;
      }
    }
    
    if (lastSelectedAccountId) {
      // Buscar la opción que coincida con el account_id guardado
      const options = Array.from(selAccount.options);
      const matchingOption = options.find(opt => 
        opt.value == lastSelectedAccountId || 
        opt.dataset.accountId == lastSelectedAccountId
      );
      
      if (matchingOption) {
        selAccount.value = matchingOption.value;
        
        // Sincronizar localStorage con BD
        await stSet({ selected_account_id: lastSelectedAccountId });
        
        // Disparar evento change para cargar los anuncios
        const changeEvent = new Event('change', { bubbles: true });
        selAccount.dispatchEvent(changeEvent);
        return; // Salir sin resetear
      } else {
      }
    }
  } catch (err) {
  }
  
  // Si no hay cuenta guardada o no se encontró, resetear al placeholder
  selAccount.value = '';
  selAccount.selectedIndex = 0;
  
}

export function getSelectedAlias(){
  const opt = selAccount?.selectedOptions?.[0];
  return opt?.dataset?.alias || null;
}

/* ======================
   Run box (progreso UI)
   ====================== */

// --- NUEVO: suavizado + creep + indeterminado ---
let _barTarget = 0;      // objetivo %
let _barNow    = 0;      // valor pintado
let _barAnim   = null;   // interval id
let _lastRealTs = 0;     // timestamp del último latido real

function _clamp(n){ return Math.max(0, Math.min(100, n)); }

function _tickBar(){
  if (!runBar) return;
  const now = Date.now();

  // creep optimista si hace >2.5s sin latido real
  let target = _barTarget;
  if ((now - _lastRealTs) > 2500 && _barNow < Math.min(_barTarget + 8, 95)) {
    target = Math.min(_barTarget + 8, 95);
  }

  const diff = target - _barNow;
  if (Math.abs(diff) < 0.5) _barNow = target;
  else _barNow += diff * 0.18; // easing

  runBar.style.width = `${Math.round(_clamp(_barNow))}%`;

  if (_barNow >= 100) {
    clearInterval(_barAnim);
    _barAnim = null;
  }
}

// ✅ v5.1.0: Funciones run UI eliminadas (barra de progreso eliminada en v4.x):
// - setBar(), setBarIndeterminate(), showRunUI(), hideRunUI()

// Barra de progreso eliminada
export function renderRunStatus(j) {
  // Capturar validation_warnings si existen
  if (j?.validation_warnings) {
    setUIState('validation', j.validation_warnings);
  }
}

/* =============
   Legal overlay
   ============= */
export function openLegal(which){
  if (!LEGAL.overlay || !LEGAL.frame || !LEGAL.title) return;
  if (which === 'terminos') {
    LEGAL.title.textContent = 'Términos y condiciones';
    LEGAL.frame.src = 'legal/terminos.html';
  } else if (which === 'cookies') {
    LEGAL.title.textContent = 'Política de cookies';
    LEGAL.frame.src = 'legal/cookies.html';
  } else { return; }
  LEGAL.overlay.hidden = false;
}

export function closeLegal(){
  if (!LEGAL.overlay || !LEGAL.frame) return;
  LEGAL.overlay.hidden = true;
  LEGAL.frame.src = 'about:blank';
}

/* =========================================================
   ✅ Marcar backup como visto (persistencia multi-dispositivo)
   ========================================================= */
/**
 * Marca backup como visto en BD + sincroniza multi-dispositivo.
 * Llamado automáticamente 2s después de que backup completa.
 */
async function markBackupAsViewed(run_id) {
  if (!run_id) return;
  
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'API_CALL',
      endpoint: `/api/walla/runs/${run_id}/mark-viewed`,
      method: 'POST'
    });
    
    if (response?.ok) {
      
      // Actualizar estado local para incluir viewed_at
      import('./state.js').then(STATE => {
        STATE.setBackupState({ viewed_at: response.viewed_at });
      });
    }
  } catch (error) {
    // No es crítico, no mostrar error al usuario
  }
}

/* ==========
   Init del UI
   ========== */
export function init() {
  // ✅ Registrar callback de renderizado automático
  setRenderCallback(renderUIState);
  
  // Inicializar estado del botón de backup
  renderUIState();
}

/* ============================================
   🔐 MENSAJE WALLAPOP CONNECTION
   ============================================ */

function showWallapopConnectionMessage() {
  // Eliminar mensaje anterior si existe
  hideWallapopConnectionMessage();
  
  
  // Crear contenedor del mensaje
  const container = document.createElement('div');
  container.id = 'wallapop-connection-message';
  container.style.cssText = `
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 40px 30px;
    text-align: center;
    min-height: 300px;
    background: linear-gradient(135deg, rgba(13, 110, 253, 0.05) 0%, rgba(13, 110, 253, 0.02) 100%);
    border-radius: 12px;
    margin: 20px;
  `;
  
  container.innerHTML = `
    <div style="font-size: 56px; margin-bottom: 20px; animation: pulse 2s infinite;">
      🔐
    </div>
    <h2 style="font-size: 24px; font-weight: 600; color: #ffffff; margin: 0 0 12px 0; letter-spacing: -0.5px;">
      ¡Un paso más!
    </h2>
    <p style="font-size: 15px; color: rgba(255, 255, 255, 0.85); margin: 0 0 8px 0; max-width: 380px; line-height: 1.6;">
      MitikLive necesita descargar tus anuncios de Wallapop
    </p>
    <p style="font-size: 14px; color: rgba(255, 255, 255, 0.6); margin: 0 0 24px 0; max-width: 380px; line-height: 1.5;">
      📥 Haz clic en "Bajar Anuncios" para empezar<br>
      <span style="font-size: 13px;">(Solo toma unos segundos)</span>
    </p>
    <button id="btn-backup-welcome" style="
      background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
      color: white;
      border: none;
      padding: 14px 32px;
      font-size: 16px;
      font-weight: 600;
      border-radius: 8px;
      cursor: pointer;
      transition: all 0.3s ease;
      box-shadow: 0 4px 12px rgba(13, 110, 253, 0.3);
    <button id="btn-backup-welcome" style="
      background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
      color: white;
      border: none;
      padding: 14px 32px;
      font-size: 16px;
      font-weight: 600;
      border-radius: 8px;
      cursor: pointer;
      transition: all 0.3s ease;
      box-shadow: 0 4px 12px rgba(13, 110, 253, 0.3);
    ">
      📥 Bajar Anuncios
    </button>
    <style>
      @keyframes pulse {
        0%, 100% { transform: scale(1); opacity: 1; }
        50% { transform: scale(1.1); opacity: 0.8; }
      }
    </style>
  `;
  
  // 🔥 CORRECCIÓN: Insertar en el tab-panel de listings
  const listingsTabPanel = document.querySelector('[data-tab-panel="listings"]');
  
  if (listingsTabPanel) {
    // Insertar como primer hijo del panel de listings
    listingsTabPanel.insertBefore(container, listingsTabPanel.firstChild);
    
    // Añadir event listeners al botón (sin inline handlers para evitar CSP)
    const btnBackupWelcome = document.getElementById('btn-backup-welcome');
    if (btnBackupWelcome) {
      btnBackupWelcome.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-2px)';
        this.style.boxShadow = '0 6px 16px rgba(13, 110, 253, 0.4)';
      });
      btnBackupWelcome.addEventListener('mouseleave', function() {
        this.style.transform = '';
        this.style.boxShadow = '0 4px 12px rgba(13, 110, 253, 0.3)';
      });
    }
  } else {
    // Fallback: buscar otros contenedores
    const listingsSection = document.getElementById('listings-section');
    const blockApp = document.getElementById('block-app');
    
    if (listingsSection && listingsSection.parentNode) {
      listingsSection.parentNode.insertBefore(container, listingsSection);
    } else if (blockApp) {
      blockApp.appendChild(container);
    } else {
      logger.error('[UI] ❌ No se encontró contenedor para el mensaje');
    }
  }
  
  // ✅ Event listener para ejecutar acción de Backup
  const btn = document.getElementById('btn-backup-welcome');
  if (btn) {
    btn.addEventListener('click', () => {
      
      // 🔄 Mostrar action-bar para que se vea la barra de progreso
      const actionBar = document.getElementById('action-bar');
      if (actionBar) {
        DOMUtils.show(actionBar);
      }
      
      // 🎨 Ocultar mensaje del candado
      hideWallapopConnectionMessage();
      
      // Simular click en el botón de backup real
      const btnBackup = document.getElementById('btn-backup');
      if (btnBackup) {
        btnBackup.click();
      } else {
        logger.error('[UI] ❌ No se encontró el botón backup');
      }
    });
  }
}

function hideWallapopConnectionMessage() {
  const msg = document.getElementById('wallapop-connection-message');
  if (msg) msg.remove();
}

/* =============================================
   ✅ MODAL DE BIENVENIDA Y GESTIÓN DE BOTONES
   ============================================= */
import { USER_STATES } from './constants.js';

/**
 * Muestra modal de bienvenida para usuarios nuevos
 * @param {Number} credits - Créditos disponibles del usuario
 */
export async function showWelcomeModal(credits = 0) {
  // ✅ Backend puede manejar welcome_shown en tabla users si se necesita
  
  
  // Validar créditos (evitar NaN)
  const validCredits = isNaN(credits) || credits < 0 ? 0 : credits;
  
  const result = await showModal({
    title: '🎉 ¡Bienvenido a MitikLive!',
    html: `
      <div style="text-align: center; padding: 10px;">
        <p style="font-size: 16px; margin-bottom: 16px; color: rgba(255,255,255,0.9);">
          Es tu primera vez usando el gestor de anuncios de Wallapop
        </p>
        <div style="background: linear-gradient(135deg, rgba(13, 110, 253, 0.15) 0%, rgba(13, 110, 253, 0.05) 100%); 
                    padding: 20px; border-radius: 8px; margin: 20px 0;">
          <div style="font-size: 32px; margin-bottom: 8px;">💳</div>
          <p style="font-size: 18px; font-weight: 600; color: #ffffff; margin: 0;">
            Tienes <span style="color: #0d6efd; font-size: 24px;">${validCredits}</span> créditos
          </p>
        </div>
        <p style="font-size: 15px; color: rgba(255,255,255,0.7); line-height: 1.6;">
          <strong style="color: rgba(255,255,255,0.9);">Paso 1:</strong> Primero debes iniciar sesión en Wallapop<br>
          <strong style="color: rgba(255,255,255,0.9);">Paso 2:</strong> Luego podrás gestionar tus anuncios desde aquí
        </p>
      </div>
    `,
    buttons: [
      { text: 'Entendido', value: 'ok', primary: true }
    ]
  });
}

/**
 * Muestra modal de bienvenida para usuarios nuevos después del login de Google
 * Esta función es llamada desde auth.js cuando is_new_user === true
 * 
 * Diferencias con showWelcomeModal():
 * - Espera a que los créditos estén disponibles (máximo 5 segundos)
 * - Se puede llamar independientemente de si hay cuentas o no
 * - Maneja el caso de race condition al obtener créditos
 * 
 * @returns {Promise<void>}
 */
export async function showWelcomeModalForNewUser() {
  // El backend ya maneja welcome_shown, no necesitamos verificar storage local
  
  
  // Esperar un momento para que el backend termine de configurar los créditos
  await sleep(500);
  
  // Obtener créditos del CreditsManager que ya los tiene cargados
  let credits = 0;
  
  try {
    if (window.CreditsManager && typeof window.CreditsManager.getState === 'function') {
      // Usar getState() que retorna el estado actual en memoria
      const creditsState = window.CreditsManager.getState();
      credits = creditsState?.credits || 0;  // ← getState() retorna {credits: X, ...}
      
      // Si no hay créditos en memoria, hacer un fetch
      if (credits === 0 && typeof window.CreditsManager.fetchCredits === 'function') {
        await window.CreditsManager.fetchCredits();
        const newState = window.CreditsManager.getState();
        credits = newState?.credits || 0;
      }
    } else {
    }
  } catch (err) {
    logger.error('[UI] ❌ Error obteniendo créditos:', err);
    credits = 0;
  }
  
  // Mostrar modal con los créditos obtenidos
  const validCredits = isNaN(credits) || credits < 0 ? 0 : credits;
  
  
  await showModal({
    title: '🎉 ¡Bienvenido a MitikLive!',
    html: `
      <div style="text-align: center; padding: 10px;">
        <p style="font-size: 16px; margin-bottom: 16px; color: rgba(255,255,255,0.9);">
          Gracias por registrarte en nuestro gestor de anuncios de Wallapop
        </p>
        
        <div style="background: linear-gradient(135deg, rgba(13, 110, 253, 0.15) 0%, rgba(13, 110, 253, 0.05) 100%); 
                    padding: 20px; border-radius: 8px; margin: 20px 0;">
          <div style="font-size: 32px; margin-bottom: 8px;">💳</div>
          <p style="font-size: 18px; font-weight: 600; color: #ffffff; margin: 0;">
            ¡Has recibido <span style="color: #0d6efd; font-size: 24px;">${validCredits}</span> créditos de bienvenida!
          </p>
        </div>
        
        <div style="text-align: left; background: rgba(255,255,255,0.03); padding: 20px; border-radius: 8px; margin: 20px 0;">
          <p style="font-size: 16px; font-weight: 600; color: #ffffff; margin: 0 0 16px 0; text-align: center;">
            📝 Pasos para empezar:
          </p>
          
          <div style="display: flex; align-items: start; margin-bottom: 14px;">
            <span style="background: #0d6efd; color: white; width: 24px; height: 24px; border-radius: 50%; 
                         display: inline-flex; align-items: center; justify-content: center; 
                         font-weight: 600; font-size: 13px; flex-shrink: 0; margin-right: 12px;">1</span>
            <div style="flex: 1;">
              <p style="margin: 0; font-size: 14px; color: rgba(255,255,255,0.9); line-height: 1.5;">
                <strong>Haz clic en "Bajar Anuncios"</strong><br>
                <span style="font-size: 13px; color: rgba(255,255,255,0.6);">
                  Descarga tus productos de Wallapop
                </span>
              </p>
            </div>
          </div>
          
          <div style="display: flex; align-items: start; margin-bottom: 14px;">
            <span style="background: #0d6efd; color: white; width: 24px; height: 24px; border-radius: 50%; 
                         display: inline-flex; align-items: center; justify-content: center; 
                         font-weight: 600; font-size: 13px; flex-shrink: 0; margin-right: 12px;">2</span>
            <div style="flex: 1;">
              <p style="margin: 0; font-size: 14px; color: rgba(255,255,255,0.9); line-height: 1.5;">
                <strong>Espera unos segundos</strong><br>
                <span style="font-size: 13px; color: rgba(255,255,255,0.6);">
                  Mientras se descargan tus anuncios
                </span>
              </p>
            </div>
          </div>
          
          <div style="display: flex; align-items: start;">
            <span style="background: #0d6efd; color: white; width: 24px; height: 24px; border-radius: 50%; 
                         display: inline-flex; align-items: center; justify-content: center; 
                         font-weight: 600; font-size: 13px; flex-shrink: 0; margin-right: 12px;">3</span>
            <div style="flex: 1;">
              <p style="margin: 0; font-size: 14px; color: rgba(255,255,255,0.9); line-height: 1.5;">
                <strong>¡Listo! Ya puedes gestionar y republicar</strong><br>
                <span style="font-size: 13px; color: rgba(255,255,255,0.6);">
                  Cada publicación consume 1 crédito
                </span>
              </p>
            </div>
          </div>
        </div>
      </div>
    `,
    buttons: [
      { text: '¡Entendido! 🚀', value: 'ok', primary: true }
    ]
  });
  
  // Marcar en backend que el modal fue mostrado
  try {
    // ✅ v6.4.2: sw está en runs.js, no en auth.js
    const { sw } = await import('./runs.js');
    await sw({ 
      type: 'API.FETCH_JSON', 
      url: '/api/user/welcome-shown', 
      method: 'POST' 
    });
  } catch (err) {
  }
  
}

/**
 * Abre o activa una pestaña de Wallapop
 */
async function openWallapopTab() {
  try {
    const tabs = await chrome.tabs.query({ url: 'https://es.wallapop.com/*' });
    
    if (tabs.length > 0) {
      await chrome.tabs.update(tabs[0].id, { active: true });
      await chrome.windows.update(tabs[0].windowId, { focused: true });
    } else {
      await chrome.tabs.create({ 
        url: 'https://es.wallapop.com',
        active: true 
      });
    }
  } catch (err) {
    logger.error('[UI] Error abriendo Wallapop:', err);
    window.open('https://es.wallapop.com', '_blank');
  }
}

/**
 * Renderiza los botones de acción según el estado del usuario
 * @param {String} userState - Estado del usuario (de USER_STATES)
 * @param {Number} credits - Créditos disponibles
 * ✅ v5.0.4: action-bar eliminado en v5.0.0 - función obsoleta
 */
export function renderActionButtons(userState, credits = 0) {
  // Esta función ya no hace nada porque action-bar fue eliminado
  // Los botones ahora están en .selection-info y se gestionan por ButtonManager
}

/* ============================================
   🔄 LOADER GLOBAL
   ============================================ */

// ✅ v10.5.27: showLoader/hideLoader eliminados - usar utils.js
// import { showLoader, hideLoader } from '../utils.js';

/* ============================================
   ✅ BARRA DE PROGRESO UNIFICADA
   ============================================ */

// Debouncing para evitar actualizaciones múltiples
let progressUpdateTimeout = null;
let lastProgressUpdate = { operation: null, current: 0, total: 0, active: false };

/**
 * Actualiza la barra de progreso unificada para backup/publish/delete
 * @param {Object} options - Configuración del progreso
 * @param {string} options.operation - Tipo: 'backup'|'publish'|'delete'|null (null oculta)
 * @param {number} options.current - Items procesados
 * @param {number} options.total - Total de items
 * @param {boolean} options.active - Si está activo (border highlight)
 */
/**
 * ⭐ BARRA DE PROGRESO - Función centralizada
 * 
 * UBICACIÓN ÚNICA: No duplicar esta funcionalidad
 * - HTML: panel.html línea ~188 (#ml-progress-bar)
 * - CSS: panel.css líneas 1792-1865 (.ml-progress-bar)
 * - JS: ESTE ARCHIVO (ui.js)
 * 
 * USO:
 * ```javascript
 * const UI = await import('./ui.js');
 * 
 * // Mostrar
 * UI.updateProgressBar({
 *   operation: 'backup' | 'publish' | 'delete',
 *   current: 5,
 *   total: 10,
 *   active: true
 * });
 * 
 * // Ocultar
 * UI.updateProgressBar({ operation: null });
 * ```
 * 
 * USADO EN:
 * - panel.js (backup_progress, publish_progress)
 * - delete-handler.js (delete)
 * 
 * @param {Object} params - Parámetros de progreso
 * @param {string|null} params.operation - Tipo: 'backup', 'publish', 'delete', null
 * @param {number} params.current - Items procesados actualmente
 * @param {number} params.total - Total de items
 * @param {boolean} params.active - Si está en progreso activo
 * @param {number} params.accountId - ID de la cuenta (para aislamiento)
 */
// ✅ v10.5.99: Variables para throttle de progreso
let lastProgressTime = 0;
let lastProgressValue = 0;

export function updateProgressBar({ operation = null, current = 0, total = 0, active = false, accountId = null } = {}) {
  // Guardar última actualización con accountId
  lastProgressUpdate = { operation, current, total, active, accountId };
  
  // Si es null (ocultar), ejecutar inmediatamente
  if (!operation) {
    _updateProgressBarDOM({ operation: null, current: 0, total: 0, active: false, accountId: null });
    lastProgressTime = 0;
    lastProgressValue = 0;
    return;
  }
  
  const now = Date.now();
  
  // ✅ v10.5.99: Ejecutar inmediatamente en estos casos:
  // - Primer evento (current <= 3) → feedback visual rápido
  // - Último evento (current === total) → mostrar 100% inmediatamente  
  // - Han pasado más de 200ms desde la última actualización
  // - El progreso ha avanzado más del 5%
  const timeSinceLastUpdate = now - lastProgressTime;
  const progressJump = total > 0 ? ((current - lastProgressValue) / total) * 100 : 0;
  
  if (current <= 3 || current === total || timeSinceLastUpdate > 200 || progressJump >= 5) {
    _updateProgressBarDOM({ operation, current, total, active, accountId });
    lastProgressTime = now;
    lastProgressValue = current;
    return;
  }
  
  // Cancelar timeout anterior
  if (progressUpdateTimeout) {
    clearTimeout(progressUpdateTimeout);
  }
  
  // Debounce de 50ms para actualizaciones muy frecuentes
  progressUpdateTimeout = setTimeout(() => {
    _updateProgressBarDOM(lastProgressUpdate);
    lastProgressTime = Date.now();
    lastProgressValue = lastProgressUpdate.current;
  }, 50);
}

/**
 * Actualiza el DOM de la barra de progreso (función interna)
 */
function _updateProgressBarDOM({ operation, current, total, active, accountId }) {
  const progressBar = document.getElementById('ml-progress-bar');
  const progressOperation = document.getElementById('progress-operation');
  const progressCount = document.getElementById('progress-count');
  const progressFill = document.getElementById('progress-bar-fill');
  
  if (!progressBar || !progressOperation || !progressCount || !progressFill) {
    return;
  }
  
  // ✅ v6.8.2: Guardar accountId en el elemento para aislamiento
  if (accountId) {
    progressBar.dataset.accountId = accountId;
  }
  
  // Si operation es null, ocultar barra
  if (!operation) {
    progressBar.classList.add('ml-hidden');  // ✅ v6.8.2: Usar clase, no style.display
    progressBar.classList.remove('active');
    delete progressBar.dataset.accountId;
    return;
  }
  
  // Mostrar barra
  progressBar.classList.remove('ml-hidden');  // ✅ v6.8.2: Usar clase, no style.display
  
  // Actualizar clase active
  if (active) {
    progressBar.classList.add('active');
  } else {
    progressBar.classList.remove('active');
  }
  
  // Textos según operación
  const operationTexts = {
    backup: '💾 Haciendo backup',
    publish: '📤 Publicando anuncios',
    delete: '🗑️ Eliminando anuncios'
  };
  
  progressOperation.textContent = operationTexts[operation] || 'Procesando...';
  progressCount.textContent = `${current}/${total}`;
  
  // Calcular y actualizar porcentaje
  const percent = total > 0 ? Math.round((current / total) * 100) : 0;
  progressFill.style.width = `${percent}%`;
  
}

/**
 * ✅ v6.8.2: Oculta la barra de progreso si es de otra cuenta
 * @param {number} currentAccountId - ID de la cuenta actual
 * @returns {boolean} true si se ocultó
 */
export function hideProgressBarIfDifferentAccount(currentAccountId) {
  const progressBar = document.getElementById('ml-progress-bar');
  if (!progressBar) return false;
  
  const barAccountId = progressBar.dataset.accountId;
  
  // Si la barra tiene cuenta asignada y es diferente, ocultarla
  if (barAccountId && String(barAccountId) !== String(currentAccountId)) {
    progressBar.classList.add('ml-hidden');
    return true;
  }
  
  return false;
}

/**
 * ✅ v6.8.2: Muestra la barra de progreso si es de esta cuenta
 * @param {number} currentAccountId - ID de la cuenta actual
 * @returns {boolean} true si se mostró
 */
export function showProgressBarIfSameAccount(currentAccountId) {
  const progressBar = document.getElementById('ml-progress-bar');
  if (!progressBar) return false;
  
  const barAccountId = progressBar.dataset.accountId;
  
  // Si la barra tiene cuenta asignada y es la misma, mostrarla
  if (barAccountId && String(barAccountId) === String(currentAccountId)) {
    progressBar.classList.remove('ml-hidden');
    return true;
  }
  
  return false;
}

/**
 * ✅ v6.8.2: Obtiene el estado actual de la barra
 * @returns {Object|null} Estado de la barra o null si no activa
 */
export function getProgressBarState() {
  return lastProgressUpdate;
}

