/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/tabs.js – Rol: Sistema de pestañas (Anuncios/Logs/Stats)
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/

import { logger, formatLogEntry, formatTime, getLogIcon, getLogClass } from './logger.js';
import * as RUNS from './runs.js';  // ✅ Para guardar preferencias

// ========================================
// ESTADO DE PESTAÑAS
// ========================================

let currentTab = 'listings'; // 'listings' | 'logs' | 'config'
let logsContainer = null;
let logsFilter = {
  level: null,
  search: ''
};

// ========================================
// INICIALIZACIÓN
// ========================================

/**
 * Inicializar sistema de pestañas
 */
export async function initTabs() {
  const tabButtons = document.querySelectorAll('[data-tab]');
  const tabPanels = document.querySelectorAll('[data-tab-panel]');
  
  if (!tabButtons.length || !tabPanels.length) {
    return;
  }

  // Event listeners para botones de pestañas
  tabButtons.forEach(btn => {
    btn.addEventListener('click', () => {
      const tab = btn.dataset.tab;
      switchTab(tab);
    });
  });

  // Inicializar contenedor de logs
  logsContainer = document.getElementById('logs-container');
  
  // Suscribirse a nuevos logs
  logger.subscribe(onNewLog);

  // Event listeners para filtros de logs
  const levelFilter = document.getElementById('log-level-filter');
  const searchFilter = document.getElementById('log-search-filter');
  const clearBtn = document.getElementById('btn-clear-logs');
  const exportBtn = document.getElementById('btn-export-logs');

  if (levelFilter) {
    levelFilter.addEventListener('change', (e) => {
      logsFilter.level = e.target.value || null;
      renderLogs();
    });
  }

  if (searchFilter) {
    searchFilter.addEventListener('input', (e) => {
      logsFilter.search = e.target.value;
      renderLogs();
    });
  }

  if (clearBtn) {
    clearBtn.addEventListener('click', () => {
      logger.clear();
      renderLogs();
    });
  }

  if (exportBtn) {
    exportBtn.addEventListener('click', exportLogs);
  }

  // ✅ Pestaña por defecto (se cambiará después del login si hay preferencia guardada)
  currentTab = 'listings';

  // Mostrar pestaña inicial
  switchTab(currentTab);
}

// ========================================
// CAMBIO DE PESTAÑAS
// ========================================

/**
 * Cambiar pestaña activa
 * @param {String} tab - 'listings' | 'logs' | 'stats'
 */
export async function switchTab(tab) {
  
  if (currentTab === tab) {
    return;
  }
  
  currentTab = tab;

  // Actualizar botones
  document.querySelectorAll('[data-tab]').forEach(btn => {
    const isActive = btn.dataset.tab === tab;
    btn.classList.toggle('active', isActive);
  });

  // Actualizar paneles
  document.querySelectorAll('[data-tab-panel]').forEach(panel => {
    const shouldHide = panel.dataset.tabPanel !== tab;
    panel.classList.toggle('hidden', shouldHide);
    
    // FORZAR display para config
    if (panel.dataset.tabPanel === 'config' && !shouldHide) {
      panel.style.display = 'block';
      panel.style.visibility = 'visible';
    }
  });

  // ✅ Guardar pestaña activa en BD
  try {
    const result = await RUNS.sw({
      type: 'USER.PREFERENCES.UPDATE',
      payload: { 
        ui_preferences: { 
          active_tab: tab 
        } 
      }
    });
    
    // Ignorar error NO_AUTH silenciosamente
    if (result?.error === 'NO_AUTH' || result?.error === 'UNAUTHENTICATED') {
      return;
    }
    
  } catch (e) {
    // Solo mostrar warning si NO es error de autenticación
    if (e?.message && !e.message.includes('NO_AUTH')) {
    }
  }

  // Acciones específicas por pestaña
  if (tab === 'logs') {
    renderStats();  // ✅ Renderizar stats arriba
    renderLogs();   // ✅ Renderizar logs abajo
  } else if (tab === 'in_progress') {
    // ✅ v6.12.0: Cargar estado del progreso desde BD
    if (window.ProgressTab && typeof window.ProgressTab.init === 'function') {
      const accountId = document.getElementById('sel-account')?.value;
      if (accountId) {
        await window.ProgressTab.init(parseInt(accountId));
      }
    }
  } else if (tab === 'config') {
    // ✅ Cargar configuración desde BD si aún no se ha cargado
    // Esto solo pasa si se entra a Config sin haber hecho login primero
    try {
      if (window.CONFIG && typeof window.CONFIG.loadCurrentConfig === 'function') {
        await window.CONFIG.loadCurrentConfig();
      }
    } catch (e) {
      // Error esperado si no hay login
      if (!e?.message?.includes('NO_AUTH')) {
      }
    }
  }

}

// ========================================
// RENDERIZADO DE LOGS
// ========================================

/**
 * Callback para nuevos logs (tiempo real)
 */
function onNewLog(entry) {
  // Solo renderizar si estamos en la pestaña de logs
  if (currentTab !== 'logs') return;
  if (entry.type === 'clear') {
    renderLogs();
    return;
  }
  
  // Verificar si el log pasa los filtros
  if (!passesFilters(entry)) return;

  // Añadir al contenedor (scroll al final)
  if (logsContainer) {
    const html = formatLogEntry(entry);
    logsContainer.insertAdjacentHTML('beforeend', html);
    
    // Auto-scroll al final
    logsContainer.scrollTop = logsContainer.scrollHeight;

    // ✅ v4.84.2: Limitar elementos en DOM (últimos 100 para mejor performance)
    const entries = logsContainer.querySelectorAll('.log-entry');
    if (entries.length > 100) {
      // Eliminar los más antiguos
      for (let i = 0; i < entries.length - 100; i++) {
        entries[i].remove();
      }
    }
  }
}

/**
 * Verificar si un log pasa los filtros actuales
 */
function passesFilters(entry) {
  // Filtro por nivel
  if (logsFilter.level) {
    const minLevel = logger.constructor.LOG_LEVELS?.[logsFilter.level] ?? 0;
    const entryLevel = logger.constructor.LOG_LEVELS?.[entry.level] ?? 0;
    if (entryLevel < minLevel) return false;
  }

  // Filtro por búsqueda
  if (logsFilter.search) {
    const search = logsFilter.search.toLowerCase();
    const matches = 
      entry.message.toLowerCase().includes(search) ||
      JSON.stringify(entry.context).toLowerCase().includes(search);
    if (!matches) return false;
  }

  return true;
}

/**
 * Renderizar todos los logs (con filtros)
 */
function renderLogs() {
  if (!logsContainer) return;

  const logs = logger.getLogs({
    level: logsFilter.level,
    search: logsFilter.search
  });

  if (logs.length === 0) {
    logsContainer.innerHTML = '<div class="log-empty">🕐 No hay actividad reciente para mostrar</div>';
    return;
  }

  logsContainer.innerHTML = logs.map(formatLogEntry).join('');
  
  // Auto-scroll al final
  logsContainer.scrollTop = logsContainer.scrollHeight;
}

// ========================================
// RENDERIZADO DE STATS
// ========================================

/**
 * Renderizar estadísticas (clickeables para filtrar)
 * ✅ v4.84.3: Añadido SUCCESS y reordenado para mejor UX
 */
function renderStats() {
  const statsContainer = document.getElementById('stats-container');
  if (!statsContainer) return;

  const stats = logger.getStats();
  
  let html = '<div class="stats-grid">';
  
  // Total (sin filtro)
  html += `<div class="stat-card ${!logsFilter.level ? 'active' : ''}" data-filter="">
    <div class="stat-label">Total</div>
    <div class="stat-value">${stats.total}</div>
  </div>`;
  
  // ✅ SUCCESS - NUEVO (primero para destacar)
  html += `<div class="stat-card stat-success ${logsFilter.level === 'SUCCESS' ? 'active' : ''}" data-filter="SUCCESS">
    <div class="stat-label">✅ Exitosas</div>
    <div class="stat-value">${stats.success || 0}</div>
  </div>`;
  
  // Info
  html += `<div class="stat-card stat-info ${logsFilter.level === 'INFO' ? 'active' : ''}" data-filter="INFO">
    <div class="stat-label">ℹ️ Info</div>
    <div class="stat-value">${stats.info}</div>
  </div>`;
  
  // Warn
  html += `<div class="stat-card stat-warn ${logsFilter.level === 'WARN' ? 'active' : ''}" data-filter="WARN">
    <div class="stat-label">⚠️ Avisos</div>
    <div class="stat-value">${stats.warn}</div>
  </div>`;
  
  // Error
  html += `<div class="stat-card stat-error ${logsFilter.level === 'ERROR' ? 'active' : ''}" data-filter="ERROR">
    <div class="stat-label">❌ Errores</div>
    <div class="stat-value">${stats.error}</div>
  </div>`;
  
  // Debug (último, menos importante para usuario)
  html += `<div class="stat-card stat-debug ${logsFilter.level === 'DEBUG' ? 'active' : ''}" data-filter="DEBUG">
    <div class="stat-label">🔍 Debug</div>
    <div class="stat-value">${stats.debug}</div>
  </div>`;
  
  html += '</div>';

  statsContainer.innerHTML = html;
  
  // ✅ Añadir event listeners para filtrar al hacer click
  statsContainer.querySelectorAll('.stat-card').forEach(card => {
    card.addEventListener('click', () => {
      const filter = card.dataset.filter;
      logsFilter.level = filter || null;
      
      // Actualizar select
      const levelFilter = document.getElementById('log-level-filter');
      if (levelFilter) {
        levelFilter.value = filter || '';
      }
      
      // Re-renderizar
      renderStats();
      renderLogs();
    });
  });
}

// ========================================
// EXPORTAR LOGS
// ========================================

/**
 * Exportar logs como archivo de texto
 */
function exportLogs() {
  const content = logger.export();
  const blob = new Blob([content], { type: 'text/plain' });
  const url = URL.createObjectURL(blob);
  
  const a = document.createElement('a');
  a.href = url;
  a.download = `mitiklive-logs-${Date.now()}.txt`;
  a.click();
  
  URL.revokeObjectURL(url);
  
}

// ========================================
// API PÚBLICA
// ========================================

export { currentTab };
