/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/state-manager.js — Rol: Gestor centralizado de estado de la aplicación
    Descripcion de todo el proyecto: 
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
            (login JWT, export/import, backups/renovaciones).  
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol 
*/

import { logger } from './logger.js';
import * as ButtonManager from './publish-button-manager.js';

/**
 * GESTOR CENTRALIZADO DE ESTADO
 * ==============================
 * 
 * UN ÚNICO punto de verdad para todo el estado de la aplicación.
 * Todos los componentes leen y escriben aquí.
 * 
 * Beneficios:
 * - No más estados contradictorios
 * - Sincronización automática
 * - Suscriptores para reactividad
 * - Debugging simplificado
 */

// ============================================
// ESTADO GLOBAL
// ============================================

const AppState = {
  // Estado de Wallapop (pestaña y sesión)
  wallapop: {
    tabId: null,
    sessionActive: false,
    profileDetected: false,
    avatarUrl: null,
    lastCheck: null,
    checking: false
  },
  
  // Cuenta seleccionada
  account: {
    id: null,
    name: null,
    validated: false,
    lastValidation: null
  },
  
  // Operaciones en curso
  operations: {
    current: null, // 'backup' | 'publish' | 'delete' | null
    progress: {
      current: 0,
      total: 0,
      percentage: 0
    },
    startedAt: null,
    lastUpdate: null,
    canInteract: true,
    error: null
  },
  
  // ✅ v6.0.0: Stats de BD para listings
  // ✅ v10.5.82: selectAllByFilter movido a listings/module-state.js
  selection: {
    dbStats: {
      total: 0,
      active: 0,
      deleted: 0,
      reserved: 0,
      sold: 0
    }
  },
  
  // UI
  ui: {
    currentTab: 'listings',
    loaderVisible: false,
    loaderMessage: ''
  }
};

// ============================================
// SUSCRIPTORES (Reactividad)
// ============================================

const subscribers = {
  wallapop: [],
  account: [],
  operations: [],
  ui: []
};

/**
 * Suscribirse a cambios en una parte del estado
 * @param {string} key - 'wallapop' | 'account' | 'operations' | 'ui'
 * @param {function} callback - Función a llamar cuando cambia
 * @returns {function} - Función para desuscribirse
 */
export function subscribe(key, callback) {
  if (!subscribers[key]) {
    logger.error('[STATE] Key inválida para suscripción:', key);
    return () => {};
  }
  
  subscribers[key].push(callback);
  
  // Retornar función de desuscripción
  return () => {
    const index = subscribers[key].indexOf(callback);
    if (index > -1) {
      subscribers[key].splice(index, 1);
    }
  };
}

/**
 * Notificar a suscriptores de cambios
 */
function notify(key) {
  const callbacks = subscribers[key] || [];
  const stateSlice = AppState[key];
  
  
  callbacks.forEach(cb => {
    try {
      cb(stateSlice);
    } catch (error) {
      logger.error('[STATE] Error en callback:', error);
    }
  });
}

// ============================================
// WALLAPOP STATE
// ============================================

/**
 * Actualizar estado de Wallapop
 */
export function setWallapopState(updates) {
  Object.assign(AppState.wallapop, updates);
  AppState.wallapop.lastCheck = Date.now();
  notify('wallapop');
}

/**
 * Obtener estado de Wallapop
 */
export function getWallapopState() {
  return { ...AppState.wallapop };
}

/**
 * Wallapop está listo para operar
 */
export function isWallapopReady() {
  const { tabId, sessionActive, profileDetected } = AppState.wallapop;
  return tabId !== null && sessionActive && profileDetected;
}

// ============================================
// ACCOUNT STATE
// ============================================

/**
 * Actualizar cuenta seleccionada
 */
export function setAccountState(updates) {
  Object.assign(AppState.account, updates);
  AppState.account.lastValidation = Date.now();
  notify('account');
}

/**
 * Obtener cuenta seleccionada
 */
export function getAccountState() {
  return { ...AppState.account };
}

/**
 * Hay cuenta válida seleccionada
 */
export function hasValidAccount() {
  return AppState.account.id !== null && AppState.account.validated;
}

/**
 * Limpiar cuenta (al hacer logout)
 */
export function clearAccount() {
  AppState.account = {
    id: null,
    name: null,
    validated: false,
    lastValidation: null
  };
  notify('account');
}

// ============================================
// OPERATIONS STATE
// ============================================

/**
 * Iniciar operación
 */
export function startOperation(type, total = 0) {
  AppState.operations = {
    current: type,
    progress: {
      current: 0,
      total: total,
      percentage: 0
    },
    startedAt: Date.now(),
    lastUpdate: Date.now(),
    canInteract: false,
    error: null
  };
  
  // ✅ Informar a ButtonManager que hay operación en curso
  ButtonManager.updateState({
    operationRunning: true,
    operationType: type
  });
  
  notify('operations');
}

/**
 * Actualizar progreso de operación
 */
export function updateOperationProgress(current, total) {
  if (!AppState.operations.current) {
    return;
  }
  
  const percentage = total > 0 ? Math.round((current / total) * 100) : 0;
  
  AppState.operations.progress = {
    current,
    total,
    percentage
  };
  AppState.operations.lastUpdate = Date.now();
  
  notify('operations');
}

/**
 * Completar operación
 */
export function completeOperation(success = true, error = null) {
  const type = AppState.operations.current;
  
  AppState.operations = {
    current: null,
    progress: { current: 0, total: 0, percentage: 0 },
    startedAt: null,
    lastUpdate: null,
    canInteract: true,
    error: error
  };
  
  // ⚠️ NO habilitar botones aquí - lo maneja action-bar.js
  // El action-bar tiene control completo del estado de botones
  
  notify('operations');
}

/**
 * Obtener operación actual
 */
export function getOperationState() {
  return { ...AppState.operations };
}

/**
 * Hay operación en curso
 */
export function hasActiveOperation() {
  return AppState.operations.current !== null;
}

/**
 * Operación atascada (> 30s sin actualizaciones)
 */
export function isOperationStuck() {
  if (!AppState.operations.current) return false;
  
  const now = Date.now();
  const lastUpdate = AppState.operations.lastUpdate || AppState.operations.startedAt;
  const elapsed = now - lastUpdate;
  
  return elapsed > 30000; // 30 segundos
}

// ============================================
// UI STATE
// ============================================

/**
 * Actualizar estado de UI
 */
export function setUIState(updates) {
  Object.assign(AppState.ui, updates);
  notify('ui');
}

/**
 * Obtener estado de UI
 */
export function getUIState() {
  return { ...AppState.ui };
}

// ============================================
// VALIDACIONES
// ============================================

/**
 * Validar que todo está listo para operar
 * @returns {object} { ready: boolean, reason: string }
 */
export function validateReadyToOperate() {
  // 1. Verificar cuenta
  if (!hasValidAccount()) {
    return {
      ready: false,
      reason: 'no_account',
      message: '⚠️ Selecciona una cuenta primero'
    };
  }
  
  // 2. Verificar Wallapop
  if (!isWallapopReady()) {
    const walla = getWallapopState();
    
    if (!walla.tabId) {
      return {
        ready: false,
        reason: 'no_wallapop_tab',
        message: '🔴 Necesitas tener Wallapop abierto'
      };
    }
    
    if (!walla.sessionActive) {
      return {
        ready: false,
        reason: 'no_wallapop_session',
        message: '🟡 Inicia sesión en Wallapop primero'
      };
    }
    
    if (!walla.profileDetected) {
      return {
        ready: false,
        reason: 'no_profile',
        message: '⚠️ No se detectó tu perfil de Wallapop'
      };
    }
  }
  
  // 3. Verificar que no hay operación en curso
  if (hasActiveOperation()) {
    return {
      ready: false,
      reason: 'operation_running',
      message: '⏳ Ya hay una operación en curso'
    };
  }
  
  // ✅ Todo OK
  return {
    ready: true,
    reason: 'ok',
    message: '✅ Listo para operar'
  };
}

// ============================================
// DEBUGGING
// ============================================

/**
 * Obtener estado completo (para debugging)
 */
export function getFullState() {
  return JSON.parse(JSON.stringify(AppState));
}

/**
 * Exportar estado como JSON (para soporte)
 */
export function exportState() {
  return JSON.stringify(AppState, null, 2);
}

/**
 * Resetear estado completo (para testing)
 */
export function resetState() {
  AppState.wallapop = {
    tabId: null,
    sessionActive: false,
    profileDetected: false,
    avatarUrl: null,
    lastCheck: null,
    checking: false
  };
  
  AppState.account = {
    id: null,
    name: null,
    validated: false,
    lastValidation: null
  };
  
  AppState.operations = {
    current: null,
    progress: { current: 0, total: 0, percentage: 0 },
    startedAt: null,
    lastUpdate: null,
    canInteract: true,
    error: null
  };
  
   AppState.ui = {
    currentTab: 'listings',
    loaderVisible: false,
    loaderMessage: ''
  };
  
  // ✅ v6.0.0: Reset selection state
  // ✅ v10.5.82: selectAllByFilter movido a module-state.js
  AppState.selection = {
    dbStats: { total: 0, active: 0, deleted: 0, reserved: 0, sold: 0 }
  };
  
  
  // Notificar a todos
  Object.keys(subscribers).forEach(key => notify(key));
}

// ============================================
// SELECTION STATE (Stats de BD para listings)
// ✅ v10.5.82: selectAllByFilter movido a listings/module-state.js
// ============================================

/**
 * Actualiza los stats de la BD
 * @param {Object} stats - { total, active, deleted, reserved, sold }
 */
export function setDbStats(stats) {
  Object.assign(AppState.selection.dbStats, stats);
}

/**
 * Obtiene los stats de la BD
 * @returns {Object} Stats
 */
export function getDbStats() {
  return { ...AppState.selection.dbStats };
}

// ============================================
// LOG DEL ESTADO INICIAL
// ============================================

