/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/runs.js – Rol: Puente con SW y acciones de runs (resume/cancel/poll + postBackupAskDelete)
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/

import { toast, showModal } from '../utils.js';
import { stSet, stRemove } from './storage.js';
import {
  currentRunId, lastNotifiedTerminal,
  set_currentRunId, set_lastNotifiedTerminal,
  setBackupState
} from './state.js';
import * as UI from './ui.js';
import * as ButtonManager from './publish-button-manager.js';
// ✅ v6.3.0: startPolling y stopPolling eliminados - backup usa WebSocket
import { getActiveAccountId } from './auth.js';
import { logger } from './logger.js';
import ResourceManager from './resource-manager.js';

/* =========================================================
   WebSocket: Listener para actualizaciones en tiempo real
   ========================================================= */

// Escuchar mensajes de WebSocket desde el Service Worker
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // Solo procesar mensajes desde el Service Worker
  if (!sender || sender.id !== chrome.runtime.id) return;
  
  try {
    if (message.type === 'RUN_PROGRESS') {
      handleRunProgressWebSocket(message.payload);
    } else if (message.type === 'RUN_COMPLETE') {
      handleRunCompleteWebSocket(message.payload);
    }
  } catch (error) {
    logger.error('[RUNS-WS] Error procesando mensaje WebSocket:', error);
  }
});

/** Procesa actualizaciones de progreso de run desde WebSocket */
function handleRunProgressWebSocket(data) {
  const { run_id, status, progress, items_processed, items_total, errors, current_item } = data;
  
  
  // ✅ Iniciar fallback polling si es el primer progreso de este run
  if (run_id && !fallbackPollingTimer) {
    // Esperar 3 segundos antes de iniciar fallback (dar tiempo al WebSocket)
    setTimeout(() => {
      // Solo iniciar si el run sigue activo y WebSocket no ha completado
      if (status === 'running' && !fallbackPollingTimer) {
        startBackupFallbackPolling(run_id);
      }
    }, 3000);
  }
  
  // Actualizar estado centralizado
  setBackupState({
    status: status || 'running',
    pending: (items_total || 0) - (items_processed || 0),
    errors: errors || 0,
    seen: items_processed || 0,
    total: items_total || 0,
    run_id: run_id
  });
  
  // 🗑️ Widget de backup eliminado - ya no se usa
  /*
  import('./backup-widget.js').then(({ isBackupWidgetActive, updateBackupProgress }) => {
    if (isBackupWidgetActive()) {
      updateBackupProgress({
        seen: items_processed || 0,
        total: items_total || 0,
        errors: errors || 0
      });
    }
  */
  
  // Actualizar UI directamente
  const runStatus = document.getElementById('run-status');
  const runNums = document.getElementById('run-nums');
  const runErrs = document.getElementById('run-errs');
  
  if (runStatus) runStatus.textContent = status || 'Procesando...';
  if (runNums) runNums.textContent = `${items_processed || 0} / ${items_total || 0}`;
  if (runErrs) runErrs.textContent = `${errors || 0} errores`;
  
  // ✅ OPTIMIZADO: Refrescar tabla solo cada 5 segundos (throttling)
  if (status === 'running') {
    const accountId = getActiveAccountId();
    if (accountId) {
      // Throttle: solo refrescar cada 5 segundos
      const now = Date.now();
      if (!handleRunProgressWebSocket._lastRefresh || now - handleRunProgressWebSocket._lastRefresh > 5000) {
        handleRunProgressWebSocket._lastRefresh = now;
        
        import('./listings/index.js').then(({ loadListingsTable }) => {
          loadListingsTable(accountId).catch(() => {});
        });
      }
    }
  }
}

/** Procesa completación de run desde WebSocket */
async function handleRunCompleteWebSocket(data) {
  const { run_id, status, items_processed, errors } = data;
  
  
  // 🗑️ Widget de backup eliminado - ya no se usa
  /*
  import('./backup-widget.js').then(({ isBackupWidgetActive, completeBackupWidget, failBackupWidget }) => {
    if (isBackupWidgetActive()) {
      if (status === 'ok' || status === 'completed') {
        completeBackupWidget({
          seen: items_processed || 0,
          errors: errors || 0
        });
      } else if (status === 'error') {
        failBackupWidget(`Error en backup: ${errors || 0} errores`);
      } else {
        completeBackupWidget({
          seen: items_processed || 0,
          errors: errors || 0
        });
      }
    }
  */
  
  // ✅ v6.3.0: polling eliminado - WebSocket proporciona actualizaciones
  set_currentRunId(null);
  // ✅ No limpiar ml.run_id - ya no se usa
  
  // Estado terminal
  const keepRunId = ['cancelled', 'canceled', 'incomplete'].includes(status) && 
                    (data.pending || 0) > 0;
  
  setBackupState({
    status: status || 'completed',
    pending: data.pending || 0,
    errors: errors || 0,
    seen: items_processed || 0,
    total: data.items_total || items_processed || 0,
    run_id: keepRunId ? run_id : null
  });
  
  // Refrescar UI
  const accountId = getActiveAccountId();
  if (accountId) {
    // ✅ OPTIMIZADO: Sin setTimeout, con loader visual
    (async () => {
      try {
        const { loadListingsTable, showListingsLoader, hideListingsLoader } = await import('./listings/index.js');
        
        // Mostrar loader inmediatamente
        showListingsLoader('🔄 Actualizando anuncios tras backup...');
        
        await loadListingsTable(accountId);
        
        if (status === 'ok' || status === 'completed') {
          await refreshPublishUI(accountId);
        }
        
      } catch (err) {
        logger.error('[RUNS-WS] Error refrescando UI:', err);
      } finally {
        const { hideListingsLoader } = await import('./listings/index.js');
        hideListingsLoader();
      }
    })();
  }
  
  // Toast de notificación
  if (status === 'ok' || status === 'completed') {
    toast(`Backup completado: ${items_processed || 0} items`, 'ok');
    
    // ✅ v10.5.82: Actualizar fecha/hora de último sync (antes backup)
    const lastBackupDate = document.getElementById('last-backup-date');
    if (lastBackupDate) {
      const now = new Date();
      const day = now.getDate();
      const month = now.getMonth() + 1;
      const hours = now.getHours();
      const minutes = String(now.getMinutes()).padStart(2, '0');
      // Formato: "28/11 14:30" - más legible
      const formatted = `${day}/${month} ${hours}:${minutes}`;
      lastBackupDate.textContent = formatted;
      lastBackupDate.title = `Última sincronización: ${now.toLocaleString('es-ES')}`;
    }
  } else if (status === 'error') {
    toast(`Backup con errores: ${errors || 0}`, 'err');
  }
  
  // ✅ Limpiar fallback polling si estaba activo
  stopBackupFallbackPolling();
}

// ---------------------------------------------------------
// ✅ POLLING DE FALLBACK HÍBRIDO
// ---------------------------------------------------------
let fallbackPollingTimerId = null;
let fallbackPollingAttempts = 0;
const MAX_FALLBACK_ATTEMPTS = 60; // ✅ 60 intentos x 3s = 3 minutos (antes 1min)

/**
 * Inicia polling HTTP de fallback si el WebSocket no responde.
 * Se llama automáticamente 3 segundos después de iniciar un run.
 */
export function startBackupFallbackPolling(runId) {
  if (fallbackPollingTimerId) return; // Ya está activo
  
  fallbackPollingAttempts = 0;
  
  
  // Primer check inmediato, luego cada 3 segundos
  checkRunStatusFallback(runId);
  
  fallbackPollingTimerId = ResourceManager.addInterval(() => {
    checkRunStatusFallback(runId);
  }, 3000, 'backup-fallback-polling');
}

/** Detiene el polling de fallback */
export function stopBackupFallbackPolling() {
  if (fallbackPollingTimerId) {
    ResourceManager.remove(fallbackPollingTimerId);
    fallbackPollingTimerId = null;
    fallbackPollingAttempts = 0;
  }
}

/** Verifica estado del run vía HTTP (fallback) */
async function checkRunStatusFallback(runId) {
  fallbackPollingAttempts++;
  
  // Límite de intentos alcanzado
  if (fallbackPollingAttempts > MAX_FALLBACK_ATTEMPTS) {
    stopBackupFallbackPolling();
    return;
  }
  
  try {
    const response = await sw({ type: 'RUN.STATUS', runId });
    
    if (!response?.ok || !response?.data) {
      return;
    }
    
    const { status, items_processed, items_total, errors } = response.data;
    
    
    // Si el run no está corriendo, procesarlo y detener polling
    if (status !== 'running') {
      
      // Disparar handleRunCompleteWebSocket manualmente
      handleRunCompleteWebSocket(response.data);
      
      // Detener polling
      stopBackupFallbackPolling();
    } else {
      // Actualizar progreso si está corriendo
      handleRunProgressWebSocket(response.data);
    }
    
  } catch (err) {
    logger.error(`❌ [FALLBACK] Excepción al consultar run #${runId}:`, err);
  }
}

/** Envío seguro de mensajes al Service Worker (resuelve {ok:false} si falla). */
export async function sw(msg) {
  return new Promise(resolve => {
    try {
      chrome.runtime.sendMessage(msg, r => resolve(r || { ok: false }));
    } catch {
      resolve({ ok: false });
    }
  });
}

/** Reanuda un run (backup/publicación) en backend vía SW. */
export async function resumeRun(runId, account_id = null) {
  try {
    const acc = account_id ?? getActiveAccountId();
    const r = await sw({ type: 'RUN.RESUME', runId, account_id: acc });
    if (!r?.ok) {
      logger.error(`❌ Error al reanudar run #${runId}`, { runId, response: r });
      throw new Error('resume_failed');
    }
    toast('Reanudando…', 'ok');

    // ✅ v6.3.0: Mostrar barra de progreso - WebSocket actualizará en tiempo real
    const UI = await import('./ui.js');
    const data = r.data || {};
    const total = Number(data.total || data.pending || 0);
    const seen = Number(data.seen || 0);
    
    UI.updateProgressBar({
      operation: 'backup',
      current: seen,
      total: total,
      active: true
    });

    // ✅ v7.2.7: Cambiar a tab "En Curso" automáticamente
    try {
      const ProgressTab = await import('./progress-tab.js');
      ProgressTab.showProgressTab();
    } catch (e) {
    }

    return r;
  } catch (e) {
    toast('No se pudo reanudar', 'err');
    logger.error(`❌ Excepción al reanudar run #${runId}`, { runId, error: e.message });
    throw e;
  }
}

/** Cancela un run en ejecución. */
export async function cancelRun(runId, account_id = null) {
  try {
    const acc = account_id ?? getActiveAccountId();
    const r = await sw({ type: 'RUN.CANCEL', runId, account_id: acc });
    if (!r?.ok) {
      logger.error(`❌ Error al cancelar run #${runId}`, { runId, response: r });
      throw new Error('cancel_failed');
    }
    toast('Cancelando…', 'ok');

    // ✅ v6.3.0: Ocultar barra de progreso
    const UI = await import('./ui.js');
    UI.updateProgressBar({ operation: null });

    return r;
  } catch (e) {
    toast('No se pudo cancelar', 'err');
    throw e;
  }
}

/**
 * 🔄 Refresca la UI de publicación (Reanudar/Cancelar) para una cuenta concreta.
 * - Si no hay cuenta válida, oculta los botones.
 * - Si hay cuenta, consulta /publish2/status y muestra/oculta según corresponda.
 * ✅ v1.8.0: Respeta processingCount actual - no muestra botones si hay publicación activa
 */
export async function refreshPublishUI(accountId) {
  const id = Number(accountId);
  if (!Number.isFinite(id) || id <= 0) {
    // ✅ Informar a ButtonManager: no hay proceso
    ButtonManager.updateState({
      hasPendingProcess: false,
      pendingProcessId: null,
      pendingType: null
    });
    return;
  }

  // ✅ v1.8.0: Si hay publicación activa (processingCount > 0), no hacer nada
  // Los botones ya están ocultos y no debemos interferir
  const currentState = ButtonManager.getState();
  if (currentState.processingCount > 0) {
    return;
  }

  const r = await sw({
    type: 'API.FETCH_JSON',
    url: `/api/walla/publish2/status?account_id=${id}`,
    method: 'GET'
  });


  // ✅ v5.0.8: Backend NO devuelve 'ok', solo devuelve {status, pending, processing, errors}
  if (r?.ok && r.data) {
    const pending = Number(r.data?.pending || 0);
    const errors = Number(r.data?.errors || 0);
    const processing = Number(r.data?.processing || 0);
    const resumable = pending > 0;
    
    
    // ✅ Informar a ButtonManager
    if (resumable) {
      ButtonManager.updateState({
        hasPendingProcess: true,
        pendingProcessId: processing || null,
        pendingType: 'publish'
      });
    } else {
      ButtonManager.updateState({
        hasPendingProcess: false,
        pendingProcessId: null,
        pendingType: null
      });
    }
  } else {
    // ✅ Error o no hay datos: no hay proceso
    ButtonManager.updateState({
      hasPendingProcess: false,
      pendingProcessId: null,
      pendingType: null
    });
  }
}

// ✅ v6.3.0: pollRun eliminado - WebSocket proporciona actualizaciones en tiempo real
// El backup ahora usa backup_progress y backup_complete vía WebSocket

// Función postBackupAskDelete eliminada - usuario controla publicación desde el panel
