/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/resume.js – Rol: Lógica de reanudación backup/publicación
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/

import { PUBLISH_ASKED_KEY } from './constants.js';
import { logger } from './logger.js';

import { stGet, stRemove } from './storage.js';
import { toast, showModal, showLoader as _showLoader, hideLoader as _hideLoader } from '../utils.js';
import { sw, resumeRun } from './runs.js';
import { getActiveAccountId } from './auth.js';
import { updateBackupButtonUI } from './ui.js'; // ✅ v5.1.0: setResumeAffordance eliminado
import * as ButtonManager from './publish-button-manager.js';

// ✅ v6.11.8: Flags para evitar llamadas simultáneas (race condition)
let isCheckingPending = false;
let lastCheckTime = 0;
const MIN_CHECK_INTERVAL = 1000; // Mínimo 1s entre checks

/**
 * Pregunta para reanudar un RUN de backup si quedó pendiente.
 * Pinta siempre el estado del botón de Backup (badges).
 */
export async function promptResumeIfPending() {
  const r = await sw({ type: 'RUN.LAST', account_id: getActiveAccountId() });
  if (!r?.ok || !r.data?.exists) {
    // ✅ v5.1.0: setResumeAffordance eliminado (era no-op)
    // updateBackupButtonUI({ status: 'idle' }); // ❌ DISABLED: No llamar directamente
    return;
  }

  // ✅ v1.8.0: Si hay publicación activa, no interferir con botones
  const currentState = ButtonManager.getState();
  if (currentState.processingCount > 0) {
    return;
  }

  const { run_id } = r.data;
  const statusRaw = String(r.data.status || '').toLowerCase();
  const status = (statusRaw === 'canceled') ? 'cancelled' : statusRaw;

  // fallback por si la API no manda pending/errors
  const seen   = Number(r.data?.seen   || 0);
  const total  = Number(r.data?.total  || 0);
  const pendFB = Math.max(0, total - seen);
  const pending = Number(r.data?.pending ?? pendFB);
  const errors  = Number(r.data?.errors  || 0);

  // ¿Es reanudable?
  const resumable =
    ['blocked','incomplete'].includes(status) ||
    ((status === 'cancelled' || status === 'error') && pending > 0);

  if (!resumable) {
    // ✅ v5.1.0: setResumeAffordance eliminado (era no-op)
    // ✅ Informar al ButtonManager que NO hay proceso pendiente
    ButtonManager.updateState({
      hasPendingProcess: false,
      pendingProcessId: null,
      pendingType: null
    });
    return;
  }

  // Si está 'running' activamente, NO mostrar modal (sigue en background)
  if (status === 'running') {
    // ✅ v5.1.0: setResumeAffordance eliminado (era no-op)
    // ✅ Informar al ButtonManager que HAY proceso pendiente
    ButtonManager.updateState({
      hasPendingProcess: true,
      pendingProcessId: run_id,
      pendingType: 'backup'
    });
    return;
  }

  // Botón "Reanudar" y modal una sola vez
  // ✅ v5.1.0: setResumeAffordance eliminado (era no-op)
  
  // ✅ Informar al ButtonManager que HAY proceso pendiente
  ButtonManager.updateState({
    hasPendingProcess: true,
    pendingProcessId: run_id,
    pendingType: 'backup'
  });

  // ✅ Backend maneja viewed_at para evitar mostrar modal repetidamente
  
  const choice = await showModal({
    title: 'Proceso pendiente',
    html: `Hay un proceso a medias <b>#${run_id}</b> (estado: <b>${status}</b>).`,
    buttons: [
      { text:'Reanudar ahora', value:'resume', primary:true },
      { text:'Más tarde', value:'later' },
      { text:'Cancelar', value:'cancel' }
    ]
  });

  if (choice === 'resume') {
    await resumeRun(run_id);
  }
}

/**
 * Revisa si hay publicación pendiente para la cuenta activa y
 * muestra/actualiza el botón “Reanudar publicación”.
 */
export async function promptPublishResumeIfPending() {
  try {
    // ✅ v6.11.8: Evitar llamadas simultáneas
    if (isCheckingPending) {
      return;
    }
    
    // ✅ v6.11.8: Debounce - ignorar si fue llamado hace menos de 1s
    const now = Date.now();
    if (now - lastCheckTime < MIN_CHECK_INTERVAL) {
      return;
    }
    
    // ✅ v1.8.0: Si hay publicación activa, no interferir con botones
    const currentState = ButtonManager.getState();
    if (currentState.processingCount > 0) {
      return;
    }
    
    // ✅ v2.6.0: El estado de delay ahora se gestiona via WebSocket
    // No es necesario consultar nada aquí, el WebSocket actualiza automáticamente
    
    isCheckingPending = true;
    lastCheckTime = now;
    
    
    const account_id = getActiveAccountId();
    
    if (!account_id) {
      return;
    }
    
    const r = await sw({ type: 'PUBLISH.PENDING', account_id });
    
    
    const data = r?.data ?? r ?? {};
    const pending = Number(data.pending || 0);
    const errors  = Number(data.errors  || 0);
    const resumable = !!(data.resumable || pending > 0 || errors > 0);
    const run_id = data.run_id ?? null;


    // ✅ v5.0.10: Informar al ButtonManager del estado de publicación
    // NO requiere run_id porque el nuevo sistema no lo usa
    if (resumable) {
      ButtonManager.updateState({
        hasPendingProcess: true,
        pendingProcessId: null, // Nuevo sistema sin run_id
        pendingType: 'publish',
        pendingCount: pending // ✅ v6.6.0: Número de pendientes
      });
    } else {
      ButtonManager.updateState({
        hasPendingProcess: false,
        pendingProcessId: null,
        pendingType: null,
        pendingCount: 0 // ✅ v6.6.0: Resetear contador
      });
    }

    // ❌ DESHABILITADO: Modal automática ya no se usa
    // Ahora se usan los botones del panel directamente
    /*
    if (resumable && !sessionStorage.getItem(PUBLISH_ASKED_KEY)) {
      const msgExtra = errors > 0 ? ` (con <b>${errors}</b> en error)` : '';
      const choice = await showModal({
        title: 'Publicación pendiente',
        html: `Tienes <b>${pending}</b> anuncio(s) en curso${msgExtra}.<br>¿Quieres reanudar ahora?`,
        buttons: [
          { text:'Reanudar ahora', value:'resume', primary:true },
          { text:'Más tarde', value:'later' },
          { text:'Cancelar', value:'cancel' }
        ]
      });

      if (choice === 'resume') {
        const res = await sw(run_id
          ? { type:'RUN.RESUME', runId: run_id, account_id }
          : { type:'PUBLISH.RESUME', account_id });
        if (!res?.ok) { toast('No se pudo reanudar la publicación','err'); return; }
        sessionStorage.removeItem(PUBLISH_ASKED_KEY);
      } else if (choice === 'cancel') {
        if (run_id) {
          await sw({ type:'PUBLISH.CANCEL', runId: run_id });
        }
        sessionStorage.setItem(PUBLISH_ASKED_KEY, '1');
      } else {
        sessionStorage.setItem(PUBLISH_ASKED_KEY, '1');
      }
    } else if (!resumable) {
    }
    */
  } catch {
    /* silencio */
  } finally {
    // ✅ v6.11.8: Liberar flag
    isCheckingPending = false;
  }
}

/**
 * Lanza un backup fresco y, al terminar, reanuda publicación.
 * Devuelve { ok:true } si se lanzó correctamente.
 */
export async function runFreshBackupThenResume({ runId = null, account_id = null } = {}) {
  // 1) Confirmación
  const choice = await showModal({
    title: 'Reanudar con backup previo',
    html: 'Se realizará un <b>backup</b> antes de reanudar para publicar con datos actualizados. ¿Continuar?',
    buttons: [
      { text:'Sí, hacer backup y reanudar', value:'yes', primary:true },
      { text:'Cancelar', value:'no' }
    ]
  });
  if (choice !== 'yes') return { ok:false, code:'user_cancel' };

  // 2) Obtener URL de perfil (rápido desde content_script)
  const [active] = await chrome.tabs.query({ active:true, currentWindow:true });
  if (!active) { toast('No hay pestaña activa','warn'); return { ok:false }; }

  let profileUrl = null;
  try {
    // ✅ v10.5.82: Envolver para silenciar lastError
    const info = await new Promise((resolve) => {
      chrome.tabs.sendMessage(active.id, { type:'DOM.CURRENT_PROFILE' }, (r) => {
        void chrome.runtime.lastError;
        resolve(r);
      });
    });
    profileUrl = info?.profileUrl || null;
    if (!profileUrl) {
      const r0 = await new Promise((resolve) => {
        chrome.tabs.sendMessage(active.id, { type:'DOM.GET_PROFILE_URL' }, (r) => {
          void chrome.runtime.lastError;
          resolve(r);
        });
      });
      profileUrl = r0?.url || null;
    }
  } catch {}

  if (!profileUrl) {
    toast('No se pudo obtener el perfil para el backup previo','err');
    return { ok:false };
  }

  // 3) Lanzar backup (export+import)
  // Loader eliminado
  // Ya hay toast al final, no hace falta loader intermedio
  
  const bk = await sw({ type:'EXPORT.PROFILE_AND_IMPORT', profileUrl });
  // ❌ hideLoader() ya no necesario
  
  if (!bk?.ok) {
    // ✅ v10.5.81: Detectar error de cuenta duplicada (409)
    const errorDetail = bk?.error || bk?.data?.import?.detail || '';
    if (errorDetail.includes('ya está vinculada')) {
      await showModal({
        title: '⚠️ Cuenta ya vinculada',
        message: errorDetail,
        buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
      });
    } else {
      toast('Backup previo falló','err');
    }
    return { ok:false };
  }

  // 4) Reanudar publicación
  const res = await sw(runId
    ? { type:'RUN.RESUME', runId, account_id }
    : { type:'PUBLISH.RESUME', account_id });
  if (!res?.ok) { toast('No se pudo reanudar después del backup','err'); return { ok:false }; }

  toast('Reanudando publicación…', 'ok');
  return { ok:true };
}

/**
 * Reanudar publicación pendiente (llamada desde panel.js)
 */
export async function resumePublishing() {
  const account_id = getActiveAccountId();
  if (!account_id) {
    toast('Selecciona una cuenta primero', 'warn');
    return { ok: false };
  }
  
  // Verificar que hay pendientes
  const statusRes = await sw({ type: 'PUBLISH.PENDING', account_id });
  if (!statusRes?.ok || !statusRes.pending || statusRes.pending <= 0) {
    toast('No hay publicación pendiente', 'warn');
    return { ok: false };
  }
  
  const { pending, errors } = statusRes;
  
  // ✅ v7.2.7: Ya NO es necesario marcar flag - el SW detecta automáticamente
  // La detección se hace en publish-process-next.js verificando el estado en BD
  
  try {
    // Iniciar procesamiento de cola
    const res = await sw({ 
      type: 'PUBLISH.PROCESS_NEXT', 
      account_id 
    });
    
    if (!res?.ok) {
      toast('No se pudo reanudar la publicación', 'err');
      return { ok: false };
    }
    
    toast('✅ Publicación reanudada', 'ok');
    
    // ✅ Iniciar polling para actualizar UI
    
    return { ok: true };
  } catch (err) {
    logger.error('[RESUME] Error reanudando:', err);
    toast('Error al reanudar publicación', 'err');
    return { ok: false };
  }
}

/** Inicialización opcional del módulo */
export function init() {
  // Aquí no conectamos listeners (viven en el orquestador o en UI).
  return;
}
