/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
  Archivo: panel/resource-manager.js — Rol: Gestor centralizado de recursos para prevenir memory leaks
  Descripcion de todo el proyecto: 
       MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI 
          (login JWT, export/import, backups/renovaciones).  
  Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
  
  ✅ v6.11.0: Sistema centralizado para cleanup de recursos (intervals, listeners, observers)
*/

/**
 * ResourceManager - Gestor centralizado de recursos
 * 
 * Propósito: Prevenir memory leaks asegurando que todos los recursos se limpien
 * 
 * Uso:
 * ```javascript
 * import ResourceManager from './resource-manager.js';
 * 
 * // Registrar interval
 * const id = ResourceManager.addInterval(() => check(), 1000);
 * 
 * // Al terminar, limpiar
 * ResourceManager.cleanup(); // Limpia TODOS
 * ```
 */

class ResourceManager {
  constructor() {
    this.intervals = new Map();
    this.timeouts = new Map();
    this.listeners = new Map();
    this.observers = new Map();
    this.nextId = 1;
  }

  /**
   * Registrar interval que se limpiará automáticamente
   */
  addInterval(callback, delay, name = 'anonymous') {
    const id = this.nextId++;
    const intervalId = setInterval(callback, delay);
    
    this.intervals.set(id, {
      intervalId,
      name,
      created: Date.now()
    });
    
    return id;
  }

  /**
   * Registrar timeout que se limpiará automáticamente
   */
  addTimeout(callback, delay, name = 'anonymous') {
    const id = this.nextId++;
    const timeoutId = setTimeout(() => {
      callback();
      this.timeouts.delete(id); // Auto-cleanup
    }, delay);
    
    this.timeouts.set(id, {
      timeoutId,
      name,
      created: Date.now()
    });
    
    return id;
  }

  /**
   * Registrar listener que se limpiará automáticamente
   */
  addListener(target, event, handler, options, name = 'anonymous') {
    const id = this.nextId++;
    
    target.addEventListener(event, handler, options);
    
    this.listeners.set(id, {
      target,
      event,
      handler,
      options,
      name,
      created: Date.now()
    });
    
    return id;
  }

  /**
   * Registrar MutationObserver que se limpiará automáticamente
   */
  addObserver(observer, name = 'anonymous') {
    const id = this.nextId++;
    
    this.observers.set(id, {
      observer,
      name,
      created: Date.now()
    });
    
    return id;
  }

  /**
   * Limpiar un recurso específico
   */
  remove(id) {
    // Interval
    if (this.intervals.has(id)) {
      const { intervalId } = this.intervals.get(id);
      clearInterval(intervalId);
      this.intervals.delete(id);
      return true;
    }
    
    // Timeout
    if (this.timeouts.has(id)) {
      const { timeoutId } = this.timeouts.get(id);
      clearTimeout(timeoutId);
      this.timeouts.delete(id);
      return true;
    }
    
    // Listener
    if (this.listeners.has(id)) {
      const { target, event, handler, options } = this.listeners.get(id);
      target.removeEventListener(event, handler, options);
      this.listeners.delete(id);
      return true;
    }
    
    // Observer
    if (this.observers.has(id)) {
      const { observer } = this.observers.get(id);
      observer.disconnect();
      this.observers.delete(id);
      return true;
    }
    
    return false;
  }

  /**
   * Limpiar TODOS los recursos
   */
  cleanup() {
    
    // Limpiar intervals
    for (const [id, { intervalId, name }] of this.intervals) {
      clearInterval(intervalId);
    }
    this.intervals.clear();
    
    // Limpiar timeouts
    for (const [id, { timeoutId, name }] of this.timeouts) {
      clearTimeout(timeoutId);
    }
    this.timeouts.clear();
    
    // Limpiar listeners
    for (const [id, { target, event, handler, options, name }] of this.listeners) {
      target.removeEventListener(event, handler, options);
    }
    this.listeners.clear();
    
    // Limpiar observers
    for (const [id, { observer, name }] of this.observers) {
      observer.disconnect();
    }
    this.observers.clear();
    
  }

  /**
   * Obtener estadísticas de recursos activos
   */
  getStats() {
    return {
      intervals: this.intervals.size,
      timeouts: this.timeouts.size,
      listeners: this.listeners.size,
      observers: this.observers.size,
      total: this.intervals.size + this.timeouts.size + 
             this.listeners.size + this.observers.size
    };
  }

  /**
   * Listar recursos activos (debug)
   */
  listActive() {
    
    for (const [id, { name, created }] of this.intervals) {
      const age = Math.round((Date.now() - created) / 1000);
    }
    
    for (const [id, { name, created }] of this.timeouts) {
      const age = Math.round((Date.now() - created) / 1000);
    }
    
    for (const [id, { name, event, created }] of this.listeners) {
      const age = Math.round((Date.now() - created) / 1000);
    }
    
    for (const [id, { name, created }] of this.observers) {
      const age = Math.round((Date.now() - created) / 1000);
    }
  }
}

// Instancia singleton
const manager = new ResourceManager();

// Cleanup automático al cerrar página
if (typeof window !== 'undefined') {
  window.addEventListener('beforeunload', () => {
    manager.cleanup();
  });
}

// Cleanup cada 5 minutos de recursos antiguos (>10 min)
manager.addInterval(() => {
  const now = Date.now();
  const maxAge = 10 * 60 * 1000; // 10 minutos
  
  // Limpiar intervals viejos
  for (const [id, { created, name }] of manager.intervals) {
    if (now - created > maxAge) {
      manager.remove(id);
    }
  }
  
  // Limpiar listeners viejos
  for (const [id, { created, name }] of manager.listeners) {
    if (now - created > maxAge) {
      manager.remove(id);
    }
  }
}, 5 * 60 * 1000, 'auto-cleanup');

export default manager;
