/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/renewal-reminder.js – Rol: Sistema de recordatorio inteligente para renovar anuncios
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
    
    ✅ v2.5.0: Sistema de recordatorio NO intrusivo para anuncios no renovados
*/

import { createModal, closeModal } from './base-modal.js';
import { logger } from './logger.js';

// Registro de cuentas donde ya se mostró el recordatorio en esta sesión
const shownForAccounts = new Set();

/**
 * Analiza los anuncios y determina cuántos necesitan renovación
 * @param {Array} listings - Array de anuncios
 * @returns {Object} - { needsRenewal, today, recent, old, total }
 */
function analyzeListings(listings) {
  if (!Array.isArray(listings) || listings.length === 0) {
    return { needsRenewal: 0, today: 0, recent: 0, old: 0, total: 0 };
  }
  
  const now = new Date();
  const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
  
  let todayCount = 0;
  let recentCount = 0; // 1-3 días
  let oldCount = 0;    // >3 días
  
  listings.forEach(listing => {
    // Solo contar anuncios activos
    if (listing.status !== 'active' && listing.status !== 'onhold') {
      return;
    }
    
    const publishedAt = listing.last_published_at;
    if (!publishedAt) {
      oldCount++; // Nunca publicado = viejo
      return;
    }
    
    try {
      const date = new Date(publishedAt);
      if (isNaN(date.getTime())) {
        oldCount++;
        return;
      }
      
      const dateOnly = new Date(date.getFullYear(), date.getMonth(), date.getDate());
      const diffDays = Math.floor((today - dateOnly) / (1000 * 60 * 60 * 24));
      
      if (diffDays === 0) {
        todayCount++;
      } else if (diffDays >= 1 && diffDays <= 3) {
        recentCount++;
      } else {
        oldCount++;
      }
    } catch {
      oldCount++;
    }
  });
  
  const needsRenewal = recentCount + oldCount;
  const total = listings.filter(l => l.status === 'active' || l.status === 'onhold').length;
  
  return { needsRenewal, today: todayCount, recent: recentCount, old: oldCount, total };
}

/**
 * Muestra modal de recordatorio de renovación
 * @param {Object} stats - Estadísticas de anuncios
 */
function showRenewalModal(stats) {
  const { needsRenewal, today, recent, old, total } = stats;
  
  // Determinar mensaje según antigüedad
  let urgencyClass = 'reminder-low';
  let urgencyIcon = '📅';
  let urgencyText = 'Tienes anuncios que podrías renovar';
  
  if (old > 0) {
    urgencyClass = 'reminder-high';
    urgencyIcon = '⚠️';
    urgencyText = old === 1 
      ? 'Tienes 1 anuncio sin renovar hace más de 3 días'
      : `Tienes ${old} anuncios sin renovar hace más de 3 días`;
  } else if (recent > 0) {
    urgencyClass = 'reminder-medium';
    urgencyIcon = '🔔';
    urgencyText = recent === 1
      ? 'Tienes 1 anuncio publicado hace 1-3 días'
      : `Tienes ${recent} anuncios publicados hace 1-3 días`;
  }
  
  const content = `
    <div class="renewal-reminder-modal ${urgencyClass}">
      <div class="reminder-header">
        <span class="reminder-icon">${urgencyIcon}</span>
        <h3 class="reminder-title">Recordatorio de renovación</h3>
      </div>
      
      <div class="reminder-body">
        <p class="reminder-message">${urgencyText}</p>
        
        <div class="reminder-stats">
          <div class="reminder-stat">
            <span class="stat-value">${needsRenewal}</span>
            <span class="stat-label">Por renovar</span>
          </div>
          <div class="reminder-stat reminder-stat-success">
            <span class="stat-value">${today}</span>
            <span class="stat-label">Renovados hoy</span>
          </div>
          <div class="reminder-stat reminder-stat-muted">
            <span class="stat-value">${total}</span>
            <span class="stat-label">Total activos</span>
          </div>
        </div>
        
        <div class="reminder-info">
          <p><strong>💡 Para renovar tus anuncios:</strong></p>
          <ol class="reminder-steps">
            <li>Selecciona los anuncios haciendo click en el checkbox ☑️</li>
            <li>O usa "Seleccionar todos" para renovar todos a la vez</li>
            <li>Pulsa el botón <strong>"Publicar"</strong> que aparecerá arriba</li>
          </ol>
          <p class="reminder-note">
            <span class="note-icon">ℹ️</span>
            Renovar tus anuncios los pone de nuevo al principio de los resultados de búsqueda
          </p>
        </div>
      </div>
      
      <div class="reminder-footer">
        <button class="reminder-btn reminder-btn-secondary" data-action="close">
          Ahora no
        </button>
        <button class="reminder-btn reminder-btn-primary" data-action="select-all">
          Seleccionar todos
        </button>
      </div>
    </div>
  `;
  
  const modal = createModal({
    id: 'renewal-reminder-modal',
    overlayClass: 'renewal-reminder-overlay',
    content,
    closeOnEsc: true,
    closeOnOverlayClick: true,
    onOpen: () => {
      logger.debug('[RenewalReminder] Modal mostrado', stats);
      
      // Añadir event listeners a los botones
      const overlay = document.getElementById('renewal-reminder-modal');
      if (!overlay) return;
      
      // Botón "Ahora no"
      const closeBtn = overlay.querySelector('[data-action="close"]');
      if (closeBtn) {
        closeBtn.addEventListener('click', () => {
          closeModal('renewal-reminder-modal');
        });
      }
      
      // Botón "Seleccionar todos"
      const selectAllBtn = overlay.querySelector('[data-action="select-all"]');
      if (selectAllBtn) {
        selectAllBtn.addEventListener('click', () => {
          // Cerrar modal
          closeModal('renewal-reminder-modal');
          
          // Trigger select all
          setTimeout(() => {
            const selectAllCheckbox = document.getElementById('select-all-listings');
            if (selectAllCheckbox && !selectAllCheckbox.checked) {
              selectAllCheckbox.click();
              logger.debug('[RenewalReminder] Seleccionados todos los anuncios');
            }
          }, 300);
        });
      }
    }
  });
  
  return modal;
}

/**
 * Verifica si debe mostrar el recordatorio y lo muestra si es necesario
 * @param {string} accountId - ID de la cuenta actual
 * @param {Array} listings - Array de anuncios
 * @param {Object} options - Opciones adicionales
 */
export function checkAndShowReminder(accountId, listings, options = {}) {
  const {
    force = false,           // Forzar mostrar aunque ya se haya mostrado
    minToShow = 3            // Mínimo de anuncios sin renovar para mostrar
  } = options;
  
  // No mostrar si no hay cuenta
  if (!accountId) {
    return;
  }
  
  // No mostrar si ya se mostró para esta cuenta en esta sesión
  if (!force && shownForAccounts.has(accountId)) {
    logger.debug('[RenewalReminder] Ya mostrado para cuenta:', accountId);
    return;
  }
  
  // Analizar anuncios
  const stats = analyzeListings(listings);
  
  // No mostrar si no hay suficientes anuncios para renovar
  if (stats.needsRenewal < minToShow) {
    logger.debug('[RenewalReminder] No hay suficientes anuncios para renovar:', stats.needsRenewal, '<', minToShow);
    return;
  }
  
  // Marcar como mostrado
  shownForAccounts.add(accountId);
  
  // Mostrar modal
  showRenewalModal(stats);
}

/**
 * Resetea el registro de cuentas mostradas (útil para testing o cambio de sesión)
 */
export function resetShownAccounts() {
  shownForAccounts.clear();
  logger.debug('[RenewalReminder] Registro de cuentas mostradas reseteado');
}

/**
 * Verifica si ya se mostró para una cuenta
 * @param {string} accountId - ID de la cuenta
 * @returns {boolean}
 */
export function wasShownForAccount(accountId) {
  return shownForAccounts.has(accountId);
}
