/**
 * publish-validation.js
 * Validación de créditos antes de publicar
 */

import { showModal } from '../utils.js';
import { logger } from './logger.js';


/**
 * Validar si el usuario tiene créditos suficientes para publicar
 * @param {number} totalToPublish - Número de anuncios que se van a publicar
 * @returns {Promise<boolean>} - true si puede publicar, false si no
 */
export async function validateCreditsBeforePublish(totalToPublish) {
  try {
    
    // Obtener estado actual de créditos
    const credits = window.CreditsManager?.getState();
    
    if (!credits) {
      logger.error('[PublishValidation] CreditsManager no disponible');
      return true; // Por seguridad, dejar continuar si el sistema falla
    }
    
    
    // Si es plan ilimitado → Siempre puede publicar
    if (credits.is_unlimited) {
      return true;
    }
    
    // Calcular publicaciones disponibles del plan
    const publicationsRemaining = Math.max(0, credits.publications_limit - credits.publications_used);
    
    // Calcular créditos disponibles totales
    const totalCreditsAvailable = publicationsRemaining + (credits.credits || 0);
    
    
    // Si tiene suficientes créditos → Puede publicar
    if (totalCreditsAvailable >= totalToPublish) {
      return true;
    }
    
    // ❌ NO tiene créditos suficientes → Mostrar modal
    
    await showInsufficientCreditsModal({
      needed: totalToPublish,
      available: totalCreditsAvailable,
      plan_name: credits.plan || 'Free',
      plan_remaining: publicationsRemaining,
      extra_credits: credits.credits || 0
    });
    
    return false;
    
  } catch (error) {
    logger.error('[PublishValidation] Error validando créditos:', error);
    return true; // Por seguridad, dejar continuar si hay error
  }
}

/**
 * Mostrar modal de créditos insuficientes
 */
async function showInsufficientCreditsModal({ needed, available, plan_name, plan_remaining, extra_credits }) {
  const deficit = needed - available;
  
  let detailsHTML = `
    <div style="text-align: left; margin: 20px 0;">
      <p><strong>❌ No tienes créditos suficientes para esta operación</strong></p>
      <hr style="margin: 15px 0; border: none; border-top: 1px solid rgba(255,255,255,0.1);">
      <p><strong>Necesitas publicar:</strong> ${needed} anuncios</p>
      <p><strong>Créditos disponibles:</strong> ${available}</p>
      <p style="color: #ef4444;"><strong>Faltan:</strong> ${deficit} créditos</p>
      <hr style="margin: 15px 0; border: none; border-top: 1px solid rgba(255,255,255,0.1);">
      <p style="font-size: 0.9em; opacity: 0.8;">
        📋 Plan actual: <strong>${plan_name}</strong><br>
        💳 Publicaciones del plan: ${plan_remaining}<br>
        ➕ Créditos extra: ${extra_credits}
      </p>
  `;
  
  // Mensaje específico según el caso
  if (plan_remaining === 0 && extra_credits === 0) {
    detailsHTML += `
      <hr style="margin: 15px 0; border: none; border-top: 1px solid rgba(255,255,255,0.1);">
      <p style="color: #f59e0b;">
        ⚠️ Has agotado las publicaciones de tu plan y no tienes créditos extra.
      </p>
      <p>
        <strong>Opciones:</strong><br>
        • Recarga créditos en <a href="https://mitiklive.com/wallapop/" target="_blank" style="color: #60a5fa;">mitiklive.com</a><br>
        • Actualiza tu plan a uno superior
      </p>
    `;
  } else if (plan_remaining > 0) {
    detailsHTML += `
      <hr style="margin: 15px 0; border: none; border-top: 1px solid rgba(255,255,255,0.1);">
      <p style="color: #60a5fa;">
        💡 <strong>Sugerencia:</strong> Publica ${available} anuncios ahora y recarga créditos para el resto.
      </p>
    `;
  }
  
  detailsHTML += `</div>`;
  
  await showModal({
    title: '❌ Créditos insuficientes',
    html: detailsHTML,
    buttons: [
      { 
        text: '🔗 Ir a recargar créditos', 
        value: 'recharge', 
        primary: true 
      },
      { 
        text: 'Cerrar', 
        value: 'close', 
        primary: false 
      }
    ]
  }).then(result => {
    if (result === 'recharge') {
      // Abrir página de recarga en nueva pestaña
      window.open('https://mitiklive.com/wallapop/', '_blank');
    }
  });
}

/**
 * Descontar crédito después de publicar exitosamente
 * Wrapper para mayor claridad en el código
 */
export async function deductCreditAfterPublish(wallaAccountId, listingId) {
  try {
    if (!window.CreditsManager) {
      logger.error('[PublishValidation] CreditsManager no disponible para deducción');
      return false;
    }
    
    
    const result = await window.CreditsManager.deductCredit(wallaAccountId, listingId);
    
    if (result) {
    } else {
    }
    
    return result;
    
  } catch (error) {
    logger.error('[PublishValidation] Error descontando crédito:', error);
    return false;
  }
}
