/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/publish-confirmation-modal.js – Rol: Modal educativo de confirmación antes de publicar
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función.
    
    ✅ v2.5.1: Modal de confirmación educativo que explica el proceso de renovación
*/

import { createModal, closeModal } from './base-modal.js';
import { logger } from './logger.js';

const STORAGE_KEY = 'mitiklive_hide_publish_confirmation';

/**
 * Verifica si el usuario ha marcado "No volver a mostrar"
 * @returns {Promise<boolean>}
 */
async function shouldShowModal() {
  try {
    const result = await chrome.storage.local.get(STORAGE_KEY);
    return !result[STORAGE_KEY]; // Mostrar si NO está marcado como oculto
  } catch (e) {
    logger.error('[PublishConfirmation] Error leyendo storage:', e);
    return true; // Por defecto, mostrar
  }
}

/**
 * Verifica si el modal educativo está habilitado (para uso externo)
 * @returns {Promise<boolean>}
 */
export async function isEducationalModalEnabled() {
  return await shouldShowModal();
}

/**
 * Guarda la preferencia de "No volver a mostrar"
 */
async function saveHidePreference() {
  try {
    await chrome.storage.local.set({ [STORAGE_KEY]: true });
    logger.debug('[PublishConfirmation] Preferencia guardada: no volver a mostrar');
  } catch (e) {
    logger.error('[PublishConfirmation] Error guardando preferencia:', e);
  }
}

/**
 * Resetea la preferencia (útil para testing o configuración)
 */
export async function resetHidePreference() {
  try {
    await chrome.storage.local.remove(STORAGE_KEY);
    logger.debug('[PublishConfirmation] Preferencia reseteada');
  } catch (e) {
    logger.error('[PublishConfirmation] Error reseteando preferencia:', e);
  }
}

/**
 * Muestra modal de confirmación educativo antes de publicar
 * @param {Object} options - Opciones
 * @param {number} options.selectedCount - Número de anuncios seleccionados
 * @param {Function} options.onConfirm - Callback al confirmar
 * @param {Function} options.onCancel - Callback al cancelar
 */
function showPublishConfirmationModal({ selectedCount, onConfirm, onCancel }) {
  const content = `
    <div class="publish-confirmation-modal">
      <div class="confirmation-header">
        <div class="confirmation-icon">🚀</div>
        <h2 class="confirmation-title">¡Multiplica la visibilidad de tus anuncios!</h2>
      </div>
      
      <div class="confirmation-body">
        <div class="confirmation-main">
          <p class="confirmation-lead">
            <strong>Vas a renovar ${selectedCount} anuncio${selectedCount !== 1 ? 's' : ''}.</strong>
            Esta estrategia profesional te permitirá:
          </p>
          
          <div class="confirmation-benefits">
            <div class="benefit-item">
              <div class="benefit-icon">📈</div>
              <div class="benefit-content">
                <h4>Primeras posiciones garantizadas</h4>
                <p>Tus anuncios aparecerán al principio de los resultados de búsqueda, como si fueran nuevos.</p>
              </div>
            </div>
            
            <div class="benefit-item">
              <div class="benefit-icon">👀</div>
              <div class="benefit-content">
                <h4>Hasta x10 más visibilidad</h4>
                <p>Los anuncios en primeras posiciones reciben hasta 10 veces más visualizaciones que los antiguos.</p>
              </div>
            </div>
            
            <div class="benefit-item">
              <div class="benefit-icon">🌐</div>
              <div class="benefit-content">
                <h4>Mejor posicionamiento en Google</h4>
                <p>Los anuncios renovados se indexan como contenido fresco, mejorando su ranking en buscadores.</p>
              </div>
            </div>
            
            <div class="benefit-item">
              <div class="benefit-icon">🔔</div>
              <div class="benefit-content">
                <h4>Notificaciones a usuarios interesados</h4>
                <p>Wallapop notifica a usuarios que tienen búsquedas guardadas relacionadas con tus productos.</p>
              </div>
            </div>
          </div>
        </div>
        
        <div class="confirmation-process">
          <h3 class="process-title">📋 Proceso de renovación:</h3>
          <ol class="process-steps">
            <li><strong>Actualización inteligente:</strong> Tus anuncios se actualizan manteniendo toda su información, fotos, precios y descripción intactos.</li>
            <li><strong>Reposicionamiento automático:</strong> Wallapop los coloca como anuncios recientes en primeras posiciones.</li>
            <li><strong>Visibilidad maximizada:</strong> Aparecen en las primeras páginas de resultados y en búsquedas de Google.</li>
          </ol>
          <p class="process-note">
            <span class="note-icon">💡</span>
            <strong>Estrategia profesional:</strong> Grandes vendedores en Wallapop renuevan sus anuncios diariamente para mantener máxima visibilidad. Esta técnica legal y efectiva te permite competir al más alto nivel.
          </p>
        </div>
        
        <div class="confirmation-assurance">
          <div class="assurance-badge">
            <span class="badge-icon">✓</span>
            <span class="badge-text">Proceso 100% seguro y autorizado</span>
          </div>
          <div class="assurance-badge">
            <span class="badge-icon">✓</span>
            <span class="badge-text">Mantiene todas tus fotos y descripción</span>
          </div>
          <div class="assurance-badge">
            <span class="badge-icon">✓</span>
            <span class="badge-text">No afecta a tu historial de ventas</span>
          </div>
        </div>
        
        <div class="confirmation-checkbox">
          <label class="checkbox-container">
            <input type="checkbox" id="hide-publish-confirmation" />
            <span class="checkmark"></span>
            <span class="checkbox-label">No volver a mostrar este mensaje</span>
          </label>
        </div>
      </div>
      
      <div class="confirmation-footer">
        <button class="confirmation-btn confirmation-btn-secondary" data-action="cancel">
          Cancelar
        </button>
        <button class="confirmation-btn confirmation-btn-primary" data-action="confirm">
          <span class="btn-icon">🚀</span>
          <span class="btn-text">Renovar ${selectedCount} anuncio${selectedCount !== 1 ? 's' : ''}</span>
        </button>
      </div>
    </div>
  `;
  
  const modal = createModal({
    id: 'publish-confirmation-modal',
    overlayClass: 'publish-confirmation-overlay',
    content,
    closeOnEsc: true,
    closeOnOverlayClick: false, // No cerrar con click fuera (forzar decisión)
    onOpen: () => {
      logger.debug('[PublishConfirmation] Modal mostrado', { selectedCount });
      
      const overlay = document.getElementById('publish-confirmation-modal');
      if (!overlay) return;
      
      // Botón "Cancelar"
      const cancelBtn = overlay.querySelector('[data-action="cancel"]');
      if (cancelBtn) {
        cancelBtn.addEventListener('click', () => {
          closeModal('publish-confirmation-modal');
          if (onCancel) onCancel();
        });
      }
      
      // Botón "Confirmar"
      const confirmBtn = overlay.querySelector('[data-action="confirm"]');
      if (confirmBtn) {
        confirmBtn.addEventListener('click', async () => {
          // Verificar si marcó "No volver a mostrar"
          const hideCheckbox = overlay.querySelector('#hide-publish-confirmation');
          if (hideCheckbox && hideCheckbox.checked) {
            await saveHidePreference();
          }
          
          closeModal('publish-confirmation-modal');
          if (onConfirm) onConfirm();
        });
      }
    }
  });
  
  return modal;
}

/**
 * Muestra el modal de confirmación si es necesario
 * @param {Object} options - Opciones
 * @param {number} options.selectedCount - Número de anuncios seleccionados
 * @returns {Promise<boolean>} - true si confirma, false si cancela
 */
export async function showPublishConfirmation({ selectedCount }) {
  // Verificar si debe mostrar
  const shouldShow = await shouldShowModal();
  
  if (!shouldShow) {
    logger.debug('[PublishConfirmation] Modal no se muestra (preferencia guardada)');
    return true; // No mostrar modal = proceder directamente
  }
  
  // Mostrar modal y esperar decisión
  return new Promise((resolve) => {
    showPublishConfirmationModal({
      selectedCount,
      onConfirm: () => resolve(true),
      onCancel: () => resolve(false)
    });
  });
}
