/**
 * Gestor unificado de botones de publicación y backup
 * @module panel/publish-button-manager
 * 
 * ✅ v10.5.29: Añadido control de btnBackup (antes en ui.js)
 * ✅ v10.5.82: Estado de selección proviene de module-state.js (fuente única de verdad)
 *              Este módulo recibe los datos via updateState() llamado desde listings/index.js
 * 
 * ARQUITECTURA:
 * - module-state.js → Fuente única de verdad para selección de listings
 * - listings/index.js → Llama updateSelectionUI() que sincroniza a este módulo
 * - Este módulo → Solo renderiza botones según el estado recibido
 */

import { logger } from './logger.js';
import DOMUtils from './dom-utils.js';

// ============================================
// Estado (recibido desde listings/index.js via updateState)
// ============================================

let state = {
  // Datos de selección (sincronizados desde module-state.js)
  filter: 'active',
  selectedCount: 0,
  selectedActive: 0,
  selectedDeleted: 0,
  isSelectAll: false,
  totalActive: 0,
  totalDeleted: 0,
  
  // Datos de BD (sincronizados desde state-manager.js)
  dbTotalActive: 0,
  dbTotalDeleted: 0,
  selectAllByFilter: false,
  excludedCount: 0, // ✅ v2.0.0: Número de excluidos
  
  // Datos propios de este módulo
  hasPendingProcess: false,
  pendingProcessId: null,
  pendingType: null,
  pendingCount: 0,
  processingCount: 0,
  operationRunning: false,
  
  // ✅ v2.6.0: Estado de delay (reactivo via WebSocket)
  publishInDelay: false,
  delayUntil: null,
  delaySeconds: null,
  
  // Créditos
  availableCredits: 0,
  
  // Estado de backup
  backupStatus: 'idle',
  backupPending: 0,
  backupErrors: 0,
  backupSeen: 0,
  backupTotal: 0,
  backupViewedAt: null,
  backupRunId: null,
  
  // Estado de autenticación
  isAuthenticated: false
};

// ============================================
// Referencias DOM (cacheadas)
// ============================================

let refs = null;

function getRefs() {
  if (refs) return refs;
  
  refs = {
    // ✅ v5.0.0: Botón unificado publicar/republicar
    btnPublishDynamic: document.getElementById('btn-publish-dynamic'),
    publishLabel: document.getElementById('publish-label'),
    
    // Botones de gestión
    btnBackup: document.getElementById('btn-backup'),
    btnDeleteSelected: document.getElementById('btn-delete-selected'),
    btnResume: document.getElementById('btn-resume-pending'),
    btnCancel: document.getElementById('btn-cancel-pending'),
    
    // ✅ v1.3.0: Botón Reanudar en tab "En Curso" (REGLA #41)
    btnResumeProgress: document.getElementById('btn-resume-progress'),
    btnResumeProgressContainer: document.getElementById('progress-resume-container'),
    
    // Contadores
    deleteCount: document.getElementById('delete-count'),
    
    // ✅ v10.5.29: Pill de backup
    pillBackup: document.getElementById('pill-backup')
  };
  
  return refs;
}

// ============================================
// API pública
// ============================================

/**
 * Actualiza el estado y refresca todos los botones
 * @param {Object} partialState - Estado parcial a actualizar
 */
export function updateState(partialState) {
  state = { ...state, ...partialState };
  
  // Estado actualizado - log solo si DEBUG activado
  
  refreshAllButtons();
}

/**
 * Obtiene el estado actual
 * @returns {Object} Estado actual
 */
export function getState() {
  return { ...state };
}

/**
 * Refresca todos los botones según estado actual
 */
export function refreshAllButtons() {
  const r = getRefs();
  if (!r) return;
  
  updatePublishDynamicButton(r);
  updateDeleteSelectedButton(r);
  updateResumeButton(r);
  updateCancelButton(r);
  updateResumeProgressButton(r); // ✅ v1.3.0: Botón Reanudar en tab "En Curso"
  updateBackupButton(r); // ✅ v10.5.29: Añadido
  
  // Log solo si DEBUG
}

/**
 * Inicializa el gestor
 */
export function init() {
  refreshAllButtons();
}

/**
 * ✅ v10.5.29: Actualiza estado de backup y refresca botón
 * Reemplaza updateBackupButtonUI de ui.js
 * @param {Object} backupState - Estado del backup
 */
export function updateBackupState(backupState) {
  if (!backupState) return;
  
  state.backupStatus = String(backupState.status || 'idle').toLowerCase();
  state.backupPending = backupState.pending || 0;
  state.backupErrors = backupState.errors || 0;
  state.backupSeen = backupState.seen || 0;
  state.backupTotal = backupState.total || 0;
  state.backupViewedAt = backupState.viewed_at || null;
  state.backupRunId = backupState.run_id || null;
  
  const r = getRefs();
  if (r) updateBackupButton(r);
}

/**
 * ✅ v10.5.29: Actualiza estado de autenticación
 * @param {boolean} isAuth - Si está autenticado
 */
export function setAuthenticated(isAuth) {
  state.isAuthenticated = !!isAuth;
  refreshAllButtons();
}

// ============================================
// Actualización de botones individuales
// ============================================

/**
 * ✅ v5.0.0: Botón UNIFICADO Publicar/Republicar (dinámico)
 * Cambia según filtro y selección
 */
function updatePublishDynamicButton(r) {
  if (!r.btnPublishDynamic || !r.publishLabel) return;
  
  const filter = state.filter;
  const hasPending = state.hasPendingProcess;
  const isRunning = state.operationRunning;
  
  // Ocultar en filtros de solo lectura
  if (filter === 'featured' || filter === 'reserved') {
    r.btnPublishDynamic.classList.add('ml-hidden');
    return;
  }
  
  // Ocultar si no hay selección
  if (state.selectedCount === 0) {
    r.btnPublishDynamic.classList.add('ml-hidden');
    return;
  }
  
  // Ocultar si hay proceso pendiente u operación en curso
  if (hasPending || isRunning) {
    r.btnPublishDynamic.classList.add('ml-hidden');
    return;
  }
  
  // ✅ MOSTRAR y configurar según filtro
  r.btnPublishDynamic.classList.remove('ml-hidden');
  
  // ✅ v2.0.0: Determinar cantidad a mostrar
  // Si selectAllByFilter: total de BD menos excluidos
  // Si no: conteo de seleccionados locales
  const excludedCount = state.excludedCount || 0;
  const displayActiveCount = state.selectAllByFilter 
    ? (state.dbTotalActive - excludedCount) 
    : state.selectedActive;
  const displayDeletedCount = state.selectAllByFilter 
    ? (state.dbTotalDeleted - excludedCount) 
    : state.selectedDeleted;
  
  // ✅ v2.0.0: Determinar si mostrar "Todos" o número específico
  const hasExclusions = state.selectAllByFilter && excludedCount > 0;
  
  if (filter === 'deleted') {
    // MODO: Republicar eliminados
    r.btnPublishDynamic.setAttribute('data-action', 'republish');
    r.btnPublishDynamic.classList.remove('btn-success');
    r.btnPublishDynamic.classList.add('btn-info');
    
    // ✅ v2.0.0: Si hay exclusiones, no mostrar "Todos"
    const text = (state.isSelectAll || state.selectAllByFilter) && !hasExclusions
      ? `Republicar (Todos ${displayDeletedCount})`
      : `Republicar (${displayDeletedCount})`;
    
    r.publishLabel.textContent = text;
    r.btnPublishDynamic.title = (state.isSelectAll || state.selectAllByFilter) && !hasExclusions
      ? `Republicar todos los ${displayDeletedCount} anuncios eliminados`
      : `Republicar ${displayDeletedCount} anuncio${displayDeletedCount !== 1 ? 's' : ''} eliminado${displayDeletedCount !== 1 ? 's' : ''}`;
    
  } else {
    // MODO: Publicar activos (filtros: active, total)
    r.btnPublishDynamic.setAttribute('data-action', 'publish');
    r.btnPublishDynamic.classList.remove('btn-info');
    r.btnPublishDynamic.classList.add('btn-success');
    
    // ✅ v2.0.0: Si hay exclusiones, no mostrar "Todos"
    const text = (state.isSelectAll || state.selectAllByFilter) && !hasExclusions
      ? `Publicar (Todos ${displayActiveCount})`
      : `Publicar (${displayActiveCount})`;
    
    r.publishLabel.textContent = text;
    r.btnPublishDynamic.title = (state.isSelectAll || state.selectAllByFilter) && !hasExclusions
      ? `Publicar todos los ${displayActiveCount} anuncios activos`
      : `Publicar ${displayActiveCount} anuncio${displayActiveCount !== 1 ? 's' : ''} activo${displayActiveCount !== 1 ? 's' : ''}`;
  }
}

/**
 * Botón "D Eliminar (n)" - Seleccionados activos
 * ✅ v2.0.0: Soporte para excludedCount
 */
function updateDeleteSelectedButton(r) {
  if (!r.btnDeleteSelected || !r.deleteCount) return;
  
  // ✅ v2.0.0: Usar total de BD menos excluidos si selectAllByFilter está activo
  const excludedCount = state.excludedCount || 0;
  const displayActiveCount = state.selectAllByFilter 
    ? (state.dbTotalActive - excludedCount) 
    : state.selectedActive;
  
  // Solo visible en filtros active/total con selección de activos
  const visible = displayActiveCount > 0 && 
                  (state.filter === 'active' || state.filter === 'total') &&
                  !state.hasPendingProcess &&
                  !state.operationRunning;
  
  if (visible) {
    r.btnDeleteSelected.classList.remove('ml-hidden');
    r.deleteCount.textContent = displayActiveCount;
    r.btnDeleteSelected.title = `Eliminar ${displayActiveCount} anuncio${displayActiveCount !== 1 ? 's' : ''} de Wallapop`;
  } else {
    r.btnDeleteSelected.classList.add('ml-hidden');
  }
}


/**
 * Botón "Reanudar" - Proceso pendiente
 * ✅ v7.2.47: Oculta durante publicación activa (no solo deshabilita)
 */
function updateResumeButton(r) {
  if (!r.btnResume) return;
  
  const hasPending = state.hasPendingProcess;
  const isPublishing = state.processingCount > 0;
  const inDelay = state.publishInDelay;  // ✅ v2.6.0: Estado reactivo via WebSocket
  
  // Ocultar si: no hay pendientes O hay publicación activa O está en delay
  if (!hasPending || isPublishing || inDelay) {
    r.btnResume.classList.add('ml-hidden');
    r.btnResume.disabled = false;
    r.btnResume.classList.remove('disabled');
    const textSpan = r.btnResume.querySelector('.btn-text');
    if (textSpan) textSpan.textContent = 'Reanudar';
    return;
  }
  
  // Mostrar botón (hay pendientes y NO hay publicación activa)
  r.btnResume.classList.remove('ml-hidden');
  r.btnResume.disabled = false;
  r.btnResume.classList.remove('disabled');
  
  // Actualizar texto con número de pendientes
  const textSpan = r.btnResume.querySelector('.btn-text');
  if (textSpan && state.pendingCount > 0) {
    textSpan.textContent = `Reanudar (${state.pendingCount})`;
  } else if (textSpan) {
    textSpan.textContent = 'Reanudar';
  }
  
  r.btnResume.title = state.pendingCount > 0 
    ? `Reanudar ${state.pendingCount} pendiente${state.pendingCount !== 1 ? 's' : ''}`
    : 'Reanudar proceso pendiente';
}

/**
 * Botón "Cancelar" - Proceso pendiente
 * ✅ v7.2.47: Oculta durante publicación activa (no solo deshabilita)
 */
function updateCancelButton(r) {
  if (!r.btnCancel) return;
  
  const hasPending = state.hasPendingProcess;
  const isPublishing = state.processingCount > 0;
  const inDelay = state.publishInDelay;  // ✅ v2.6.0: Estado reactivo via WebSocket
  
  // Ocultar si: no hay pendientes O hay publicación activa O está en delay
  if (!hasPending || isPublishing || inDelay) {
    r.btnCancel.classList.add('ml-hidden');
    r.btnCancel.disabled = false;
    r.btnCancel.classList.remove('disabled');
    const textSpan = r.btnCancel.querySelector('.btn-text');
    if (textSpan) textSpan.textContent = 'Cancelar';
    return;
  }
  
  // Mostrar botón (hay pendientes y NO hay publicación activa)
  r.btnCancel.classList.remove('ml-hidden');
  r.btnCancel.disabled = false;
  r.btnCancel.classList.remove('disabled');
  
  // Actualizar texto con número de pendientes
  const textSpan = r.btnCancel.querySelector('.btn-text');
  if (textSpan && state.pendingCount > 0) {
    textSpan.textContent = `Cancelar (${state.pendingCount})`;
  } else if (textSpan) {
    textSpan.textContent = 'Cancelar';
  }
  
  r.btnCancel.title = state.pendingCount > 0
    ? `Cancelar ${state.pendingCount} pendiente${state.pendingCount !== 1 ? 's' : ''} en cola`
    : 'Cancelar procesos pendientes en cola';
}

/**
 * ✅ v1.3.0: Botón "Reanudar" en tab "En Curso"
 * Misma lógica que btnResume pero en el tab de progreso (REGLA #41)
 * Al hacer click, dispara el botón original btn-resume-pending
 */
function updateResumeProgressButton(r) {
  if (!r.btnResumeProgress || !r.btnResumeProgressContainer) return;
  
  const hasPending = state.hasPendingProcess;
  const isPublishing = state.processingCount > 0;
  const inDelay = state.publishInDelay;  // ✅ v2.6.0: Estado reactivo via WebSocket
  
  // Ocultar si: no hay pendientes O hay publicación activa O está en delay
  if (!hasPending || isPublishing || inDelay) {
    r.btnResumeProgress.classList.add('ml-hidden');
    r.btnResumeProgressContainer.classList.add('ml-hidden');
    return;
  }
  
  // Mostrar botón y contenedor
  r.btnResumeProgress.classList.remove('ml-hidden');
  r.btnResumeProgressContainer.classList.remove('ml-hidden');
  
  // Actualizar texto con número de pendientes
  const textSpan = r.btnResumeProgress.querySelector('.btn-text');
  if (textSpan && state.pendingCount > 0) {
    textSpan.textContent = `Reanudar publicación (${state.pendingCount} pendientes)`;
  } else if (textSpan) {
    textSpan.textContent = 'Reanudar publicación';
  }
  
  r.btnResumeProgress.title = state.pendingCount > 0 
    ? `Reanudar ${state.pendingCount} pendiente${state.pendingCount !== 1 ? 's' : ''}`
    : 'Reanudar publicación pendiente';
}

// ============================================
// ✅ v10.5.29: Botón Backup (migrado de ui.js)
// ============================================

/**
 * Actualiza el botón de backup según estado
 * Migrado de ui.js para centralización
 * 
 * ✅ v10.5.144: Deshabilitar cuando hay publicación activa
 */
function updateBackupButton(r) {
  if (!r.btnBackup) return;
  
  const status = state.backupStatus;
  const { backupPending, backupErrors, backupSeen, backupTotal, backupViewedAt } = state;
  
  // Normalizar estados
  const isIdle = status === 'idle';
  const isRunning = status === 'running';
  const isIncomplete = ['incomplete', 'cancelled', 'canceled', 'error'].includes(status);
  const isOk = status === 'ok';
  
  // ✅ v10.5.144: Detectar si hay publicación activa
  const hasActivePublish = state.hasPendingProcess || state.pendingCount > 0 || state.processingCount > 0;
  
  // Limpiar clases
  r.btnBackup.classList.remove('is-running', 'btn-danger', 'btn-warning', 'btn-success');
  
  // Habilitar/deshabilitar según autenticación, estado de backup Y publicación
  const isBackupActive = isRunning;
  const shouldDisable = !state.isAuthenticated || isBackupActive || hasActivePublish;
  r.btnBackup.disabled = shouldDisable;
  r.btnBackup.setAttribute('aria-disabled', shouldDisable ? 'true' : 'false');
  
  // ✅ v10.5.144: Tooltip explicativo cuando está deshabilitado por publicación
  if (hasActivePublish && !isBackupActive) {
    r.btnBackup.title = 'No disponible durante la publicación';
  } else if (!state.isAuthenticated) {
    r.btnBackup.title = 'Inicia sesión para sincronizar';
  } else if (isBackupActive) {
    r.btnBackup.title = 'Sincronización en curso...';
  } else {
    r.btnBackup.title = 'Sincronizar anuncios desde Wallapop';
  }
  
  // ===== TEXTO Y COLOR DEL BOTÓN =====
  let buttonText = 'Backup';
  const label = r.btnBackup.querySelector('.label');
  
  if (isRunning) {
    const progress = backupTotal > 0 ? ` 🔄 ${backupSeen}/${backupTotal}` : ' 🔄 ...';
    buttonText = `Backup${progress}`;
    r.btnBackup.classList.add('btn-primary', 'is-running');
  } else if (isIncomplete && backupPending > 0) {
    buttonText = `Reanudar ⚠️ ${backupPending} pendientes`;
    r.btnBackup.classList.add('btn-warning');
  } else if (isOk && !backupViewedAt) {
    buttonText = 'Backup ✅';
    r.btnBackup.classList.add('btn-success');
    
    // Auto-reset visual después de 2s
    setTimeout(() => {
      if (label?.textContent === 'Backup ✅' && state.backupStatus === 'ok') {
        label.textContent = 'Backup';
        r.btnBackup.classList.remove('btn-success');
        // Notificar para marcar como visto (via evento)
        window.dispatchEvent(new CustomEvent('backup-viewed', { detail: { runId: state.backupRunId } }));
      }
    }, 2000);
  }
  
  if (label) label.textContent = buttonText;
  
  // ===== PILL DE PROGRESO =====
  updateBackupPill(r);
  
  // Limpiar badges pequeños (no usados)
  const bPend = ensureBadge(r.btnBackup, 'badge-pending');
  const bErr = ensureBadge(r.btnBackup, 'badge-errors');
  if (bPend) DOMUtils.hide(bPend);
  if (bErr) DOMUtils.hide(bErr);
}

/**
 * Actualiza el pill de backup (barra de progreso)
 */
function updateBackupPill(r) {
  const pill = r.pillBackup;
  if (!pill) return;
  
  const status = state.backupStatus;
  const { backupSeen, backupTotal, backupViewedAt } = state;
  
  const isRunning = status === 'running';
  const isIncomplete = ['incomplete', 'cancelled', 'canceled', 'error'].includes(status);
  const isOk = status === 'ok';
  
  let pillText = '';
  let pillClass = '';
  let showProgressBar = false;
  let progressPercent = 0;
  
  if (isRunning) {
    showProgressBar = true;
    progressPercent = backupTotal > 0 ? Math.round((backupSeen / backupTotal) * 100) : 0;
    pillText = `${backupSeen}/${backupTotal}`;
    pillClass = 'with-progress';
  } else if (isIncomplete && backupTotal > 0 && backupSeen < backupTotal) {
    pillText = `Backup incompleto: ${backupSeen}/${backupTotal}`;
    pillClass = 'err';
  } else if (isOk && !backupViewedAt) {
    pillText = 'Backup completo';
    pillClass = 'ok';
    // Auto-limpiar después de 2s
    const currentText = pillText;
    setTimeout(() => {
      if (pill.textContent === currentText || pill.querySelector('.progress-text')?.textContent === currentText) {
        pill.textContent = '';
        pill.className = 'ml-pill ml-action-pill';
        pill.innerHTML = '';
      }
    }, 2000);
  }
  
  pill.className = `ml-pill ml-action-pill ${pillClass}`;
  
  if (showProgressBar) {
    pill.innerHTML = `
      <div class="progress-bar" style="width: ${progressPercent}%"></div>
      <span class="progress-text">${pillText}</span>
    `;
  } else {
    pill.innerHTML = pillText;
  }
}

/**
 * Asegura que existe un badge en un botón
 */
function ensureBadge(btn, className) {
  if (!btn) return null;
  let badge = btn.querySelector(`.${className}`);
  if (!badge) {
    badge = document.createElement('span');
    badge.className = className;
    btn.appendChild(badge);
  }
  return badge;
}

// ============================================
// Validación de créditos
// ============================================

/**
 * Verifica si hay suficientes créditos
 * @param {number} needed - Créditos necesarios
 * @returns {boolean} True si hay suficientes
 */
export function hasEnoughCredits(needed) {
  return state.availableCredits >= needed;
}

/**
 * Muestra modal de créditos insuficientes
 * @param {number} needed - Créditos necesarios
 */
export function showInsufficientCreditsModal(needed) {
  const available = state.availableCredits;
  const missing = needed - available;
  
  // Este modal ya existe, solo lo invocamos
  if (window.showInsufficientCreditsModal) {
    window.showInsufficientCreditsModal(needed, available, missing);
  }
}
