/**
 * ============================================
 * Tab "En Curso" - Visualización de progreso de publicación
 * ============================================
 * v6.12.0 - Migración de modal a tab
 * 
 * REUTILIZA código de publishing-monitor.js:
 * - renderItem()
 * - updateStats()  
 * - loadStateFromDatabase()
 * 
 * ✅ Sigue GUIA-REUTILIZACION-CODIGO v3.6
 * ✅ TODO desde BD (0% localStorage)
 * ✅ Multi-cuenta (aislado por account_id)
 * ✅ WebSocket para updates en tiempo real
 */

import { logger } from './logger.js';
import { 
  getImageUrl,
  formatPrice,
  truncateTitle 
} from './listings/helpers.js'; // ✅ v7.2.0: Reutilizar funciones centralizadas
import { 
  initDelaySlider as initDelaySliderComponent,
  getCurrentDelay 
} from './delay-slider.js'; // ✅ v10.5.34: Componente centralizado
import {
  clearAllSavedSelections,
  clearSelection,
  clearSelectAllByFilter
} from './listings/module-state.js'; // ✅ v10.5.144: Para resetPublishState
import { syncRowSelectionVisuals } from './listings/events.js'; // ✅ v10.5.144: Sincronizar visual
import * as ButtonManager from './publish-button-manager.js'; // ✅ v10.5.144: Para resetPublishState

// ============================================
// CONFIGURACIÓN
// ============================================

// ✅ v7.2.0: Eliminar hardcodeo - API_BASE se obtiene de session
// ✅ v10.5.52: Items por página para scroll infinito
const ITEMS_PER_PAGE = 20;

// ============================================
// ESTADO INTERNO
// ============================================

// ✅ v10.5.33: API_BASE cacheado desde storage
let _cachedApiBase = 'https://www.mitiklive.com/fa'; // Fallback

/**
 * ✅ v10.5.33: Carga API_BASE desde chrome.storage (fuente única de verdad)
 * Si no existe aún (primer arranque), usa fallback silenciosamente
 */
async function loadApiBase() {
  try {
    const result = await chrome.storage.local.get(['cfg.API_BASE']);
    if (result['cfg.API_BASE']) {
      _cachedApiBase = result['cfg.API_BASE'];
    }
    // Si no existe, usa fallback silenciosamente
  } catch (err) {
    // Error silencioso - usa fallback hardcodeado
  }
}

const STATE = {
  accountId: null,       // ID de la cuenta actual
  items: [],             // Lista de items {id, title, price, image_url, status, error}
  total: 0,              // Total de anuncios a publicar
  processed: 0,          // Anuncios procesados
  isActive: false,       // Si hay publicación en curso
  isPaused: false,       // ✅ v2.2.0: Si la publicación está pausada
  filter: 'pending',     // ✅ v10.5.52: Filtro activo: 'success', 'error', 'pending' (sin 'all')
  startTime: null,       // Timestamp de inicio
  delay: 0,              // Delay configurado (segundos)
  currentPage: 1,        // ✅ v7.2.52: Página actual para scroll infinito
  stats: {},             // ✅ v7.2.53: Stats del backend (pending, processing, success, errors)
  _lastProcessingId: null, // ✅ v10.5.82: Track del último item processing para transiciones suaves
  _lastProcessingTime: 0   // ✅ v10.5.132: Timestamp del último item en processing (evita parpadeo)
};

// ============================================
// FUNCIONES PÚBLICAS
// ============================================

/**
 * Mostrar tab "En Curso" programáticamente
 * ✅ v7.2.9: Mejorado con delay para asegurar cambio de tab
 */
export function showProgressTab() {
  
  // Mostrar tab si estaba oculto
  showTab();
  
  // Pequeño delay para que el DOM se actualice
  setTimeout(() => {
    const tabButton = document.querySelector('.tab-button[data-tab="in_progress"]');
    
    if (!tabButton) {
      return;
    }
    
    if (tabButton.style.display === 'none') {
      tabButton.style.display = 'flex';
    }
    
    if (!tabButton.classList.contains('active')) {
      tabButton.click();
    } else {
    }
  }, 100);
}

// ============================================
// INICIALIZACIÓN
// ============================================

/**
 * Inicializar tab de progreso
 * @param {number} accountId - ID de la cuenta actual
 */
export async function initProgressTab(accountId) {
  
  // ✅ v10.5.33: Cargar API_BASE desde storage primero
  await loadApiBase();
  
  // ✅ v2.2.0: Cargar estado de pausa
  await loadPauseState();
  
  STATE.accountId = accountId;
  
  // Cargar estado desde BD
  await loadState();
  
  // Inicializar listeners de contadores
  initStatsListeners();
  
  // ✅ v2.2.2: Inicializar listener de pausa (solo una vez)
  initPauseListener();
  
  // Inicializar slider de delay
  await initDelaySlider();
  
  // Iniciar timer si hay publicación activa
  if (STATE.isActive) {
    startTimer();
  }
  
  // Renderizar
  render();
  
}

/**
 * Actualizar tab cuando se cambia de cuenta
 * @param {number} accountId - ID de la nueva cuenta
 */
export async function switchAccount(accountId) {
  // ✅ v10.5.50: Si es la misma cuenta, solo refrescar datos (preservar filtro)
  const isSameAccount = STATE.accountId === accountId;
  const currentFilter = STATE.filter; // Guardar filtro actual
  
  STATE.accountId = accountId;
  
  // Solo resetear si cambió de cuenta
  if (!isSameAccount) {
    STATE.items = [];
    STATE.total = 0;
    STATE.processed = 0;
    STATE.isActive = false;
    STATE.filter = 'pending'; // ✅ v10.5.52: Default a pendientes
    STATE.startTime = null;
    stopTimer();
  }
  
  // Cargar estado de la cuenta
  await loadState();
  
  // ✅ v10.5.50: Restaurar filtro si es la misma cuenta
  if (isSameAccount) {
    STATE.filter = currentFilter;
  }
  
  // Re-inicializar listeners solo si cambió de cuenta
  if (!isSameAccount) {
    initStatsListeners();
    await initDelaySlider();
  }
  
  // Iniciar timer si hay publicación activa
  if (STATE.isActive && !isSameAccount) {
    startTimer();
  }
  
  // Renderizar
  render();
  
  // Actualizar visibilidad del tab
  updateTabVisibility();
}

/**
 * Limpiar tab al hacer logout
 */
export function clearProgressTab() {
  
  STATE.accountId = null;
  STATE.items = [];
  STATE.total = 0;
  STATE.processed = 0;
  STATE.isActive = false;
  STATE.isPaused = false; // ✅ v2.2.0
  STATE.filter = 'pending'; // ✅ v10.5.52: Default a pendientes
  STATE.startTime = null;
  
  stopTimer();
  render();
  hideTab();
}

// ============================================
// ✅ v2.2.0: SISTEMA DE PAUSA
// ============================================

/**
 * Alternar pausa/reanudar publicación
 * ✅ v2.2.2: Spinner en botón + notificación al content script
 */
export async function togglePause() {
  // Obtener botón y mostrar spinner
  const pauseBtn = document.getElementById('btn-pause-publish');
  if (pauseBtn) {
    pauseBtn.disabled = true;
    pauseBtn.textContent = '⏳';
  }
  
  STATE.isPaused = !STATE.isPaused;
  
  // Guardar estado en storage para que el Service Worker lo lea
  await chrome.storage.local.set({ 'publish.paused': STATE.isPaused });
  
  // Notificar al Service Worker
  chrome.runtime.sendMessage({ 
    type: 'PUBLISH.SET_PAUSED', 
    paused: STATE.isPaused 
  });
  
  // Notificar al content script para mostrar/ocultar modal en Wallapop
  try {
    const tabs = await chrome.tabs.query({ url: '*://*.wallapop.com/*' });
    if (tabs.length > 0) {
      const pendingCount = STATE.stats?.pending || 0;
      if (STATE.isPaused) {
        await chrome.tabs.sendMessage(tabs[0].id, { 
          type: 'SHOW_PAUSE_OVERLAY', 
          pendingCount 
        });
      } else {
        await chrome.tabs.sendMessage(tabs[0].id, { 
          type: 'HIDE_PAUSE_OVERLAY' 
        });
      }
    }
  } catch (e) {
    // Ignorar error si no hay tab de Wallapop
  }
  
  logger.info(`[ProgressTab] Publicación ${STATE.isPaused ? 'PAUSADA' : 'REANUDADA'}`);
  
  // Actualizar UI (esto restaurará el botón con el icono correcto)
  renderProcessingFixed();
}

/**
 * ✅ v2.2.2: Actualizar estado de pausa desde fuente externa (ej: botón en Wallapop)
 */
export async function setPaused(paused) {
  STATE.isPaused = paused;
  await chrome.storage.local.set({ 'publish.paused': paused });
  logger.info(`[ProgressTab] Publicación ${paused ? 'PAUSADA' : 'REANUDADA'} (externo)`);
  renderProcessingFixed();
}

/**
 * Obtener estado de pausa
 */
export function isPaused() {
  return STATE.isPaused;
}

/**
 * Cargar estado de pausa desde storage
 */
async function loadPauseState() {
  try {
    const result = await chrome.storage.local.get(['publish.paused']);
    STATE.isPaused = !!result['publish.paused'];
  } catch (e) {
    STATE.isPaused = false;
  }
}

/**
 * ✅ v10.5.144: Función centralizada para limpiar TODO el estado de publicación
 * REGLA #11: Centralización estricta de funciones relacionadas
 * REGLA #8: Reutilizar código existente (syncRowSelectionVisuals)
 * 
 * Se usa al:
 * - Cancelar publicación
 * - Completar publicación
 * - Cambiar de cuenta durante publicación
 * 
 * Limpia:
 * 1. Estado de selección (selectedIds, selectAllByFilter, selectionByFilter)
 * 2. Visual de selección (checkboxes, clase .selected) via syncRowSelectionVisuals
 * 3. Botón Eliminar (via ButtonManager)
 * 4. Barra de progreso
 * 5. Pestaña "En curso"
 * 6. Badge de pestaña Actividad
 */
export function resetPublishState() {
  logger.info('[ProgressTab] resetPublishState() - Limpiando todo el estado de publicación');
  
  // ✅ v2.2.0: Limpiar estado de pausa
  STATE.isPaused = false;
  chrome.storage.local.remove(['publish.paused']);
  
  // ✅ v2.6.0: La limpieza de delay se hace via API/WebSocket, no aquí
  
  // 1. Limpiar TODA la selección en module-state
  clearAllSavedSelections();
  
  // 2. Resetear checkbox de cabecera
  const selectAllCheckbox = document.getElementById('select-all-listings');
  if (selectAllCheckbox) {
    selectAllCheckbox.checked = false;
    selectAllCheckbox.indeterminate = false;
  }
  
  // 3. Desmarcar checkboxes individuales
  const checkboxes = document.querySelectorAll('.listing-checkbox');
  checkboxes.forEach(cb => cb.checked = false);
  
  // 4. Sincronizar clase .selected en filas (REUTILIZA función existente)
  syncRowSelectionVisuals();
  
  // 5. Forzar actualización de ButtonManager con selección vacía
  // ✅ v2.4.9: processingCount se actualiza automáticamente desde updateStats() (REGLA #41)
  ButtonManager.updateState({
    selectedCount: 0,
    selectedActive: 0,
    selectedDeleted: 0,
    selectAllByFilter: false,
    hasPendingProcess: false,
    pendingCount: 0
  });
  
  // 6. Ocultar barra de progreso
  const progressBar = document.getElementById('ml-progress-bar');
  if (progressBar) {
    progressBar.classList.add('ml-hidden');
  }
  
  // 7. Limpiar pestaña de progreso (En curso)
  clearProgressTab();
  
  // 8. Quitar badge de pestaña Actividad
  const activityBadge = document.querySelector('#tab-activity .tab-badge');
  if (activityBadge) {
    activityBadge.remove();
  }
  
  logger.debug('[ProgressTab] ✅ Estado de publicación limpiado');
}

// ============================================
// CARGA DE DATOS DESDE BD
// ============================================

/**
 * Cargar estado desde BD
 * ✅ REUTILIZA endpoint /monitor-state (ya funciona en backend v2.9.3)
 */
async function loadState() {
  if (!STATE.accountId) {
    return;
  }
  
  try {
    
    // ✅ REGLA #36: Usar Service Worker para autenticación
    const response = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/walla/publish2/monitor-state/${STATE.accountId}`,
      method: 'GET'
    });
    
    // ✅ v7.2.6: Si hay error de conexión, mantener tab visible con estado actual
    if (!response?.ok) {
      if (response?.error?.includes('CONNECTION_ERROR')) {
        // No ocultar tab, solo skip actualización
        return;
      }
      
      // Otros errores (401, 404, etc) → ocultar tab
      STATE.isActive = false;
      hideTab();
      return;
    }
    
    if (!response.data) {
      STATE.isActive = false;
      hideTab();
      return;
    }
    
    const data = response.data;
    
    // Verificar si hay cola activa
    if (!data.has_active_queue || data.total === 0) {
      STATE.isActive = false;
      hideTab();
      return;
    }
    
    // Actualizar estado
    STATE.items = data.items || [];
    STATE.total = data.total || 0;
    STATE.processed = data.processed || 0;
    STATE.isActive = true;
    // ✅ v7.2.54: Construir stats desde respuesta del backend
    STATE.stats = {
      pending: data.pending_count || 0,
      processing: data.processing_count || 0,
      success: data.success_count || 0,
      errors: data.error_count || 0
    };
    
    
    // Mostrar tab si hay proceso activo
    if (STATE.isActive) {
      showTab();
    }
    
  } catch (error) {
    logger.error('[ProgressTab] Error cargando estado:', error);
    STATE.isActive = false;
    hideTab();
  }
}

// ============================================
// ACTUALIZACIÓN EN TIEMPO REAL (WebSocket)
// ============================================

/**
 * Procesar update de progreso (desde WebSocket)
 * @param {Object} data - {item_id, status, processed, total, errors}
 * ✅ v10.5.82: Contenedor fijo se actualiza SIEMPRE, independiente del tab activo
 * ✅ v1.0.0: Actualiza ButtonManager cuando hay error para mostrar botón Reanudar
 */
export function handleProgressUpdate(data) {
  
  // Actualizar contadores
  if (data.processed !== undefined) STATE.processed = data.processed;
  if (data.total !== undefined) STATE.total = data.total;
  
  // Actualizar item específico
  if (data.item_id && data.status) {
    const item = STATE.items.find(i => i.id === data.item_id);
    if (item) {
      // ✅ v1.0.0: Si pasa a processing, inicializar tiempo total
      if (data.status === 'processing' && item.status !== 'processing') {
        item.itemStartTime = Date.now();
      }
      // ✅ v1.0.0: Si sale de processing, limpiar tiempo
      if (data.status !== 'processing' && item.status === 'processing') {
        item.itemStartTime = null;
      }
      
      // Actualizar todos los campos
      item.status = data.status;
      if (data.title) item.title = data.title;
      if (data.price !== undefined) item.price = data.price;
      if (data.image_url) item.image_url = data.image_url;
      if (data.error) item.error = data.error;
      if (data.missingFields) item.missingFields = data.missingFields;
      if (data.stepMessage) item.stepMessage = data.stepMessage;
    } else {
      // Nuevo item
      STATE.items.push({
        id: data.item_id,
        title: data.title || `Item ${data.item_id}`,
        price: data.price || 0,
        image_url: data.image_url || null,
        status: data.status,
        error: data.error || null,
        missingFields: data.missingFields || [],
        stepMessage: data.stepMessage || null,
        itemStartTime: data.status === 'processing' ? Date.now() : null // ✅ v1.0.0
      });
    }
    
    // ✅ v1.0.0: Cuando un item pasa a error_retry, actualizar ButtonManager
    // ✅ v2.4.9: processingCount se actualiza automáticamente desde updateStats() (REGLA #41)
    // Si hay errores pero la sesión sigue activa, el botón Reanudar NO debe aparecer
    // Solo actualizamos hasPendingProcess y pendingCount para tracking
    if (data.status === 'error_retry') {
      const pendingItems = STATE.items.filter(i => 
        i.status === 'pending' || i.status === 'queued_delete'
      );
      const errorItems = STATE.items.filter(i => i.status === 'error_retry');
      
      // Actualizar contadores pero NO processingCount
      ButtonManager.updateState({
        hasPendingProcess: true,
        pendingType: 'publish',
        pendingCount: pendingItems.length + errorItems.length
      });
      logger.debug(`[ProgressTab] Error detectado - pendingCount actualizado: ${errorItems.length} errores, ${pendingItems.length} pendientes`);
    }
  }
  
  // ✅ v10.5.82: SIEMPRE actualizar contenedor fijo (independiente del tab)
  renderProcessingFixed();
  
  // Renderizar lista solo si tab está activo
  const currentTab = document.querySelector('.tab-button.active')?.dataset.tab;
  if (currentTab === 'in_progress') {
    renderItemsListOnly();
  }
  
  // Actualizar badge siempre
  updateBadge();
  
  // Actualizar métricas de tiempo
  updateTimeMetrics();
}

/**
 * ✅ v10.5.53: Actualizar mensaje de paso (sin render completo)
 * ✅ v1.0.0: Tiempo total desde inicio del item (no se resetea entre pasos)
 * @param {string} message - Mensaje a mostrar (ej: "Seleccionando categoría...")
 * @param {number} step - Número de paso (1-9) para color progresivo
 */
export function updateStepMessage(message, step = 1) {
  // Actualizar en STATE (para el item processing actual)
  const processingItem = STATE.items.find(i => i.status === 'processing');
  if (processingItem) {
    processingItem.stepMessage = message;
    processingItem.stepNumber = step;
    
    // ✅ v1.0.0: Solo inicializar tiempo si es el PRIMER paso del item
    // El tiempo NO se resetea entre pasos - es tiempo TOTAL
    if (!processingItem.itemStartTime) {
      processingItem.itemStartTime = Date.now();
    }
  }
  
  // ✅ v10.5.54: Iniciar interval para actualizar tiempo
  startStepTimeUpdater();
  
  // Actualizar directamente en DOM (sin re-render completo)
  const stepEl = document.getElementById('progress-step-message');
  if (stepEl) {
    stepEl.setAttribute('data-step', step);
    const textEl = stepEl.querySelector('.progress-step-text');
    if (textEl) {
      // ✅ v1.0.0: Usar tiempo total desde inicio del item
      const elapsed = processingItem?.itemStartTime 
        ? Math.floor((Date.now() - processingItem.itemStartTime) / 1000)
        : 0;
      textEl.textContent = `${message} (${elapsed}s)`;
    }
  }
}

// ✅ v10.5.53: Actualizar tiempo cada segundo
// ✅ v1.0.0: Usa itemStartTime (tiempo total) en vez de stepStartTime
let _stepTimeInterval = null;
function startStepTimeUpdater() {
  if (_stepTimeInterval) return;
  _stepTimeInterval = setInterval(() => {
    const processingItem = STATE.items.find(i => i.status === 'processing');
    if (!processingItem?.itemStartTime) return;
    
    const stepEl = document.getElementById('progress-step-message');
    const textEl = stepEl?.querySelector('.progress-step-text');
    if (textEl && processingItem.stepMessage) {
      // ✅ v1.0.0: Tiempo total desde inicio del item
      const elapsed = Math.floor((Date.now() - processingItem.itemStartTime) / 1000);
      textEl.textContent = `${processingItem.stepMessage} (${elapsed}s)`;
    }
  }, 1000);
}

function stopStepTimeUpdater() {
  if (_stepTimeInterval) {
    clearInterval(_stepTimeInterval);
    _stepTimeInterval = null;
  }
}

/**
 * Notificar inicio de publicación
 * @param {number} total - Total de anuncios a publicar
 * @param {boolean} isResume - ✅ v1.1.0: Si es reanudación, cargar datos de BD
 */
export async function startPublication(total, isResume = false) {
  
  STATE.isActive = true;
  STATE.total = total;
  STATE.processed = 0;
  STATE.filter = 'pending'; // ✅ v10.5.52: Default a pendientes
  
  // ✅ v1.1.0: Si es reanudación, cargar datos existentes de BD
  if (isResume) {
    await loadState();
  } else {
    STATE.items = [];
  }
  
  showTab();
  render();
  updateBadge();
  
  // Iniciar timer
  startTimer();
  
  // ✅ v10.5.53: Iniciar actualizador de tiempo de paso
  startStepTimeUpdater();
  
  // ✅ v7.2.9: Cambiar a tab "En Curso" automáticamente
  showProgressTab();
}

/**
 * Notificar fin de publicación
 */
export function finishPublication() {
  
  STATE.isActive = false;
  
  // Detener timer
  stopTimer();
  
  // ✅ v10.5.53: Detener actualizador de tiempo de paso
  stopStepTimeUpdater();
  
  // Mantener datos pero ocultar tab tras 30 segundos
  setTimeout(() => {
    if (!STATE.isActive) {
      hideTab();
    }
  }, 30000);
  
  updateBadge();
}

// ============================================
// RENDERIZADO
// ============================================

// ============================================
// ✅ v10.5.82: RENDERIZADO INDEPENDIENTE DEL CONTENEDOR FIJO
// ============================================

/**
 * ✅ v10.5.82: Actualizar SOLO el contenedor fijo de processing
 * Se ejecuta SIEMPRE, independiente del tab activo
 * Usa transiciones suaves para evitar parpadeo
 * ✅ v10.5.132: No ocultar durante gaps entre items
 */
function renderProcessingFixed() {
  const processingContainer = document.getElementById('progress-processing-fixed');
  if (!processingContainer) return;
  
  const processingItems = STATE.items.filter(i => i.status === 'processing');
  const currentProcessingId = processingItems[0]?.id || null;
  
  // ✅ Detectar si cambió el anuncio que se está procesando
  const itemChanged = currentProcessingId !== STATE._lastProcessingId;
  
  if (processingItems.length > 0) {
    const newHTML = processingItems.map(item => renderItem(item)).join('');
    
    // ✅ Solo actualizar DOM si el contenido cambió
    if (processingContainer.innerHTML !== newHTML) {
      if (itemChanged && STATE._lastProcessingId !== null) {
        // ✅ Transición suave al cambiar de anuncio
        processingContainer.classList.add('transitioning');
        setTimeout(() => {
          processingContainer.innerHTML = newHTML;
          processingContainer.classList.remove('transitioning');
          addImageErrorHandlers(processingContainer);
        }, 150); // Fade out -> update -> fade in
      } else {
        // ✅ Actualización directa (mismo anuncio, solo cambió stepMessage)
        processingContainer.innerHTML = newHTML;
        addImageErrorHandlers(processingContainer);
      }
    }
    
    STATE._lastProcessingId = currentProcessingId;
    STATE._lastProcessingTime = Date.now(); // ✅ v10.5.132: Marcar tiempo
  } else {
    // No hay items procesando
    // ✅ v10.5.132: No ocultar inmediatamente si la publicación está activa
    // ✅ v1.0.0: Tampoco ocultar si hay errores pendientes de reintento
    const timeSinceLastProcessing = Date.now() - (STATE._lastProcessingTime || 0);
    const hasErrors = STATE.items.filter(i => i.status === 'error_retry').length > 0;
    const hasPending = STATE.items.filter(i => i.status === 'pending' || i.status === 'queued_delete').length > 0;
    const shouldKeepVisible = STATE.isActive && timeSinceLastProcessing < 3000; // 3 segundos de gracia
    
    // ✅ v1.0.0: Mantener visible si hay errores o pendientes (proceso pausado)
    if (!shouldKeepVisible && !hasErrors && !hasPending && processingContainer.innerHTML !== '') {
      processingContainer.classList.add('transitioning');
      setTimeout(() => {
        // ✅ v10.5.132: Verificar de nuevo antes de ocultar
        const stillNoProcessing = STATE.items.filter(i => i.status === 'processing').length === 0;
        const stillNoErrors = STATE.items.filter(i => i.status === 'error_retry').length === 0;
        const stillNoPending = STATE.items.filter(i => i.status === 'pending' || i.status === 'queued_delete').length === 0;
        if (stillNoProcessing && stillNoErrors && stillNoPending && !STATE.isActive) {
          processingContainer.innerHTML = '';
        }
        processingContainer.classList.remove('transitioning');
      }, 150);
      STATE._lastProcessingId = null;
    }
  }
}

// ✅ v1.3.0: updateResumeButton y handleResumeClick ELIMINADOS
// Ahora gestionados por ButtonManager (REGLA #41)

/**
 * ✅ v10.5.82: Añadir handlers de error a imágenes
 */
function addImageErrorHandlers(container) {
  container.querySelectorAll('img.progress-item-image').forEach(img => {
    img.addEventListener('error', function() {
      this.style.display = 'none';
      const fallback = this.nextElementSibling;
      if (fallback && fallback.classList.contains('progress-item-image-fallback')) {
        fallback.style.display = 'flex';
      }
    });
  });
}

/**
 * ✅ v10.5.82: Renderizar SOLO la lista de items (sin el contenedor fijo)
 * Para llamadas desde handleProgressUpdate cuando el tab está activo
 */
/**
 * ✅ v2.4.4: Adjuntar event listeners a botones "Cargar más"
 * Usa event delegation para cumplir con CSP (Content Security Policy)
 * Según guía: addEventListener en vez de onclick inline
 */
function attachLoadMoreListeners(container) {
  const buttons = container.querySelectorAll('button[data-action="load-more"]');
  buttons.forEach(btn => {
    btn.addEventListener('click', loadMore);
  });
}

function renderItemsListOnly() {
  const container = document.getElementById('progress-items-container');
  if (!container) return;
  
  // Actualizar contadores
  updateStats();
  
  // Aplicar filtro visual
  applyFilterVisual();
  
  // ✅ v1.3.0: Botón Reanudar ahora gestionado por ButtonManager
  // Se actualiza automáticamente via refreshAllButtons()
  
  // Actualización incremental de la lista
  const currentHTML = renderItemsList();
  
  if (container.dataset.lastRender !== currentHTML) {
    container.innerHTML = currentHTML;
    container.dataset.lastRender = currentHTML;
    addImageErrorHandlers(container);
    
    // ✅ v2.4.4: Añadir event listeners para botones "Cargar más" (CSP compliance)
    // Según guía (líneas 2719, 2777, 2882): usar addEventListener en vez de onclick inline
    attachLoadMoreListeners(container);
  }
}

/**
 * Renderizar contenido del tab
 * ✅ v10.5.82: Refactorizado para usar funciones independientes
 */
function render() {
  const container = document.getElementById('progress-items-container');
  if (!container) return;
  
  // ✅ v10.5.82: Actualizar contenedor fijo (siempre)
  renderProcessingFixed();
  
  // ✅ v10.5.82: Actualizar lista de items
  renderItemsListOnly();
}

/**
 * ✅ v10.5.49: Aplicar filtro visual a los stats
 */
function applyFilterVisual() {
  const stats = document.querySelectorAll('.progress-stat[data-filter], .progress-stat-special[data-filter]');
  stats.forEach(stat => {
    stat.classList.toggle('active', stat.dataset.filter === STATE.filter);
  });
}

/**
 * Actualizar contadores
 * ✅ v7.2.8: Añade clase 'publishing' al container si hay items processing
 * ✅ v1.0.0: Sincroniza STATE.stats con items locales para evitar desincronización
 */
function updateStats() {
  // ✅ v1.0.0: SIEMPRE calcular desde items locales para consistencia
  // Los items se actualizan en tiempo real, los stats del backend pueden estar desactualizados
  const successCount = STATE.items.filter(i => i.status === 'success' || i.status === 'published').length;
  const errorCount = STATE.items.filter(i => i.status === 'error_retry').length;
  const processingCount = STATE.items.filter(i => i.status === 'processing').length;
  const pendingCount = STATE.items.filter(i => i.status === 'pending' || i.status === 'queued_delete').length;
  
  // ✅ v1.0.0: Actualizar STATE.stats para consistencia
  STATE.stats = {
    success: successCount,
    errors: errorCount,
    processing: processingCount,
    pending: pendingCount
  };
  
  // ✅ Procesados = Solo exitosos (los que SÍ se publicaron)
  const processed = successCount;
  
  // Calcular porcentajes
  const successPct = STATE.total > 0 ? Math.round((successCount / STATE.total) * 100) : 0;
  const errorPct = STATE.total > 0 ? Math.round((errorCount / STATE.total) * 100) : 0;
  const pendingPct = STATE.total > 0 ? Math.round((pendingCount / STATE.total) * 100) : 0;
  
  // Actualizar valores con null-safety
  const processedEl = document.getElementById('progress-stat-processed');
  const processedTotalEl = document.getElementById('progress-stat-processed-total');
  const successEl = document.getElementById('progress-stat-success');
  const successPctEl = document.getElementById('progress-stat-success-pct');
  const errorsEl = document.getElementById('progress-stat-errors');
  const errorsPctEl = document.getElementById('progress-stat-errors-pct');
  const pendingEl = document.getElementById('progress-stat-pending');
  const pendingPctEl = document.getElementById('progress-stat-pending-pct');
  
  if (processedEl) processedEl.textContent = processed;
  if (processedTotalEl) processedTotalEl.textContent = `/ ${STATE.total}`;
  if (successEl) successEl.textContent = successCount;
  if (successPctEl) successPctEl.textContent = `${successPct}%`;
  if (errorsEl) errorsEl.textContent = errorCount;
  if (errorsPctEl) errorsPctEl.textContent = `${errorPct}%`;
  if (pendingEl) pendingEl.textContent = pendingCount;
  if (pendingPctEl) pendingPctEl.textContent = `${pendingPct}%`;
  
  // ✅ v7.2.8: Añadir/quitar clase 'publishing' al container
  const statsContainer = document.querySelector('.progress-stats-container, .progress-stats-container-special');
  if (statsContainer) {
    if (processingCount > 0) {
      statsContainer.classList.add('publishing');
    } else {
      statsContainer.classList.remove('publishing');
    }
  }
  
  // ✅ v2.4.9: Actualizar ButtonManager con processingCount (REGLA #41)
  // Esto deshabilita automáticamente los botones Reanudar/Cancelar durante publicación
  ButtonManager.updateState({
    processingCount: processingCount,
    pendingCount: pendingCount
  });
}

/**
 * Renderizar lista de items
 * ✅ v10.5.52: Solo 3 filtros (success, error, pending) - Sin 'all'
 */
function renderItemsList() {
  // ✅ v10.5.49: Normalizar status del backend ('published' → 'success')
  const normalizeStatus = (status) => {
    if (status === 'published') return 'success';
    return status;
  };
  
  // Separar items por estado (normalizando) - Excluir 'processing' (va en sección fija)
  const successItems = STATE.items.filter(i => normalizeStatus(i.status) === 'success');
  const errorItems = STATE.items.filter(i => normalizeStatus(i.status) === 'error_retry');
  const pendingItems = STATE.items.filter(i => i.status === 'pending' || i.status === 'queued_publish');
  
  switch (STATE.filter) {
    case 'success':
      // ✅ v10.5.52: Filtro exitosos con scroll infinito
      if (successItems.length > 0) {
        const totalSuccess = successItems.length;
        const itemsToShow = successItems.slice(0, STATE.currentPage * ITEMS_PER_PAGE);
        const hiddenCount = totalSuccess - itemsToShow.length;
        
        return `
          <div class="progress-info-notice progress-info-notice-success">
            <span class="progress-info-icon">✅</span>
            <div class="progress-info-content">
              <p class="progress-info-title">${totalSuccess} anuncios publicados</p>
              <p class="progress-info-text">Estos anuncios se han publicado correctamente en Wallapop.</p>
            </div>
          </div>
          ${itemsToShow.map(item => renderItem(item)).join('')}
          ${hiddenCount > 0 ? `
            <div class="progress-load-more">
              <button class="load-more-btn" data-action="load-more">
                Cargar más (${hiddenCount} restantes)
              </button>
            </div>
          ` : ''}
        `;
      } else {
        return `
          <div class="progress-empty">
            <span class="progress-empty-icon">📭</span>
            <p class="progress-empty-title">No hay anuncios publicados</p>
            <p class="progress-empty-help">Los anuncios exitosos aparecerán aquí</p>
          </div>
        `;
      }
      
    case 'error':
      // ✅ v10.5.50: Mostrar aviso de reintento para errores
      if (errorItems.length > 0) {
        const totalErrors = errorItems.length;
        const itemsToShow = errorItems.slice(0, STATE.currentPage * ITEMS_PER_PAGE);
        const hiddenCount = totalErrors - itemsToShow.length;
        
        return `
          <div class="progress-info-notice progress-info-notice-error">
            <span class="progress-info-icon">🔄</span>
            <div class="progress-info-content">
              <p class="progress-info-title">${totalErrors} anuncios con errores</p>
              <p class="progress-info-text">Estos anuncios se reintentarán automáticamente al finalizar el ciclo. No se cobrará crédito adicional.</p>
            </div>
          </div>
          ${itemsToShow.map(item => renderItem(item)).join('')}
          ${hiddenCount > 0 ? `
            <div class="progress-load-more">
              <button class="load-more-btn" data-action="load-more">
                Cargar más (${hiddenCount} restantes)
              </button>
            </div>
          ` : ''}
        `;
      } else {
        return `
          <div class="progress-empty">
            <span class="progress-empty-icon">✓</span>
            <p class="progress-empty-title">Sin errores</p>
            <p class="progress-empty-help">Todos los anuncios se procesaron correctamente</p>
          </div>
        `;
      }
      
    case 'pending':
    default:
      // ✅ v10.5.52: Filtro pendientes (default)
      if (pendingItems.length > 0) {
        const totalPending = pendingItems.length;
        const itemsToShow = pendingItems.slice(0, STATE.currentPage * ITEMS_PER_PAGE);
        const hiddenCount = totalPending - itemsToShow.length;
        
        return `
          ${itemsToShow.map(item => renderItem(item)).join('')}
          ${hiddenCount > 0 ? `
            <div class="progress-load-more">
              <button class="load-more-btn" data-action="load-more">
                Cargar más (${hiddenCount} restantes)
              </button>
            </div>
          ` : ''}
        `;
      } else if (STATE.stats?.pending > 0) {
        // Hay pendientes según stats pero no en items (aún no cargados)
        return `
          <div class="progress-empty">
            <span class="progress-empty-icon">⏳</span>
            <p class="progress-empty-title">Cargando anuncios pendientes...</p>
            <p class="progress-empty-help">${STATE.stats.pending} anuncios en cola se cargarán cuando empiecen a procesarse</p>
          </div>
        `;
      } else {
        return `
          <div class="progress-empty">
            <span class="progress-empty-icon">✓</span>
            <p class="progress-empty-title">No hay anuncios pendientes</p>
            <p class="progress-empty-help">Todos los anuncios en cola han sido procesados</p>
          </div>
        `;
      }
  }
}

/**
 * Renderizar item individual
 * ✅ REUTILIZADO de publishing-monitor.js
 * ✅ v7.0.3: Mejorado manejo de imágenes
 */
function renderItem(item) {
  // ✅ v10.5.33: Usar _cachedApiBase (cargado de storage en init)
  const apiBase = _cachedApiBase;
  const imageUrl = getImageUrl(item.image_url, apiBase);
  
  const imageHtml = imageUrl
    ? `<img src="${imageUrl}" alt="${item.title || 'Producto'}" class="progress-item-image" loading="lazy" />
       <div class="progress-item-image progress-item-image-fallback" style="display:none;">📷</div>`
    : `<div class="progress-item-image progress-item-image-fallback">📷</div>`;
  
  const statusText = {
    'processing': 'Procesando',
    'success': 'Exitoso',
    'published': 'Exitoso',
    'error_retry': 'Error',
    'pending': 'Pendiente',
    'queued_publish': 'Pendiente'
  }[item.status] || 'Pendiente';
  
  // ✅ v10.5.52: Normalizar status para CSS
  let statusClass = item.status;
  if (statusClass === 'error_retry') statusClass = 'error';
  if (statusClass === 'published') statusClass = 'success';
  if (statusClass === 'queued_publish') statusClass = 'pending';
  
  // ✅ v7.2.4: Mostrar precio correctamente
  let priceText = 'Sin precio';
  if (item.price > 0) {
    priceText = formatPrice(item.price);
  } else if (item.price === 0) {
    priceText = 'Gratis';
  }
  
  // Mapeo de nombres técnicos a nombres amigables
  const fieldNames = {
    'weight_tier': 'Peso del envío',
    'brand': 'Marca',
    'model': 'Modelo',
    'storage_capacity': 'Capacidad de almacenamiento',
    'condition': 'Estado/Condición',
    'color': 'Color',
    'size': 'Talla',
    'bulky_measures': 'Medidas (ancho/largo/alto)',
    'supports_shipping': 'Envío activado/desactivado'
  };
  
  let errorHtml = '';
  if (item.error) {
    // ✅ v7.2.35: Omitir error "deleted_pending_publish" (es interno, no visible al usuario)
    const isDeletedPending = item.error.includes('eliminado') || 
                             item.error.includes('deleted_pending_publish') ||
                             item.error.includes('pendiente republicar');
    
    if (!isDeletedPending) {
      if (item.missingFields && Array.isArray(item.missingFields) && item.missingFields.length > 0) {
        const missingList = item.missingFields
          .map(f => fieldNames[f] || f)
          .join(', ');
        errorHtml = `<div class="progress-item-error">❌ Faltan campos: ${missingList}</div>`;
      } else {
        errorHtml = `<div class="progress-item-error">❌ ${item.error}</div>`;
      }
    }
  }
  
  return `
    <div class="progress-item ${statusClass}" data-item-id="${item.id}">
      ${imageHtml}
      <div class="progress-item-info">
        <div class="progress-item-title" title="${item.title || ''}">${truncateTitle(item.title, 50)}</div>
        <div class="progress-item-price">${priceText}</div>
        ${errorHtml}
      </div>
      ${statusClass === 'processing' ? `
        <button class="progress-pause-btn" id="btn-pause-publish" title="${STATE.isPaused ? 'Reanudar publicación' : 'Pausar publicación'}">
          ${STATE.isPaused ? '▶️' : '⏸️'}
        </button>
        <div class="progress-item-step" id="progress-step-message" data-step="${item.stepNumber || 1}">
          <span class="progress-step-spinner ${STATE.isPaused ? 'paused' : ''}"></span>
          <span class="progress-step-text">${STATE.isPaused ? 'Pausado' : (item.stepMessage || 'Iniciando...')}</span>
        </div>
      ` : `
        <div class="progress-item-status ${statusClass}">
          ${statusText}
        </div>
      `}
    </div>
  `;
}

// ============================================
// GESTIÓN DE VISIBILIDAD DEL TAB
// ============================================

/**
 * Mostrar tab "En Curso"
 */
function showTab() {
  const tabButton = document.querySelector('[data-tab="in_progress"]');
  if (tabButton) {
    tabButton.style.display = 'flex';
  }
}

/**
 * Ocultar tab "En Curso"
 */
function hideTab() {
  const tabButton = document.querySelector('[data-tab="in_progress"]');
  if (tabButton) {
    tabButton.style.display = 'none';
  }
}

/**
 * Actualizar visibilidad según estado
 */
function updateTabVisibility() {
  if (STATE.isActive) {
    showTab();
  } else {
    hideTab();
  }
}

/**
 * Actualizar badge del tab
 * ✅ v7.2.8: Añade clase 'publishing' si hay items processing
 */
function updateBadge() {
  const badge = document.getElementById('tab-in-progress-badge');
  if (!badge) return;
  
  const pendingCount = STATE.total - STATE.processed;
  const processingCount = STATE.items.filter(i => i.status === 'processing').length;
  
  if (pendingCount > 0) {
    badge.textContent = pendingCount;
    badge.style.display = 'inline-flex';
    
    // ✅ v7.2.8: Animación si hay publicación activa
    if (processingCount > 0) {
      badge.classList.add('publishing');
    } else {
      badge.classList.remove('publishing');
    }
  } else {
    badge.style.display = 'none';
    badge.classList.remove('publishing');
  }
}

// ============================================
// FILTRADO POR CLICK EN CONTADORES
// ============================================

/**
 * Inicializar listeners de contadores clickeables
 */
function initStatsListeners() {
  const stats = document.querySelectorAll('.progress-stat[data-filter], .progress-stat-special[data-filter]');
  stats.forEach(stat => {
    stat.addEventListener('click', () => {
      const filter = stat.dataset.filter;
      setFilter(filter);
    });
  });
}

// ✅ v2.2.2: Listener global para botón pausa (solo se registra una vez)
let _pauseListenerInitialized = false;
function initPauseListener() {
  if (_pauseListenerInitialized) return;
  _pauseListenerInitialized = true;
  
  document.addEventListener('click', (e) => {
    const pauseBtn = e.target.closest('#btn-pause-publish');
    if (pauseBtn) {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
      togglePause();
    }
  }, true); // Captura en fase de captura
}

/**
 * Establecer filtro activo
 * ✅ v7.2.52: Resetea currentPage para scroll infinito
 */
function setFilter(filter) {
  STATE.filter = filter;
  STATE.currentPage = 1; // ✅ v7.2.52: Resetear página al cambiar filtro
  
  // Actualizar UI de stats
  const stats = document.querySelectorAll('.progress-stat[data-filter], .progress-stat-special[data-filter]');
  stats.forEach(stat => {
    stat.classList.toggle('active', stat.dataset.filter === filter);
  });
  
  // Re-renderizar lista
  render();
}

// ============================================
// TIMER Y MÉTRICAS EN TIEMPO REAL
// ============================================

let timerInterval = null;

/**
 * Iniciar timer de tiempo transcurrido
 */
function startTimer() {
  if (timerInterval) clearInterval(timerInterval);
  
  STATE.startTime = Date.now();
  
  timerInterval = setInterval(() => {
    updateTimeMetrics();
  }, 1000); // ✅ Actualizar cada segundo
  
  // Primera actualización inmediata
  updateTimeMetrics();
}

/**
 * Detener timer
 */
function stopTimer() {
  if (timerInterval) {
    clearInterval(timerInterval);
    timerInterval = null;
  }
}

/**
 * Actualizar métricas de tiempo
 */
function updateTimeMetrics() {
  if (!STATE.startTime || !STATE.isActive) return;
  
  const elapsed = Math.floor((Date.now() - STATE.startTime) / 1000); // segundos
  const minutes = Math.floor(elapsed / 60);
  const seconds = elapsed % 60;
  
  // Tiempo transcurrido
  const timeElapsedEl = document.getElementById('progress-time-elapsed');
  if (timeElapsedEl) {
    timeElapsedEl.textContent = `${minutes}m ${seconds}s`;
  }
  
  // ✅ v10.5.26: Velocidad mejorada - no mostrar valores absurdos
  let speedText = '—';
  let speed = 0;
  
  if (STATE.processed > 0) {
    speed = (STATE.processed / elapsed) * 60; // anuncios por minuto
    
    // Solo mostrar velocidad si tenemos suficientes datos (al menos 2 anuncios o 30s)
    if (STATE.processed >= 2 || elapsed >= 30) {
      if (elapsed < 60) {
        speedText = `~${speed.toFixed(1)}/min`;
      } else {
        speedText = `${speed.toFixed(1)}/min`;
      }
    } else {
      speedText = 'Calculando...';
    }
  } else if (elapsed < 30) {
    speedText = 'Iniciando...';
  } else {
    speedText = '0/min';
  }
  
  const speedEl = document.getElementById('progress-speed');
  if (speedEl) {
    speedEl.textContent = speedText;
  }
  
  // ✅ v7.2.10: ETA mejorado
  const pending = STATE.total - STATE.processed;
  const etaElement = document.getElementById('progress-eta');
  
  if (etaElement) {
    if (pending === 0) {
      etaElement.textContent = '¡Completado!';
    } else if (STATE.processed === 0) {
      // Sin procesar aún: mostrar "Calculando..."
      etaElement.textContent = 'Calculando...';
    } else if (speed > 0) {
      // Calcular ETA en minutos
      const etaMinutes = Math.ceil(pending / speed);
      
      if (etaMinutes < 1) {
        etaElement.textContent = '<1min';
      } else if (etaMinutes === 1) {
        etaElement.textContent = '~1min';
      } else {
        etaElement.textContent = `~${etaMinutes}min`;
      }
    } else {
      etaElement.textContent = '--';
    }
  }
  
  // Barra de progreso visual (si existe)
  const percentage = STATE.total > 0 ? Math.round((STATE.processed / STATE.total) * 100) : 0;
  const barFill = document.getElementById('progress-bar-fill');
  const barText = document.getElementById('progress-bar-text');
  if (barFill) barFill.style.width = `${percentage}%`;
  if (barText) barText.textContent = `${percentage}%`;
}

// ============================================
// DELAY SLIDER
// ============================================

/** Cleanup function para el slider */
let sliderCleanup = null;

/**
 * Inicializar slider de delay
 * ✅ v10.5.34: Usa componente centralizado delay-slider.js
 * Sincronizado con slider de Config en tiempo real
 */
async function initDelaySlider() {
  // Limpiar slider anterior si existe
  if (sliderCleanup) {
    sliderCleanup();
  }

  // ✅ v10.5.34: Usar componente centralizado
  sliderCleanup = initDelaySliderComponent({
    sliderId: 'progress-delay-slider',
    displayId: 'progress-delay-value',
    messageId: 'progress-delay-message',
    showToast: false,  // No mostrar toast en pestaña Progress (solo "✓")
    autoSave: true
  });
  
  // Actualizar STATE.delay con el valor actual
  STATE.delay = getCurrentDelay();
}

// ============================================
// ✅ v7.2.52: SCROLL INFINITO - Cargar más items
// ============================================

function loadMore() {
  STATE.currentPage++;
  render();
}

// ============================================
// ✅ v2.2.2: Listener para sincronizar estado de pausa
// ============================================

chrome.runtime.onMessage.addListener((msg) => {
  if (msg?.type === 'PUBLISH.PAUSED_CHANGED') {
    STATE.isPaused = !!msg.paused;
    renderProcessingFixed();
    
    // Si se reanudó desde Wallapop, ocultar modal allí también
    if (!msg.paused) {
      chrome.tabs.query({ url: '*://*.wallapop.com/*' }).then(tabs => {
        if (tabs.length > 0) {
          chrome.tabs.sendMessage(tabs[0].id, { type: 'HIDE_PAUSE_OVERLAY' }).catch(() => {});
        }
      });
    }
  }
});

// ============================================
// EXPORTAR
// ============================================

export const ProgressTab = {
  init: initProgressTab,
  switchAccount,
  clear: clearProgressTab,
  resetPublishState,  // ✅ v10.5.144: Limpieza completa centralizada
  handleProgressUpdate,
  startPublication,
  finishPublication,
  updateStepMessage,  // ✅ v10.5.53: Mensaje de paso dinámico
  loadMore            // ✅ v7.2.52: Exportar para onclick
};

// ✅ v2.4.1: Exponer a window para onclick handlers
window.ProgressTab = ProgressTab;

