/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/polling.js – Rol: Polling de estado de runs
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/
import { toast } from '../utils.js';
import { logger } from './logger.js';
import ResourceManager from './resource-manager.js';

import { sw } from './runs.js';
import { getActiveAccountId } from './auth.js';

// Timer interno del módulo (ahora gestionado por ResourceManager)
let _timerId = null;

/**
 * Arranca un polling sencillo llamando a `callback` cada `intervalMs`.
 * Ignora arranques duplicados.
 */
export function startPolling(callback, intervalMs = 5000) {
  if (_timerId) return;
  // Ejecuta con try/catch y permite callback async
  _timerId = ResourceManager.addInterval(() => {
    Promise.resolve()
      .then(() => callback && callback())
      .catch(() => { /* silencio */ });
  }, Number(intervalMs) || 5000, 'polling-runs');
}

/** Detiene el polling si está activo. */
export function stopPolling() {
  if (_timerId) {
    ResourceManager.remove(_timerId);
    _timerId = null;
  }
}

/** Alias semántico. */
export function cancelPolling() {
  stopPolling();
}

/**
 * Muestra un aviso si el último run quedó incompleto y hay pendientes.
 * (Usado tras login/arranque)
 */
export async function checkLastRunAndWarn() {
  try {
    const r = await sw({ type: 'RUN.LAST', account_id: getActiveAccountId() });
    if (!r?.ok || !r?.data?.exists) {
      // No hay run previo, resetear a idle
      const { setBackupState } = await import('./state.js');
      setBackupState({ status: 'idle', pending: 0, errors: 0, seen: 0, total: 0, run_id: null });
      return;
    }

    const { run_id, status, total, seen, pending, errors } = r.data;
    const st = String(status || '').toLowerCase();


    // ✅ Actualizar estado centralizado (trigger automático de UI)
    const { setBackupState } = await import('./state.js');
    setBackupState({ 
      status: st, 
      pending: Number(pending || 0), 
      errors: Number(errors || 0),
      seen: Number(seen || 0),
      total: Number(total || 0),
      run_id
    });
    

    // ✅ Toast informativo si hay pendientes
    if (pending > 0 && (st === 'cancelled' || st === 'incomplete' || st === 'error')) {
      toast(`Último backup incompleto: ${seen}/${total}. Pulsa el botón Backup para reanudar.`, 'warn');
    }
  } catch {
    // silencio
  }
}

/** Inicialización opcional del módulo (si el orquestador la invoca) */
export function init() {
  // Nada que hacer por defecto; el orquestador decide cuándo arrancar un polling concreto.
  return;
}
