/**
 * Módulo de tour guiado interactivo
 * @module onboarding/tour
 * 
 * Tour con spotlight que aparece solo la primera vez.
 * Destaca elementos clave del panel paso a paso.
 * 
 * ✅ Sigue REGLA #7: Usa sleep de utils.js
 * ✅ Sigue REGLA #4: Usa logger.js
 * ✅ v10.5.148: Tour inicial completo
 */

import { logger } from '../logger.js';
import { sleep } from '../../utils.js';
import { TOUR_STEPS } from './config.js';
import * as State from './state.js';

const filepath = 'onboarding/tour.js';

// ============================================
// ESTADO DEL TOUR
// ============================================

let currentStep = 0;
let overlay = null;
let tooltip = null;

// ============================================
// INICIALIZACIÓN
// ============================================

/**
 * Iniciar tour si es primera vez
 * @returns {Promise<boolean>} true si se inició el tour
 */
export async function init() {
  const completed = await State.isTourCompleted();
  
  if (completed) {
    logger.debug(`[${filepath}] Tour ya completado`);
    return false;
  }

  logger.debug(`[${filepath}] Iniciando tour...`);
  await start();
  return true;
}

/**
 * Iniciar tour
 */
export async function start() {
  State.setTourActive(true);
  currentStep = 0;
  
  createOverlay();
  await sleep(100); // Esperar animación de entrada
  await showStep(0);
}

// ============================================
// OVERLAY
// ============================================

/**
 * Crear overlay oscuro
 */
function createOverlay() {
  overlay = document.createElement('div');
  overlay.className = 'ml-tour-overlay';
  overlay.id = 'tour-overlay';
  document.body.appendChild(overlay);
  
  // Animación de entrada
  setTimeout(() => overlay.classList.add('active'), 10);
}

// ============================================
// PASOS DEL TOUR
// ============================================

/**
 * Mostrar un paso del tour
 */
async function showStep(stepIndex) {
  if (stepIndex >= TOUR_STEPS.length) {
    await complete();
    return;
  }

  const step = TOUR_STEPS[stepIndex];
  currentStep = stepIndex;

  // Esperar a que el elemento esté visible
  const element = await waitForElement(step.selector);
  if (!element) {
    logger.error(`[${filepath}] Elemento no encontrado: ${step.selector}`);
    await next(); // Saltar al siguiente
    return;
  }

  // Highlight del elemento
  if (step.highlight) {
    highlightElement(element);
  }

  // Mostrar tooltip
  showTooltip(step, element);
}

/**
 * Esperar a que un elemento esté en el DOM
 */
function waitForElement(selector, timeout = 5000) {
  return new Promise((resolve) => {
    const element = document.querySelector(selector);
    if (element) {
      resolve(element);
      return;
    }

    const observer = new MutationObserver(() => {
      const element = document.querySelector(selector);
      if (element) {
        observer.disconnect();
        resolve(element);
      }
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true
    });

    // Timeout
    setTimeout(() => {
      observer.disconnect();
      resolve(null);
    }, timeout);
  });
}

/**
 * Destacar elemento con spotlight
 */
function highlightElement(element) {
  // Remover highlight anterior
  document.querySelectorAll('.ml-tour-spotlight').forEach(el => {
    el.classList.remove('ml-tour-spotlight');
  });

  // Añadir clase de spotlight
  element.classList.add('ml-tour-spotlight');
  
  // Hacer scroll suave al elemento
  element.scrollIntoView({ 
    behavior: 'smooth', 
    block: 'center' 
  });
}

// ============================================
// TOOLTIP DEL TOUR
// ============================================

/**
 * Mostrar tooltip del paso
 */
function showTooltip(step, element) {
  // Remover tooltip anterior
  if (tooltip) {
    tooltip.remove();
  }

  tooltip = document.createElement('div');
  tooltip.className = 'ml-tour-tooltip';
  tooltip.innerHTML = `
    <div class="ml-tour-tooltip-header">
      <span class="ml-tour-tooltip-step">${step.step} de ${step.total}</span>
      <button class="ml-tour-tooltip-close" aria-label="Saltar tour" title="Saltar tour">×</button>
    </div>
    <div class="ml-tour-tooltip-body">
      <h3 class="ml-tour-tooltip-title">${step.title}</h3>
      <p class="ml-tour-tooltip-message">${step.message}</p>
    </div>
    <div class="ml-tour-tooltip-footer">
      ${currentStep > 0 ? '<button class="ml-tour-btn ml-tour-btn-back">← Anterior</button>' : ''}
      <button class="ml-tour-btn ml-tour-btn-next">${currentStep < TOUR_STEPS.length - 1 ? 'Siguiente →' : '¡Entendido!'}</button>
    </div>
  `;

  document.body.appendChild(tooltip);

  // Posicionar tooltip
  positionTooltip(tooltip, element, step.position);

  // Event listeners
  tooltip.querySelector('.ml-tour-tooltip-close').onclick = () => skip();
  tooltip.querySelector('.ml-tour-btn-next').onclick = () => next();
  
  const btnBack = tooltip.querySelector('.ml-tour-btn-back');
  if (btnBack) {
    btnBack.onclick = () => previous();
  }

  // Animación
  setTimeout(() => tooltip.classList.add('active'), 10);
}

/**
 * Posicionar tooltip relativo al elemento
 */
function positionTooltip(tooltip, element, position) {
  const rect = element.getBoundingClientRect();
  const tooltipRect = tooltip.getBoundingClientRect();
  
  let top, left;

  switch (position) {
    case 'bottom':
      top = rect.bottom + 16;
      left = rect.left + (rect.width / 2) - (tooltipRect.width / 2);
      break;
    case 'top':
      top = rect.top - tooltipRect.height - 16;
      left = rect.left + (rect.width / 2) - (tooltipRect.width / 2);
      break;
    case 'right':
      top = rect.top + (rect.height / 2) - (tooltipRect.height / 2);
      left = rect.right + 16;
      break;
    case 'left':
      top = rect.top + (rect.height / 2) - (tooltipRect.height / 2);
      left = rect.left - tooltipRect.width - 16;
      break;
    default:
      top = rect.bottom + 16;
      left = rect.left;
  }

  // Ajustar si se sale de la pantalla
  const maxLeft = window.innerWidth - tooltipRect.width - 16;
  const maxTop = window.innerHeight - tooltipRect.height - 16;
  
  left = Math.max(16, Math.min(left, maxLeft));
  top = Math.max(16, Math.min(top, maxTop));

  tooltip.style.top = `${top}px`;
  tooltip.style.left = `${left}px`;
}

// ============================================
// NAVEGACIÓN
// ============================================

/**
 * Ir al siguiente paso
 */
async function next() {
  await showStep(currentStep + 1);
}

/**
 * Volver al paso anterior
 */
async function previous() {
  if (currentStep > 0) {
    await showStep(currentStep - 1);
  }
}

/**
 * Saltar tour
 */
export async function skip() {
  logger.debug(`[${filepath}] Tour saltado en paso ${currentStep + 1}`);
  await complete();
}

/**
 * Completar tour
 */
async function complete() {
  State.setTourActive(false);
  
  // Marcar como completado
  await State.markTourCompleted();
  
  // Limpiar UI primero y ESPERAR a que termine
  await cleanupAsync();
  
  logger.debug(`[${filepath}] Tour completado`);
  
  // ✅ Inicializar resto del sistema ahora que el tour terminó
  // Importación dinámica para evitar ciclos
  const { init: initTooltips } = await import('./tooltips.js');
  const { init: initHelp } = await import('./help.js');
  const { ensureHintsContainer, evaluateAndShowHint } = await import('./hints.js');
  
  await initTooltips();
  initHelp();
  
  // Inicializar hints contextuales
  ensureHintsContainer();
  await evaluateAndShowHint();
}

// ============================================
// LIMPIEZA
// ============================================

/**
 * Limpiar elementos del tour de forma asíncrona
 */
async function cleanupAsync() {
  return new Promise(resolve => {
    // Remover clase active inmediatamente
    if (overlay) {
      overlay.classList.remove('active');
    }
    if (tooltip) {
      tooltip.classList.remove('active');
    }
    
    // Esperar la transición y eliminar
    setTimeout(() => {
      if (overlay) {
        overlay.remove();
        overlay = null;
      }
      if (tooltip) {
        tooltip.remove();
        tooltip = null;
      }
      
      // Remover spotlights
      document.querySelectorAll('.ml-tour-spotlight').forEach(el => {
        el.classList.remove('ml-tour-spotlight');
      });
      
      resolve();
    }, 350); // Un poco más que la transición (300ms)
  });
}

/**
 * Limpiar elementos del tour (versión síncrona para skip/cancel)
 */
function cleanup() {
  // Remover overlay
  if (overlay) {
    overlay.classList.remove('active');
    setTimeout(() => {
      if (overlay) overlay.remove();
    }, 300);
    overlay = null;
  }

  // Remover tooltip
  if (tooltip) {
    tooltip.classList.remove('active');
    setTimeout(() => {
      if (tooltip) tooltip.remove();
    }, 300);
    tooltip = null;
  }

  // Remover spotlights
  document.querySelectorAll('.ml-tour-spotlight').forEach(el => {
    el.classList.remove('ml-tour-spotlight');
  });
}

// ============================================
// API PÚBLICA
// ============================================

/**
 * Resetear tour (para testing)
 */
export async function reset() {
  await State.resetTour();
  logger.debug(`[${filepath}] Tour reseteado`);
}
