/**
 * Módulo de tooltips interactivos
 * @module onboarding/tooltips
 * 
 * Gestiona badges pulsantes que aparecen en botones importantes la primera vez.
 * Se ocultan automáticamente al hacer click en el elemento.
 * 
 * ✅ Sigue REGLA #4: Usa logger.js
 * ✅ v10.5.147: Tooltips con badges pulsantes
 */

import { logger } from '../logger.js';
import { TOOLTIP_CONFIG, TIMEOUTS } from './config.js';
import * as State from './state.js';

const filepath = 'onboarding/tooltips.js';

// ============================================
// INICIALIZACIÓN
// ============================================

/**
 * Inicializar tooltips mejorados con badges
 */
export async function init() {
  logger.debug(`[${filepath}] Inicializando tooltips...`);
  
  // NO inicializar tooltips si el tour está activo
  if (State.isTourActive()) {
    logger.debug(`[${filepath}] Tour activo, tooltips pospuestos`);
    return;
  }
  
  // Añadir badge a tooltips no vistos
  for (const [key, config] of Object.entries(TOOLTIP_CONFIG)) {
    const wasSeen = await State.isTooltipSeen(key);
    
    if (!wasSeen) {
      addTooltipBadge(config);
    }
  }
}

// ============================================
// BADGES
// ============================================

/**
 * Añadir badge pulsante a un elemento
 */
function addTooltipBadge(config) {
  let attempts = 0;
  const maxAttempts = Math.floor(TIMEOUTS.TOOLTIP_RETRY / TIMEOUTS.TOOLTIP_WAIT);
  
  // Esperar a que el elemento esté en el DOM
  const checkElement = () => {
    attempts++;
    
    const element = document.querySelector(config.selector);
    if (!element) {
      // Si no ha superado los intentos, reintentar
      if (attempts < maxAttempts) {
        setTimeout(checkElement, TIMEOUTS.TOOLTIP_WAIT);
      } else {
        // Timeout - elemento probablemente no existe o está oculto
        logger.debug(`[${filepath}] Elemento no disponible para tooltip: ${config.selector} (puede estar oculto)`);
      }
      return;
    }

    // Verificar que no tenga ya un badge
    if (element.querySelector('.ml-tooltip-badge')) {
      return;
    }

    // Crear badge
    const badge = document.createElement('span');
    badge.className = 'ml-tooltip-badge';
    badge.setAttribute('data-tooltip-id', config.id);
    
    // Posicionar el badge - el padre debe ser relative
    const parent = element.parentElement;
    if (parent) {
      const computedStyle = window.getComputedStyle(parent);
      if (computedStyle.position === 'static') {
        parent.style.position = 'relative';
      }
      parent.appendChild(badge);
    }

    // Mejorar el tooltip nativo
    const originalTitle = element.getAttribute('title') || '';
    element.setAttribute('title', config.message);
    element.setAttribute('data-original-title', originalTitle);

    // Marcar como visto al hacer click
    element.addEventListener('click', async () => {
      await State.markTooltipSeen(config.id);
      badge.remove();
    }, { once: true });

    logger.debug(`[${filepath}] Badge añadido: ${config.selector}`);
  };

  checkElement();
}
