/**
 * Estado del módulo de onboarding
 * @module onboarding/state
 * 
 * Fuente única de verdad para el estado del sistema de onboarding.
 * Gestiona hint actual, tooltips vistos y configuración del tour.
 * 
 * ✅ Sigue REGLA #11: Estado centralizado en UN solo lugar
 * ✅ v10.5.146: Sistema de onboarding fase 1
 */

import { logger } from '../logger.js';
import { STORAGE_KEYS } from './config.js';

const filepath = 'onboarding/state.js';

// ============================================
// ESTADO INTERNO
// ============================================

let currentHint = null;
let tourActive = false;

// ============================================
// GETTERS
// ============================================

/**
 * Obtener hint actualmente mostrado
 * @returns {string|null} ID del hint actual
 */
export function getCurrentHint() {
  return currentHint;
}

/**
 * Verificar si el tour está activo
 * @returns {boolean}
 */
export function isTourActive() {
  return tourActive;
}

// ============================================
// SETTERS
// ============================================

/**
 * Establecer hint actual
 * @param {string|null} hintId - ID del hint
 */
export function setCurrentHint(hintId) {
  currentHint = hintId;
  logger.debug(`[${filepath}] Hint actual: ${hintId || 'ninguno'}`);
}

/**
 * Limpiar hint actual
 */
export function clearCurrentHint() {
  currentHint = null;
}

/**
 * Establecer estado del tour
 * @param {boolean} active
 */
export function setTourActive(active) {
  tourActive = !!active;
  logger.debug(`[${filepath}] Tour activo: ${tourActive}`);
}

// ============================================
// STORAGE (chrome.storage.local)
// ============================================

/**
 * Verificar si un hint fue cerrado recientemente
 * @param {string} hintId - ID del hint
 * @param {number|null} hoursValid - Horas válidas (null = indefinido)
 * @returns {Promise<boolean>}
 */
export async function isHintClosed(hintId, hoursValid = null) {
  const key = `ml-hint-${hintId}-closed`;
  const result = await chrome.storage.local.get([key]);
  const closedAt = result[key];
  
  if (!closedAt) return false;
  if (!hoursValid) return true;
  
  const hoursSince = (Date.now() - closedAt) / (1000 * 60 * 60);
  return hoursSince < hoursValid;
}

/**
 * Marcar hint como cerrado
 * @param {string} hintId
 */
export async function markHintClosed(hintId) {
  const key = `ml-hint-${hintId}-closed`;
  await chrome.storage.local.set({ [key]: Date.now() });
  logger.debug(`[${filepath}] Hint cerrado: ${hintId}`);
}

/**
 * Verificar si hint está deshabilitado permanentemente
 * @param {string} hintId
 * @returns {Promise<boolean>}
 */
export async function isHintDisabled(hintId) {
  const key = `ml-hint-${hintId}-disabled`;
  const result = await chrome.storage.local.get([key]);
  return result[key] === true;
}

/**
 * Deshabilitar hint permanentemente
 * @param {string} hintId
 */
export async function disableHint(hintId) {
  const key = `ml-hint-${hintId}-disabled`;
  await chrome.storage.local.set({ [key]: true });
  logger.debug(`[${filepath}] Hint deshabilitado: ${hintId}`);
}

/**
 * Obtener días desde un timestamp guardado
 * @param {string} key - Storage key
 * @returns {Promise<number>} Días transcurridos (Infinity si no existe)
 */
export async function getDaysSince(key) {
  const result = await chrome.storage.local.get([key]);
  const timestamp = result[key];
  
  if (!timestamp) return Infinity;
  
  const days = (Date.now() - timestamp) / (1000 * 60 * 60 * 24);
  return Math.floor(days);
}

/**
 * Actualizar timestamp de última sincronización
 */
export async function updateLastSync() {
  await chrome.storage.local.set({ [STORAGE_KEYS.LAST_SYNC]: Date.now() });
  logger.debug(`[${filepath}] Actualizado timestamp de sync`);
}

/**
 * Actualizar timestamp de última publicación
 */
export async function updateLastPublish() {
  await chrome.storage.local.set({ [STORAGE_KEYS.LAST_PUBLISH]: Date.now() });
  logger.debug(`[${filepath}] Actualizado timestamp de publish`);
}

// ============================================
// TOOLTIPS
// ============================================

/**
 * Verificar si tooltip fue visto
 * @param {string} tooltipId
 * @returns {Promise<boolean>}
 */
export async function isTooltipSeen(tooltipId) {
  const key = STORAGE_KEYS[`TOOLTIP_${tooltipId.toUpperCase()}_SEEN`];
  if (!key) return false;
  
  const result = await chrome.storage.local.get([key]);
  return result[key] === true;
}

/**
 * Marcar tooltip como visto
 * @param {string} tooltipId
 */
export async function markTooltipSeen(tooltipId) {
  const key = STORAGE_KEYS[`TOOLTIP_${tooltipId.toUpperCase()}_SEEN`];
  if (!key) return;
  
  await chrome.storage.local.set({ [key]: true });
  logger.debug(`[${filepath}] Tooltip visto: ${tooltipId}`);
}

/**
 * Resetear todos los tooltips
 */
export async function resetTooltips() {
  const tooltipKeys = Object.values(STORAGE_KEYS).filter(k => k.includes('tooltip'));
  const resetData = {};
  tooltipKeys.forEach(key => resetData[key] = false);
  await chrome.storage.local.set(resetData);
  logger.debug(`[${filepath}] Tooltips reseteados`);
}

// ============================================
// TOUR
// ============================================

/**
 * Verificar si tour fue completado
 * @returns {Promise<boolean>}
 */
export async function isTourCompleted() {
  const result = await chrome.storage.local.get([STORAGE_KEYS.TOUR_COMPLETED]);
  return result[STORAGE_KEYS.TOUR_COMPLETED] === true;
}

/**
 * Marcar tour como completado
 */
export async function markTourCompleted() {
  await chrome.storage.local.set({ [STORAGE_KEYS.TOUR_COMPLETED]: true });
  logger.debug(`[${filepath}] Tour completado`);
}

/**
 * Resetear tour
 */
export async function resetTour() {
  await chrome.storage.local.set({ [STORAGE_KEYS.TOUR_COMPLETED]: false });
  logger.debug(`[${filepath}] Tour reseteado`);
}

// ============================================
// ESTADÍSTICAS
// ============================================

/**
 * Obtener estadísticas del estado global
 * @returns {Promise<Object>}
 */
export async function getStats() {
  // Obtener del estado global del panel
  const pending = window.STATE?.pendingCount || 0;
  const totalListings = window.STATE?.listings?.length || 0;
  
  return {
    pending,
    totalListings
  };
}
