/**
 * Módulo onboarding - Orquestador principal
 * @module onboarding/index
 * 
 * API pública del módulo de sistema de ayuda y onboarding.
 * Coordina hints, tooltips, tour y modal de ayuda.
 * 
 * ✅ Sigue estructura de listings/index.js
 * ✅ Sigue REGLA #1: Importa desde módulos centralizados
 * ✅ Sigue REGLA #4: Usa logger.js
 * ✅ v10.5.146-149: Sistema completo de onboarding
 */

import { logger } from '../logger.js';
import * as State from './state.js';
import * as Hints from './hints.js';
import * as Tooltips from './tooltips.js';
import * as Tour from './tour.js';
import * as Help from './help.js';

const filepath = 'onboarding/index.js';

// ============================================
// INICIALIZACIÓN
// ============================================

/**
 * Inicializar sistema de onboarding
 */
export async function init() {
  logger.debug(`[${filepath}] Inicializando sistema de onboarding...`);
  
  // v10.5.148: Tour inicial (primera vez)
  // Si el tour se inicia, esperar a que termine antes de mostrar hints
  const tourStarted = await Tour.init();
  if (tourStarted) {
    logger.debug(`[${filepath}] Tour iniciado, hints y tooltips pospuestos`);
    // El tour bloqueará los hints hasta que termine
    // Los tooltips se inicializarán cuando el tour complete
    return;
  }
  
  // Crear contenedor de hints si no existe
  Hints.ensureHintsContainer();
  
  // Evaluar y mostrar hint contextual
  await Hints.evaluateAndShowHint();
  
  // v10.5.147: Inicializar tooltips mejorados (solo si NO hay tour activo)
  await Tooltips.init();
  
  // v10.5.149: Inicializar botón de ayuda
  Help.init();
  
  logger.debug(`[${filepath}] Sistema de onboarding inicializado`);
}

// ============================================
// API PÚBLICA - HINTS
// ============================================

/**
 * Re-evaluar hints (llamar cuando cambien los datos)
 */
export async function refresh() {
  await Hints.refresh();
}

/**
 * Actualizar timestamp de última sincronización
 */
export async function updateLastSync() {
  await State.updateLastSync();
}

/**
 * Actualizar timestamp de última publicación
 */
export async function updateLastPublish() {
  await State.updateLastPublish();
}

// ============================================
// API PÚBLICA - TOOLTIPS
// ============================================

/**
 * Resetear tooltips (para testing)
 */
export async function resetTooltips() {
  await State.resetTooltips();
  logger.debug(`[${filepath}] Tooltips reseteados - recarga para ver badges`);
}

// ============================================
// API PÚBLICA - TOUR
// ============================================

/**
 * Iniciar tour manualmente
 */
export async function startTour() {
  await Tour.start();
}

/**
 * Resetear tour (para testing)
 */
export async function resetTour() {
  await Tour.reset();
  logger.debug(`[${filepath}] Tour reseteado - recarga para iniciarlo`);
}

/**
 * Saltar tour
 */
export async function skipTour() {
  await Tour.skip();
}

// ============================================
// API PÚBLICA - AYUDA
// ============================================

/**
 * Abrir modal de ayuda
 */
export function openHelp() {
  Help.open();
}

/**
 * Cerrar modal de ayuda
 */
export function closeHelp() {
  Help.close();
}
