/**
 * Módulo de hints contextuales
 * @module onboarding/hints
 * 
 * Gestiona los hints inteligentes que aparecen según el contexto del usuario.
 * Solo muestra 1 hint a la vez, priorizando por importancia.
 * 
 * ✅ Sigue REGLA #1: Importa desde módulos centralizados
 * ✅ Sigue REGLA #4: Usa logger.js
 * ✅ Sigue REGLA #11: Todo relacionado con hints en este módulo
 * ✅ v10.5.146: Sistema de onboarding fase 1
 */

import { logger } from '../logger.js';
import { HINT_CONFIG, THRESHOLDS } from './config.js';
import * as State from './state.js';
import * as Actions from './actions.js';

const filepath = 'onboarding/hints.js';

// ============================================
// EVALUACIÓN DE HINTS
// ============================================

/**
 * Evaluar contexto y mostrar el hint más prioritario
 */
export async function evaluateAndShowHint() {
  try {
    // No mostrar hints durante el tour
    if (State.isTourActive()) {
      logger.debug(`[${filepath}] Tour activo, hints pospuestos`);
      return;
    }

    // Obtener estadísticas
    const stats = await State.getStats();
    
    // Evaluar hints por prioridad
    const hints = [
      await evaluatePendingHint(stats),
      await evaluateWelcomeHint(stats),
      await evaluateBackupHint(stats),
      await evaluatePublishHint(stats)
    ];

    // Filtrar hints no aplicables
    const applicableHints = hints.filter(h => h !== null);
    
    if (applicableHints.length === 0) {
      logger.debug(`[${filepath}] No hay hints aplicables`);
      return;
    }

    // Ordenar por prioridad y mostrar el primero
    applicableHints.sort((a, b) => a.priority - b.priority);
    const hintToShow = applicableHints[0];
    
    logger.debug(`[${filepath}] Mostrando hint: ${hintToShow.id}`);
    showHint(hintToShow);
    
  } catch (error) {
    logger.error(`[${filepath}] Error evaluando hints`, { error: error.message });
  }
}

/**
 * Evaluar si mostrar hint de pendientes
 */
async function evaluatePendingHint(stats) {
  if (stats.pending <= 0) return null;
  
  const config = HINT_CONFIG.pending;
  return {
    ...config,
    message: `📊 ${stats.pending} anuncio${stats.pending > 1 ? 's' : ''} esperando publicación`,
    data: { count: stats.pending }
  };
}

/**
 * Evaluar si mostrar hint de bienvenida
 */
async function evaluateWelcomeHint(stats) {
  // Solo si tiene 0 anuncios
  if (stats.totalListings > 0) return null;
  
  // No mostrar si tour ya completado (usuario ya vio el onboarding)
  if (await State.isTourCompleted()) return null;
  
  // No mostrar si fue cerrado
  if (await State.isHintClosed('welcome')) return null;
  
  const config = HINT_CONFIG.welcome;
  return {
    ...config,
    message: '🎯 Para empezar, haz backup de tus anuncios de Wallapop.',
    data: {}
  };
}

/**
 * Evaluar si mostrar hint de backup antiguo
 */
async function evaluateBackupHint(stats) {
  // Solo si tiene anuncios
  if (stats.totalListings === 0) return null;
  
  // Verificar si está deshabilitado
  if (await State.isHintDisabled('no-backup')) return null;
  
  // Verificar si fue cerrado recientemente
  if (await State.isHintClosed('no-backup', THRESHOLDS.SNOOZE_BACKUP)) return null;
  
  const daysSinceBackup = await State.getDaysSince('ml-last-sync');
  if (daysSinceBackup < THRESHOLDS.BACKUP_DAYS) return null;
  
  const config = HINT_CONFIG['no-backup'];
  return {
    ...config,
    message: `📅 Hace ${daysSinceBackup} día${daysSinceBackup > 1 ? 's' : ''} que no haces backup`,
    data: { days: daysSinceBackup }
  };
}

/**
 * Evaluar si mostrar hint de publicación antigua
 */
async function evaluatePublishHint(stats) {
  // Solo si tiene anuncios
  if (stats.totalListings === 0) return null;
  
  // Verificar si está deshabilitado
  if (await State.isHintDisabled('no-publish')) return null;
  
  // Verificar si fue cerrado recientemente
  if (await State.isHintClosed('no-publish', THRESHOLDS.SNOOZE_PUBLISH)) return null;
  
  const daysSincePublish = await State.getDaysSince('ml-last-publish');
  if (daysSincePublish < THRESHOLDS.PUBLISH_DAYS) return null;
  
  const config = HINT_CONFIG['no-publish'];
  return {
    ...config,
    message: `📅 Hace ${daysSincePublish} día${daysSincePublish > 1 ? 's' : ''} que no publicas anuncios`,
    data: { days: daysSincePublish }
  };
}

// ============================================
// RENDERIZADO DE HINTS
// ============================================

/**
 * Mostrar un hint en el panel
 */
function showHint(hintData) {
  const container = document.getElementById('hints-container');
  if (!container) {
    logger.warn(`[${filepath}] Contenedor de hints no encontrado`);
    return;
  }

  // Limpiar hints anteriores
  container.innerHTML = '';

  const hint = document.createElement('div');
  hint.className = `ml-hint ml-hint-${hintData.color}`;
  hint.setAttribute('data-hint-id', hintData.id);
  
  // IMPORTANTE: El hint debe poder recibir clicks aunque el contenedor no
  hint.style.pointerEvents = 'auto';

  const actions = hintData.getActions(hintData.data);
  
  hint.innerHTML = `
    <div class="ml-hint-header">
      <span class="ml-hint-icon">${hintData.icon}</span>
      <span class="ml-hint-title">${hintData.title}</span>
      <button class="ml-hint-close" aria-label="Cerrar" title="Cerrar">×</button>
    </div>
    <div class="ml-hint-body">
      <p class="ml-hint-message">${hintData.message}</p>
    </div>
    <div class="ml-hint-actions">
      ${actions.map(action => `
        <button class="ml-hint-btn ${action.primary ? 'ml-hint-btn-primary' : ''}" 
                data-action="${action.action}">
          ${action.text}
        </button>
      `).join('')}
    </div>
  `;

  // Event listeners
  hint.querySelector('.ml-hint-close').onclick = () => {
    closeHint(hintData.id);
    hint.remove();
  };
  
  actions.forEach((action, i) => {
    const btn = hint.querySelectorAll('.ml-hint-btn')[i];
    btn.onclick = () => {
      handleHintAction(action.action, hintData.id);
      hint.remove();
    };
  });

  container.appendChild(hint);
  State.setCurrentHint(hintData.id);
}

/**
 * Cerrar hint
 */
async function closeHint(hintId) {
  await State.markHintClosed(hintId);
  State.clearCurrentHint();
}

/**
 * Manejar acción de hint
 */
async function handleHintAction(actionName, hintId) {
  logger.debug(`[${filepath}] Acción de hint: ${actionName}`);
  
  switch (actionName) {
    case 'resumePending':
      Actions.resumePending();
      break;
    case 'goToProgressTab':
      Actions.goToProgressTab();
      break;
    case 'doBackup':
      Actions.doBackup();
      break;
    case 'goToPublishTab':
      Actions.goToPublishTab();
      break;
    case 'snooze':
      await closeHint(hintId);
      break;
    case 'disable':
      await State.disableHint(hintId);
      break;
    default:
      logger.warn(`[${filepath}] Acción desconocida: ${actionName}`);
  }
}

// ============================================
// UTILIDADES
// ============================================

/**
 * Asegurar que existe el contenedor de hints
 */
export function ensureHintsContainer() {
  let container = document.getElementById('hints-container');
  if (!container) {
    // Crear contenedor antes del listings-section
    const listingsSection = document.getElementById('listings-section');
    if (listingsSection) {
      container = document.createElement('div');
      container.id = 'hints-container';
      // CRÍTICO: Asegurar que NUNCA bloquea clicks (con !important inline)
      container.style.setProperty('pointer-events', 'none', 'important');
      listingsSection.parentNode.insertBefore(container, listingsSection);
      logger.debug(`[${filepath}] Contenedor de hints creado`);
    } else {
      logger.error(`[${filepath}] No se pudo crear contenedor de hints`);
    }
  } else {
    // Si ya existe, asegurar que no bloquea clicks
    container.style.setProperty('pointer-events', 'none', 'important');
  }
  
  return container;
}

/**
 * Re-evaluar hints (llamar cuando cambien los datos)
 */
export async function refresh() {
  logger.debug(`[${filepath}] Refrescando hints...`);
  await evaluateAndShowHint();
}
