/**
 * Módulo de modal de ayuda
 * @module onboarding/help
 * 
 * Botón flotante (?) que abre modal con recursos de ayuda.
 * Siempre accesible para el usuario.
 * 
 * ✅ Sigue REGLA #4: Usa logger.js
 * ✅ v10.5.149: Modal de ayuda y recursos
 */

import { logger } from '../logger.js';
import * as Tour from './tour.js';

const filepath = 'onboarding/help.js';

// ============================================
// ESTADO
// ============================================

let helpButton = null;
let helpModal = null;

// ============================================
// INICIALIZACIÓN
// ============================================

/**
 * Inicializar botón de ayuda
 */
export function init() {
  logger.debug(`[${filepath}] Inicializando botón de ayuda...`);
  createHelpButton();
}

// ============================================
// BOTÓN FLOTANTE
// ============================================

/**
 * Crear botón flotante de ayuda
 */
function createHelpButton() {
  // Verificar que no exista ya (evitar duplicados)
  if (helpButton || document.getElementById('ml-help-button')) {
    logger.debug(`[${filepath}] Botón de ayuda ya existe`);
    return;
  }
  
  // Crear botón
  // NOTA: La verificación de autenticación se hace en panel.js (STATE.AUTH.state)
  // Este módulo solo se llama si el usuario está autenticado
  helpButton = document.createElement('button');
  helpButton.id = 'ml-help-button';
  helpButton.className = 'ml-help-button';
  helpButton.innerHTML = '?';
  helpButton.setAttribute('aria-label', 'Ayuda');
  helpButton.setAttribute('title', 'Ayuda y guía rápida');
  
  helpButton.onclick = () => open();
  
  document.body.appendChild(helpButton);
  logger.debug(`[${filepath}] Botón de ayuda creado`);
}

// ============================================
// MODAL
// ============================================

/**
 * Abrir modal de ayuda
 */
export function open() {
  if (helpModal) {
    helpModal.classList.add('active');
    return;
  }

  createModal();
}

/**
 * Crear modal de ayuda
 */
function createModal() {
  helpModal = document.createElement('div');
  helpModal.className = 'ml-help-modal';
  helpModal.innerHTML = `
    <div class="ml-help-modal-overlay"></div>
    <div class="ml-help-modal-content">
      <div class="ml-help-modal-header">
        <h2 class="ml-help-modal-title">📖 Guía Rápida MitikLive</h2>
        <button class="ml-help-modal-close" aria-label="Cerrar">×</button>
      </div>
      <div class="ml-help-modal-body">
        
        <div class="ml-help-section">
          <button class="ml-help-item" data-action="restart-tour">
            <span class="ml-help-icon">🚀</span>
            <div class="ml-help-text">
              <strong>Tutorial Paso a Paso</strong>
              <span>Vuelve a ver el tour de introducción</span>
            </div>
            <span class="ml-help-arrow">→</span>
          </button>
        </div>

        <div class="ml-help-section">
          <h3 class="ml-help-section-title">💡 Consejos Rápidos</h3>
          <div class="ml-help-tips">
            <div class="ml-help-tip">
              <span class="ml-help-tip-icon">💾</span>
              <p><strong>Haz backup regularmente</strong><br>
              Evita perder tus anuncios. Recomendado cada 7 días.</p>
            </div>
            <div class="ml-help-tip">
              <span class="ml-help-tip-icon">📤</span>
              <p><strong>Publica en horarios óptimos</strong><br>
              Mejor visibilidad: 9-11h y 18-21h.</p>
            </div>
            <div class="ml-help-tip">
              <span class="ml-help-tip-icon">⏳</span>
              <p><strong>Delay entre publicaciones</strong><br>
              Usa 3-5 segundos para evitar bloqueos.</p>
            </div>
            <div class="ml-help-tip">
              <span class="ml-help-tip-icon">📊</span>
              <p><strong>Monitorea "En Curso"</strong><br>
              Sigue el progreso de tus publicaciones en tiempo real.</p>
            </div>
          </div>
        </div>

        <div class="ml-help-section">
          <h3 class="ml-help-section-title">❓ Preguntas Frecuentes</h3>
          <div class="ml-help-faqs">
            <details class="ml-help-faq">
              <summary>¿Cómo hago mi primer backup?</summary>
              <p>Click en el botón <strong>💾 Backup</strong> en la parte superior. Esto guardará todos tus anuncios localmente. Es el primer paso esencial.</p>
            </details>
            <details class="ml-help-faq">
              <summary>¿Cómo publico anuncios?</summary>
              <p>1. Selecciona anuncios (checkbox)<br>
              2. Click en <strong>📤 Publicar</strong><br>
              3. Monitorea el progreso en "En Curso"</p>
            </details>
            <details class="ml-help-faq">
              <summary>¿Qué pasa si se interrumpe una publicación?</summary>
              <p>Aparecerá un hint de <strong>⚠️ Pendientes</strong>. Click en "Reanudar" para continuar donde lo dejaste.</p>
            </details>
            <details class="ml-help-faq">
              <summary>¿Puedo usar varias cuentas?</summary>
              <p>Sí, usa el <strong>selector de cuentas</strong> arriba. Cada cuenta se gestiona por separado.</p>
            </details>
            <details class="ml-help-faq">
              <summary>¿Es seguro?</summary>
              <p>Sí. MitikLive es privado y solo tú tienes acceso. Los datos se guardan localmente y en tu cuenta.</p>
            </details>
          </div>
        </div>

        <div class="ml-help-section">
          <h3 class="ml-help-section-title">⌨️ Atajos de Teclado</h3>
          <div class="ml-help-shortcuts">
            <div class="ml-help-shortcut">
              <kbd>Ctrl</kbd> + <kbd>Click</kbd>
              <span>Selección múltiple</span>
            </div>
            <div class="ml-help-shortcut">
              <kbd>Doble Click</kbd>
              <span>Editar anuncio</span>
            </div>
            <div class="ml-help-shortcut">
              <kbd>Esc</kbd>
              <span>Cerrar modales</span>
            </div>
          </div>
        </div>

      </div>
      <div class="ml-help-modal-footer">
        <p class="ml-help-version">MitikLive v2.2.9 · Ayuda y soporte</p>
      </div>
    </div>
  `;

  document.body.appendChild(helpModal);

  // Event listeners
  helpModal.querySelector('.ml-help-modal-close').onclick = () => close();
  helpModal.querySelector('.ml-help-modal-overlay').onclick = () => close();
  
  const restartTourBtn = helpModal.querySelector('[data-action="restart-tour"]');
  if (restartTourBtn) {
    restartTourBtn.onclick = () => {
      close();
      restartTour();
    };
  }

  // Animación de entrada
  setTimeout(() => helpModal.classList.add('active'), 10);
  logger.debug(`[${filepath}] Modal de ayuda abierto`);
}

/**
 * Cerrar modal de ayuda
 */
export function close() {
  if (!helpModal) return;
  
  helpModal.classList.remove('active');
  setTimeout(() => {
    helpModal.remove();
    helpModal = null;
  }, 300);
  
  logger.debug(`[${filepath}] Modal de ayuda cerrado`);
}

/**
 * Reiniciar tour
 */
async function restartTour() {
  logger.debug(`[${filepath}] Reiniciando tour desde ayuda`);
  await Tour.reset();
  await Tour.start();
}
