/*
  Archivo: scripts/panel/maintenance.js
  Rol: Verificar modo mantenimiento y bloquear panel
*/

import { logger } from './logger.js';
import { toast } from '../utils.js';
import ResourceManager from './resource-manager.js';

let maintenanceCheckIntervalId = null;

/**
 * Verificar si la extensión necesita actualización
 * ✅ SIN CACHE LOCAL - Siempre consulta al backend
 * @param {boolean} forceCheck - No usado (mantenido por compatibilidad)
 * @returns {Promise<boolean>} true si necesita actualizar
 */
export async function checkVersion(forceCheck = false) {
  try {
    const manifestData = chrome.runtime.getManifest();
    const currentVersion = manifestData.version;
    
    logger.debug(`[VERSION] Verificando versión: ${currentVersion}`);
    
    const response = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: `/api/system/version-check?current_version=${currentVersion}`,
      method: 'GET'
    });
    
    if (response?.ok && response.data) {
      const { needs_update, required_version, download_url } = response.data;
      
      logger.debug(`[VERSION] Respuesta: needs_update=${needs_update}, required=${required_version}`);
      
      if (needs_update) {
        showUpdateModal(required_version, download_url);
        return true;
      }
    }
    
    return false;
  } catch (err) {
    logger.error('[VERSION] Error verificando:', err);
    // En caso de error, NO bloquear usuario
    return false;
  }
}

/**
 * Mostrar modal de actualización requerida
 */
export function showUpdateModal(minimumVersion, downloadUrl) {
  // Remover modal anterior si existe
  const existing = document.getElementById('update-modal');
  if (existing) existing.remove();
  
  // Crear overlay
  const overlay = document.createElement('div');
  overlay.id = 'update-modal';
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.95);
    backdrop-filter: blur(10px);
    z-index: 999999;
    display: flex;
    align-items: center;
    justify-content: center;
    animation: fadeIn 0.3s ease;
  `;
  
  // Crear modal
  const modal = document.createElement('div');
  modal.style.cssText = `
    background: linear-gradient(135deg, #1a1d29 0%, #0f1419 100%);
    border: 2px solid rgba(59, 130, 246, 0.3);
    border-radius: 16px;
    padding: 32px 24px;
    max-width: 320px;
    width: 90%;
    text-align: center;
    box-shadow: 0 20px 60px rgba(59, 130, 246, 0.2);
    animation: modalPop 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
  `;
  
  modal.innerHTML = `
    <div style="font-size: 48px; margin-bottom: 16px;">🔄</div>
    <h2 class="modal-title-info">
      Actualización Requerida
    </h2>
    <p class="modal-desc">
      Versión requerida: <strong class="text-white">${minimumVersion}</strong>
    </p>
    <p class="modal-desc-sm">
      Debes usar EXACTAMENTE esta versión para continuar.
    </p>
    <a href="https://www.mitiklive.com/wallapop/descargar" target="_blank" id="update-instructions-btn" style="
      display: block;
      background: #3b82f6;
      color: #fff;
      border: none;
      padding: 12px 24px;
      border-radius: 8px;
      cursor: pointer;
      font-size: 14px;
      font-weight: 600;
      text-decoration: none;
      transition: transform 0.2s, box-shadow 0.2s;
      margin-bottom: 8px;
    ">Descargar + Instrucciones</a>
    <a href="${downloadUrl}" target="_blank" id="update-download-btn" style="
      display: block;
      background: transparent;
      color: #3b82f6;
      border: 1px solid #3b82f6;
      padding: 10px 24px;
      border-radius: 8px;
      cursor: pointer;
      font-size: 13px;
      font-weight: 600;
      text-decoration: none;
      transition: transform 0.2s, box-shadow 0.2s;
    ">Descargar Directamente</a>
  `;
  
  // Animaciones CSS
  const style = document.createElement('style');
  style.textContent = `
    @keyframes fadeIn {
      from { opacity: 0; }
      to { opacity: 1; }
    }
    @keyframes modalPop {
      from {
        opacity: 0;
        transform: scale(0.9) translateY(-20px);
      }
      to {
        opacity: 1;
        transform: scale(1) translateY(0);
      }
    }
    #update-instructions-btn:hover, #update-download-btn:hover {
      transform: scale(1.05);
      box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
    }
  `;
  
  document.head.appendChild(style);
  overlay.appendChild(modal);
  document.body.appendChild(overlay);
  
  // Event listeners
  document.getElementById('update-instructions-btn').addEventListener('click', () => {
    toast('📖 Abriendo instrucciones...', 'info');
  });
  
  document.getElementById('update-download-btn').addEventListener('click', () => {
    toast('⬇️ Descargando actualización...', 'info');
  });
}

/**
 * Mostrar modal FORZADA (anti-bypass) desde backend
 * Se re-crea automáticamente si usuario la quita
 */
export function showUpdateModalForced(minimumVersion, downloadUrl) {
  
  const createModal = () => {
    // Remover anterior
    const existing = document.getElementById('update-modal-forced');
    if (existing) existing.remove();
    
    // Crear overlay (sin poder cerrar)
    const overlay = document.createElement('div');
    overlay.id = 'update-modal-forced';
    overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.98);
      backdrop-filter: blur(15px);
      z-index: 9999999;
      display: flex;
      align-items: center;
      justify-content: center;
      animation: fadeIn 0.3s ease;
    `;
    
    // Bloquear eventos
    overlay.addEventListener('click', (e) => e.stopPropagation());
    overlay.addEventListener('contextmenu', (e) => e.preventDefault());
    
    // Modal
    const modal = document.createElement('div');
    modal.style.cssText = `
      background: linear-gradient(135deg, #1a1d29 0%, #0f1419 100%);
      border: 3px solid rgba(239, 68, 68, 0.5);
      border-radius: 16px;
      padding: 32px 24px;
      max-width: 320px;
      width: 90%;
      text-align: center;
      box-shadow: 0 20px 60px rgba(239, 68, 68, 0.3);
      animation: modalPulse 2s ease-in-out infinite;
    `;
    
    modal.innerHTML = `
      <div style="font-size: 56px; margin-bottom: 16px; animation: rotate 3s linear infinite;">🔄</div>
      <h2 class="modal-title-error">
        ACTUALIZACIÓN OBLIGATORIA
      </h2>
      <p class="modal-desc">
        Versión requerida: <strong class="text-white">${minimumVersion}</strong>
      </p>
      <p class="modal-desc-error">
        El servidor rechazó tu petición. Debes usar EXACTAMENTE esta versión.
      </p>
      <a href="https://www.mitiklive.com/wallapop/descargar" target="_blank" style="
        display: block;
        background: #ef4444;
        color: #fff;
        border: none;
        padding: 14px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 15px;
        font-weight: 700;
        text-decoration: none;
        transition: all 0.2s;
        margin-bottom: 10px;
        box-shadow: 0 4px 12px rgba(239, 68, 68, 0.4);
      ">📖 Descargar + Instrucciones</a>
      <a href="${downloadUrl}" target="_blank" style="
        display: block;
        background: transparent;
        color: #ef4444;
        border: 2px solid #ef4444;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        transition: all 0.2s;
      ">⬇️ Descargar Directamente</a>
    `;
    
    // CSS animations
    const style = document.createElement('style');
    style.textContent = `
      @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
      }
      @keyframes modalPulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.02); }
      }
      @keyframes rotate {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
      }
    `;
    
    document.head.appendChild(style);
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    
    // Bloquear teclas
    const preventKeys = (e) => {
      if (e.key === 'Escape' || e.key === 'F5' || (e.ctrlKey && e.key === 'r')) {
        e.preventDefault();
        e.stopPropagation();
        toast('⚠️ Debes actualizar la extensión', 'warning');
      }
    };
    
    document.addEventListener('keydown', preventKeys, true);
  };
  
  // Crear modal
  createModal();
  
  // Re-crear cada 2 segundos si usuario la quita (anti-bypass)
  const guardInterval = setInterval(() => {
    if (!document.getElementById('update-modal-forced')) {
      createModal();
    }
  }, 2000);
  
  // Guardar intervalo globalmente para limpieza
  window.__updateModalGuard = guardInterval;
}

/**
 * Verificar si el sistema está en mantenimiento
 * @returns {Promise<boolean>} true si está en mantenimiento (y NO es admin)
 */
export async function checkMaintenance() {
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'API.FETCH_JSON',
      url: '/api/system/maintenance',
      method: 'GET'
    });
    
    if (response?.ok && response.data) {
      const { maintenance, message } = response.data;
      
      if (maintenance) {
        
        // ✅ VERIFICAR SI ES ADMIN (bypass)
        const { AUTH } = await import('./state.js');
        const isAdmin = AUTH?.user?.role === 'admin';
        
        if (isAdmin) {
          
          // Mostrar banner/icono en header (sin bloquear)
          showMaintenanceBanner(message);
          
          return false; // NO bloquear
        }
        
        // Usuario normal → bloquear con modal
        showMaintenanceModal(message);
        return true;
      }
    }
    
    // Si NO hay mantenimiento, quitar banner si existe
    removeMaintenanceBanner();
    
    return false;
  } catch (err) {
    logger.error('[MAINTENANCE] Error verificando:', err);
    // En caso de error, asumir NO mantenimiento para no bloquear usuarios
    return false;
  }
}

/**
 * Mostrar banner de mantenimiento en header (solo para admins)
 */
function showMaintenanceBanner(message) {
  // Remover banner anterior si existe
  removeMaintenanceBanner();
  
  // Buscar header donde está el debug toggle
  const header = document.querySelector('.ml-header') || 
                 document.querySelector('header') || 
                 document.getElementById('ml-top-bar');
  
  if (!header) {
    return;
  }
  
  // Crear banner - ✅ v6.4.5: Usa clase CSS centralizada
  const banner = document.createElement('div');
  banner.id = 'ml-maintenance-banner';
  // Estilos vienen de panel.css (#ml-maintenance-banner)
  
  banner.innerHTML = `
    <span style="font-size: 14px;">⚠️</span>
    <span>Mantenimiento</span>
  `;
  
  // Hover effect - manejado por CSS (:hover)
  
  // Click para mostrar info
  banner.addEventListener('click', () => {
    toast(`🔧 Modo mantenimiento activo. Usuarios bloqueados.\n${message || ''}`, 'warning');
  });
  
  // Insertar después del debug toggle o al final del header
  const debugToggle = header.querySelector('#ml-debug-toggle') || 
                      header.querySelector('[data-debug]');
  
  if (debugToggle && debugToggle.parentElement) {
    debugToggle.parentElement.appendChild(banner);
  } else {
    header.appendChild(banner);
  }
  
}

/**
 * Quitar banner de mantenimiento
 */
function removeMaintenanceBanner() {
  const banner = document.getElementById('ml-maintenance-banner');
  if (banner) {
    banner.remove();
  }
}

/**
 * Mostrar modal de mantenimiento (bloquea todo el panel)
 */
function showMaintenanceModal(message = 'Sistema en mantenimiento. Vuelve pronto.') {
  // Remover modal anterior si existe
  const existing = document.getElementById('maintenance-modal');
  if (existing) existing.remove();
  
  // Crear overlay
  const overlay = document.createElement('div');
  overlay.id = 'maintenance-modal';
  overlay.className = 'ml-maintenance-modal-overlay';
  overlay.setAttribute('role', 'dialog');
  overlay.setAttribute('aria-modal', 'true');
  overlay.setAttribute('aria-labelledby', 'maintenance-modal-title');
  
  // Crear modal
  const modal = document.createElement('div');
  modal.className = 'ml-maintenance-modal';
  
  modal.innerHTML = `
    <div class="ml-maintenance-modal__icon">⚠️</div>
    <h2 id="maintenance-modal-title" class="ml-maintenance-modal__title">
      Mantenimiento
    </h2>
    <p class="ml-maintenance-modal__message">
      ${message}
    </p>
    <button id="maintenance-retry-btn" class="ml-maintenance-modal__btn" aria-label="Reintentar conexión">
      <span class="ml-maintenance-modal__btn-icon">🔄</span>
      Reintentar
    </button>
    <p class="ml-maintenance-modal__footer">
      El sistema estará disponible pronto
    </p>
  `;
  
  // Añadir estilos si no existen
  if (!document.getElementById('ml-maintenance-modal-styles')) {
    const styles = document.createElement('style');
    styles.id = 'ml-maintenance-modal-styles';
    styles.textContent = `
      .ml-maintenance-modal-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.95);
        backdrop-filter: blur(10px);
        z-index: 999999;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 16px;
        animation: fadeIn 0.3s ease;
      }
      
      .ml-maintenance-modal {
        background: linear-gradient(135deg, #1a1d29 0%, #0f1419 100%);
        border: 2px solid rgba(251, 191, 36, 0.3);
        border-radius: 16px;
        padding: 32px 24px;
        max-width: 400px;
        width: 100%;
        text-align: center;
        box-shadow: 0 20px 60px rgba(251, 191, 36, 0.2);
        animation: modalPop 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
      }
      
      .ml-maintenance-modal__icon {
        font-size: 56px;
        margin-bottom: 16px;
        animation: pulse 2s ease-in-out infinite;
        line-height: 1;
      }
      
      .ml-maintenance-modal__title {
        margin: 0 0 12px 0;
        color: #fbbf24;
        font-size: 22px;
        font-weight: 700;
        line-height: 1.2;
      }
      
      .ml-maintenance-modal__message {
        color: #9ca3af;
        font-size: 15px;
        line-height: 1.6;
        margin: 0 0 24px 0;
      }
      
      .ml-maintenance-modal__btn {
        background: linear-gradient(135deg, #fbbf24 0%, #f59e0b 100%);
        color: #000;
        border: none;
        padding: 12px 24px;
        border-radius: 8px;
        cursor: pointer;
        font-size: 15px;
        font-weight: 600;
        transition: all 0.2s ease;
        width: 100%;
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        box-shadow: 0 4px 12px rgba(251, 191, 36, 0.3);
      }
      
      .ml-maintenance-modal__btn-icon {
        font-size: 16px;
        display: inline-block;
        transition: transform 0.3s ease;
      }
      
      .ml-maintenance-modal__btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 16px rgba(251, 191, 36, 0.4);
      }
      
      .ml-maintenance-modal__btn:hover .ml-maintenance-modal__btn-icon {
        transform: rotate(180deg);
      }
      
      .ml-maintenance-modal__btn:active {
        transform: translateY(0);
      }
      
      .ml-maintenance-modal__footer {
        margin: 16px 0 0 0;
        color: #6b7280;
        font-size: 12px;
        font-style: italic;
      }
      
      @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
      }
      
      @keyframes fadeOut {
        from { opacity: 1; }
        to { opacity: 0; }
      }
      
      @keyframes modalPop {
        from {
          opacity: 0;
          transform: scale(0.9) translateY(-20px);
        }
        to {
          opacity: 1;
          transform: scale(1) translateY(0);
        }
      }
      
      @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.1); }
      }
      
      /* Responsive - Tablet */
      @media (max-width: 768px) {
        .ml-maintenance-modal {
          padding: 28px 20px;
          max-width: 360px;
        }
        
        .ml-maintenance-modal__icon {
          font-size: 48px;
        }
        
        .ml-maintenance-modal__title {
          font-size: 20px;
        }
        
        .ml-maintenance-modal__message {
          font-size: 14px;
        }
      }
      
      /* Responsive - Mobile */
      @media (max-width: 480px) {
        .ml-maintenance-modal-overlay {
          padding: 12px;
        }
        
        .ml-maintenance-modal {
          padding: 24px 16px;
          max-width: 100%;
          border-radius: 12px;
        }
        
        .ml-maintenance-modal__icon {
          font-size: 40px;
          margin-bottom: 12px;
        }
        
        .ml-maintenance-modal__title {
          font-size: 18px;
          margin-bottom: 8px;
        }
        
        .ml-maintenance-modal__message {
          font-size: 13px;
          margin-bottom: 20px;
        }
        
        .ml-maintenance-modal__btn {
          padding: 10px 20px;
          font-size: 14px;
        }
        
        .ml-maintenance-modal__footer {
          font-size: 11px;
          margin-top: 12px;
        }
      }
    `;
    document.head.appendChild(styles);
  }
  
  overlay.appendChild(modal);
  document.body.appendChild(overlay);
  
  // Event listener botón reintentar
  document.getElementById('maintenance-retry-btn').addEventListener('click', async () => {
    toast('Verificando...', 'info');
    const stillMaintenance = await checkMaintenance();
    
    if (!stillMaintenance) {
      // Ya no está en mantenimiento
      overlay.remove();
      toast('✅ Sistema disponible', 'success');
      
      // Recargar página para iniciar normal
      window.location.reload();
    } else {
      toast('⚠️ Aún en mantenimiento', 'warning');
    }
  });
}

/**
 * Iniciar verificación periódica cada 30 segundos (si está en mantenimiento)
 */
export function startMaintenanceCheck() {
  if (maintenanceCheckIntervalId) {
    ResourceManager.remove(maintenanceCheckIntervalId);
  }
  
  // Verificar cada 30 segundos
  maintenanceCheckIntervalId = ResourceManager.addInterval(async () => {
    const inMaintenance = await checkMaintenance();
  }, 30000, 'maintenance-check');
}

/**
 * Detener verificación periódica
 */
export function stopMaintenanceCheck() {
  if (maintenanceCheckIntervalId) {
    ResourceManager.remove(maintenanceCheckIntervalId);
    maintenanceCheckIntervalId = null;
  }
}

/**
 * ============================================
 * SISTEMA DE GRACEFUL SHUTDOWN
 * ============================================
 * Manejo inteligente del modo mantenimiento:
 * - Si usuario está trabajando → Banner de advertencia (no bloquea)
 * - Si usuario está idle → Modal de bloqueo inmediato
 * - Cuando termina tarea → Bloqueo automático
 * ============================================
 */

let operationWatcherId = null;

/**
 * Mostrar banner de ADVERTENCIA para usuarios trabajando
 * Banner amarillo arriba, NO bloquea la interfaz
 */
export function showMaintenanceWarningBanner(message = 'Mantenimiento programado. Termina tu tarea actual.') {
  // Remover banner anterior si existe
  removeMaintenanceWarningBanner();
  
  // Crear banner contenedor
  const banner = document.createElement('div');
  banner.id = 'ml-maintenance-warning';
  banner.className = 'ml-maintenance-warning';
  banner.setAttribute('role', 'alert');
  banner.setAttribute('aria-live', 'polite');
  
  banner.innerHTML = `
    <div class="ml-maintenance-warning__content">
      <div class="ml-maintenance-warning__icon">⚠️</div>
      <div class="ml-maintenance-warning__text">
        <strong>Mantenimiento programado</strong>
        <span>${message}</span>
      </div>
      <div class="ml-maintenance-warning__pulse"></div>
    </div>
  `;
  
  // Añadir estilos si no existen
  if (!document.getElementById('ml-maintenance-warning-styles')) {
    const styles = document.createElement('style');
    styles.id = 'ml-maintenance-warning-styles';
    styles.textContent = `
      .ml-maintenance-warning {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        background: linear-gradient(135deg, rgba(251, 191, 36, 0.95) 0%, rgba(245, 158, 11, 0.95) 100%);
        color: #000;
        padding: 12px 16px;
        box-shadow: 0 4px 12px rgba(251, 191, 36, 0.3);
        z-index: 999998;
        animation: slideDown 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        backdrop-filter: blur(10px);
      }
      
      .ml-maintenance-warning__content {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 12px;
        max-width: 800px;
        margin: 0 auto;
        position: relative;
      }
      
      .ml-maintenance-warning__icon {
        font-size: 24px;
        animation: pulse 2s ease-in-out infinite;
        flex-shrink: 0;
      }
      
      .ml-maintenance-warning__text {
        display: flex;
        flex-direction: column;
        gap: 2px;
        flex: 1;
        min-width: 0;
      }
      
      .ml-maintenance-warning__text strong {
        font-weight: 700;
        font-size: 14px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
      }
      
      .ml-maintenance-warning__text span {
        font-weight: 500;
        font-size: 13px;
        opacity: 0.9;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
      }
      
      .ml-maintenance-warning__pulse {
        position: absolute;
        right: 8px;
        width: 8px;
        height: 8px;
        background: #000;
        border-radius: 50%;
        animation: pulseScale 2s ease-in-out infinite;
      }
      
      @keyframes slideDown {
        from {
          opacity: 0;
          transform: translateY(-100%);
        }
        to {
          opacity: 1;
          transform: translateY(0);
        }
      }
      
      @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.1); }
      }
      
      @keyframes pulseScale {
        0%, 100% { 
          transform: scale(1);
          opacity: 1;
        }
        50% { 
          transform: scale(1.5);
          opacity: 0.5;
        }
      }
      
      /* Responsive */
      @media (max-width: 768px) {
        .ml-maintenance-warning {
          padding: 10px 12px;
        }
        
        .ml-maintenance-warning__content {
          gap: 8px;
        }
        
        .ml-maintenance-warning__icon {
          font-size: 20px;
        }
        
        .ml-maintenance-warning__text strong {
          font-size: 13px;
        }
        
        .ml-maintenance-warning__text span {
          font-size: 12px;
        }
      }
      
      @media (max-width: 480px) {
        .ml-maintenance-warning__text {
          gap: 0;
        }
        
        .ml-maintenance-warning__text span {
          display: none;
        }
      }
    `;
    document.head.appendChild(styles);
  }
  
  // Insertar al principio del body
  document.body.insertBefore(banner, document.body.firstChild);
  
}

/**
 * Quitar banner de advertencia
 */
export function removeMaintenanceWarningBanner() {
  const banner = document.getElementById('ml-maintenance-warning');
  if (banner) {
    banner.style.animation = 'slideUp 0.3s ease-out';
    setTimeout(() => banner.remove(), 300);
  }
}

/**
 * Escuchar cuando termine la operación actual para entonces bloquear
 * Usa el state-manager para detectar cuando completeOperation() se llama
 */
export function watchForOperationComplete(onComplete) {
  // Limpiar watcher anterior si existe
  if (operationWatcherId) {
    ResourceManager.remove(operationWatcherId);
  }
  
  
  // Importar state-manager dinámicamente
  import('./state-manager.js').then(({ hasActiveOperation, subscribe }) => {
    // Suscribirse a cambios en operaciones
    const unsubscribe = subscribe('operations', (newState) => {
      // Cuando la operación termina (current === null)
      if (newState.current === null && operationWatcherId) {
        
        // Limpiar watcher
        ResourceManager.remove(operationWatcherId);
        operationWatcherId = null;
        
        // Quitar banner de advertencia
        removeMaintenanceWarningBanner();
        
        // Ejecutar callback (mostrar modal)
        if (onComplete) {
          onComplete();
        }
        
        // Desuscribirse
        unsubscribe();
      }
    });
    
    // También hacer polling cada 2 segundos como backup
    operationWatcherId = ResourceManager.addInterval(() => {
      if (!hasActiveOperation()) {
        
        ResourceManager.remove(operationWatcherId);
        operationWatcherId = null;
        
        removeMaintenanceWarningBanner();
        
        if (onComplete) {
          onComplete();
        }
      }
    }, 2000, 'operation-watcher');
  }).catch(err => {
    logger.error('[MAINTENANCE] Error importando state-manager:', err);
    
    // Fallback: bloquear después de 30 segundos
    setTimeout(() => {
      removeMaintenanceWarningBanner();
      if (onComplete) onComplete();
    }, 30000);
  });
}

/**
 * Detener watcher de operaciones
 */
export function stopOperationWatcher() {
  if (operationWatcherId) {
    ResourceManager.remove(operationWatcherId);
    operationWatcherId = null;
  }
}

/**
 * Manejar modo mantenimiento de forma inteligente
 * Esta es la función principal que decide qué hacer según el estado del usuario
 * 
 * @param {boolean} enabled - Si el mantenimiento está activado
 * @param {string} message - Mensaje a mostrar
 */
export async function handleMaintenanceMode(enabled, message = 'Sistema en mantenimiento') {
  
  // Si se DESACTIVA el mantenimiento
  if (!enabled) {
    // Quitar todos los modales y banners
    hideMaintenanceModal();
    removeMaintenanceWarningBanner();
    removeMaintenanceBanner();
    stopOperationWatcher();
    
    toast('✅ Mantenimiento finalizado', 'success');
    return;
  }
  
  // Mantenimiento ACTIVADO → verificar rol y estado
  const { AUTH } = await import('./state.js');
  const isAdmin = AUTH?.user?.role === 'admin';
  
  // Admin: Solo banner informativo (sin bloqueo)
  if (isAdmin) {
    showMaintenanceBanner(message);
    return;
  }
  
  // Usuario normal: Verificar si está trabajando
  const { hasActiveOperation } = await import('./state-manager.js');
  const isWorking = hasActiveOperation();
  
  if (isWorking) {
    // Usuario trabajando → Mostrar advertencia y esperar
    
    showMaintenanceWarningBanner(message);
    toast('⚠️ Termina tu tarea. Sistema entrará en mantenimiento.', 'warning', 5000);
    
    // Esperar a que termine para bloquear
    watchForOperationComplete(() => {
      showMaintenanceModal(message);
    });
  } else {
    // Usuario idle → Bloquear inmediatamente
    showMaintenanceModal(message);
  }
}

/**
 * Ocultar modal de mantenimiento
 */
export function hideMaintenanceModal() {
  const modal = document.getElementById('maintenance-modal');
  if (modal) {
    modal.style.animation = 'fadeOut 0.3s ease';
    setTimeout(() => modal.remove(), 300);
  }
}

/**
 * Limpiar cache de validación de versión
 * Útil para debugging o forzar nueva validación
 */
export function clearVersionCache() {
  sessionStorage.removeItem(VERSION_CHECK_KEY);
}
