/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/logger.js – Rol: Sistema de actividad y eventos del usuario (historial visual)
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/

import { escapeHtml } from '../utils.js'; // ✅ v10.5.27: Centralizado

// ========================================
// ACTIVITY LOGGER - Historial de actividad del usuario
// ========================================
// ✅ v4.84.3: Convertido de sistema de debug logs a historial de actividad
// Propósito: Mostrar eventos importantes al USUARIO, no logs técnicos

const LOG_LEVELS = {
  DEBUG: 0,   // Eventos técnicos (solo si debugMode activo)
  INFO: 1,    // Información general (login, cambio cuenta)
  WARN: 2,    // Advertencias (sesión expirando, sin créditos)
  ERROR: 3,   // Errores que afectan al usuario
  SUCCESS: 4  // ✅ NUEVO: Acciones completadas exitosamente
};

const LOG_LEVEL_NAMES = ['DEBUG', 'INFO', 'WARN', 'ERROR', 'SUCCESS'];
const LOG_LEVEL_ICONS = {
  DEBUG: '🔍',
  INFO: 'ℹ️',
  WARN: '⚠️',
  ERROR: '❌',
  SUCCESS: '✅'  // ✅ NUEVO
};

const LOG_LEVEL_CLASSES = {
  DEBUG: 'log-debug',
  INFO: 'log-info',
  WARN: 'log-warn',
  ERROR: 'log-error',
  SUCCESS: 'log-success'  // ✅ NUEVO
};

class Logger {
  constructor(options = {}) {
    this.maxSize = options.maxSize || 500; // Buffer circular
    this.minLevel = LOG_LEVELS[options.minLevel] ?? LOG_LEVELS.INFO; // Nivel mínimo para buffer
    this.debugMode = options.debugMode || false; // ✅ v2.5.1: Controla logs en consola
    this.buffer = []; // Array de logs: { timestamp, level, message, context }
    this.listeners = []; // Callbacks para render en tiempo real
  }

  /**
   * Añade un log al buffer con timestamp y nivel
   */
  _log(level, message, context = {}) {
    // ✅ v10.5.114: NO generar logs en absoluto cuando estamos en Wallapop
    const isWallapopContext = typeof window !== 'undefined' && 
      window.location?.hostname?.includes('wallapop');
    
    if (isWallapopContext) {
      return; // No hacer NADA en contexto de Wallapop
    }
    
    // Siempre añadir al buffer (para pestaña Logs)
    const entry = {
      timestamp: Date.now(),
      level: LOG_LEVEL_NAMES[level],
      message: String(message),
      context: context || {}
    };

    this.buffer.push(entry);

    // Buffer circular: si excede el máximo, eliminar el más antiguo
    if (this.buffer.length > this.maxSize) {
      this.buffer.shift();
    }

    // Notificar a los listeners (para UI en tiempo real)
    this._notify(entry);

    // ✅ v2.5.1: debugMode controla TODOS los logs en consola
    // Solo mostrar en consola si debugMode está activo O es un ERROR crítico
    if (!this.debugMode && level < LOG_LEVELS.ERROR) return;
    
    // Solo usar console.error para todos los niveles (logs eliminados)
    const consoleFn = console.error;
    
    const logLevel = entry?.level || 'UNKNOWN';
    const logMessage = entry?.message || String(message) || 'No message';
    
    // Formatear y loggear según tipo de context
    if (!context || context === undefined) {
      // Sin context
      consoleFn(`[${logLevel}] ${logMessage}`);
    } else if (context instanceof Error) {
      // ✅ v6.6.0: Context es un Error - extraer mensaje y stack
      const errorInfo = {
        name: context.name,
        message: context.message,
        stack: context.stack?.split('\n').slice(0, 5).join('\n') // Primeras 5 líneas del stack
      };
      try {
        consoleFn(`[${logLevel}] ${logMessage}\n${JSON.stringify(errorInfo, null, 2)}`);
      } catch (err) {
        consoleFn(`[${logLevel}] ${logMessage}`, context);
      }
    } else if (typeof context === 'object') {
      // ✅ v10.5.82: No mostrar objetos vacíos {}
      const keys = Object.keys(context);
      if (keys.length === 0) {
        consoleFn(`[${logLevel}] ${logMessage}`);
        return;
      }
      
      // Context es objeto con datos - intentar stringify
      try {
        // ✅ v7.2.5: Sanitizar nulls y undefined para evitar errores
        const sanitized = JSON.parse(JSON.stringify(context));
        consoleFn(`[${logLevel}] ${logMessage}\n${JSON.stringify(sanitized, null, 2)}`);
      } catch (err) {
        // Si falla stringify, loggear objeto tal cual
        consoleFn(`[${logLevel}] ${logMessage}`, context);
      }
    } else {
      // Context es primitivo
      consoleFn(`[${logLevel}] ${logMessage}`, context);
    }
  }

  /**
   * Métodos públicos por nivel
   */
  debug(message, context) {
    this._log(LOG_LEVELS.DEBUG, message, context);
  }

  info(message, context) {
    this._log(LOG_LEVELS.INFO, message, context);
  }

  warn(message, context) {
    this._log(LOG_LEVELS.WARN, message, context);
  }

  error(message, context) {
    this._log(LOG_LEVELS.ERROR, message, context);
  }

  /**
   * ✅ v4.84.3: NUEVO - Para eventos exitosos importantes
   */
  success(message, context) {
    this._log(LOG_LEVELS.SUCCESS, message, context);
  }

  /**
   * ✅ v4.84.3: NUEVO - Métodos específicos para actividades del usuario
   */
  
  // Publicaciones
  publishStarted(count, accountId) {
    this.success(`🚀 Publicación iniciada: ${count} anuncio${count !== 1 ? 's' : ''}`, { accountId, count });
  }

  publishCompleted(published, errors, total) {
    if (errors > 0) {
      this.warn(`Publicación completada: ${published}/${total} anuncios (${errors} errores)`, { published, errors, total });
    } else {
      this.success(`✅ ${published} anuncio${published !== 1 ? 's' : ''} publicado${published !== 1 ? 's' : ''} correctamente`, { published, total });
    }
  }

  publishError(itemId, error) {
    this.error(`Error publicando anuncio #${itemId}: ${error}`, { itemId, error });
  }

  // Backups
  backupStarted(accountId) {
    this.info(`📦 Backup iniciado`, { accountId });
  }

  backupCompleted(stats) {
    const { total, ok, errors } = stats;
    if (errors > 0) {
      this.warn(`Backup completado: ${ok}/${total} anuncios (${errors} errores)`, stats);
    } else {
      this.success(`✅ Backup completado: ${total} anuncios guardados`, stats);
    }
  }

  // Autenticación
  userLoggedIn(alias) {
    this.success(`👤 Sesión iniciada: ${alias || 'Usuario'}`, { alias });
  }

  userLoggedOut() {
    this.info(`👋 Sesión cerrada`, {});
  }

  sessionExpiring() {
    this.warn(`⏰ Tu sesión está a punto de expirar`, {});
  }

  // Créditos
  insufficientCredits(needed, available) {
    this.error(`💳 Créditos insuficientes: necesitas ${needed}, tienes ${available}`, { needed, available });
  }

  creditsUpdated(remaining) {
    this.info(`💳 Créditos actualizados: ${remaining} disponibles`, { remaining });
  }

  /**
   * Obtener logs filtrados
   * @param {Object} filters - { level: 'INFO', search: 'backup', limit: 100 }
   * @returns {Array} Logs filtrados
   */
  getLogs(filters = {}) {
    let logs = [...this.buffer];

    // Filtrar por nivel
    if (filters.level) {
      const minLevel = LOG_LEVELS[filters.level] ?? 0;
      logs = logs.filter(log => LOG_LEVELS[log.level] >= minLevel);
    }

    // Filtrar por búsqueda de texto
    if (filters.search) {
      const search = filters.search.toLowerCase();
      logs = logs.filter(log => 
        log.message.toLowerCase().includes(search) ||
        JSON.stringify(log.context).toLowerCase().includes(search)
      );
    }

    // Limitar resultados
    if (filters.limit) {
      logs = logs.slice(-filters.limit);
    }

    return logs;
  }

  /**
   * Limpiar buffer de logs
   */
  clear() {
    this.buffer = [];
    this._notify({ type: 'clear' });
  }

  /**
   * Exportar logs como texto plano
   * @returns {String} Logs formateados
   */
  export() {
    return this.buffer.map(log => {
      const time = new Date(log.timestamp).toISOString();
      const ctx = Object.keys(log.context).length > 0 
        ? ` | ${JSON.stringify(log.context)}` 
        : '';
      return `[${time}] [${log.level}] ${log.message}${ctx}`;
    }).join('\n');
  }

  /**
   * Cambiar nivel mínimo de logging en runtime
   */
  setMinLevel(level) {
    const levelNum = LOG_LEVELS[level];
    if (levelNum !== undefined) {
      this.minLevel = levelNum;
      this.info(`Nivel de logging cambiado a: ${level}`);
    }
  }

  /**
   * Suscribir un listener para cambios en tiempo real
   * @param {Function} callback - (entry) => void
   */
  subscribe(callback) {
    this.listeners.push(callback);
    return () => {
      this.listeners = this.listeners.filter(cb => cb !== callback);
    };
  }

  /**
   * Notificar a todos los listeners
   */
  _notify(entry) {
    this.listeners.forEach(cb => {
      try {
        cb(entry);
      } catch (err) {
        console.error('[Logger] Error en listener:', err);
      }
    });
  }

  /**
   * Obtener estadísticas de logs
   */
  getStats() {
    const stats = {
      total: this.buffer.length,
      success: 0,  // ✅ v4.84.3: NUEVO
      debug: 0,
      info: 0,
      warn: 0,
      error: 0
    };

    this.buffer.forEach(log => {
      const level = log.level.toLowerCase();
      if (stats[level] !== undefined) {
        stats[level]++;
      }
    });

    return stats;
  }
}

// ========================================
// INSTANCIA GLOBAL
// ========================================

// ✅ v4.84.3: Logger convertido en Activity Logger
export const logger = new Logger({
  maxSize: 200,  // Últimos 200 eventos
  minLevel: 'INFO'  // ✅ Mostrar INFO, WARN, ERROR, SUCCESS (no DEBUG)
});

// ✅ v2.5.1: debugMode controla si los logs se muestran en consola
// Cargar debugMode al inicio
chrome.storage.local.get(['debugMode'], (result) => {
  logger.debugMode = !!result.debugMode;
});

// Escuchar cambios en debugMode
chrome.storage.onChanged.addListener((changes) => {
  if (changes.debugMode) {
    logger.debugMode = !!changes.debugMode.newValue;
  }
});

// ========================================
// HELPERS DE FORMATEO PARA UI
// ========================================

/**
 * Formatear timestamp para mostrar en UI
 * @param {Number} timestamp - Milisegundos desde epoch
 * @returns {String} Formato "HH:MM:SS"
 */
export function formatTime(timestamp) {
  const date = new Date(timestamp);
  const h = String(date.getHours()).padStart(2, '0');
  const m = String(date.getMinutes()).padStart(2, '0');
  const s = String(date.getSeconds()).padStart(2, '0');
  return `${h}:${m}:${s}`;
}

/**
 * Obtener icono para nivel de log
 */
export function getLogIcon(level) {
  return LOG_LEVEL_ICONS[level] || '📝';
}

/**
 * Obtener clase CSS para nivel de log
 */
export function getLogClass(level) {
  return LOG_LEVEL_CLASSES[level] || 'log-info';
}

/**
 * Formatear entrada de log como HTML
 * @param {Object} entry - { timestamp, level, message, context }
 * @returns {String} HTML string
 */
export function formatLogEntry(entry) {
  const time = formatTime(entry.timestamp);
  const icon = getLogIcon(entry.level);
  const cls = getLogClass(entry.level);
  
  let html = `<div class="log-entry ${cls}">`;
  html += `<span class="log-time">${time}</span>`;
  html += `<span class="log-icon">${icon}</span>`;
  html += `<span class="log-level">${entry.level}</span>`;
  html += `<span class="log-message">${escapeHtml(entry.message)}</span>`;
  
  // Mostrar contexto si existe
  if (entry.context && Object.keys(entry.context).length > 0) {
    const contextStr = JSON.stringify(entry.context, null, 2);
    html += `<span class="log-context" title="${escapeHtml(contextStr)}">${escapeHtml(JSON.stringify(entry.context))}</span>`;
  }
  
  html += `</div>`;
  return html;
}

// ✅ v10.5.27: escapeHtml eliminado - usar import de utils.js

// ========================================
// EXPORTAR CONSTANTES
// ========================================

export { LOG_LEVELS, LOG_LEVEL_NAMES };
