/**
 * Sistema WebSocket y vista publishing en tiempo real
 * @module listings/websocket
 */

import { logger } from '../logger.js';
import { toast } from '../../utils.js';
import { sw, showError } from './helpers.js';
import {  getCurrentAccountId, getCurrentFilter } from './module-state.js';
import { FILTERS } from './config.js';

// ============================================
// Vista Publishing
// ============================================

/**
 * Fetch listings en estado de publicación
 */
async function fetchPublishingListings(accountId) {
  try {
    const response = await sw({
      type: 'API.FETCH_JSON',
      url: `/api/walla/listings/publishing?account_id=${accountId}`,
      method: 'GET'
    });
    
    if (!response.ok) {
      throw new Error(response.error || 'Error al obtener listings en publicación');
    }
    
    return response.data;
  } catch (err) {
    logger.error('[PUBLISHING] Error:', err);
    toast('Error al cargar anuncios en publicación', 'error');
    return {};
  }
}

/**
 * Renderiza vista especial de "Publicando"
 * @param {number} accountId - ID de cuenta
 */
export async function renderPublishingView(accountId) {
  if (!accountId) return;
  
  const tbody = document.querySelector('#listings-tbody');
  if (!tbody) return;
  
  try {
    const grouped = await fetchPublishingListings(accountId);
    tbody.innerHTML = '';
    
    const dates = Object.keys(grouped);
    if (dates.length === 0) {
      tbody.innerHTML = `
        <tr class="empty-row">
          <td colspan="8" class="empty-cell">
            🚀 No hay anuncios en publicación
          </td>
        </tr>
      `;
      return;
    }
    
    dates.sort((a, b) => b.localeCompare(a));
    
    for (const date of dates) {
      const items = grouped[date];
      if (!items || items.length === 0) continue;
      
      const dateLabel = formatDateLabel(date);
      const headerRow = document.createElement('tr');
      headerRow.className = 'date-header';
      headerRow.innerHTML = `
        <td colspan="8">
          📅 ${dateLabel} (${items.length} anuncios)
        </td>
      `;
      tbody.appendChild(headerRow);
      
      items.forEach(item => {
        const row = createPublishingRow(item);
        tbody.appendChild(row);
      });
    }
    
  } catch (err) {
    logger.error('[PUBLISHING] Error:', err);
    showError('Error al cargar vista de publicación');
  }
}

/**
 * Crea row para vista publishing
 */
function createPublishingRow(item) {
  const row = document.createElement('tr');
  row.dataset.listingId = item.listing_id;
  row.className = `publishing-row status-${item.publish_status || 'unknown'}`;
  
  const statusIcons = {
    'processing': '⏳',
    'published': '✅',
    'error_retry': '❌'
  };
  const statusTexts = {
    'processing': 'Publicando...',
    'published': 'Publicado',
    'error_retry': 'Error'
  };
  
  const statusIcon = statusIcons[item.publish_status] || '•';
  const statusText = statusTexts[item.publish_status] || 'Desconocido';
  const imageUrl = item.image_url || '';
  // ✅ v6.8.1: Manejar título como objeto {"original": "..."} o string
  let titleText = item.title || 'Sin título';
  if (typeof titleText === 'object' && titleText.original) {
    titleText = titleText.original;
  }
  const imageHtml = imageUrl 
    ? `<img src="${imageUrl}" alt="${titleText}" loading="lazy" />`
    : '<div class="no-image">📷</div>';
  const timeText = item.last_published_at ? formatTime(item.last_published_at) : '-';
  const priceText = item.price_amount ? `${item.price_amount}€` : '-';
  
  row.innerHTML = `
    <td class="col-img">${imageHtml}</td>
    <td class="col-title">${titleText}</td>
    <td class="col-price">${priceText}</td>
    <td class="col-status-pub">
      ${statusIcon} <span>${statusText}</span>
    </td>
    <td class="col-time-pub">${timeText}</td>
    <td class="col-actions">
      ${item.publish_error ? `<span class="error-icon" title="${item.publish_error}">⚠️</span>` : ''}
    </td>
  `;
  
  return row;
}

function formatDateLabel(dateStr) {
  try {
    const date = new Date(dateStr);
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);
    
    const isSameDay = (d1, d2) => {
      return d1.getFullYear() === d2.getFullYear() &&
             d1.getMonth() === d2.getMonth() &&
             d1.getDate() === d2.getDate();
    };
    
    if (isSameDay(date, today)) return 'HOY';
    if (isSameDay(date, yesterday)) return 'AYER';
    
    const options = { weekday: 'short', day: 'numeric', month: 'short' };
    return date.toLocaleDateString('es-ES', options);
  } catch (err) {
    return dateStr;
  }
}

function formatTime(isoString) {
  try {
    const date = new Date(isoString);
    const hours = date.getHours().toString().padStart(2, '0');
    const minutes = date.getMinutes().toString().padStart(2, '0');
    return `${hours}:${minutes}`;
  } catch (err) {
    return '-';
  }
}

// ============================================
// WebSocket Handlers
// ============================================

/**
 * Handler de mensajes WebSocket
 * @param {Function} loadListingsTable - Callback para recargar tabla
 */
function createWebSocketHandler(loadListingsTable) {
  return function handleWebSocketMessages(msg, sender, sendResponse) {
    const accountId = getCurrentAccountId();
    const filter = getCurrentFilter();
    
    // ✅ v10.5.20: Durante backup NO tocamos la tabla - solo progreso
    // La tabla se recarga completa al finalizar (backup_complete en panel.js)
    if (msg?.type === 'backup_item_saved') {
      // Solo responder OK, la barra de progreso se actualiza en panel.js
      sendResponse({ ok: true });
      return true;
    }
    
    // ✅ v10.5.86: HANDLERS DE backup_complete, publish_complete, publish_progress
    // ELIMINADOS - Ahora manejados ÚNICAMENTE en panel.js para evitar:
    // - loadListingsTable() llamado múltiples veces
    // - Duplicación de lógica de UI
    // Ver: ANALISIS-BARRAS-PROGRESO.md
    
    // ✅ Estado de cuenta cambiado (CONTRATO ÚNICO - sin prefijo WS.)
    // MANTENIDO: Este evento es específico de listings y no se maneja en panel.js
    if (msg?.type === 'account_state_changed') {
      const { account_id } = msg.payload || {};
      
      if (account_id && account_id === accountId) {
        loadListingsTable(accountId);
      }
      
      sendResponse({ ok: true });
      return true;
    }
  };
}

let webSocketHandler = null;

/**
 * Configura listeners de WebSocket
 * @param {Function} loadListingsTable - Callback para recargar tabla
 */
export function setupWebSocketListeners(loadListingsTable) {
  // Remover listener anterior
  if (webSocketHandler) {
    chrome.runtime.onMessage.removeListener(webSocketHandler);
  }
  
  // Crear y agregar nuevo handler
  webSocketHandler = createWebSocketHandler(loadListingsTable);
  chrome.runtime.onMessage.addListener(webSocketHandler);
  
}

/**
 * Actualiza row en vista publishing (tiempo real)
 */
function updatePublishingRow(listingId, newStatus) {
  if (getCurrentFilter() !== FILTERS.PUBLISHING) return;
  
  const row = document.querySelector(`tr[data-listing-id="${listingId}"]`);
  if (!row) return;
  
  row.className = `publishing-row status-${newStatus}`;
  
  const statusCell = row.querySelector('.col-status-pub');
  if (statusCell) {
    const statusIcons = {
      'processing': '⏳',
      'published': '✅',
      'error_retry': '❌'
    };
    const statusTexts = {
      'processing': 'Publicando...',
      'published': 'Publicado',
      'error_retry': 'Error'
    };
    
    const icon = statusIcons[newStatus] || '•';
    const text = statusTexts[newStatus] || 'Desconocido';
    
    statusCell.innerHTML = `${icon} <span>${text}</span>`;
  }
  
  // Trigger animación
  row.style.animation = 'none';
  setTimeout(() => {
    row.style.animation = '';
  }, 10);
}
