/**
 * Funciones de renderizado de listings
 * @module listings/render
 */

import DOMUtils from '../dom-utils.js';
import { logger } from '../logger.js';
import { initImageModalListeners } from '../image-modal.js';
import { FILTERS, CONFIG } from './config.js';
import { 
  truncateTitle, 
  formatPrice, 
  getStatusBadge, 
  getImageUrl,
  updateListingsCounter as updateCounter
} from './helpers.js';
import {
  hasSelection,
  getCurrentFilter,
  getCurrentSearch,
  getApiBase
} from './module-state.js';

// ============================================
// Referencias DOM
// ============================================

const tbody = document.getElementById('listings-tbody');
const selectAllCheckbox = document.getElementById('select-all-listings');
const statTotal = document.getElementById('stat-total');
const statActive = document.getElementById('stat-active');
const statSold = document.getElementById('stat-sold');
const statReserved = document.getElementById('stat-reserved');
const statDeleted = document.getElementById('stat-deleted');

// ============================================
// Helpers internos
// ============================================

/**
 * ✅ v10.3.9: Configura fallback para imágenes (sin onerror inline por CSP)
 */
function setupImageFallbacks() {
  if (!tbody) return;
  tbody.querySelectorAll('img.listing-thumb-fallback').forEach(img => {
    if (img.dataset.fallbackSetup) return; // Evitar duplicados
    img.dataset.fallbackSetup = 'true';
    img.addEventListener('error', function() {
      this.style.display = 'none';
      const placeholder = this.nextElementSibling;
      if (placeholder) placeholder.style.display = 'flex';
    }, { once: true });
  });
}

function isDeletedFilter() {
  return getCurrentFilter() === FILTERS.DELETED;
}

/**
 * ✅ v10.5.136: Resalta palabras de búsqueda en el texto
 * @param {string} text - Texto original
 * @param {string} search - Texto de búsqueda (puede tener múltiples palabras)
 * @returns {string} HTML con palabras resaltadas
 */
function highlightSearchTerms(text, search) {
  if (!search || !text) return text;
  
  const words = search.trim().split(/\s+/).filter(w => w.length > 0);
  if (words.length === 0) return text;
  
  let result = text;
  
  // Resaltar cada palabra (case-insensitive)
  words.forEach(word => {
    // Escapar caracteres especiales de regex
    const escaped = word.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    const regex = new RegExp(`(${escaped})`, 'gi');
    result = result.replace(regex, '<mark class="search-highlight">$1</mark>');
  });
  
  return result;
}

/**
 * ✅ v10.5.139: Calcula un color consistente para una categoría
 * Usa un hash simple del nombre para asignar 1 de 8 colores
 * @param {string} categoryName - Nombre de la categoría
 * @returns {string} Número del color (1-8)
 */
function getCategoryColor(categoryName) {
  if (!categoryName) return '';
  
  // Hash simple: suma de códigos de caracteres
  let hash = 0;
  for (let i = 0; i < categoryName.length; i++) {
    hash = ((hash << 5) - hash) + categoryName.charCodeAt(i);
    hash = hash & hash; // Convertir a 32-bit integer
  }
  
  // Mapear a 1-8
  return String((Math.abs(hash) % 8) + 1);
}

function shouldEnableCheckbox(itemStatus) {
  const isActive = itemStatus === 'active';
  const isDeleted = itemStatus === 'deleted';
  return isActive || (isDeleted && isDeletedFilter());
}

/**
 * ✅ v6.6.0: Formatea fecha de última publicación
 * ✅ v10.5.136: Añade clase CSS según antigüedad (hoy, reciente, antiguo)
 * @param {string} isoString - Fecha ISO del backend
 * @returns {Object} { text: string, cssClass: string }
 */
function formatPublishedDate(isoString) {
  if (!isoString) {
    return { text: '—', cssClass: 'date-never' };
  }
  
  try {
    const date = new Date(isoString);
    if (isNaN(date.getTime())) {
      return { text: '—', cssClass: 'date-never' };
    }
    
    const now = new Date();
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const dateOnly = new Date(date.getFullYear(), date.getMonth(), date.getDate());
    const diffDays = Math.floor((today - dateOnly) / (1000 * 60 * 60 * 24));
    
    const day = date.getDate().toString().padStart(2, '0');
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const hours = date.getHours().toString().padStart(2, '0');
    const minutes = date.getMinutes().toString().padStart(2, '0');
    const text = `${day}/${month} ${hours}:${minutes}`;
    
    // Determinar clase CSS según antigüedad
    let cssClass = 'date-old';
    if (diffDays === 0) {
      cssClass = 'date-today';
    } else if (diffDays >= 1 && diffDays <= 3) {
      cssClass = 'date-recent';
    }
    
    return { text, cssClass };
  } catch (err) {
    return { text: '—', cssClass: 'date-never' };
  }
}

// ============================================
// Renderizado de rows
// ============================================

/**
 * Genera el HTML de una fila de listing
 * @param {Object} item - Datos del listing
 * @returns {string} HTML de la fila
 */
export function renderListingRow(item) {
  const isSelected = hasSelection(item.listing_id);
  // ✅ v6.8.1: Manejar título como objeto {"original": "..."} o string
  let titleText = item.title || '(sin título)';
  if (typeof titleText === 'object' && titleText.original) {
    titleText = titleText.original;
  }
  
  // ✅ v10.5.136: Resaltar términos de búsqueda
  const currentSearch = getCurrentSearch();
  const titleTruncated = truncateTitle(titleText, 40);
  const titleDisplay = highlightSearchTerms(titleTruncated, currentSearch);
  
  const price = formatPrice(item.price_amount, item.price_currency);
  const statusBadge = getStatusBadge(item.status);
  const wallaId = item.id_wallapop || '—';
  
  // ✅ v10.5.138: Categoría en lugar de ID
  // ✅ v10.5.139: Con color basado en hash del nombre
  const category = item.category || '';
  const categoryTruncated = truncateTitle(category, 18);
  const categoryDisplay = highlightSearchTerms(categoryTruncated, currentSearch);
  const categoryColor = category ? getCategoryColor(category) : '';
  
  const raw = item.first_image_url || item.first_image || '';
  const imageUrl = getImageUrl(raw, getApiBase());
  const checkboxDisabled = !shouldEnableCheckbox(item.status);
  
  // ✅ v10.5.136: Resaltar inventario también
  const inventoryNote = item.inventory_note || '';
  
  let checkboxTitle = '';
  if (item.status === 'active') {
    checkboxTitle = 'Seleccionar para publicar/eliminar';
  } else if (item.status === 'deleted' && isDeletedFilter()) {
    checkboxTitle = 'Seleccionar para republicar';
  } else {
    checkboxTitle = '⚠️ Solo activos o eliminados (en filtro Eliminados)';
  }
  
  // Estados de publicación (solo en filtro publishing)
  let publishClass = '';
  let publishStatusBadge = '';
  
  if (item.publish_status && getCurrentFilter() === FILTERS.PUBLISHING) {
    switch (item.publish_status) {
      case 'queued':
      case 'queued_delete':
        publishClass = 'row-pending';
        publishStatusBadge = '<span class="publish-badge publish-queued" title="En cola">🕐 Cola</span>';
        break;
      case 'processing':
      case 'ready_publish':
      case 'publishing':
        publishClass = 'row-publishing';
        publishStatusBadge = '<span class="publish-badge publish-processing" title="Publicando...">⏳ Publicando</span>';
        break;
      case 'published':
        publishClass = 'row-published';
        publishStatusBadge = '<span class="publish-badge publish-success" title="Publicado correctamente">✅ Publicado</span>';
        break;
      case 'error':
      case 'error_retry':
        publishClass = 'row-error';
        const errorMsg = item.publish_error || 'Error desconocido';
        publishStatusBadge = `<span class="publish-badge publish-error" title="${errorMsg}">❌ Error</span>`;
        break;
    }
  }
  
  // ✅ v10.5.136: Fecha con clase de color
  const publishedDate = formatPublishedDate(item.last_published_at);

  return `
    <tr data-listing-id="${item.listing_id}" data-status="${item.status}" data-published="${item.last_published_at || ''}" data-price="${item.price_amount || 0}" data-category="${category}" class="${isSelected ? 'selected' : ''} ${publishClass}">
      <td class="col-check">
        <input 
          type="checkbox" 
          class="listing-checkbox" 
          data-id="${item.listing_id}"
          ${isSelected ? 'checked' : ''}
          ${checkboxDisabled ? 'disabled' : ''}
          title="${checkboxTitle}"
        >
      </td>
      <td class="col-image">
        ${imageUrl
          ? `<img src="${imageUrl}" class="listing-thumb listing-thumb-fallback" alt="" title="${imageUrl}"><div class="listing-thumb-placeholder" style="display:none;" title="(imagen no disponible)">📷</div>`
          : `<div class="listing-thumb-placeholder" title="(sin imagen)">📷</div>`
        }
      </td>
      <td class="col-title">
        <span class="listing-title" title="${titleText || ''}">${titleDisplay}</span>
        ${publishStatusBadge}
      </td>
      <td class="col-price">
        <span class="listing-price">${price}</span>
      </td>
      <td class="col-category">
        <span class="listing-category" ${categoryColor ? `data-cat-color="${categoryColor}"` : ''} title="${category || 'Sin categoría'}">${categoryDisplay || '—'}</span>
      </td>
      <td class="col-status">
        ${statusBadge}
      </td>
      <td class="col-published">
        <span class="published-date ${publishedDate.cssClass}" title="${item.last_published_at || 'Nunca publicado'}">${publishedDate.text}</span>
      </td>
      <td class="col-inventory">
        <input 
          type="text" 
          class="inventory-input" 
          data-listing-id="${item.listing_id}"
          data-web-id="${wallaId}"
          placeholder="Ej: Estantería 8-5"
          value="${inventoryNote}"
        >
      </td>
    </tr>
  `;
}

/**
 * Renderiza lista completa de listings
 * @param {Array} items - Array de listings
 * @param {number} total - Total de items
 * @param {Function} attachCheckboxEvents - Callback para attach events
 */
export function renderListings(items, total, attachCheckboxEvents) {
  if (!tbody) {
    logger.error('[RENDER] tbody not found');
    return;
  }

  if (!items || items.length === 0) {
    tbody.innerHTML = `
      <tr class="empty-row">
        <td colspan="8" class="empty-cell">
          No hay anuncios para mostrar
        </td>
      </tr>
    `;
    if (selectAllCheckbox) selectAllCheckbox.checked = false;
    // ✅ v10.5.82: Actualizar contador a 0
    const search = getCurrentSearch();
    updateCounter(0, 0, !!(search && search.trim()));
    return;
  }

  const itemsToRender = items;
  
  if (itemsToRender.length <= CONFIG.CHUNK_SIZE) {
    tbody.innerHTML = itemsToRender.map(renderListingRow).join('');
    setupImageFallbacks(); // ✅ v10.3.9
    attachCheckboxEvents();
    initImageModalListeners(); // ✅ v10.5.17: Click para ampliar imágenes
  } else {
    const firstChunk = itemsToRender.slice(0, CONFIG.CHUNK_SIZE);
    tbody.innerHTML = firstChunk.map(renderListingRow).join('');
    setupImageFallbacks(); // ✅ v10.3.9
    
    const remainingItems = itemsToRender.slice(CONFIG.CHUNK_SIZE);
    renderChunksProgressively(remainingItems, CONFIG.CHUNK_SIZE, attachCheckboxEvents);
  }

  const allChecked = itemsToRender.length > 0 && itemsToRender.every(it => hasSelection(it.listing_id));
  if (selectAllCheckbox) selectAllCheckbox.checked = allChecked;

  // ✅ v10.5.82: Detectar si hay búsqueda activa para el formato del contador
  const search = getCurrentSearch();
  const isSearchResult = !!(search && search.trim());
  updateCounter(itemsToRender.length, total || items.length, isSearchResult);
}

/**
 * Renderiza chunks progresivamente
 * @param {Array} items - Items a renderizar
 * @param {number} chunkSize - Tamaño del chunk
 * @param {Function} attachCheckboxEvents - Callback para events
 */
function renderChunksProgressively(items, chunkSize, attachCheckboxEvents) {
  let offset = 0;
  
  function renderNextChunk() {
    if (offset >= items.length) {
      setupImageFallbacks(); // ✅ v10.3.9
      attachCheckboxEvents();
      initImageModalListeners(); // ✅ v10.5.17: Click para ampliar imágenes
      return;
    }
    
    const chunk = items.slice(offset, offset + chunkSize);
    const html = chunk.map(renderListingRow).join('');
    tbody.insertAdjacentHTML('beforeend', html);
    
    offset += chunkSize;
    
    if ('requestIdleCallback' in window) {
      requestIdleCallback(renderNextChunk);
    } else {
      setTimeout(renderNextChunk, 0);
    }
  }
  
  renderNextChunk();
}

/**
 * Agrega items al final (scroll infinito)
 * @param {Array} items - Items a agregar
 * @param {Function} attachCheckboxEvents - Callback para events
 */
export function appendListings(items, attachCheckboxEvents) {
  if (!tbody || !items || items.length === 0) return;

  const newRows = items.map(renderListingRow).join('');
  tbody.insertAdjacentHTML('beforeend', newRows);
  setupImageFallbacks(); // ✅ v10.3.9
  attachCheckboxEvents();
  initImageModalListeners(); // ✅ v10.5.17: Click para ampliar imágenes
}

/**
 * Actualiza contador de listings
 * @param {number} loaded - Cargados
 * @param {number} total - Total
 * @param {boolean} isSearchResult - Si es resultado de búsqueda
 */
export function updateListingsCounter(loaded, total = null, isSearchResult = false) {
  updateCounter(loaded, total, isSearchResult);
}

/**
 * Renderiza estadísticas
 * @param {Object} stats - Estadísticas del backend
 * @param {number} total - Total de items
 */
export function renderStats(stats, total) {
  const statsWrap = document.querySelector('.ml-listings-stats');
  if (statsWrap) DOMUtils.show(statsWrap);
  
  if (statTotal) statTotal.textContent = total || 0;
  if (statActive) statActive.textContent = stats.active || 0;
  if (statSold) statSold.textContent = (stats.sold || 0) + (stats.bumped || 0);
  if (statReserved) statReserved.textContent = stats.reserved || 0;
  if (statDeleted) statDeleted.textContent = stats.deleted || 0;
}

/**
 * ✅ v6.8.1: Incrementa contadores de stats cuando llega un nuevo item (centralizado)
 * @param {string} status - Estado del listing (active, reserved, sold, deleted)
 */
export function incrementStatCounter(status) {
  // Incrementar TOTAL
  if (statTotal) {
    statTotal.textContent = (parseInt(statTotal.textContent) || 0) + 1;
  }
  
  // Incrementar según status
  const statusMap = {
    'active': statActive,
    'reserved': statReserved,
    'sold': statSold,
    'featured': statSold,
    'deleted': statDeleted,
    'removed': statDeleted
  };
  
  const targetEl = statusMap[status];
  if (targetEl) {
    targetEl.textContent = (parseInt(targetEl.textContent) || 0) + 1;
  }
}

/**
 * Actualiza una row existente
 * @param {string} listingId - ID del listing
 * @param {Object} updates - Cambios a aplicar
 */
export function updateListingRow(listingId, updates) {
  if (!tbody) return;
  
  const row = tbody.querySelector(`tr[data-listing-id="${listingId}"]`);
  if (!row) {
    return;
  }
  
  // Aplicar updates según tipo
  if (updates.status) {
    row.setAttribute('data-status', updates.status);
    const statusCell = row.querySelector('.col-status');
    if (statusCell) {
      statusCell.innerHTML = getStatusBadge(updates.status);
    }
  }
  
  if (updates.inventory_note !== undefined) {
    const input = row.querySelector('.inventory-input');
    if (input) input.value = updates.inventory_note || '';
  }
  
  if (updates.publish_status && getCurrentFilter() === FILTERS.PUBLISHING) {
    row.className = row.className.replace(/row-(pending|publishing|published|error)/g, '');
    
    let publishClass = '';
    switch (updates.publish_status) {
      case 'queued':
      case 'queued_delete':
        publishClass = 'row-pending';
        break;
      case 'processing':
      case 'ready_publish':
      case 'publishing':
        publishClass = 'row-publishing';
        break;
      case 'published':
        publishClass = 'row-published';
        break;
      case 'error':
      case 'error_retry':
        publishClass = 'row-error';
        break;
    }
    
    if (publishClass) row.classList.add(publishClass);
  }
}

/**
 * Mueve una row al inicio de la tabla
 * @param {string} listingId - ID del listing
 */
export function moveRowToTop(listingId) {
  if (!tbody) return;
  
  const row = tbody.querySelector(`tr[data-listing-id="${listingId}"]`);
  if (!row) return;
  
  tbody.insertBefore(row, tbody.firstChild);
}

/**
 * Limpia el estado visual de una row
 * @param {string} listingId - ID del listing
 */
export function clearRowState(listingId) {
  if (!tbody) return;
  
  const row = tbody.querySelector(`tr[data-listing-id="${listingId}"]`);
  if (!row) return;
  
  row.classList.remove('row-pending', 'row-publishing', 'row-published', 'row-error', 'selected');
}

/**
 * Establece estado visual de una row
 * @param {string} listingId - ID del listing
 * @param {string} state - Estado (pending|publishing|published|error)
 */
export function setRowState(listingId, state) {
  if (!tbody) return;
  
  const row = tbody.querySelector(`tr[data-listing-id="${listingId}"]`);
  if (!row) return;
  
  clearRowState(listingId);
  
  const stateClass = `row-${state}`;
  row.classList.add(stateClass);
}
