/**
 * Sistema de carga progresiva / lazy loading
 * @module listings/progressive
 */

import DOMUtils from '../dom-utils.js';
import { logger } from '../logger.js';
import { toast } from '../../utils.js';
import { stRemove } from '../storage.js';
import { CONFIG } from './config.js';
import { 
  sw, 
  ensureApiBase,
  showError,
  showListingsLoader,
  hideListingsLoader
} from './helpers.js';
import {
  getCurrentAccountId,
  getCurrentFilter,
  getCurrentSearch,
  getCurrentSortBy,
  getCurrentOffset,
  getCurrentTotal,
  getHasMoreItems,
  getIsLoadingMore,
  getCurrentListings,
  setCurrentListings,
  setCurrentOffset,
  setCurrentTotal,
  setHasMoreItems,
  setIsLoadingMore,
  resetScrollState,
  getApiBase,
  getApiBaseReady,
  setApiBase,
  setApiBaseReady,
  // ✅ v10.5.142: Para auto-seleccionar nuevos items
  getSelectAllByFilter,
  addSelection
} from './module-state.js';
import { setDbStats } from '../state-manager.js';

// ============================================
// Fetch & Render
// ============================================

/**
 * Obtención inicial - limpia todo y carga primera página
 * @param {Function} renderListings - Callback para renderizar
 * @param {Function} renderStats - Callback para stats
 * @param {Function} highlightFilter - Callback para highlight
 * @param {Function} toggleButtons - Callback para botones
 * @param {Function} updateSelectionUI - Callback para selección
 */
export async function fetchAndRender(renderListings, renderStats, highlightFilter, toggleButtons, updateSelectionUI) {
  const accountId = getCurrentAccountId();
  if (!accountId) return;
  
  resetScrollState();
  
  await ensureApiBase(getApiBase, getApiBaseReady, setApiBase, setApiBaseReady);
  
  // ✅ v10.5.81: No mostrar loader si hay búsqueda activa (evita parpadeo)
  const currentSearch = getCurrentSearch();
  const showLoader = !currentSearch || currentSearch.trim() === '';
  
  if (showLoader) {
    showListingsLoader('🔄 Cargando anuncios...');
  }

  try {
    const statusParam = getCurrentFilter() === 'all' ? '' : getCurrentFilter();
    
    const params = new URLSearchParams({
      account_id: String(accountId),
      status: statusParam,
      search: getCurrentSearch(),
      limit: CONFIG.ITEMS_PER_PAGE,
      offset: 0
    });
    
    // ✅ v6.6.0: Agregar ordenación si está definida
    const sortBy = getCurrentSortBy();
    if (sortBy) {
      params.set('sort_by', sortBy);
    }

    const r = await sw({
      type: 'API.FETCH_JSON',
      url: `/api/walla/listings/by-account?${params}`,
      method: 'GET'
    });

    // Manejar 403 Forbidden
    if (r?.status === 403 || r?.error === 'account_not_owned') {
      await stRemove(['selected_account_id', 'account_id']).catch(() => {});
      
      const tbody = document.getElementById('listings-tbody');
      if (tbody) tbody.innerHTML = '';
      
      renderStats({}, 0);
      toast('Esta cuenta no te pertenece', 'warn');
      return;
    }

    if (!r?.ok || !r.data) throw new Error(r?.error || 'Error cargando anuncios');

    const items = r.data.items || [];
    const total = r.data.total || 0;
    const stats = r.data.stats || {};
    
    // ✅ Guardar stats reales de la BD
    setDbStats({
      total: total,
      active: stats.active || 0,
      deleted: stats.deleted || 0,
      reserved: stats.reserved || 0,
      sold: stats.sold || 0
    });
    
    setCurrentListings(items);
    setCurrentTotal(total);
    setCurrentOffset(items.length);
    setHasMoreItems(items.length < total);
    
    renderListings(items, total);
    renderStats(stats, total);
    highlightFilter(getCurrentFilter());
    toggleButtons(getCurrentFilter());
    updateSelectionUI();

  } catch (err) {
    logger.error('[PROGRESSIVE] Error:', err);
    showError('No se pudieron cargar los anuncios');
  } finally {
    if (showLoader) {
      hideListingsLoader();
    }
  }
}

/**
 * Carga más items (scroll infinito)
 * @param {Function} appendListings - Callback para agregar rows
 * @param {Function} updateCounter - Callback para contador
 */
export async function loadMoreItems(appendListings, updateCounter) {
  const accountId = getCurrentAccountId();
  if (!accountId || getIsLoadingMore() || !getHasMoreItems()) {
    return;
  }

  setIsLoadingMore(true);
  
  const scrollLoader = document.getElementById('scroll-loader');
  if (scrollLoader) scrollLoader.style.display = 'flex';

  await ensureApiBase(getApiBase, getApiBaseReady, setApiBase, setApiBaseReady);

  try {
    const statusParam = getCurrentFilter() === 'all' ? '' : getCurrentFilter();
    
    const params = new URLSearchParams({
      account_id: String(accountId),
      status: statusParam,
      search: getCurrentSearch(),
      limit: CONFIG.ITEMS_PER_PAGE,
      offset: getCurrentOffset()
    });
    
    // ✅ v6.6.0: Mantener ordenación al cargar más
    const sortBy = getCurrentSortBy();
    if (sortBy) {
      params.set('sort_by', sortBy);
    }

    const r = await sw({
      type: 'API.FETCH_JSON',
      url: `/api/walla/listings/by-account?${params}`,
      method: 'GET'
    });

    if (r?.ok && r.data) {
      const newItems = r.data.items || [];
      
      const allListings = getCurrentListings().concat(newItems);
      setCurrentListings(allListings);
      setCurrentOffset(getCurrentOffset() + newItems.length);
      setHasMoreItems(allListings.length < getCurrentTotal());
      
      // ✅ v10.5.142: Auto-seleccionar nuevos items si "Seleccionar todos" está activo
      if (getSelectAllByFilter() && newItems.length > 0) {
        const currentFilter = getCurrentFilter();
        const isDeletedFilter = currentFilter === 'deleted';
        
        newItems.forEach(item => {
          const isActive = item.status === 'active';
          const isDeleted = item.status === 'deleted';
          // Seleccionar según filtro actual
          if (isActive || (isDeleted && isDeletedFilter)) {
            addSelection(item.listing_id);
          }
        });
      }
      
      appendListings(newItems);
      updateCounter(allListings.length, getCurrentTotal());
    }

  } catch (err) {
    logger.error('[PROGRESSIVE] Error loading more:', err);
  } finally {
    setIsLoadingMore(false);
    if (scrollLoader) DOMUtils.hide(scrollLoader);
  }
}

/**
 * Configura scroll infinito
 * @param {Function} loadMoreItems - Callback para cargar más
 */
export function setupScrollInfinite(loadMoreItemsCallback) {
  const tableWrapper = document.querySelector('.ml-listings-table-wrapper');
  if (!tableWrapper) return;
  
  // ✅ v10.5.137: Debounce para actualizar indicador de posición
  let scrollTimeout = null;
  
  tableWrapper.addEventListener('scroll', () => {
    const { scrollTop, scrollHeight, clientHeight } = tableWrapper;
    const distanceToBottom = scrollHeight - scrollTop - clientHeight;
    
    // Cargar más si estamos cerca del final
    if (distanceToBottom < CONFIG.SCROLL_THRESHOLD && getHasMoreItems() && !getIsLoadingMore()) {
      loadMoreItemsCallback();
    }
    
    // ✅ v10.5.137: Actualizar indicador de posición (debounced)
    clearTimeout(scrollTimeout);
    scrollTimeout = setTimeout(() => {
      updateScrollPositionIndicator(tableWrapper);
    }, 100);
  });
}

/**
 * ✅ v10.5.137: Actualiza el indicador de posición en scroll
 * Muestra qué rango de filas está visible actualmente
 */
function updateScrollPositionIndicator(wrapper) {
  const tbody = document.getElementById('listings-tbody');
  if (!tbody) return;
  
  const rows = tbody.querySelectorAll('tr[data-listing-id]');
  if (rows.length === 0) return;
  
  const wrapperRect = wrapper.getBoundingClientRect();
  let firstVisible = null;
  let lastVisible = null;
  
  // Encontrar primera y última fila visible
  rows.forEach((row, index) => {
    const rowRect = row.getBoundingClientRect();
    const isVisible = rowRect.top < wrapperRect.bottom && rowRect.bottom > wrapperRect.top;
    
    if (isVisible) {
      if (firstVisible === null) firstVisible = index + 1;
      lastVisible = index + 1;
    }
  });
  
  // Actualizar contador si hay filas visibles
  if (firstVisible !== null && lastVisible !== null) {
    const loadedEl = document.getElementById('listings-loaded-count');
    const total = getCurrentTotal();
    
    if (loadedEl && total > 0) {
      // Mostrar rango si hay más de una fila visible y no estamos al inicio
      if (firstVisible > 1 || lastVisible < rows.length) {
        loadedEl.textContent = `${firstVisible}-${lastVisible}`;
      } else {
        loadedEl.textContent = rows.length;
      }
    }
  }
}
