/**
 * Estado del módulo listings
 * @module listings/module-state
 * 
 * ✅ v10.5.82: Fuente única de verdad para estado de listings
 * Incluye: selección, filtros, búsqueda, ordenación, selectAllByFilter
 * ✅ v2.0.0: Añadido excludedIds para "todos excepto estos"
 */

import { FILTERS } from './config.js';

// ============================================
// Estado interno del módulo
// ============================================

let currentAccountId = null;
let currentListings = [];
let selectedIds = new Set();
let currentFilter = FILTERS.ACTIVE;
let currentSearch = '';
let currentSortBy = ''; // ✅ v6.6.0: Ordenación actual

// ✅ v10.5.82: Selección persistente por filtro
let selectionByFilter = new Map();

// ✅ v10.5.82: Flag "seleccionar todos por filtro" (movido desde state-manager.js)
// Indica que el usuario quiere operar sobre TODOS los items del filtro (no solo los cargados)
let selectAllByFilter = false;

// ✅ v2.0.0: IDs excluidos cuando selectAllByFilter está activo
// Permite "seleccionar todos excepto estos"
let excludedIds = new Set();

// ✅ v10.5.82: totalItems usado por setTotalItems (único)
let totalItems = 0;

// Scroll infinito
let currentOffset = 0;
let currentTotal = 0;
let isLoadingMore = false;
let hasMoreItems = true;

// API Base (cacheada)
let API_BASE = '';
let _apiBaseReady = null;

// ============================================
// Getters
// ============================================

export function getCurrentAccountId() {
  return currentAccountId;
}

export function getCurrentListings() {
  return currentListings;
}

export function getSelectedIds() {
  return new Set(selectedIds);
}

export function getCurrentFilter() {
  return currentFilter;
}

export function getCurrentSearch() {
  return currentSearch;
}

export function getCurrentSortBy() {
  return currentSortBy;
}

export function getTotalItems() {
  return totalItems;
}

export function getCurrentOffset() {
  return currentOffset;
}

export function getCurrentTotal() {
  return currentTotal;
}

export function getHasMoreItems() {
  return hasMoreItems;
}

export function getIsLoadingMore() {
  return isLoadingMore;
}

export function getApiBase() {
  return API_BASE;
}

// ============================================
// Setters
// ============================================

export function setCurrentAccountId(id) {
  currentAccountId = id;
}

export function setCurrentListings(listings) {
  currentListings = listings;
}

export function setCurrentFilter(filter) {
  currentFilter = filter;
}

export function setCurrentSearch(search) {
  currentSearch = search;
}

export function setCurrentSortBy(sortBy) {
  currentSortBy = sortBy;
}

export function setTotalItems(total) {
  totalItems = total;
}

export function setCurrentOffset(offset) {
  currentOffset = offset;
}

export function setCurrentTotal(total) {
  currentTotal = total;
}

export function setHasMoreItems(hasMore) {
  hasMoreItems = hasMore;
}

export function setIsLoadingMore(loading) {
  isLoadingMore = loading;
}

export function setApiBase(base) {
  API_BASE = base;
}

export function setApiBaseReady(promise) {
  _apiBaseReady = promise;
}

export function getApiBaseReady() {
  return _apiBaseReady;
}

// ============================================
// Operaciones de selección
// ============================================

export function addSelection(id) {
  selectedIds.add(id);
}

export function removeSelection(id) {
  selectedIds.delete(id);
}

export function clearSelection() {
  selectedIds.clear();
}

export function hasSelection(id) {
  return selectedIds.has(id);
}

export function getSelectionCount() {
  return selectedIds.size;
}

// ✅ v10.5.82: Selección persistente por filtro
/**
 * Guarda la selección actual para el filtro actual
 */
export function saveSelectionForCurrentFilter() {
  if (selectedIds.size > 0) {
    selectionByFilter.set(currentFilter, new Set(selectedIds));
  } else {
    selectionByFilter.delete(currentFilter);
  }
}

/**
 * Restaura la selección guardada para un filtro
 * @param {string} filter - Filtro del cual restaurar
 * @returns {boolean} - True si había selección guardada
 */
export function restoreSelectionForFilter(filter) {
  const saved = selectionByFilter.get(filter);
  if (saved && saved.size > 0) {
    selectedIds = new Set(saved);
    return true;
  }
  selectedIds.clear();
  return false;
}

/**
 * Limpia todas las selecciones guardadas (al cambiar cuenta)
 */
export function clearAllSavedSelections() {
  selectionByFilter.clear();
  selectedIds.clear();
  selectAllByFilter = false;
}

// ============================================
// ✅ v10.5.82: selectAllByFilter (movido desde state-manager.js)
// ============================================

/**
 * Activa/desactiva el flag de "seleccionar todos por filtro"
 * @param {boolean} value
 */
export function setSelectAllByFilter(value) {
  selectAllByFilter = !!value;
  // ✅ v2.0.0: Al activar selectAllByFilter, limpiar excluidos
  if (value) {
    excludedIds.clear();
  }
}

/**
 * Obtiene el flag de "seleccionar todos por filtro"
 * @returns {boolean}
 */
export function getSelectAllByFilter() {
  return selectAllByFilter;
}

/**
 * Limpia el flag de "seleccionar todos por filtro"
 */
export function clearSelectAllByFilter() {
  selectAllByFilter = false;
  excludedIds.clear(); // ✅ v2.0.0: También limpiar excluidos
}

// ✅ v2.0.0: Gestión de IDs excluidos
/**
 * Añade un ID a la lista de excluidos (cuando se desmarca con selectAllByFilter activo)
 * @param {number} id
 */
export function addExcludedId(id) {
  excludedIds.add(id);
}

/**
 * Quita un ID de la lista de excluidos (cuando se vuelve a marcar)
 * @param {number} id
 */
export function removeExcludedId(id) {
  excludedIds.delete(id);
}

/**
 * Obtiene los IDs excluidos
 * @returns {Set<number>}
 */
export function getExcludedIds() {
  return new Set(excludedIds);
}

/**
 * Obtiene el número de excluidos
 * @returns {number}
 */
export function getExcludedCount() {
  return excludedIds.size;
}

/**
 * Verifica si un ID está excluido
 * @param {number} id
 * @returns {boolean}
 */
export function isExcluded(id) {
  return excludedIds.has(id);
}

// ============================================
// Limpieza
// ============================================

/**
 * Limpia totalItems (usado al cambiar cuenta)
 */
export function clearMetadata() {
  totalItems = 0;
}

// ============================================
// Reset de estado
// ============================================

export function resetScrollState() {
  currentOffset = 0;
  currentTotal = 0;
  hasMoreItems = true;
  currentListings = [];
}

export function resetAll() {
  currentAccountId = null;
  currentListings = [];
  selectedIds.clear();
  excludedIds.clear(); // ✅ v2.0.0
  currentFilter = FILTERS.ACTIVE;
  currentSearch = '';
  totalItems = 0;
  currentOffset = 0;
  currentTotal = 0;
  isLoadingMore = false;
  hasMoreItems = true;
  selectionByFilter.clear();
  selectAllByFilter = false; // ✅ v2.0.0
}

// ============================================
// ✅ v10.5.82: Estado de selección para ButtonManager
// Fuente única de verdad - elimina duplicación
// ✅ v2.0.0: Incluye excludedCount para cálculo correcto con exclusiones
// ============================================

/**
 * Calcula y retorna todo el estado de selección necesario para los botones
 * @returns {Object} Estado completo de selección
 */
export function getSelectionState() {
  const filter = currentFilter;
  const isDeletedFilter = filter === FILTERS.DELETED;
  
  // Contar seleccionados por tipo
  let selectedActive = 0;
  let selectedDeleted = 0;
  
  selectedIds.forEach(id => {
    const listing = currentListings.find(l => l.listing_id === id);
    if (listing) {
      if (listing.status === 'active') selectedActive++;
      else if (listing.status === 'deleted') selectedDeleted++;
    }
  });
  
  // Contar totales cargados
  const totalActive = currentListings.filter(l => l.status === 'active').length;
  const totalDeleted = currentListings.filter(l => l.status === 'deleted').length;
  
  // Determinar si es "seleccionar todos"
  const relevantLoaded = isDeletedFilter ? totalDeleted : totalActive;
  const relevantSelected = isDeletedFilter ? selectedDeleted : selectedActive;
  const isSelectAll = relevantSelected === relevantLoaded && relevantLoaded > 0;
  
  return {
    filter,
    selectedCount: selectedIds.size,
    selectedActive,
    selectedDeleted,
    totalActive,
    totalDeleted,
    isSelectAll,
    selectAllByFilter,  // ✅ v10.5.82: Incluir flag
    excludedCount: excludedIds.size,  // ✅ v2.0.0: Número de excluidos
    excludedIds: new Set(excludedIds)  // ✅ v2.0.0: Copia del Set
  };
}
