/**
 * Módulo listings - Orquestador principal
 * @module listings/index
 * 
 * API pública del módulo de gestión de anuncios.
 * Coordina todos los submódulos y expone las funciones públicas.
 * 
 * ✅ v10.5.82: module-state.js es fuente única de verdad para selección
 */

import DOMUtils from '../dom-utils.js';
import { logger } from '../logger.js';
import { FILTERS } from './config.js';
import * as State from './module-state.js';
import * as ButtonManager from '../publish-button-manager.js';
import { getDbStats } from '../state-manager.js';
import { renderListings, appendListings, renderStats, renderListingRow, updateListingRow as updateRow, moveRowToTop as moveRow, clearRowState as clearRow, setRowState as setRow, updateListingsCounter } from './render.js';
import { initFilters, highlightActiveFilter, togglePublishButtons } from './filters.js';
import { setupEventListeners, attachCheckboxEvents, syncRowSelectionVisuals } from './events.js';
import { setupWebSocketListeners, renderPublishingView as renderPublishing } from './websocket.js';
import { fetchAndRender as fetchData, loadMoreItems as loadMore, setupScrollInfinite } from './progressive.js';
import { checkAndShowReminder } from '../renewal-reminder.js'; // ✅ v2.5.0

// ============================================
// Referencias DOM
// ============================================

const section = document.getElementById('listings-section');
const tbody = document.getElementById('listings-tbody');

// ============================================
// API Pública
// ============================================

/**
 * Carga la tabla de anuncios para una cuenta
 * @param {number} accountId - ID de la cuenta
 * @public
 */
export async function loadListingsTable(accountId) {
  
  const idNum = Number(accountId);
  const id = Number.isFinite(idNum) && idNum > 0 ? idNum : null;

  if (!id) {
    State.setCurrentAccountId(null);
    State.clearMetadata();
    State.setTotalItems(0);
    if (section) DOMUtils.hide(section);
    State.clearAllSavedSelections(); // ✅ v10.5.82: Limpiar todas las selecciones guardadas
    State.setCurrentListings([]);
    if (tbody) tbody.innerHTML = '';
    updateSelectionUI();
    return;
  }

  State.setCurrentAccountId(id);
  State.setCurrentFilter(FILTERS.ACTIVE);
  
  if (section) section.style.display = 'block';
  State.clearAllSavedSelections(); // ✅ v10.5.82: Limpiar selecciones al cambiar cuenta
  updateSelectionUI();

  highlightActiveFilter(FILTERS.ACTIVE);
  
  await fetchData(
    (items, total) => renderListings(items, total, () => attachCheckboxEvents(updateSelectionUI)),
    renderStats,
    highlightActiveFilter,
    togglePublishButtons,
    updateSelectionUI
  );
  
  // ✅ v2.5.0: Verificar y mostrar recordatorio de renovación (NO intrusivo)
  // Solo se muestra una vez por cuenta y sesión, después de cargar los anuncios
  setTimeout(() => {
    const currentListings = State.getCurrentListings();
    checkAndShowReminder(id, currentListings, {
      minToShow: 3  // Mínimo 3 anuncios sin renovar para mostrar
    });
  }, 1000); // Delay de 1s para no interferir con la carga
}

/**
 * Obtiene IDs de listings seleccionados
 * @returns {Set} Set de IDs seleccionados
 * @public
 */
export function getSelectedListingIds() {
  return State.getSelectedIds();
}

/**
 * Limpia la selección completamente (estado + visual)
 * ✅ v10.5.144: Añadido clearSelectAllByFilter y syncRowSelectionVisuals
 * @public
 */
export function clearSelection() {
  // 1. Limpiar estado en module-state
  State.clearSelection();
  State.clearSelectAllByFilter();
  
  // 2. Desmarcar checkboxes individuales
  const checkboxes = document.querySelectorAll('.listing-checkbox');
  checkboxes.forEach(cb => cb.checked = false);
  
  // 3. Resetear checkbox de cabecera
  const selectAllCheckbox = document.getElementById('select-all-listings');
  if (selectAllCheckbox) {
    selectAllCheckbox.checked = false;
    selectAllCheckbox.indeterminate = false;
  }
  
  // 4. Quitar clase .selected (fondo verde) de las filas
  syncRowSelectionVisuals();
  
  // 5. Actualizar ButtonManager
  updateSelectionUI();
}

/**
 * Obtiene IDs seleccionados (alias para compatibilidad)
 * @returns {Array} Array de IDs
 * @public
 */
export function getSelectedIds() {
  return Array.from(State.getSelectedIds());
}

/**
 * Obtiene el filtro actual
 * @returns {string} Filtro actual
 * @public
 */
export function getCurrentFilter() {
  return State.getCurrentFilter();
}

/**
 * Obtiene los listings actuales
 * @returns {Array} Array de listings
 * @public
 */
export function getCurrentListings() {
  return State.getCurrentListings();
}

/**
 * Actualiza una row de listing
 * @param {string} listingId - ID del listing
 * @param {Object} updates - Actualizaciones
 * @public
 */
export function updateListingRow(listingId, updates) {
  updateRow(listingId, updates);
  
  // También actualizar en estado
  const listings = State.getCurrentListings();
  const index = listings.findIndex(l => l.listing_id === listingId);
  if (index !== -1) {
    listings[index] = { ...listings[index], ...updates };
    State.setCurrentListings(listings);
  }
}

/**
 * Obtiene listings filtrados
 * @param {string} filter - Filtro a aplicar
 * @returns {Array} Listings filtrados
 * @public
 */
export function getFilteredListings(filter = 'all') {
  const listings = State.getCurrentListings();
  
  if (filter === FILTERS.ALL) {
    return listings;
  }
  
  return listings.filter(listing => {
    switch (filter) {
      case FILTERS.ACTIVE:
        return listing.status === 'active';
      case FILTERS.DELETED:
        return listing.status === 'deleted';
      case FILTERS.SOLD:
        return listing.status === 'sold' || listing.status === 'bumped';
      case FILTERS.RESERVED:
        return listing.status === 'reserved';
      default:
        return true;
    }
  });
}

/**
 * Obtiene todos los listings
 * @returns {Array} Todos los listings
 * @public
 */
export function getAllListings() {
  return State.getCurrentListings();
}

/**
 * Muestra el loader de listings
 * @param {string} message - Mensaje a mostrar
 * @public
 */
export function showListingsLoader(message = '🔄 Cargando...') {
  const loader = document.getElementById('listings-loader');
  if (!loader) return;
  
  const loaderText = loader.querySelector('.loader-text');
  if (loaderText) loaderText.textContent = message;
  
  loader.style.display = 'flex';
}

/**
 * Oculta el loader de listings
 * @public
 */
export function hideListingsLoader() {
  const loader = document.getElementById('listings-loader');
  if (loader) loader.style.display = 'none';
}

/**
 * Actualiza un listing por ID
 * @param {string} listingId - ID del listing
 * @param {Object} updates - Actualizaciones
 * @public
 */
export function updateListingById(listingId, updates) {
  updateListingRow(listingId, updates);
}

/**
 * Alias para updateListingById
 * @public
 */
export const updateListingInMetadata = updateListingById;

/**
 * Refresca los listings
 * @public
 */
export async function refreshListings() {
  const accountId = State.getCurrentAccountId();
  if (accountId) {
    await loadListingsTable(accountId);
  }
}

/**
 * Mueve una row al inicio
 * @param {string} listingId - ID del listing
 * @public
 */
export function moveRowToTop(listingId) {
  moveRow(listingId);
}

/**
 * Limpia el estado de una row
 * @param {string} listingId - ID del listing
 * @public
 */
export function clearRowState(listingId) {
  clearRow(listingId);
}

/**
 * Establece el estado de una row
 * @param {string} listingId - ID del listing
 * @param {string} state - Estado
 * @public
 */
export function setRowState(listingId, state) {
  setRow(listingId, state);
}

/**
 * Renderiza una fila individual de listing
 * @param {Object} item - Datos del listing
 * @returns {HTMLElement} - Elemento TR
 * @public
 */
export { renderListingRow };

/**
 * Actualiza el contador de listings "X de Y"
 * @param {number} loaded - Cantidad cargada
 * @param {number} total - Total disponible
 * @public
 */
export { updateListingsCounter };

// ============================================
// Helpers internos
// ============================================

/**
 * Actualiza la UI de selección (usa ButtonManager centralizado)
 * ✅ v10.5.82: Simplificado usando getSelectionState() como fuente única
 * ✅ v2.1.0: Incluye excludedCount para "todos excepto estos"
 */
function updateSelectionUI() {
  // ✅ v10.5.82: Obtener estado de selección desde fuente única (module-state.js)
  const selState = State.getSelectionState();
  const { filter, selectedCount, selectedActive, selectedDeleted, totalActive, totalDeleted, isSelectAll, selectAllByFilter, excludedCount } = selState;
  
  // Actualizar estado del checkbox "Seleccionar todos"
  const selectAllCheckbox = document.getElementById('select-all-listings');
  if (selectAllCheckbox && State.getCurrentListings().length > 0) {
    const isDeletedFilter = filter === FILTERS.DELETED;
    const relevantSelected = isDeletedFilter ? selectedDeleted : selectedActive;
    const relevantTotal = isDeletedFilter ? totalDeleted : totalActive;
    
    if (relevantSelected === 0) {
      selectAllCheckbox.checked = false;
      selectAllCheckbox.indeterminate = false;
    } else if (relevantSelected === relevantTotal && relevantTotal > 0) {
      selectAllCheckbox.checked = true;
      selectAllCheckbox.indeterminate = false;
    } else {
      selectAllCheckbox.checked = false;
      selectAllCheckbox.indeterminate = true;
    }
  }
  
  // ✅ v10.5.82: dbStats viene de state-manager (stats de BD)
  const dbStats = getDbStats();
  
  // ✅ Actualizar ButtonManager con estado unificado
  // ✅ v2.1.0: Incluir excludedCount
  ButtonManager.updateState({
    filter,
    selectedCount,
    selectedActive,
    selectedDeleted,
    totalActive,
    totalDeleted,
    isSelectAll,
    dbTotalActive: dbStats.active || 0,
    dbTotalDeleted: dbStats.deleted || 0,
    selectAllByFilter,
    excludedCount: excludedCount || 0
  });
}

/**
 * Handler de cambio de filtro
 */
async function onFilterChange(filter) {
  if (filter === FILTERS.PUBLISHING) {
    const accountId = State.getCurrentAccountId();
    await renderPublishing(accountId);
    return;
  }
  
  await fetchData(
    (items, total) => renderListings(items, total, () => attachCheckboxEvents(updateSelectionUI)),
    renderStats,
    highlightActiveFilter,
    togglePublishButtons,
    updateSelectionUI
  );
}

/**
 * Handler de load more
 */
async function handleLoadMore() {
  await loadMore(
    (items) => appendListings(items, () => attachCheckboxEvents(updateSelectionUI)),
    updateListingsCounter
  );
}

// ============================================
// Inicialización
// ============================================

(async function initListingsModule() {
  // Ocultar sección inicialmente
  if (section) DOMUtils.hide(section);
  
  // Inicializar submódulos
  initFilters(onFilterChange);
  setupEventListeners(updateSelectionUI);
  setupWebSocketListeners(loadListingsTable);
  setupScrollInfinite(handleLoadMore);
  
  // ✅ v5.0.4: Listener para búsqueda en tiempo real
  document.addEventListener('listings:search', handleSearchEvent);
  
  // ✅ v6.1.0: Listener para refrescar al editar listing
  document.addEventListener('listings:refresh', handleRefreshEvent);
  
  // ✅ v6.6.0: Listener para cambio de ordenación
  document.addEventListener('listings:sort-changed', handleSortEvent);
  
})();

// ============================================
// ✅ v5.0.4: Handler de búsqueda
// ============================================

/**
 * Handler del evento de búsqueda
 * ✅ v10.5.82: Solo delega al backend (REGLA #27: debounce 300ms ya en events.js)
 * El filtrado visual causaba inconsistencias - ahora el backend hace todo
 * @param {CustomEvent} ev - Evento con searchText
 */
async function handleSearchEvent(ev) {
  // El backend hace la búsqueda completa (título, descripción, id, inventario)
  // No hacemos filtrado visual para evitar inconsistencias
  await fetchData(
    (items, total) => renderListings(items, total, () => attachCheckboxEvents(updateSelectionUI)),
    renderStats,
    highlightActiveFilter,
    togglePublishButtons,
    updateSelectionUI
  );
}

// ============================================
// ✅ v6.1.0: Handler de refresh (edición)
// ============================================

/**
 * Handler del evento de refresh tras editar un listing
 * @param {CustomEvent} ev - Evento con listingId
 */
async function handleRefreshEvent(ev) {
  const { listingId } = ev.detail;
  
  // Refrescar tabla completa para obtener datos actualizados
  const accountId = State.getCurrentAccountId();
  if (accountId) {
    await fetchData(
      (items, total) => renderListings(items, total, () => attachCheckboxEvents(updateSelectionUI)),
      renderStats,
      highlightActiveFilter,
      togglePublishButtons,
      updateSelectionUI
    );
  }
}

// ============================================
// ✅ v6.6.0: Handler de ordenación
// ============================================

/**
 * Handler del evento de cambio de ordenación
 * @param {CustomEvent} ev - Evento con sortBy
 */
async function handleSortEvent(ev) {
  const { sortBy } = ev.detail;
  
  // Recargar tabla con nuevo orden desde backend
  const accountId = State.getCurrentAccountId();
  if (accountId) {
    await fetchData(
      (items, total) => renderListings(items, total, () => attachCheckboxEvents(updateSelectionUI)),
      renderStats,
      highlightActiveFilter,
      togglePublishButtons,
      updateSelectionUI
    );
  }
}
