/**
 * Utilidades reutilizables del módulo listings
 * @module listings/helpers
 */

import { logger } from '../logger.js';

/**
 * Debounce function - retrasa la ejecución hasta que pasen X ms sin llamadas
 * @param {Function} func - Función a ejecutar
 * @param {number} wait - Tiempo de espera en ms
 * @returns {Function} Función debounced
 */
export function debounce(func, wait) {
  let timeout;
  return function executedFunction(...args) {
    const later = () => {
      clearTimeout(timeout);
      func(...args);
    };
    clearTimeout(timeout);
    timeout = setTimeout(later, wait);
  };
}

/**
 * Trunca un título a un máximo de caracteres
 * @param {string} text - Texto a truncar
 * @param {number} maxLength - Longitud máxima
 * @returns {string} Texto truncado
 */
export function truncateTitle(text, maxLength = 40) {
  if (!text) return '(sin título)';
  // ✅ v6.5.0: Asegurar que text es string
  const str = typeof text === 'string' ? text : String(text);
  if (str.length <= maxLength) return str;
  return str.substring(0, maxLength) + '...';
}

/**
 * Formatea precio con moneda
 * @param {number} amount - Cantidad
 * @param {string} currency - Moneda (EUR, USD, etc)
 * @returns {string} Precio formateado
 */
export function formatPrice(amount, currency = 'EUR') {
  if (amount === null || amount === undefined) return '—';
  
  const formatter = new Intl.NumberFormat('es-ES', {
    style: 'currency',
    currency: currency || 'EUR'
  });
  
  return formatter.format(amount);
}

/**
 * Obtiene el badge HTML para un estado
 * @param {string} status - Estado del listing
 * @returns {string} HTML del badge
 */
export function getStatusBadge(status) {
  const badges = {
    active: '<span class="badge badge-active" title="Activo">✅ Activo</span>',
    deleted: '<span class="badge badge-deleted" title="Eliminado">🗑️ Eliminado</span>',
    sold: '<span class="badge badge-sold" title="Vendido">💰 Vendido</span>',
    reserved: '<span class="badge badge-reserved" title="Reservado">⏸️ Reservado</span>',
    onhold: '<span class="badge badge-onhold" title="En espera">⏳ En espera</span>'
  };
  
  return badges[status] || '<span class="badge">❓ —</span>';
}

/**
 * Convierte URL de imagen a formato completo
 * @param {string} path - Path de la imagen
 * @param {string} apiBase - API_BASE del servidor
 * @returns {string} URL completa de imagen
 */
export function getImageUrl(path, apiBase = '') {
  if (!path) return '';
  
  // Si ya es URL completa, retornar tal cual
  if (/^https?:\/\//i.test(path)) return path;
  
  // Si no hay API_BASE, no podemos construir URL
  if (!apiBase) return '';
  
  // Si empieza con /, concatenar directamente con API_BASE
  if (path.startsWith('/')) return `${apiBase}${path}`;
  
  // Path relativo → agregar /data/ (Nginx sirve desde /data/)
  return `${apiBase}/data/${path}`;
}

/**
 * Envía mensaje al Service Worker
 * @param {Object} msg - Mensaje a enviar
 * @returns {Promise<Object>} Respuesta del SW
 */
export async function sw(msg) {
  return new Promise((resolve) => {
    try {
      chrome.runtime.sendMessage(msg, (response) => {
        resolve(response || { ok: false });
      });
    } catch {
      resolve({ ok: false });
    }
  });
}

/**
 * Carga y cachea API_BASE desde el SW (idempotente)
 * @param {Function} getApiBase - Getter del API_BASE actual
 * @param {Function} getApiBaseReady - Getter de la promesa de carga
 * @param {Function} setApiBase - Setter del API_BASE
 * @param {Function} setApiBaseReady - Setter de la promesa
 * @returns {Promise<string>} API_BASE
 */
export async function ensureApiBase(getApiBase, getApiBaseReady, setApiBase, setApiBaseReady) {
  const current = getApiBase();
  if (current) return current;
  
  const ready = getApiBaseReady();
  if (!ready) {
    const promise = (async () => {
      try {
        const s = await sw({ type: 'SESSION.GET' });
        const base = (s?.data?.config?.API_BASE || '').replace(/\/$/, '');
        setApiBase(base);
      } catch {
        setApiBase('');
      }
      return getApiBase();
    })();
    
    setApiBaseReady(promise);
    return promise;
  }
  
  return ready;
}

/**
 * Muestra un mensaje de error al usuario
 * @param {string} message - Mensaje a mostrar
 */
export function showError(message) {
  logger.error('[LISTINGS]', message);
  // Reutilizar toast del módulo utils si está disponible
  if (window.toast) {
    window.toast(message, 'error');
  }
}

/**
 * Actualiza el contador de listings cargados/total
 * ✅ v10.5.82: Maneja búsqueda vs carga normal
 * @param {number} loaded - Número de listings cargados/mostrados
 * @param {number} total - Total de listings (en BD o resultados de búsqueda)
 * @param {boolean} isSearchResult - Si es resultado de búsqueda
 */
export function updateListingsCounter(loaded, total, isSearchResult = false) {
  const loadedEl = document.getElementById('listings-loaded-count');
  const totalEl = document.getElementById('listings-total-count');
  const separatorEl = document.querySelector('.listings-counter-separator');
  
  if (loadedEl) loadedEl.textContent = loaded;
  
  if (isSearchResult && total === loaded) {
    // Búsqueda: mostrar "X resultados" (total = loaded, no hace falta "de Y")
    if (totalEl) totalEl.textContent = '';
    if (separatorEl) separatorEl.style.display = 'none';
  } else {
    // Carga normal o búsqueda con paginación: mostrar "X de Y"
    if (totalEl) totalEl.textContent = total;
    if (separatorEl) separatorEl.style.display = '';
  }
}

/**
 * Muestra el loader de listings
 * @param {string} message - Mensaje a mostrar
 */
export function showListingsLoader(message = '🔄 Cargando...') {
  const loader = document.getElementById('listings-loader');
  if (!loader) return;
  
  const loaderText = loader.querySelector('.loader-text');
  if (loaderText) loaderText.textContent = message;
  
  loader.style.display = 'flex';
}

/**
 * Oculta el loader de listings
 */
export function hideListingsLoader() {
  const loader = document.getElementById('listings-loader');
  if (loader) loader.style.display = 'none';
}
