/**
 * Sistema de filtros de listings
 * @module listings/filters
 */

import DOMUtils from '../dom-utils.js';
import { logger } from '../logger.js';
import { FILTERS } from './config.js';
import {
  getCurrentFilter,
  setCurrentFilter,
  saveSelectionForCurrentFilter,
  restoreSelectionForFilter
} from './module-state.js';

// ============================================
// UI de filtros
// ============================================

/**
 * Inicializa el sistema de filtros con delegación de eventos
 * @param {Function} onFilterChange - Callback cuando cambia el filtro
 */
export function initFilters(onFilterChange) {
  const wrap = document.querySelector('.ml-listings-stats');
  if (!wrap) {
    return;
  }

  // Remover listener anterior
  wrap.removeEventListener('click', handleStatsFilterClick);
  wrap.addEventListener('click', async (ev) => {
    await handleStatsFilterClick(ev, onFilterChange);
  });
  
  // Ocultar select antiguo si existe
  const oldSelect = document.getElementById('listings-filter');
  if (oldSelect) DOMUtils.hide(oldSelect);
  
  // ✅ Marcar filtro ACTIVE por defecto visualmente desde el inicio
  highlightActiveFilter(FILTERS.ACTIVE);
  
}

/**
 * Handler del click en filtros
 * ✅ v10.5.82: Guarda selección antes de cambiar, restaura al volver
 */
async function handleStatsFilterClick(ev, onFilterChange) {
  const stat = ev.target.closest('.stat');
  const wrap = document.querySelector('.ml-listings-stats');
  if (!stat || !wrap.contains(stat)) return;
  
  const newFilter = stat.dataset.filter || FILTERS.ALL;
  const oldFilter = getCurrentFilter();
  
  // No hacer nada si es el mismo filtro
  if (newFilter === oldFilter) return;
  
  // ✅ v10.5.82: Guardar selección del filtro actual antes de cambiar
  saveSelectionForCurrentFilter();
  
  // Cambiar al nuevo filtro
  setCurrentFilter(newFilter);
  
  // ✅ v10.5.82: Limpiar búsqueda al cambiar filtro (evita confusión)
  const searchInput = document.getElementById('listings-search');
  if (searchInput && searchInput.value) {
    searchInput.value = '';
    // Importar setCurrentSearch para limpiar estado
    import('./module-state.js').then(({ setCurrentSearch }) => {
      setCurrentSearch('');
    });
  }
  
  // ✅ v10.5.82: Restaurar selección guardada del nuevo filtro (si existe)
  restoreSelectionForFilter(newFilter);
  
  highlightActiveFilter(newFilter);
  
  if (onFilterChange) {
    await onFilterChange(newFilter);
  }
}

/**
 * Resalta el filtro activo visualmente
 * @param {string} filter - Filtro a resaltar
 */
export function highlightActiveFilter(filter) {
  document.querySelectorAll('.ml-listings-stats .stat')
    .forEach(b => b.classList.toggle('is-selected', (b.dataset.filter || FILTERS.ALL) === filter));
}

/**
 * Toggle visibilidad de botones según filtro
 * @param {string} filter - Filtro actual
 */
export function togglePublishButtons(filter) {
  const btnPublishAll = document.getElementById('btn-publish');
  const btnRepublishAll = document.getElementById('btn-republish-deleted');
  const deletedCount = document.querySelector('.stat.deleted .stat-value')?.textContent || '0';
  
  if (filter === FILTERS.DELETED) {
    // Filtro eliminados: mostrar republicar
    if (btnPublishAll) btnPublishAll.style.display = 'none';
    if (btnRepublishAll) {
      btnRepublishAll.style.display = '';
      const label = btnRepublishAll.querySelector('.label');
      if (label) label.textContent = `Republicar Todos (${deletedCount})`;
    }
  } else {
    // Otros filtros: mostrar publicar
    if (btnPublishAll) btnPublishAll.style.display = '';
    if (btnRepublishAll) btnRepublishAll.style.display = 'none';
  }
}
