/**
 * Event handlers del módulo listings
 * @module listings/events
 * ✅ v2.0.0: Soporte para excludedIds (todos excepto estos)
 */

import { logger } from '../logger.js';
import { toast } from '../../utils.js';
import {
  hasSelection,
  addSelection,
  removeSelection,
  clearSelection,
  getSelectedIds,
  getCurrentListings,
  getCurrentFilter,
  setSelectAllByFilter,
  clearSelectAllByFilter,
  getSelectAllByFilter,
  addExcludedId,
  removeExcludedId,
  isExcluded
} from './module-state.js';
import { getDbStats } from '../state-manager.js';
import { FILTERS, CONFIG } from './config.js';

// ============================================
// Setup de eventos
// ============================================

/**
 * Inicializa todos los event listeners
 * @param {Function} updateSelectionUI - Callback para actualizar UI
 */
export function setupEventListeners(updateSelectionUI) {
  setupImageModalDelegation();
  setupSelectAllListener(updateSelectionUI);
  setupSearchListener(); // ✅ v5.0.4: Buscador en tiempo real
  setupInventoryInputListeners(); // ✅ v5.9.0: Auto-save inventory notes
  setupRowClickToSelect(updateSelectionUI); // ✅ v10.5.140: Click para seleccionar
  setupRowDoubleClickListener(); // ✅ v6.1.0: Doble-click para editar
  setupSortingListeners(); // ✅ v6.6.0: Ordenación por columnas
}

/**
 * Delega eventos de modal de imagen en tbody
 */
function setupImageModalDelegation() {
  const tbody = document.getElementById('listings-tbody');
  
  if (!tbody) {
    return;
  }
  
  tbody.addEventListener('click', (e) => {
    const img = e.target.closest('.listing-thumb');
    
    if (img) {
      e.stopPropagation();
      e.preventDefault();
      
      try {
        const currentSrc = img.src;
        const fullSrc = currentSrc
          .replace('/small/', '/xlarge/')
          .replace('/medium/', '/xlarge/')
          .replace('/large/', '/xlarge/');
        
        // Abrir imagen en nueva ventana
        window.open(fullSrc, '_blank');
      } catch (error) {
        logger.error('[EVENTS] Error opening image:', error);
      }
    }
  });
}

/**
 * Configura listener para "Seleccionar todos"
 * @param {Function} updateSelectionUI - Callback para actualizar UI
 */
function setupSelectAllListener(updateSelectionUI) {
  const selectAllCheckbox = document.getElementById('select-all-listings');
  if (!selectAllCheckbox) return;
  
  selectAllCheckbox.addEventListener('change', (ev) => {
    handleSelectAll(ev, updateSelectionUI);
  });
}

/**
 * Handler de "Seleccionar todos"
 */
function handleSelectAll(ev, updateSelectionUI) {
  const checked = ev.target.checked;
  const filter = getCurrentFilter();
  const isDeletedFilter = filter === FILTERS.DELETED;
  
  if (checked) {
    // Seleccionar según filtro
    const listings = getCurrentListings();
    const dbStats = getDbStats();
    
    // Contar cuántos hay cargados vs en BD
    const loadedActive = listings.filter(l => l.status === 'active').length;
    const loadedDeleted = listings.filter(l => l.status === 'deleted').length;
    const totalActiveInDb = dbStats.active || 0;
    const totalDeletedInDb = dbStats.deleted || 0;
    
    const relevantLoaded = isDeletedFilter ? loadedDeleted : loadedActive;
    const relevantInDb = isDeletedFilter ? totalDeletedInDb : totalActiveInDb;
    
    // ✅ v6.0.0: Si hay más en BD que los cargados, activar flag automáticamente
    if (relevantInDb > relevantLoaded) {
      setSelectAllByFilter(true);
    } else {
      clearSelectAllByFilter();
    }
    
    // Seleccionar los cargados visualmente
    listings.forEach(item => {
      const isActive = item.status === 'active';
      const isDeleted = item.status === 'deleted';
      if (isActive || (isDeleted && isDeletedFilter)) {
        addSelection(item.listing_id);
      }
    });
  } else {
    clearSelection();
    clearSelectAllByFilter();
  }
  
  // Actualizar checkboxes individuales
  updateCheckboxesFromSelection();
  
  if (updateSelectionUI) {
    updateSelectionUI();
  }
}

/**
 * Actualiza checkboxes visuales según selección en estado
 */
function updateCheckboxesFromSelection() {
  const checkboxes = document.querySelectorAll('.listing-checkbox');
  checkboxes.forEach(checkbox => {
    const id = Number(checkbox.dataset.id);
    if (!checkbox.disabled && id) {
      checkbox.checked = hasSelection(id);
    }
  });
  
  // ✅ v10.5.141: Sincronizar también los backgrounds de filas
  syncRowSelectionVisuals();
}

/**
 * ✅ v10.5.141: Sincroniza clase .selected en TODAS las filas del DOM
 * según el estado de selección en module-state.js
 * Debe llamarse después de cualquier cambio de selección
 */
export function syncRowSelectionVisuals() {
  const rows = document.querySelectorAll('#listings-tbody tr[data-listing-id]');
  
  rows.forEach(row => {
    const listingId = parseInt(row.dataset.listingId, 10);
    if (!listingId) return;
    
    const isSelected = hasSelection(listingId);
    
    if (isSelected) {
      row.classList.add('selected');
    } else {
      row.classList.remove('selected');
    }
  });
}

/**
 * Adjunta eventos a checkboxes individuales
 * @param {Function} updateSelectionUI - Callback para actualizar UI
 */
export function attachCheckboxEvents(updateSelectionUI) {
  const checkboxes = document.querySelectorAll('.listing-checkbox');
  
  checkboxes.forEach(checkbox => {
    // Remover listener anterior
    checkbox.removeEventListener('change', handleCheckboxChange);
    
    // Agregar nuevo listener
    checkbox.addEventListener('change', (ev) => {
      handleCheckboxChange(ev, updateSelectionUI);
    });
  });
  
  // ✅ v10.5.141: Sincronizar backgrounds al añadir nuevas filas
  syncRowSelectionVisuals();
}

/**
 * Handler de checkbox individual
 * ✅ v5.0.0: Con validación de selección según filtro
 */
function handleCheckboxChange(ev, updateSelectionUI) {
  const checkbox = ev.target;
  const id = Number(checkbox.dataset.id);
  
  if (!id) return;
  
  // ✅ v5.0.0: Validar si se puede seleccionar este anuncio
  const listing = getCurrentListings().find(l => l.listing_id === id);
  if (!listing) {
    checkbox.checked = false;
    return;
  }
  
  const filter = getCurrentFilter();
  
  // BLOQUEAR selección en filtros de solo lectura
  if (filter === FILTERS.FEATURED || filter === FILTERS.RESERVED) {
    checkbox.checked = false;
    return;
  }
  
  // BLOQUEAR selección de anuncios no permitidos según filtro
  if ((filter === FILTERS.ACTIVE || filter === FILTERS.TOTAL) && listing.status !== 'active') {
    // En activos/total solo se pueden seleccionar activos
    checkbox.checked = false;
    return;
  }
  
  if (filter === FILTERS.DELETED && listing.status !== 'deleted') {
    // En eliminados solo se pueden seleccionar eliminados
    checkbox.checked = false;
    return;
  }
  
  // ✅ Validación pasada - proceder con selección
  const row = checkbox.closest('tr');
  const selectAllActive = getSelectAllByFilter();
  
  if (checkbox.checked) {
    addSelection(id);
    // ✅ v2.0.0: Si selectAllByFilter está activo, quitar de excluidos
    if (selectAllActive) {
      removeExcludedId(id);
    }
    // ✅ v6.3.0: Animación de feedback al seleccionar
    if (row) {
      row.classList.add('selected', 'row-selected-animation');
      setTimeout(() => row.classList.remove('row-selected-animation'), 300);
    }
  } else {
    removeSelection(id);
    // ✅ v2.0.0: Si selectAllByFilter está activo, añadir a excluidos (NO limpiar el flag)
    // Esto permite "todos excepto estos" manteniendo el conteo correcto
    if (selectAllActive) {
      addExcludedId(id);
    }
    // ✅ v6.3.0: Quitar clase selected
    if (row) {
      row.classList.remove('selected');
    }
  }
  
  // Actualizar estado de "Seleccionar todos"
  updateSelectAllState();
  
  if (updateSelectionUI) {
    updateSelectionUI();
  }
}

/**
 * Actualiza estado del checkbox "Seleccionar todos"
 */
function updateSelectAllState() {
  const selectAllCheckbox = document.getElementById('select-all-listings');
  if (!selectAllCheckbox) return;
  
  const checkboxes = Array.from(document.querySelectorAll('.listing-checkbox:not([disabled])'));
  const checkedCount = checkboxes.filter(cb => cb.checked).length;
  
  if (checkedCount === 0) {
    selectAllCheckbox.checked = false;
    selectAllCheckbox.indeterminate = false;
  } else if (checkedCount === checkboxes.length) {
    selectAllCheckbox.checked = true;
    selectAllCheckbox.indeterminate = false;
  } else {
    selectAllCheckbox.checked = false;
    selectAllCheckbox.indeterminate = true;
  }
}

// ============================================
// ✅ v5.0.4: Buscador en tiempo real
// ============================================

/**
 * Configura listener para el buscador
 */
function setupSearchListener() {
  const searchInput = document.getElementById('listings-search');
  if (!searchInput) {
    return;
  }
  
  // Usar 'input' para búsqueda en tiempo real
  searchInput.addEventListener('input', handleSearchInput);
}

/**
 * Handler de búsqueda en tiempo real
 * ✅ v10.5.82: Usa CONFIG.SEARCH_DEBOUNCE (REGLA #27)
 */
let searchDebounceTimer = null;

function handleSearchInput(ev) {
  const searchText = ev.target.value.trim(); // ✅ v10.5.134: Normalizar texto
  
  // Guardar en estado (import estático ya existe arriba)
  import('./module-state.js').then(({ setCurrentSearch }) => {
    setCurrentSearch(searchText);
  });
  
  // ✅ REGLA #27: Debounce usando CONFIG
  clearTimeout(searchDebounceTimer);
  searchDebounceTimer = setTimeout(() => {
    // Disparar evento personalizado para que el módulo principal recargue desde backend
    const searchEvent = new CustomEvent('listings:search', { 
      detail: { searchText } 
    });
    document.dispatchEvent(searchEvent);
  }, CONFIG.SEARCH_DEBOUNCE);
}

// ============================================
// ✅ v5.9.0: Inventory Notes Auto-save
// ============================================

let inventoryDebounceTimers = {}; // { listing_id: timeoutId }

/**
 * Setup listeners para inputs de inventario con auto-save
 */
function setupInventoryInputListeners() {
  const tbody = document.getElementById('listings-tbody');
  
  if (!tbody) {
    return;
  }
  
  // Event delegation para todos los inputs .inventory-input
  tbody.addEventListener('input', async (e) => {
    const input = e.target.closest('.inventory-input');
    
    if (!input) return;
    
    const listingId = input.dataset.listingId;
    const webId = input.dataset.webId;
    const note = input.value.trim();
    
    // Cancelar timer anterior para este listing
    if (inventoryDebounceTimers[listingId]) {
      clearTimeout(inventoryDebounceTimers[listingId]);
    }
    
    // ✅ v10.5.82: Usar clases CSS en lugar de inline styles
    input.classList.remove('is-saved', 'is-error');
    input.classList.add('is-saving');
    
    // Debounce de 800ms
    inventoryDebounceTimers[listingId] = setTimeout(async () => {
      try {
        
        // Enviar al service worker
        const response = await chrome.runtime.sendMessage({
          type: 'LISTING.INVENTORY.UPDATE',
          payload: {
            listing_web_id: webId,
            inventory_note: note
          }
        });
        
        if (response?.ok) {
          // Feedback: guardado exitoso
          input.classList.remove('is-saving');
          input.classList.add('is-saved');
          setTimeout(() => {
            input.classList.remove('is-saved');
          }, 1500);
          
        } else {
          throw new Error(response?.error || 'Error desconocido');
        }
      } catch (error) {
        logger.error('[INVENTORY] ❌ Error guardando:', error);
        
        // Feedback: error
        input.classList.remove('is-saving');
        input.classList.add('is-error');
        setTimeout(() => {
          input.classList.remove('is-error');
        }, 2000);
      }
      
      delete inventoryDebounceTimers[listingId];
    }, 800);
  });
  
}

// ============================================
// ✅ v6.1.0: Doble-click para editar anuncio
// ============================================

/**
 * ✅ v10.5.140: Click simple en fila para seleccionar/deseleccionar
 * Excepto: imagen (zoom), checkbox (ya tiene su handler), input inventario
 */
function setupRowClickToSelect(updateSelectionUI) {
  const tbody = document.getElementById('listings-tbody');
  if (!tbody) return;
  
  // Usar delegación de eventos
  tbody.addEventListener('click', (e) => {
    const target = e.target;
    
    // Ignorar clicks en elementos interactivos
    if (
      target.closest('.listing-checkbox') ||
      target.closest('.inventory-input') ||
      target.closest('.listing-thumb') ||
      target.closest('.listing-thumb-placeholder') ||
      target.closest('img')
    ) {
      return;
    }
    
    // Buscar la fila
    const row = target.closest('tr[data-listing-id]');
    if (!row) return;
    
    const listingId = parseInt(row.dataset.listingId, 10);
    if (!listingId) return;
    
    // Buscar el checkbox de esta fila
    const checkbox = row.querySelector('.listing-checkbox');
    if (!checkbox || checkbox.disabled) return;
    
    // Toggle selección
    checkbox.checked = !checkbox.checked;
    
    // Disparar evento change manualmente
    checkbox.dispatchEvent(new Event('change', { bubbles: true }));
  });
}

/**
 * Setup listener para doble-click en filas de la tabla
 * Abre el editor de anuncios
 */
function setupRowDoubleClickListener() {
  const tbody = document.getElementById('listings-tbody');
  
  if (!tbody) {
    return;
  }
  
  tbody.addEventListener('dblclick', async (e) => {
    // Ignorar doble-click en elementos interactivos
    const target = e.target;
    if (
      target.closest('.listing-checkbox') ||
      target.closest('.inventory-input') ||
      target.closest('.btn-actions') ||
      target.closest('.listing-thumb')
    ) {
      return;
    }
    
    // Buscar la fila
    const row = target.closest('tr[data-listing-id]');
    if (!row) return;
    
    const listingId = parseInt(row.dataset.listingId, 10);
    if (!listingId) return;
    
    // Verificar si es editable (no reservado ni destacado)
    const listing = getCurrentListings().find(l => l.listing_id === listingId);
    if (!listing) {
      return;
    }
    
    // Bloquear edición de reservados
    if (listing.status === 'reserved') {
      toast('⚠️ No se pueden editar anuncios reservados', 'warn');
      return;
    }
    
    
    // Abrir editor - usar import dinámico solo en el panel (no en SW)
    try {
      const editorModule = await import('../listing-editor/index.js');
      editorModule.openEditor(listingId);
    } catch (error) {
      logger.error('[EVENTS] Error loading listing editor:', error);
      toast('Error abriendo editor', 'err');
    }
  });
  
  // Añadir estilos de cursor para indicar que es clickeable
  addEditableRowStyles();
  
}

/**
 * Añade estilos CSS dinámicos para filas editables
 */
function addEditableRowStyles() {
  // Verificar si ya existe
  if (document.getElementById('editable-row-styles')) return;
  
  const style = document.createElement('style');
  style.id = 'editable-row-styles';
  style.textContent = `
    /* ✅ v6.1.0: Estilos para filas editables */
    #listings-tbody tr[data-listing-id] {
      cursor: pointer;
    }
    
    #listings-tbody tr[data-listing-id]:hover .col-title {
      text-decoration: underline;
      text-decoration-style: dotted;
      text-underline-offset: 2px;
    }
    
    #listings-tbody tr[data-listing-id] .col-title {
      position: relative;
    }
    
    #listings-tbody tr[data-listing-id]:hover .col-title::after {
      content: '✏️';
      position: absolute;
      right: -20px;
      top: 50%;
      transform: translateY(-50%);
      font-size: 12px;
      opacity: 0.6;
    }
    
    /* Bloquear cursor en reservados */
    #listings-tbody tr[data-status="reserved"] {
      cursor: not-allowed;
    }
    
    #listings-tbody tr[data-status="reserved"]:hover .col-title {
      text-decoration: none;
    }
    
    #listings-tbody tr[data-status="reserved"]:hover .col-title::after {
      content: '🔒';
    }
  `;
  
  document.head.appendChild(style);
}

/**
 * ✅ v6.6.0: Setup de sorting por columnas
 * ✅ v10.5.82: Persiste en localStorage
 * ✅ v10.5.137: Soporta múltiples columnas (precio, fecha)
 */
function setupSortingListeners() {
  const sortableHeaders = document.querySelectorAll('.ml-listings-table th.sortable');
  
  if (sortableHeaders.length === 0) {
    return;
  }
  
  // ✅ Restaurar ordenación guardada
  const savedSort = localStorage.getItem('ml_listings_sort');
  if (savedSort) {
    import('./module-state.js').then(({ setCurrentSortBy }) => {
      setCurrentSortBy(savedSort);
      // Actualizar UI del header correspondiente
      updateSortHeaderUI(savedSort);
    });
  }
  
  // Añadir listener a cada header sortable
  sortableHeaders.forEach(header => {
    header.addEventListener('click', async () => {
      const sortType = header.dataset.sort; // 'published' o 'price'
      await handleSortClick(sortType, header);
    });
  });
}

/**
 * ✅ v10.5.137: Maneja click en header de ordenación
 */
async function handleSortClick(sortType, clickedHeader) {
  const tbody = document.getElementById('listings-tbody');
  if (!tbody) return;
  
  const rows = Array.from(tbody.querySelectorAll('tr[data-listing-id]'));
  if (rows.length === 0) return;
  
  const { getCurrentSortBy, setCurrentSortBy } = await import('./module-state.js');
  
  const currentSort = getCurrentSortBy();
  let newSort, sortDir;
  
  // Determinar nueva ordenación
  if (sortType === 'published') {
    if (currentSort === 'published_desc') {
      newSort = 'published_asc';
      sortDir = 'asc';
    } else {
      newSort = 'published_desc';
      sortDir = 'desc';
    }
  } else if (sortType === 'price') {
    if (currentSort === 'price_desc') {
      newSort = 'price_asc';
      sortDir = 'asc';
    } else {
      newSort = 'price_desc';
      sortDir = 'desc';
    }
  } else if (sortType === 'category') {
    // ✅ v10.5.138: Ordenación por categoría
    if (currentSort === 'category_desc') {
      newSort = 'category_asc';
      sortDir = 'asc';
    } else {
      newSort = 'category_desc';
      sortDir = 'desc';
    }
  } else if (sortType === 'title') {
    // ✅ v10.5.139: Ordenación por título
    if (currentSort === 'title_desc') {
      newSort = 'title_asc';
      sortDir = 'asc';
    } else {
      newSort = 'title_desc';
      sortDir = 'desc';
    }
  }
  
  // Actualizar estado
  setCurrentSortBy(newSort);
  localStorage.setItem('ml_listings_sort', newSort);
  
  // Actualizar UI de todos los headers
  updateSortHeaderUI(newSort);
  
  // Ordenar filas del DOM
  rows.sort((a, b) => {
    let valA, valB;
    
    if (sortType === 'published') {
      valA = a.dataset.published || '';
      valB = b.dataset.published || '';
      
      if (!valA && !valB) return 0;
      if (!valA) return 1;
      if (!valB) return -1;
      
      const comparison = valA.localeCompare(valB);
      return sortDir === 'asc' ? comparison : -comparison;
    } else if (sortType === 'price') {
      valA = parseFloat(a.dataset.price) || 0;
      valB = parseFloat(b.dataset.price) || 0;
      
      const comparison = valA - valB;
      return sortDir === 'asc' ? comparison : -comparison;
    } else if (sortType === 'category') {
      // ✅ v10.5.138: Ordenación por categoría
      valA = (a.dataset.category || '').toLowerCase();
      valB = (b.dataset.category || '').toLowerCase();
      
      if (!valA && !valB) return 0;
      if (!valA) return 1;
      if (!valB) return -1;
      
      const comparison = valA.localeCompare(valB);
      return sortDir === 'asc' ? comparison : -comparison;
    } else if (sortType === 'title') {
      // ✅ v10.5.139: Ordenación por título
      // Obtener título del span dentro de la fila
      const titleA = a.querySelector('.listing-title')?.textContent || '';
      const titleB = b.querySelector('.listing-title')?.textContent || '';
      
      valA = titleA.toLowerCase();
      valB = titleB.toLowerCase();
      
      const comparison = valA.localeCompare(valB);
      return sortDir === 'asc' ? comparison : -comparison;
    }
    
    return 0;
  });
  
  // Re-insertar filas ordenadas
  rows.forEach(row => tbody.appendChild(row));
  
  // Disparar evento para que el backend lo use en scroll infinito
  document.dispatchEvent(new CustomEvent('listings:sort-changed', {
    detail: { sortBy: newSort }
  }));
}

/**
 * ✅ v10.5.137: Actualiza UI de headers de ordenación
 * ✅ v10.5.138: Añadido soporte para categoría
 * ✅ v10.5.139: Añadido soporte para título
 */
function updateSortHeaderUI(currentSort) {
  // Limpiar todos los headers
  document.querySelectorAll('.ml-listings-table th.sortable').forEach(h => {
    h.classList.remove('sort-asc', 'sort-desc');
  });
  
  // Aplicar clase al header activo
  if (currentSort.startsWith('published')) {
    const header = document.querySelector('.col-published.sortable');
    if (header) {
      header.classList.add(currentSort === 'published_asc' ? 'sort-asc' : 'sort-desc');
    }
  } else if (currentSort.startsWith('price')) {
    const header = document.querySelector('.col-price.sortable');
    if (header) {
      header.classList.add(currentSort === 'price_asc' ? 'sort-asc' : 'sort-desc');
    }
  } else if (currentSort.startsWith('category')) {
    const header = document.querySelector('.col-category.sortable');
    if (header) {
      header.classList.add(currentSort === 'category_asc' ? 'sort-asc' : 'sort-desc');
    }
  } else if (currentSort.startsWith('title')) {
    const header = document.querySelector('.col-title.sortable');
    if (header) {
      header.classList.add(currentSort === 'title_asc' ? 'sort-asc' : 'sort-desc');
    }
  }
}

/**
 * Inicializa eventos cuando DOM está listo
 */
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', setupImageModalDelegation);
} else {
  setupImageModalDelegation();
}
