/**
 * Modal de edición de anuncios
 * @module listing-editor/modal
 * @version 6.3.0
 * 
 * Renderiza la modal con campos editables y solo lectura.
 * Incluye contadores de caracteres y validación en tiempo real.
 * v6.3.0: Sistema de pestañas y mejora de contraste.
 */

import { logger } from '../logger.js';
import { initImageManager, getImageState, cleanup as cleanupImageManager } from './image-manager.js';
import { getApiBase } from '../listings/module-state.js';
import { escapeHtml } from '../../utils.js'; // ✅ v10.5.27: Centralizado

// ============================================
// Estado del módulo
// ============================================

let modalElement = null;
let currentListing = null;
let onSaveCallback = null;
let onCloseCallback = null;
let imageChanges = null; // Estado de cambios en imágenes

// Límites de caracteres
const TITLE_MAX_LENGTH = 50;
const DESCRIPTION_MAX_LENGTH = 640;

// ============================================
// Apertura de modal
// ============================================

/**
 * Abre o actualiza la modal de edición
 * @param {Object} options
 * @param {boolean} options.loading - Si está cargando
 * @param {Object} options.listing - Datos del listing (si no loading)
 * @param {Function} options.onSave - Callback al guardar
 * @param {Function} options.onClose - Callback al cerrar
 */
export function openEditorModal({ loading = false, listing = null, onSave = null, onClose = null }) {
  if (onSave) onSaveCallback = onSave;
  if (onClose) onCloseCallback = onClose;
  
  if (loading) {
    renderLoadingModal();
  } else if (listing) {
    currentListing = listing;
    renderEditorModal(listing);
  }
}

/**
 * Cierra y limpia la modal
 */
export function closeEditorModal() {
  if (modalElement) {
    modalElement.remove();
    modalElement = null;
  }
  currentListing = null;
  onSaveCallback = null;
  onCloseCallback = null;
  imageChanges = null;
  cleanupImageManager();
}

// ============================================
// Renderizado
// ============================================

/**
 * Renderiza modal en estado de carga
 */
function renderLoadingModal() {
  closeEditorModal();
  
  modalElement = document.createElement('div');
  modalElement.className = 'listing-editor-overlay';
  modalElement.innerHTML = `
    <div class="listing-editor-modal" role="dialog" aria-modal="true" aria-label="Editar Anuncio">
      <div class="listing-editor-header">
        <h3>📝 Cargando anuncio...</h3>
        <button class="listing-editor-close" title="Cerrar">✕</button>
      </div>
      <div class="listing-editor-body">
        <div class="listing-editor-loading">
          <div class="spinner"></div>
          <p>Obteniendo datos del anuncio...</p>
        </div>
      </div>
    </div>
  `;
  
  // Event listener para cerrar
  modalElement.querySelector('.listing-editor-close').addEventListener('click', () => {
    if (onCloseCallback) onCloseCallback();
  });
  
  // Click en overlay cierra
  modalElement.addEventListener('click', (e) => {
    if (e.target === modalElement) {
      if (onCloseCallback) onCloseCallback();
    }
  });
  
  document.body.appendChild(modalElement);
}

/**
 * Renderiza modal con datos del listing
 * @param {Object} listing - Datos del listing
 */
function renderEditorModal(listing) {
  if (!modalElement) {
    modalElement = document.createElement('div');
    modalElement.className = 'listing-editor-overlay';
    document.body.appendChild(modalElement);
  }
  
  // Preparar atributos para mostrar (solo lectura)
  const attributesHtml = renderAttributes(listing.type_attributes);
  
  // Preparar imágenes
  const imagesHtml = renderImages(listing.images_wallapop, listing.images_local);
  
  // ✅ v6.3.0: Mapeos de estado (consistente con listings/helpers.js)
  const statusIcons = { active: '✅', deleted: '🗑️', reserved: '⏸️', sold: '💰', onhold: '⏳' };
  const statusTexts = { active: 'Activo', deleted: 'Eliminado', reserved: 'Reservado', sold: 'Vendido', onhold: 'En espera' };
  const statusIcon = statusIcons[listing.status] || '❓';
  const statusText = statusTexts[listing.status] || listing.status;
  
  modalElement.innerHTML = `
    <div class="listing-editor-modal" role="dialog" aria-modal="true" aria-label="Editar Anuncio">
      <div class="listing-editor-header">
        <h3>📝 Editar Anuncio</h3>
        <div class="listing-editor-header-info">
          <span class="header-info-item">ID: ${listing.id}</span>
          <span class="header-info-separator">|</span>
          <span class="header-info-item">Wallapop: ${listing.id_wallapop}</span>
          <span class="header-info-separator">|</span>
          <span class="header-info-item header-info-status ${listing.status}">${statusIcon} ${statusText}</span>
        </div>
        <button class="listing-editor-close" title="Cerrar">✕</button>
      </div>
      
      <!-- ✅ v6.3.0: Sistema de pestañas -->
      <div class="listing-editor-tabs">
        <button class="listing-editor-tab active" data-tab="main">📝 Datos principales</button>
        <button class="listing-editor-tab" data-tab="details">📋 Más detalles</button>
      </div>
      
      <div class="listing-editor-body">
        <!-- ✅ Pestaña: Datos principales -->
        <div class="listing-editor-tab-content active" data-tab-content="main">
          <!-- Imágenes -->
          <div class="listing-editor-section">
            <label class="listing-editor-label">📷 Imágenes</label>
            <div class="listing-editor-images" id="editor-images-container">
              ${imagesHtml}
            </div>
            <small class="listing-editor-hint">Arrastra para reordenar • Click en 🗑️ para eliminar • Click en ➕ para añadir</small>
          </div>
          
          <!-- ✅ v6.3.0: Título y Precio en la misma línea -->
          <div class="listing-editor-row">
            <!-- Título -->
            <div class="listing-editor-section listing-editor-section-title">
              <label class="listing-editor-label" for="editor-title">Título *</label>
              <input 
                type="text" 
                id="editor-title" 
                class="listing-editor-input"
                maxlength="${TITLE_MAX_LENGTH}"
                value="${escapeHtml(listing.title)}"
                placeholder="Título del anuncio"
              >
              <div class="listing-editor-counter">
                <span id="editor-title-count">${listing.title.length}</span>/${TITLE_MAX_LENGTH}
              </div>
            </div>
            
            <!-- Precio -->
            <div class="listing-editor-section listing-editor-section-price">
              <label class="listing-editor-label" for="editor-price">Precio (€) *</label>
              <input 
                type="text" 
                id="editor-price" 
                class="listing-editor-input listing-editor-price"
                inputmode="decimal"
                value="${listing.price}"
                placeholder="0.00"
              >
              <small class="listing-editor-hint">Ej: 35.50</small>
            </div>
          </div>
          
          <!-- Descripción -->
          <div class="listing-editor-section">
            <label class="listing-editor-label" for="editor-description">Descripción</label>
            <textarea 
              id="editor-description" 
              class="listing-editor-textarea"
              maxlength="${DESCRIPTION_MAX_LENGTH}"
              rows="6"
              placeholder="Descripción del producto..."
            >${escapeHtml(listing.description)}</textarea>
            <div class="listing-editor-counter">
              <span id="editor-description-count">${listing.description.length}</span>/${DESCRIPTION_MAX_LENGTH}
            </div>
          </div>
        </div>
        
        <!-- ✅ Pestaña: Más detalles -->
        <div class="listing-editor-tab-content" data-tab-content="details">
          <!-- ✅ v6.8.3: Layout de dos columnas -->
          <div class="listing-editor-details-grid">
            <!-- Columna izquierda: Info de solo lectura -->
            <div class="listing-editor-details-left">
              <!-- Categoría (solo lectura) -->
              <div class="listing-editor-section listing-editor-readonly">
                <label class="listing-editor-label">📁 Categoría</label>
                <div class="listing-editor-value">${escapeHtml(listing.category_path)}</div>
              </div>
              
              <!-- Ubicación (solo lectura) -->
              <div class="listing-editor-section listing-editor-readonly">
                <label class="listing-editor-label">📍 Ubicación</label>
                <div class="listing-editor-value">
                  ${escapeHtml(listing.location.city)} ${listing.location.postal_code ? `(${listing.location.postal_code})` : ''}
                </div>
              </div>
              
              <!-- Atributos (solo lectura) -->
              ${attributesHtml ? `
              <div class="listing-editor-section listing-editor-readonly">
                <label class="listing-editor-label">🏷️ Atributos</label>
                <div class="listing-editor-attributes">
                  ${attributesHtml}
                </div>
              </div>
              ` : ''}
            </div>
            
            <!-- Columna derecha: Inventario editable -->
            <div class="listing-editor-details-right">
              <div class="listing-editor-section">
                <label class="listing-editor-label" for="editor-inventory">📦 Inventario / Estantería</label>
                <textarea 
                  id="editor-inventory" 
                  class="listing-editor-textarea listing-editor-inventory"
                  maxlength="500"
                  rows="12"
                  placeholder="Ej: Estantería 8-5&#10;Caja azul&#10;Stock: 3 unidades"
                >${escapeHtml(listing.inventory_note || '')}</textarea>
                <div class="listing-editor-counter">
                  <span id="editor-inventory-count">${(listing.inventory_note || '').length}</span>/500
                </div>
                <small class="listing-editor-hint">Notas internas para ubicar el producto</small>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      <div class="listing-editor-footer">
        <button class="listing-editor-btn listing-editor-btn-cancel">Cancelar</button>
        <button class="listing-editor-btn listing-editor-btn-save">💾 Guardar cambios</button>
      </div>
    </div>
  `;
  
  // Event listeners
  setupModalEvents();
}

/**
 * Renderiza atributos como badges de solo lectura
 * @param {Object} attributes - type_attributes del JSON
 * @returns {string} HTML
 */
function renderAttributes(attributes) {
  if (!attributes || Object.keys(attributes).length === 0) {
    return '';
  }
  
  const items = [];
  
  for (const [key, attr] of Object.entries(attributes)) {
    if (attr && typeof attr === 'object' && attr.text) {
      const title = attr.title || key;
      items.push(`
        <div class="listing-editor-attr">
          <span class="attr-label">${escapeHtml(title)}:</span>
          <span class="attr-value">${escapeHtml(attr.text)}</span>
        </div>
      `);
    }
  }
  
  return items.join('');
}

/**
 * Renderiza galería de imágenes
 * @param {Array} imagesWallapop - Imágenes de Wallapop
 * @param {Array} imagesLocal - Rutas locales
 * @returns {string} HTML
 */
function renderImages(imagesWallapop, imagesLocal) {
  if ((!imagesWallapop || imagesWallapop.length === 0) && (!imagesLocal || imagesLocal.length === 0)) {
    return '<div class="listing-editor-no-images">Sin imágenes</div>';
  }
  
  // Preferir imágenes de Wallapop (URLs directas)
  if (imagesWallapop && imagesWallapop.length > 0) {
    return imagesWallapop.map((img, index) => `
      <div class="listing-editor-image-item">
        <img 
          src="${img.medium || img.small || img.big}" 
          alt="Imagen ${index + 1}"
          class="listing-editor-img"
          data-fallback="true"
        >
        <span class="listing-editor-image-number">${index + 1}</span>
      </div>
    `).join('');
  }
  
  // Si no hay URLs de Wallapop, mostrar placeholder con rutas locales
  return imagesLocal.map((path, index) => `
    <div class="listing-editor-image-item listing-editor-image-local">
      <div class="listing-editor-image-placeholder">📷</div>
      <span class="listing-editor-image-number">${index + 1}</span>
    </div>
  `).join('');
}

// ============================================
// Event handlers
// ============================================

/**
 * Configura eventos de la modal
 */
function setupModalEvents() {
  if (!modalElement) return;
  
  // ✅ v6.3.0: Sistema de pestañas
  const tabs = modalElement.querySelectorAll('.listing-editor-tab');
  const tabContents = modalElement.querySelectorAll('.listing-editor-tab-content');
  
  tabs.forEach(tab => {
    tab.addEventListener('click', () => {
      const targetTab = tab.getAttribute('data-tab');
      
      // Desactivar todas las pestañas y contenidos
      tabs.forEach(t => t.classList.remove('active'));
      tabContents.forEach(c => c.classList.remove('active'));
      
      // Activar la pestaña seleccionada
      tab.classList.add('active');
      const targetContent = modalElement.querySelector(`[data-tab-content="${targetTab}"]`);
      if (targetContent) {
        targetContent.classList.add('active');
      }
    });
  });
  
  // Cerrar con X
  const closeBtn = modalElement.querySelector('.listing-editor-close');
  if (closeBtn) {
    closeBtn.addEventListener('click', () => {
      if (onCloseCallback) onCloseCallback();
    });
  }
  
  // Cancelar
  const cancelBtn = modalElement.querySelector('.listing-editor-btn-cancel');
  if (cancelBtn) {
    cancelBtn.addEventListener('click', () => {
      if (onCloseCallback) onCloseCallback();
    });
  }
  
  // Guardar
  const saveBtn = modalElement.querySelector('.listing-editor-btn-save');
  if (saveBtn) {
    saveBtn.addEventListener('click', handleSaveClick);
  }
  
  // Click en overlay cierra
  modalElement.addEventListener('click', (e) => {
    if (e.target === modalElement) {
      if (onCloseCallback) onCloseCallback();
    }
  });
  
  // Contador de título
  const titleInput = modalElement.querySelector('#editor-title');
  const titleCount = modalElement.querySelector('#editor-title-count');
  if (titleInput && titleCount) {
    titleInput.addEventListener('input', () => {
      titleCount.textContent = titleInput.value.length;
      updateCounterColor(titleCount, titleInput.value.length, TITLE_MAX_LENGTH);
    });
  }
  
  // Contador de descripción
  const descInput = modalElement.querySelector('#editor-description');
  const descCount = modalElement.querySelector('#editor-description-count');
  if (descInput && descCount) {
    descInput.addEventListener('input', () => {
      descCount.textContent = descInput.value.length;
      updateCounterColor(descCount, descInput.value.length, DESCRIPTION_MAX_LENGTH);
    });
  }
  
  // ✅ v6.8.3: Contador de inventario
  const inventoryInput = modalElement.querySelector('#editor-inventory');
  const inventoryCount = modalElement.querySelector('#editor-inventory-count');
  if (inventoryInput && inventoryCount) {
    inventoryInput.addEventListener('input', () => {
      inventoryCount.textContent = inventoryInput.value.length;
      updateCounterColor(inventoryCount, inventoryInput.value.length, 500);
    });
  }
  
  // Validación de precio: solo números, punto y coma
  const priceInput = modalElement.querySelector('#editor-price');
  if (priceInput) {
    priceInput.addEventListener('input', (e) => {
      // Permitir solo números, punto y coma
      let value = e.target.value;
      // Eliminar caracteres no válidos
      value = value.replace(/[^0-9.,]/g, '');
      // Solo permitir un separador decimal
      const parts = value.split(/[.,]/);
      if (parts.length > 2) {
        value = parts[0] + '.' + parts.slice(1).join('');
      }
      e.target.value = value;
    });
    
    // Prevenir caracteres no válidos al escribir
    priceInput.addEventListener('keypress', (e) => {
      const char = e.key;
      const currentValue = e.target.value;
      
      // Permitir: números, punto, coma, backspace, delete, arrows
      if (!/[\d.,]/.test(char) && !['Backspace', 'Delete', 'ArrowLeft', 'ArrowRight', 'Tab'].includes(e.key)) {
        e.preventDefault();
        return;
      }
      
      // No permitir más de un separador decimal
      if ((char === '.' || char === ',') && (currentValue.includes('.') || currentValue.includes(','))) {
        e.preventDefault();
      }
    });
  }
  
  // Inicializar gestor de imágenes
  if (currentListing) {
    const imagesContainer = modalElement.querySelector('#editor-images-container');
    if (imagesContainer) {
      // ✅ v10.3.9: Handler de fallback para imágenes (sin onerror inline por CSP)
      imagesContainer.querySelectorAll('img[data-fallback="true"]').forEach(img => {
        img.addEventListener('error', function() {
          this.src = 'data:image/svg+xml,' + encodeURIComponent('<svg xmlns="http://www.w3.org/2000/svg" width="80" height="80"><rect fill="#333" width="100%" height="100%"/><text fill="#666" x="50%" y="50%" text-anchor="middle" dy=".3em">📷</text></svg>');
          this.removeAttribute('data-fallback'); // Evitar bucle
        }, { once: true });
      });
      
      const apiBase = getApiBase() || '';
      initImageManager(
        imagesContainer,
        currentListing.images_wallapop,
        currentListing.images_local,
        (changes) => {
          imageChanges = changes;
        },
        apiBase
      );
    }
  }
}

/**
 * Actualiza color del contador según proximidad al límite
 * @param {HTMLElement} countElement - Elemento del contador
 * @param {number} current - Caracteres actuales
 * @param {number} max - Máximo permitido
 */
function updateCounterColor(countElement, current, max) {
  const percentage = (current / max) * 100;
  
  if (percentage >= 100) {
    countElement.parentElement.classList.add('counter-limit');
    countElement.parentElement.classList.remove('counter-warning');
  } else if (percentage >= 80) {
    countElement.parentElement.classList.add('counter-warning');
    countElement.parentElement.classList.remove('counter-limit');
  } else {
    countElement.parentElement.classList.remove('counter-warning', 'counter-limit');
  }
}

/**
 * Maneja click en guardar
 */
function handleSaveClick() {
  if (!modalElement || !onSaveCallback) return;
  
  const titleInput = modalElement.querySelector('#editor-title');
  const descInput = modalElement.querySelector('#editor-description');
  const priceInput = modalElement.querySelector('#editor-price');
  const inventoryInput = modalElement.querySelector('#editor-inventory');  // ✅ v6.8.3
  
  if (!titleInput || !descInput || !priceInput) {
    logger.error('[LISTING_EDITOR_MODAL] Inputs no encontrados');
    return;
  }
  
  // Obtener estado actual de imágenes
  const currentImageState = getImageState();
  
  const formData = {
    title: titleInput.value,
    description: descInput.value,
    // Normalizar: reemplazar coma por punto para parseFloat
    price: parseFloat(priceInput.value.replace(',', '.')) || 0,
    // Datos de imágenes
    images: currentImageState,
    // ✅ v6.8.3: Inventario
    inventory_note: inventoryInput ? inventoryInput.value : ''
  };
  
  // Validación visual
  let hasError = false;
  
  if (formData.title.trim().length === 0) {
    titleInput.classList.add('input-error');
    hasError = true;
  } else {
    titleInput.classList.remove('input-error');
  }
  
  // Validar precio: debe ser número válido y positivo
  const priceValue = priceInput.value.replace(',', '.');
  if (!priceValue || isNaN(parseFloat(priceValue)) || formData.price < 0) {
    priceInput.classList.add('input-error');
    hasError = true;
  } else {
    priceInput.classList.remove('input-error');
  }
  
  // Validar que haya al menos una imagen activa
  const activeImagesCount = currentImageState.order.length;
  if (activeImagesCount === 0) {
    alert('El anuncio debe tener al menos una imagen');
    hasError = true;
  }
  
  if (hasError) {
    return;
  }
  
  // Deshabilitar botón mientras guarda
  const saveBtn = modalElement.querySelector('.listing-editor-btn-save');
  if (saveBtn) {
    saveBtn.disabled = true;
    saveBtn.textContent = 'Guardando...';
  }
  
  onSaveCallback(formData);
}

// ============================================
// Utilidades
// ============================================

// ✅ v10.5.27: escapeHtml eliminado - usar import de utils.js
