/**
 * Módulo de edición de anuncios
 * @module listing-editor/index
 * @version 6.1.0
 * 
 * Permite al usuario editar título, descripción y precio de un anuncio.
 * Conserva la estructura JSON original para compatibilidad con publicación.
 */

import { logger } from '../logger.js';
import { toast } from '../../utils.js';
import { openEditorModal, closeEditorModal } from './modal.js';
import { fetchListingData, saveListingData } from './api.js';

// ============================================
// Estado del módulo
// ============================================

let isEditorOpen = false;
let currentListingId = null;

// ============================================
// API pública
// ============================================

/**
 * Abre la modal de edición para un listing específico
 * @param {number} listingId - ID del listing en BD
 */
export async function openEditor(listingId) {
  if (isEditorOpen) {
    return;
  }
  
  
  isEditorOpen = true;
  currentListingId = listingId;
  
  try {
    // Mostrar modal con estado de carga
    openEditorModal({
      loading: true,
      onSave: handleSave,
      onClose: handleClose
    });
    
    // Cargar datos del listing
    const data = await fetchListingData(listingId);
    
    if (!data.ok) {
      throw new Error(data.error || 'Error cargando datos');
    }
    
    // Actualizar modal con datos
    openEditorModal({
      loading: false,
      listing: data.listing,
      onSave: handleSave,
      onClose: handleClose
    });
    
    
  } catch (error) {
    logger.error('[LISTING_EDITOR] Error abriendo editor:', error);
    toast(`Error: ${error.message}`, 'err');
    closeEditor();
  }
}

/**
 * Cierra la modal de edición
 */
export function closeEditor() {
  if (!isEditorOpen) return;
  
  closeEditorModal();
  isEditorOpen = false;
  currentListingId = null;
}

/**
 * Verifica si la modal está abierta
 * @returns {boolean}
 */
export function isOpen() {
  return isEditorOpen;
}

// ============================================
// Handlers internos
// ============================================

/**
 * Maneja el guardado de cambios
 * @param {Object} formData - Datos del formulario
 */
async function handleSave(formData) {
  if (!currentListingId) {
    logger.error('[LISTING_EDITOR] No hay listing seleccionado');
    return;
  }
  
  
  try {
    const result = await saveListingData(currentListingId, formData);
    
    if (result.ok) {
      toast('✅ Anuncio actualizado correctamente', 'ok');
      
      // Cerrar modal
      closeEditor();
      
      // Disparar evento para refrescar tabla
      const refreshEvent = new CustomEvent('listings:refresh', {
        detail: { listingId: currentListingId }
      });
      document.dispatchEvent(refreshEvent);
      
    } else {
      throw new Error(result.error || 'Error guardando');
    }
    
  } catch (error) {
    logger.error('[LISTING_EDITOR] Error guardando:', error);
    toast(`Error: ${error.message}`, 'err');
  }
}

/**
 * Maneja el cierre de la modal (sin guardar)
 */
function handleClose() {
  closeEditor();
}

// ============================================
// Setup de eventos globales
// ============================================

/**
 * Inicializa el módulo de edición
 * Se llama una vez al cargar el panel
 */
export function initListingEditor() {
  
  // Listener para tecla Escape
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && isEditorOpen) {
      closeEditor();
    }
  });
}
