/**
 * Gestor de imágenes para el editor de anuncios
 * @module listing-editor/image-manager
 * @version 6.1.0
 * 
 * Permite reordenar, eliminar y añadir imágenes.
 * Usa drag & drop nativo para reordenamiento.
 */

import { logger } from '../logger.js';

// ============================================
// Estado del módulo
// ============================================

let currentImages = []; // Array de {id, url, path, markedForDelete}
let originalOrder = []; // Para detectar cambios
let containerElement = null;
let onChangeCallback = null;

// ============================================
// API pública
// ============================================

/**
 * Inicializa el gestor de imágenes
 * @param {HTMLElement} container - Contenedor de imágenes
 * @param {Array} imagesWallapop - Imágenes de Wallapop (URLs) - NO SE USA para orden
 * @param {Array} imagesLocal - Rutas locales (ESTE ES EL ORDEN REAL)
 * @param {Function} onChange - Callback cuando cambian las imágenes
 * @param {string} apiBase - Base URL de la API (ej: https://www.mitiklive.com/fa)
 */
export function initImageManager(container, imagesWallapop, imagesLocal, onChange, apiBase = '') {
  containerElement = container;
  onChangeCallback = onChange;
  
  // Construir array de imágenes basándose en el orden de images_local
  currentImages = [];
  
  if (imagesLocal && imagesLocal.length > 0) {
    // El orden correcto viene de images_local
    // Construir URLs desde las rutas locales
    imagesLocal.forEach((path, index) => {
      // Construir URL completa desde el path local
      // path puede ser:
      //   - "products/1/1116/mzn330dn50jn/img_001.jpg" (relativo)
      //   - "/data/products/1/1116/mzn330dn50jn/img_001.jpg" (absoluto)
      const baseUrl = apiBase.replace(/\/$/, '');
      // Si ya empieza con /, concatenar directamente; si no, añadir /data/
      const imageUrl = path.startsWith('/') 
        ? `${baseUrl}${path}` 
        : `${baseUrl}/data/${path}`;
      
      currentImages.push({
        id: `img_${index}`,
        url: imageUrl, // URL construida desde path local
        path: path,
        markedForDelete: false,
        isNew: false
      });
    });
  } else if (imagesWallapop && imagesWallapop.length > 0) {
    // Fallback: solo URLs de Wallapop (sin rutas locales)
    imagesWallapop.forEach((img, index) => {
      currentImages.push({
        id: img.id || `img_${index}`,
        url: img.medium || img.small || img.big,
        path: null,
        markedForDelete: false,
        isNew: false
      });
    });
  }
  
  originalOrder = currentImages.map(img => img.id);
  
  renderImages();
  setupDragAndDrop();
  
}

/**
 * Obtiene el estado actual de las imágenes
 * @returns {Object} Estado de imágenes
 */
export function getImageState() {
  const activeImages = currentImages.filter(img => !img.markedForDelete);
  const deletedImages = currentImages.filter(img => img.markedForDelete && !img.isNew);
  const newImages = currentImages.filter(img => img.isNew && !img.markedForDelete);
  
  // Detectar si hubo cambios
  const currentOrder = activeImages.map(img => img.id);
  const hasOrderChanged = JSON.stringify(currentOrder) !== JSON.stringify(originalOrder.filter(id => 
    !deletedImages.some(img => img.id === id)
  ));
  
  return {
    order: activeImages.map(img => img.path || img.id),
    deleted: deletedImages.map(img => img.path),
    newImages: newImages.map(img => img.file),
    hasChanges: hasOrderChanged || deletedImages.length > 0 || newImages.length > 0
  };
}

/**
 * Limpia el estado del gestor
 */
export function cleanup() {
  currentImages = [];
  originalOrder = [];
  containerElement = null;
  onChangeCallback = null;
}

// ============================================
// Renderizado
// ============================================

/**
 * Renderiza las imágenes en el contenedor
 */
function renderImages() {
  if (!containerElement) return;
  
  const html = currentImages.map((img, index) => {
    const deleteClass = img.markedForDelete ? 'marked-for-delete' : '';
    const newClass = img.isNew ? 'is-new' : '';
    
    return `
      <div class="listing-editor-image-item ${deleteClass} ${newClass}" 
           data-image-id="${img.id}" 
           data-index="${index}"
           draggable="${!img.markedForDelete}">
        ${img.url 
          ? `<img src="${img.url}" alt="Imagen ${index + 1}" class="img-with-fallback">`
          : img.file 
            ? `<img src="${URL.createObjectURL(img.file)}" alt="Nueva imagen">`
            : `<div class="listing-editor-image-placeholder">📷</div>`
        }
        <span class="listing-editor-image-number">${index + 1}</span>
        <button class="listing-editor-image-delete" 
                data-image-id="${img.id}" 
                title="${img.markedForDelete ? 'Restaurar' : 'Eliminar'}">
          ${img.markedForDelete ? '↩️' : '🗑️'}
        </button>
        ${img.markedForDelete ? '<div class="delete-overlay">ELIMINADA</div>' : ''}
      </div>
    `;
  }).join('');
  
  // Añadir botón de agregar imagen
  const addButton = `
    <div class="listing-editor-image-add" title="Añadir imagen">
      <label for="editor-image-upload">
        <span>➕</span>
        <small>Añadir</small>
      </label>
      <input type="file" 
             id="editor-image-upload" 
             accept="image/jpeg,image/png,image/webp" 
             multiple 
             style="display:none">
    </div>
  `;
  
  containerElement.innerHTML = html + addButton;
  
  // ✅ v10.3.9: Handler de fallback para imágenes (sin onerror inline por CSP)
  containerElement.querySelectorAll('img.img-with-fallback').forEach(img => {
    img.addEventListener('error', function() {
      this.style.display = 'none';
    }, { once: true });
  });
  
  // Setup eventos de eliminar
  setupDeleteButtons();
  
  // Setup evento de añadir
  setupAddImageButton();
}

/**
 * Configura botones de eliminar
 */
function setupDeleteButtons() {
  if (!containerElement) return;
  
  const deleteButtons = containerElement.querySelectorAll('.listing-editor-image-delete');
  deleteButtons.forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      const imageId = btn.dataset.imageId;
      toggleDeleteImage(imageId);
    });
  });
}

/**
 * Configura botón de añadir imagen
 */
function setupAddImageButton() {
  if (!containerElement) return;
  
  const fileInput = containerElement.querySelector('#editor-image-upload');
  if (fileInput) {
    fileInput.addEventListener('change', handleFileUpload);
  }
}

/**
 * Marca/desmarca imagen para eliminar
 * @param {string} imageId - ID de la imagen
 */
function toggleDeleteImage(imageId) {
  const index = currentImages.findIndex(img => img.id === imageId);
  if (index === -1) return;
  
  currentImages[index].markedForDelete = !currentImages[index].markedForDelete;
  
  
  renderImages();
  setupDragAndDrop();
  notifyChange();
}

/**
 * Maneja la subida de nuevas imágenes
 * @param {Event} e - Evento change
 */
function handleFileUpload(e) {
  const files = Array.from(e.target.files);
  
  if (files.length === 0) return;
  
  // Validar cantidad total
  const activeCount = currentImages.filter(img => !img.markedForDelete).length;
  if (activeCount + files.length > 10) {
    alert('Máximo 10 imágenes por anuncio');
    return;
  }
  
  // Añadir cada archivo
  files.forEach((file, i) => {
    // Validar tamaño (max 10MB)
    if (file.size > 10 * 1024 * 1024) {
      alert(`Imagen ${file.name} excede 10MB`);
      return;
    }
    
    // Validar tipo
    if (!['image/jpeg', 'image/png', 'image/webp'].includes(file.type)) {
      alert(`Formato no soportado: ${file.name}`);
      return;
    }
    
    currentImages.push({
      id: `new_${Date.now()}_${i}`,
      url: null,
      path: null,
      file: file,
      markedForDelete: false,
      isNew: true
    });
  });
  
  
  renderImages();
  setupDragAndDrop();
  notifyChange();
  
  // Limpiar input
  e.target.value = '';
}

// ============================================
// Drag & Drop
// ============================================

let draggedElement = null;
let draggedIndex = -1;

/**
 * Configura drag & drop para reordenar
 */
function setupDragAndDrop() {
  if (!containerElement) return;
  
  const items = containerElement.querySelectorAll('.listing-editor-image-item[draggable="true"]');
  
  items.forEach(item => {
    item.addEventListener('dragstart', handleDragStart);
    item.addEventListener('dragend', handleDragEnd);
    item.addEventListener('dragover', handleDragOver);
    item.addEventListener('dragenter', handleDragEnter);
    item.addEventListener('dragleave', handleDragLeave);
    item.addEventListener('drop', handleDrop);
  });
}

function handleDragStart(e) {
  draggedElement = e.currentTarget;
  draggedIndex = parseInt(draggedElement.dataset.index, 10);
  
  e.currentTarget.classList.add('dragging');
  e.dataTransfer.effectAllowed = 'move';
  e.dataTransfer.setData('text/plain', draggedIndex);
  
}

function handleDragEnd(e) {
  e.currentTarget.classList.remove('dragging');
  
  // Limpiar todas las clases de hover
  const items = containerElement.querySelectorAll('.listing-editor-image-item');
  items.forEach(item => item.classList.remove('drag-over'));
  
  draggedElement = null;
  draggedIndex = -1;
}

function handleDragOver(e) {
  e.preventDefault();
  e.dataTransfer.dropEffect = 'move';
}

function handleDragEnter(e) {
  e.preventDefault();
  if (e.currentTarget !== draggedElement) {
    e.currentTarget.classList.add('drag-over');
  }
}

function handleDragLeave(e) {
  e.currentTarget.classList.remove('drag-over');
}

function handleDrop(e) {
  e.preventDefault();
  e.currentTarget.classList.remove('drag-over');
  
  const targetIndex = parseInt(e.currentTarget.dataset.index, 10);
  
  if (draggedIndex === -1 || targetIndex === draggedIndex) return;
  
  // Reordenar array
  const [movedImage] = currentImages.splice(draggedIndex, 1);
  currentImages.splice(targetIndex, 0, movedImage);
  
  
  renderImages();
  setupDragAndDrop();
  notifyChange();
}

// ============================================
// Utilidades
// ============================================

/**
 * Notifica cambios al callback
 */
function notifyChange() {
  if (onChangeCallback) {
    onChangeCallback(getImageState());
  }
}
