/**
 * API client para el editor de anuncios
 * @module listing-editor/api
 * @version 6.1.0
 */

import { logger } from '../logger.js';

// ============================================
// Fetch de datos del listing
// ============================================

/**
 * Obtiene los datos completos de un listing para edición
 * @param {number} listingId - ID del listing
 * @returns {Promise<Object>} Datos del listing
 */
export async function fetchListingData(listingId) {
  
  try {
    const response = await chrome.runtime.sendMessage({
      type: 'LISTING.EDITOR.GET',
      payload: { listing_id: listingId }
    });
    
    if (!response) {
      throw new Error('Sin respuesta del service worker');
    }
    
    if (response.error) {
      throw new Error(response.error);
    }
    
    return response;
    
  } catch (error) {
    logger.error('[LISTING_EDITOR_API] Error fetching:', error);
    throw error;
  }
}

/**
 * Guarda los cambios de un listing
 * @param {number} listingId - ID del listing
 * @param {Object} data - Datos a guardar
 * @param {string} data.title - Título (máx 50 chars)
 * @param {string} data.description - Descripción (máx 640 chars)
 * @param {number} data.price - Precio en EUR
 * @param {Object} data.images - Estado de imágenes
 * @returns {Promise<Object>} Resultado de la operación
 */
export async function saveListingData(listingId, data) {
  
  // Validaciones locales antes de enviar
  if (!data.title || data.title.trim().length === 0) {
    throw new Error('El título no puede estar vacío');
  }
  
  if (data.title.length > 50) {
    throw new Error('El título excede 50 caracteres');
  }
  
  if (data.description.length > 640) {
    throw new Error('La descripción excede 640 caracteres');
  }
  
  if (isNaN(data.price) || data.price < 0) {
    throw new Error('Precio inválido');
  }
  
  try {
    // Si hay imágenes nuevas, primero subirlas
    let uploadedPaths = [];
    if (data.images && data.images.newImages && data.images.newImages.length > 0) {
      
      for (const file of data.images.newImages) {
        const uploadResult = await uploadImage(listingId, file);
        if (uploadResult.ok) {
          uploadedPaths.push(uploadResult.path);
        } else {
          throw new Error(`Error subiendo imagen: ${uploadResult.error}`);
        }
      }
    }
    
    // Construir orden final de imágenes
    // Reemplazar IDs temporales (new_*) con las rutas reales subidas
    let finalOrder = [];
    let newImageIndex = 0;
    
    if (data.images && data.images.order) {
      for (const item of data.images.order) {
        if (item.startsWith('new_')) {
          // Es una imagen nueva, usar la ruta subida
          if (newImageIndex < uploadedPaths.length) {
            finalOrder.push(uploadedPaths[newImageIndex]);
            newImageIndex++;
          }
        } else {
          // Es una imagen existente, mantener el path
          finalOrder.push(item);
        }
      }
    }
    
    
    const response = await chrome.runtime.sendMessage({
      type: 'LISTING.EDITOR.SAVE',
      payload: {
        listing_id: listingId,
        title: data.title.trim(),
        description: data.description.trim(),
        price: parseFloat(data.price),
        images_order: finalOrder,
        images_to_delete: data.images ? data.images.deleted : [],
        inventory_note: data.inventory_note || ''  // ✅ v6.8.3
      }
    });
    
    if (!response) {
      throw new Error('Sin respuesta del service worker');
    }
    
    if (response.error) {
      throw new Error(response.error);
    }
    
    return response;
    
  } catch (error) {
    logger.error('[LISTING_EDITOR_API] Error saving:', error);
    throw error;
  }
}

/**
 * Sube una imagen nueva al servidor
 * @param {number} listingId - ID del listing
 * @param {File} file - Archivo de imagen
 * @returns {Promise<Object>} Resultado con path de la imagen
 */
async function uploadImage(listingId, file) {
  
  try {
    // Convertir archivo a base64
    const base64 = await fileToBase64(file);
    
    const response = await chrome.runtime.sendMessage({
      type: 'LISTING.EDITOR.UPLOAD_IMAGE',
      payload: {
        listing_id: listingId,
        filename: file.name,
        content_type: file.type,
        base64_data: base64
      }
    });
    
    if (!response) {
      throw new Error('Sin respuesta del service worker');
    }
    
    if (response.error) {
      throw new Error(response.error);
    }
    
    return response;
    
  } catch (error) {
    logger.error('[LISTING_EDITOR_API] Error uploading image:', error);
    throw error;
  }
}

/**
 * Convierte un File a base64
 * @param {File} file - Archivo
 * @returns {Promise<string>} Base64 string (sin prefijo data:)
 */
function fileToBase64(file) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = () => {
      // Quitar el prefijo "data:image/jpeg;base64,"
      const base64 = reader.result.split(',')[1];
      resolve(base64);
    };
    reader.onerror = reject;
    reader.readAsDataURL(file);
  });
}
