/*  Cabecera obligatoria: NO BORRAR NI MODIFICAR este bloque inicial en ningún fichero.
    Archivo: scripts/panel/listeners.js – Rol: Escucha de mensajes del Service Worker
    Descripcion de todo el proyecto:
         MitikLive Wallapop (MV3): panel lateral en Wallapop para gestionar anuncios conectado a FastAPI
            (login JWT, export/import, backups/renovaciones).
    Aviso: NO BORRAR los comentarios descriptivos situados encima de cada función, solo cambiarlos si se modifica la funcion.
    En Rol (linea mas arriba): si está vacía o el fichero se modifica o reestructura, modificar esa linea de rol
*/

import { toast } from '../utils.js';
import { logger } from './logger.js';
import { aliasFromProfileUrl } from './detect.js';

import { stRemove } from './storage.js';
import { updateExtensionVersion } from './version-tracker.js';
import { 
  AUTH, LAST_GOOD, currentRunId,
  set_AUTH, set_LAST_GOOD 
} from './state.js';
import { 
  setAuthUI,
  paintConnected, paintNoLogin, setRunControls, // ✅ v5.1.0: hideRunUI eliminado 
  showValidationBadges, updateBackupButtonUI, paintApiStatus,
  updateProgressBar // ✅ v6.3.0: Barra de progreso unificada
} from './ui.js';
import { detectLoggedWalla } from './detect.js';
import { checkLastRunAndWarn } from './polling.js';
import { promptResumeIfPending, promptPublishResumeIfPending } from './resume.js';
// ✅ v6.3.0: pollRun eliminado - backup usa WebSocket en tiempo real
import { loadAccounts } from './auth.js';

let CS_WATCHDOG = null;
let NOLOGIN_HOLD = null;
const NOLOGIN_HOLD_MS = 5000;

/** Cancela el hold de degradación */
function cancelNoLoginHold() {
  if (NOLOGIN_HOLD) {
    clearTimeout(NOLOGIN_HOLD);
    NOLOGIN_HOLD = null;
  }
}

/** Programa degradación diferida */
function scheduleNoLogin(_reason = '') {
  cancelNoLoginHold();
  NOLOGIN_HOLD = setTimeout(() => {
    paintNoLogin();
    set_LAST_GOOD({ alias: '', avatar: '' });
    NOLOGIN_HOLD = null;
  }, NOLOGIN_HOLD_MS);
}

/** Extrae alias de URL de perfil */
/* ========================================
   Listener principal de mensajes del SW
   ======================================== */

let _inited = false;

export function init() {
  if (_inited) return;
  _inited = true;

  chrome.runtime.onMessage.addListener((msg) => {
    // === Cambio de estado de sesión ===
    if (msg?.type === 'SESSION.STATE') {
      const newState = msg.state;
      set_AUTH({ ...AUTH, state: newState });
      setAuthUI(newState === 'AUTHENTICATED', AUTH.user?.email || '');
      
      if (newState === 'AUTHENTICATED') {
        // ❌ VALIDACIÓN DE VERSIÓN DESHABILITADA (causaba pool exhaustion)
        // Solo verificar mantenimiento
        import('./maintenance.js').then(async ({ checkMaintenance }) => {
          const inMaintenance = await checkMaintenance();
          if (!inMaintenance) {
            // Solo si NO mantenimiento, cargar datos
            promptResumeIfPending().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
            promptPublishResumeIfPending().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
            checkLastRunAndWarn().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
          }
        }).catch(err => {
          logger.error('[LISTENERS] Error verificando mantenimiento:', err);
        });
      } else {
        // Al salir, ocultamos el botón de reanudar publicación
      }
      return;
    }

    // === Usuario logueado en MitikLive ===
    if (msg?.type === 'SESSION.LOGGED_IN') {
      set_AUTH({ ...AUTH, user: msg.user || null });
      // ✅ ml.run_id ya no se usa en localStorage
      setRunControls(false);
      // ✅ v5.1.0: hideRunUI eliminado (era no-op)
      setAuthUI(true, msg.user?.email || '');
      
      // ❌ VALIDACIÓN DE VERSIÓN DESHABILITADA (causaba pool exhaustion)
      // Solo verificar mantenimiento
      import('./maintenance.js').then(async ({ checkMaintenance }) => {
        const inMaintenance = await checkMaintenance();
        if (!inMaintenance) {
          // Solo si NO mantenimiento, cargar datos
          checkLastRunAndWarn().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
          loadAccounts().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
          detectLoggedWalla().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
          promptResumeIfPending().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
          promptPublishResumeIfPending().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
        }
      }).catch(err => {
        logger.error('[LISTENERS] Error verificando mantenimiento:', err);
      });
      
      return;
    }

    // === Usuario deslogueado de MitikLive ===
    if (msg?.type === 'SESSION.LOGGED_OUT') {
      set_AUTH({ ...AUTH, user: null });
      setAuthUI(false, '');
      detectLoggedWalla().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
      
      // ✅ v6.12.0: Limpiar tab de progreso al cerrar sesión
      if (window.ProgressTab?.clear) {
        window.ProgressTab.clear();
      }
      
      return;
    }

    // ✅ v10.5.125: Cuenta suspendida o baneada
    if (msg?.type === 'ACCOUNT.SUSPENDED') {
      import('../utils.js').then(({ showModal }) => {
        showModal({
          title: '⛔ Cuenta suspendida',
          html: `Tu cuenta ha sido suspendida.<br><br><strong>Motivo:</strong> ${msg.reason || 'Actividad sospechosa detectada'}<br><br>Si crees que es un error, contacta con soporte.`,
          buttons: [{ text: 'Entendido', value: 'ok', primary: true }]
        }).then(() => {
          // Cerrar sesión tras aceptar
          chrome.runtime.sendMessage({ type: 'AUTH.LOGOUT' });
        });
      });
      return;
    }

    // ✅ v10.5.142: Actualizar desplegable de cuentas cuando cambia pending
    if (msg?.type === 'ACCOUNT.PENDING_UPDATED') {
      const accountId = msg.account_id;
      const pendingCount = msg.data?.pending_publish_count || 0;
      
      // Actualizar el texto de la opción en el desplegable
      const selAccount = document.getElementById('sel-account');
      if (selAccount) {
        const options = Array.from(selAccount.options);
        const option = options.find(opt => 
          opt.value == accountId || opt.dataset.accountId == accountId
        );
        
        if (option) {
          const alias = option.dataset.alias || option.textContent.split(' ')[0];
          const activeCount = option.dataset.activeCount || '';
          
          // Actualizar texto sin parpadeo
          if (pendingCount > 0) {
            option.textContent = `${alias} ⏳${pendingCount}`;
          } else if (activeCount) {
            option.textContent = `${alias} (${activeCount})`;
          } else {
            option.textContent = alias;
          }
          
          option.dataset.pendingPublishCount = pendingCount;
        }
      }
      return;
    }

    // === Actualización de datos de usuario ===
    if (msg?.type === 'SESSION.UPDATED') {
      if (msg?.user?.email) {
        set_AUTH({ ...AUTH, user: msg.user });
        // ✅ v10.5.144: Usar span interno para el texto del email
        const pillUserText = document.querySelector('.ml-user-email-text');
        const userEmail = document.getElementById('user-email');
        if (pillUserText) pillUserText.textContent = msg.user.email;
        if (userEmail) userEmail.textContent = msg.user.email;
        
        // ✅ v4.65.2: Actualizar créditos cuando se refresca el token
        if (window.CreditsManager?.refresh) {
          window.CreditsManager.refresh(true).catch(err => {
            logger.error('[LISTENERS] Error actualizando créditos:', err);
          });
        }
      }
      return;
    }

    // === Estado canónico desde el SW (backup/publish por cuenta) ===
    if (msg?.type === 'ACTIVITY.UPDATE' && msg.activity) {
      const { publish } = msg.activity || {};
      
      // Publicación: si hay pending/error y NO running => mostrar "Reanudar"
      if (publish?.state === 'pending') {
        //   true, 
        //   publish.pending, 
        //   publish.errors, 
        //   publish.run_id || null
        // );
      } else if (publish?.state === 'running') {
      } else if (['ok', 'error', 'none'].includes(publish?.state)) {
      }
      return;
    }

    // === Invalidación de caché => refrescar botones ===
    if (msg?.type === 'CACHE.INVALIDATED') {
      promptPublishResumeIfPending().catch(err => { logger.error(`[${filepath.split('/').pop()}] Error:`, err); });
      return;
    }

    // === Resultado de verificación de publicación pendiente ===
    if (msg?.type === 'PUBLISH.PENDING.RESULT') {
      const pending = Number(msg.pending || 0);
      const errors = Number(msg.errors || 0);
      const resumable = !!(msg.resumable || pending > 0 || errors > 0);
      const run_id = msg.run_id ?? null;
      
      return;
    }

    // === Validation warnings ===
    if (msg?.type === 'VALIDATION_WARNINGS') {
      showValidationBadges(msg.warnings);
      return;
    }

    // === Cambio de sesión en Wallapop (desde content script) ===
    if (msg?.type === 'WALLA.SESSION_CHANGED') {
      const { logged, alias, avatar, profileUrl } = msg || {};
      
      if (!logged) {
        // Degradación diferida para evitar parpadeo en F5/navegación interna
        scheduleNoLogin('push');
      } else {
        cancelNoLoginHold();
        const lastGood = LAST_GOOD;
        const nextAlias = (alias && alias.trim()) || 
                         (profileUrl ? (aliasFromProfileUrl(profileUrl) || '') : '') || 
                         lastGood.alias || 
                         '';
        const nextAvatar = avatar || lastGood.avatar || '';
        
        set_LAST_GOOD({ alias: nextAlias, avatar: nextAvatar });
        
        if (CS_WATCHDOG) {
          clearTimeout(CS_WATCHDOG);
          CS_WATCHDOG = null;
        }
        
        paintConnected(nextAlias, nextAvatar);
      }
      return;
    }

    // === Run iniciado ===
    if (msg?.type === 'RUN.STARTED' && msg.run_id) {
      // ✅ v6.3.0: Mostrar barra de progreso - WebSocket actualizará en tiempo real
      if (currentRunId === msg.run_id) return;
      
      updateProgressBar({
        operation: 'backup',
        current: 0,
        total: msg.total || 0,
        active: true
      });
      return;
    }
    
    // === 🆕 Crédito descontado (actualización en tiempo real) ===
    if (msg?.type === 'CREDIT_DEDUCTED') {
      
      // Actualizar CreditsManager si está disponible
      if (window.CreditsManager) {
        try {
          // Forzar actualización desde el backend para tener datos precisos
          window.CreditsManager.fetchCredits(true);
        } catch (err) {
          logger.error('[Listeners] Error actualizando créditos:', err);
        }
      }
      return;
    }
    
    // === 🔴 FORCE_LOGOUT: Sesión revocada en otro dispositivo ===
    if (msg?.type === 'FORCE_LOGOUT') {
      logger.error('[Listeners] 🔴 FORCE_LOGOUT recibido:', msg.reason);
      
      const message = msg.message || 'Has iniciado sesión en otro dispositivo';
      
      // Mostrar alerta al usuario
      toast(`🔴 ${message}. Redirigiendo a login...`, 'error', 5000);
      
      // Limpiar estado local
      paintNoLogin();
      set_AUTH(null);
      set_LAST_GOOD({ alias: '', avatar: '' });
      
      // Esperar 2 segundos y recargar para ir a login
      setTimeout(() => {
        window.location.reload();
      }, 2000);
      
      return;
    }
    
    // === 🔒 MAINTENANCE_MODE_CHANGED: Modo mantenimiento activado/desactivado ===
    if (msg?.type === 'MAINTENANCE_MODE_CHANGED') {
      
      // Usar nueva función inteligente de mantenimiento
      import('./maintenance.js').then(({ handleMaintenanceMode }) => {
        handleMaintenanceMode(msg.enabled, msg.message || 'Sistema en mantenimiento');
      }).catch(err => {
        logger.error('[Listeners] Error importando maintenance.js:', err);
      });
      
      return;
    }
    
    // === 🟢 SERVER_ALIVE: Heartbeat recibido del servidor ===
    if (msg?.type === 'SERVER_ALIVE') {
      // Actualizar DOT a verde
      paintApiStatus('online');
      return;
    }
    
    // === 🔴 SERVER_DEAD: Servidor sin respuesta ===
    if (msg?.type === 'SERVER_DEAD') {
      
      // Actualizar DOT a rojo
      paintApiStatus('offline');
      
      // Mostrar toast de advertencia
      toast('⚠️ Servidor sin respuesta. Reconectando...', 'warning', 3000);
      return;
    }
    
    // ✅ v10.5.86: HANDLERS DE backup_complete, publish_complete, publish_progress
    // ELIMINADOS - Ahora manejados ÚNICAMENTE en panel.js para evitar:
    // - loadListingsTable() llamado 3 veces
    // - Duplicación de lógica
    // Ver: ANALISIS-BARRAS-PROGRESO.md
    
  });
}